package com.bcxin.tenant.open.document.domains.documents;

import com.bcxin.tenant.open.infrastructures.constants.BusinessConstants;
import com.bcxin.tenant.open.infrastructures.entities.EntityAbstract;
import com.bcxin.tenant.open.infrastructures.utils.SetUtil;
import com.redis.om.spring.annotations.*;
import com.redis.om.spring.repository.query.SearchLanguage;
import lombok.Data;
import org.springframework.data.annotation.Id;
import org.springframework.data.geo.Point;
import org.springframework.util.StringUtils;

import java.sql.Timestamp;
import java.time.Instant;
import java.util.*;
import java.util.stream.Collectors;

/**
 *  驻勤点信息
 */
@Document(language = SearchLanguage.CHINESE)
@Data
public class RdSecurityStationDocument extends EntityAbstract {
    @Id
    @Indexed
    private String id;

    /**
     * 不适合使用@Searchable 全文索引的方式, 如果是文本标题或者描述可能才比较适用用全文索引
     */
    @Indexed(weight = 22.0)
    private String name;

    /**
     * searchStream = searchStream.filter("*".concat(keyword).concat("*")); 采用这种方式实现模糊搜索
     */
    @Searchable(sortable = false, nostem = true, weight = 20.0)
    private String keyword;

    @Indexed
    @TagIndexed
    private Set<String> stationTypes;

    @Indexed
    private String organizationId;

    @Indexed
    private String companyName;

    private String principalId;

    private String principalName;

    private String principalContact;

    private String serviceScope;

    private String address;

    @Indexed
    @TagIndexed
    private Set<String> followedDeviceNos;

    @Indexed(sortable = true)
    private Integer countOfSecurityMan;

    private Integer countOfCommunityUser;

    @Indexed
    @GeoIndexed
    private Point lonLat;

    @Indexed
    private String superviseDepartId;

    private String superviseDepartName;

    private String siteState;

    /**
     * 生效驻勤点; 通过组合计算出来的结果
     */
    @Indexed
    private boolean isActivated;

    private Timestamp beginDate;

    private Timestamp endDate;

    /**
     * 资源类型: 保安,人口
     */
    @Indexed
    private Set<String> resourceTypes;

    private String industry;

    private String institutional;

    /**
     * 权限范围
     * 监管属性, 本企业, 内保单位, 内保主管单位
     */
    @Indexed
    private Set<String> scopePermissions;

    /**
     * 所属内保单位
     */
    private String proprietorCompanyName;

    private String proprietorCompanyId;

    private Timestamp lastUpdatedTime;

    /**
     * 项目Id
     */
    private String projectId;

    /**
     * 项目名称
     */
    private String projectName;

    /**
     * 执行范围(用于判断签到)
     */
    private Double performRange;

    private Boolean hasVideo;

    public void updateWord(String... text) {
        if (text == null || text.length == 0) {
            this.setKeyword("");
        }
        String content =
                Arrays.stream(text).filter(ii -> StringUtils.hasLength(ii)).collect(Collectors.joining(" "));
        this.setKeyword(content);
    }

    /**
     * 相关的内保单位
     *
     * @param companyId
     * @param companyName
     */
    public void assignProprietor(String companyId, String companyName) {
        this.setProprietorCompanyId(companyId);
        this.setProprietorCompanyName(companyName);
    }

    public void changeState(Date beginDate, Date endDate, String siteState) {
        if (beginDate != null) {
            this.setBeginDate(Timestamp.from(Instant.ofEpochMilli(beginDate.getTime())));
        } else {
            this.setBeginDate(null);
        }

        if (endDate != null) {
            this.setEndDate(Timestamp.from(Instant.ofEpochMilli(endDate.getTime())));
        } else {
            this.setEndDate(null);
        }

        this.setSiteState(siteState);
        this.setActivated(false);

        if (this.getBeginDate() != null && this.getEndDate() != null && this.getSiteState() == "已完善") {
            Timestamp now = Timestamp.from(Instant.now());
            if (this.getBeginDate().before(now) && this.getEndDate().after(now)) {
                this.setActivated(true);
            }
        }
    }

    public boolean addFollowedDeviceNo(String deviceNo) {
        this.setFollowedDeviceNos(SetUtil.merge(this.getFollowedDeviceNos(), deviceNo));

        return true;
    }

    public boolean removeFollowedDeviceNo(String deviceNo) {
        this.setFollowedDeviceNos(SetUtil.remove(this.getFollowedDeviceNos(), deviceNo));

        return true;
    }

    public void updateCountOfCommunityUser(int countOfCommunityUser) {
        this.setCountOfCommunityUser(countOfCommunityUser);
    }

    public void addResourceTypes(String... resourceTypes) {
        this.setResourceTypes(SetUtil.merge(this.getResourceTypes(), resourceTypes));
    }

    public void addScopePermissions(String... scopeIds) {
        this.setScopePermissions(SetUtil.merge(this.getScopePermissions(), scopeIds));
    }

    public void removeBusinessScopePermissions(String... scopeIds) {
        this.setScopePermissions(SetUtil.remove(this.getScopePermissions(), scopeIds));
        this.setScopePermissions(SetUtil.merge(this.getScopePermissions(), this.getOrganizationId(), this.getSuperviseDepartId()));
    }

    public void rest() {
        this.setSuperviseDepartId(BusinessConstants.DefaultEmptyValue);
        this.setScopePermissions(new HashSet<>());
        this.setResourceTypes(new HashSet<>());
        this.setProprietorCompanyName(null);
        this.setProprietorCompanyId(null);
    }
}
