package com.bcxin.tenant.open.rest.apis.impls;

import com.alibaba.nacos.shaded.com.google.common.collect.Lists;
import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.enums.TRTCActionType;
import com.bcxin.tenant.open.jdks.PoliceIncidentsRpcProvider;
import com.bcxin.tenant.open.jdks.RoomRpcProvider;
import com.bcxin.tenant.open.jdks.SocketPublishRpcProvider;
import com.bcxin.tenant.open.jdks.requests.*;
import com.bcxin.tenant.open.jdks.requests.enums.BroadcastMessageType;
import com.bcxin.tenant.open.rest.apis.components.WebSocketServer;
import com.bcxin.tenant.open.rest.apis.dtos.*;
import com.bcxin.tenant.open.rest.apis.utils.SocketMessageUtil;
import org.apache.dubbo.config.annotation.DubboService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;
import java.util.stream.Collectors;

@DubboService
public class SocketPublishRpcProviderImpl extends AbstractSocketPublishRpcProvider {
    private static final Logger logger = LoggerFactory.getLogger(SocketPublishRpcProvider.class);

    private final PoliceIncidentsRpcProvider policeIncidentsRpcProvider;

    private final RoomRpcProvider roomRpcProvider;


    public SocketPublishRpcProviderImpl(WebSocketServer webSocketServer,
                                        PoliceIncidentsRpcProvider policeIncidentsRpcProvider,
                                        JsonProvider jsonProvider, RoomRpcProvider roomRpcProvider) {
        super(webSocketServer, jsonProvider);
        this.policeIncidentsRpcProvider = policeIncidentsRpcProvider;
        this.roomRpcProvider = roomRpcProvider;
    }

    @Override
    public Collection<SocketSessionDTO> getDestinationSocketSessions(BroadcastMessageRequest request) {
        BroadcastMessageType messageType = request.getMessageType();
        Collection<SocketSessionDTO> socketSessions = Collections.emptyList();
        switch (messageType) {
            case PoliceIncidents -> socketSessions = getPoliceIncidentsSocketSessionDTOS(request);
            case TRTC -> socketSessions = getTRTCSocketSessionDTOS(request);
        }
        return socketSessions;
    }

    @Override
    public SocketSessionMessageResponse getSocketSessionMessageResponse(SocketSessionDTO st, BroadcastMessageRequest request) {
        SocketSessionMessageResponse sr = null;
        BroadcastMessageType messageType = request.getMessageType();
        switch (messageType) {
            case PoliceIncidents -> {
                SocketPoliceIncidentMessage message = getMatchedContents(request, st.getSuperviseDepartIds());
                sr = SocketSessionMessageResponse.create(request.getMessageType(), message);
            }
            case TRTC -> {
                SocketTrtcMessage message = getMatchedTrtcMessageContents(st, request);
                sr = SocketSessionMessageResponse.create(request.getMessageType(), message);

            }
        }

        return sr;
    }

    private SocketTrtcMessage getMatchedTrtcMessageContents(SocketSessionDTO st, BroadcastMessageRequest request) {
        Collection<BroadcastTrtcMessageContentItem> messageContents = Collections.EMPTY_LIST;
        BroadcastTrtcMessageContent content = (BroadcastTrtcMessageContent) request.getContent();
        TRTCActionType actionType = content.getActionType();
        switch (actionType) {
            case STOPWORD, OPENWORD, CAMON, CAMOFF -> {
                if (content.getContentItems().isEmpty()) {
                    messageContents = Lists.newArrayList(BroadcastTrtcMessageContentItem.create(content.getRoomId(), content.getActionType(), st.getTencentUserId()));
                } else {
                    messageContents = content.getContentItems().stream().filter(ix -> ix.getTencentUserId().equals(st.getTencentUserId()))
                            .collect(Collectors.toList());
                }
            }
            case HANDUP, CHEESE -> messageContents = content.getContentItems();
            case WORDALLOWED, WORDDENY, CAMALLOWED, CAMDENY, REMOVEUSER ->
                    messageContents = content.getContentItems().stream().filter(ix -> ix.getTencentUserId().equals(st.getTencentUserId()))
                            .collect(Collectors.toList());
        }
        SocketTrtcMessage message = null;
        if (messageContents.isEmpty()) {
            message = SocketTrtcMessage.create(Collections.emptyList());
        } else {
            message = SocketTrtcMessage.create(messageContents.stream().map(SocketMessageUtil::trtcRequestMessageToResponse).collect(Collectors.toList()));
        }
        return message;
    }

    public SocketPoliceIncidentMessage getMatchedContents(BroadcastMessageRequest request, String[] superviseDepartIds) {
        if (superviseDepartIds == null || superviseDepartIds.length == 0) {
            return SocketPoliceIncidentMessage.createEmptyMessage();
        }

        if (request.getContent().getContentItems() == null || request.getContent().getContentItems().isEmpty()) {
            return SocketPoliceIncidentMessage.createEmptyMessage();
        }

        Collection<String> allowedSuperviseDepartIds = Arrays.stream(superviseDepartIds).collect(Collectors.toList());
        Collection<SocketPoliceIncidentMessageItem> messageItems =
                ((BroadcastPoliceIndicentMessageContent) request.getContent()).getContentItems().stream()
                .filter(ix -> allowedSuperviseDepartIds.contains(ix.getSuperDepartId()))
                .map(SocketMessageUtil::policeIncidentRequestMessageToResponse)
                .collect(Collectors.toList());

        return SocketPoliceIncidentMessage.create(messageItems);
    }

    private Collection<SocketSessionDTO> getTRTCSocketSessionDTOS(BroadcastMessageRequest request) {

        BroadcastTrtcMessageContent content = (BroadcastTrtcMessageContent) request.getContent();
        Collection<String> specialTencentIds = content.getSendToSessionSpecialIds();
        Collection<SocketSessionDTO> socketSessionDTOS =
                this.getWebSocketServer()
                        .getMatchSocketSessions(BroadcastMessageType.TRTC, specialTencentIds);

        Collection<String> invalidUsers = specialTencentIds.stream()
                .filter(ix -> !socketSessionDTOS.stream().anyMatch(ii -> ii.getTencentUserId().equals(ix)))
                .collect(Collectors.toList());
        if (!invalidUsers.isEmpty()) {
            logger.warn("没有匹配的socket session：tencentUserId:{}",
                    invalidUsers.stream().collect(Collectors.joining(",")));
        }
        return socketSessionDTOS;
    }

    private Collection<SocketSessionDTO> getPoliceIncidentsSocketSessionDTOS(BroadcastMessageRequest request) {
        Collection<String> specialDepartIds =
                ((BroadcastPoliceIndicentMessageContent) request.getContent()).getContentItems().stream()
                        .map(ix -> ix.getSuperDepartId())
                        .collect(Collectors.toList());

        Collection<SocketSessionDTO> socketSessions =
                this.getWebSocketServer()
                        .getMatchSocketSessions(BroadcastMessageType.PoliceIncidents, specialDepartIds);
        return socketSessions;
    }
}
