package com.bcxin.tenant.open.rest.apis.controllers;

import com.bcxin.tenant.open.jdks.RdSecurityStationRailWriterRpcProvider;
import com.bcxin.tenant.open.jdks.requests.RdSecurityStationRailRequest;
import com.bcxin.tenant.open.jdks.requests.RdSecurityStationRailSearchRequest;
import com.bcxin.tenant.open.jdks.responses.RdSecurityStationRailSearchResponse;
import com.bcxin.tenant.open.jdks.responses.SecurityStationRailDetailResponse;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.Collection;

@Tag(name = "SecurityStationRailController", description = "电子围栏")
@RestController
@RequestMapping("/security-station-rails")
public class SecurityStationRailController extends ControllerAbstract {

    private final RdSecurityStationRailWriterRpcProvider securityStationRailWriterRpcProvider;

    public SecurityStationRailController(RdSecurityStationRailWriterRpcProvider securityStationRailWriterRpcProvider) {
        this.securityStationRailWriterRpcProvider = securityStationRailWriterRpcProvider;
    }

    @Operation(
            summary = "创建电子围栏", description = "成功, 则返回电子围栏id",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok."),
                    @ApiResponse(responseCode = "404", description = "找不到驻勤信息.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "accessToken/dispatchToken",
                            description = "来自V5的AccessToken或者认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping
    public ResponseEntity<String> post(@RequestBody RdSecurityStationRailRequest request) {
        long id =
                this.securityStationRailWriterRpcProvider.create(
                        request
                );

        return this.ok(String.valueOf(id));
    }

    @Operation(
            summary = "修改电子围栏", description = "成功, 则返回200",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok."),
                    @ApiResponse(responseCode = "404", description = "找不到该电子围栏.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.PATH,required = true,name = "id",description = "电子围栏Id"),
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "accessToken/dispatchToken",
                            description = "来自V5的AccessToken或者来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PutMapping("/{id}")
    public ResponseEntity put(@PathVariable long id, @RequestBody RdSecurityStationRailRequest request) {
        this.securityStationRailWriterRpcProvider.update(id, request);

        return this.ok();
    }

    @Operation(
            summary = "删除电子围栏", description = "成功, 则返回200",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok."),
                    @ApiResponse(responseCode = "404", description = "找不到该电子围栏.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.PATH,required = true,name = "id",description = "电子围栏Id"),
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "accessToken/dispatchToken",
                            description = "来自V5的AccessToken或者来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @DeleteMapping("/{id}")
    public ResponseEntity delete(@PathVariable long id) {
        this.securityStationRailWriterRpcProvider.delete(id);

        return this.ok();
    }

    @Operation(
            summary = "搜索驻勤点底下的电子围栏-不分页", description = "成功, 则返回列表; 如果驻勤点无效, 则返回空",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "accessToken/dispatchToken",
                            description = "来自V5的AccessToken或者来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/search")
    public ResponseEntity<Collection<RdSecurityStationRailSearchResponse>> search(@RequestBody RdSecurityStationRailSearchRequest request) {
        Collection<RdSecurityStationRailSearchResponse> responses =
                this.securityStationRailWriterRpcProvider.search(request);

        return this.ok(responses);
    }
    @Operation(
            summary = "获取电子围栏详情信息", description = "获取电子围栏详情信息",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回电子围栏信息."),
                    @ApiResponse(responseCode = "404", description = "找不到电子围栏信息.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "accessToken/dispatchToken",
                            description = "来自V5的AccessToken或者来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken"),
                    @Parameter(in = ParameterIn.DEFAULT, required = true, name = "id", description = "电子围栏信息的id")
            }
    )
    @GetMapping("/{id}")
    public ResponseEntity<SecurityStationRailDetailResponse> get(@PathVariable Long id){

        SecurityStationRailDetailResponse data =
                this.securityStationRailWriterRpcProvider.get(id);
        if (data == null) {
            return this.notFound();
        }

        return this.ok(data);
    }
}
