package com.bcxin.tenant.open.rest.apis.controllers;

import com.bcxin.tenant.open.infrastructures.TenantContext;
import com.bcxin.tenant.open.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.open.infrastructures.exceptions.ArgumentTenantException;
import com.bcxin.tenant.open.infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.tenant.open.jdks.RollCallWriterRpcProvider;
import com.bcxin.tenant.open.jdks.requests.CreateRollCallRequest;
import com.bcxin.tenant.open.jdks.requests.AppReplyRollCallEmployeeStatusRequest;
import com.bcxin.tenant.open.jdks.requests.GetBatchRollCallStatusRequest;
import com.bcxin.tenant.open.jdks.responses.CreateRollCallResponse;
import com.bcxin.tenant.open.jdks.responses.RollCallProgressStatusResponse;
import com.bcxin.tenant.open.rest.apis.controllers.requests.DoCreateRollCallRequest;
import com.bcxin.tenant.open.rest.apis.controllers.requests.RollCallDoReplyRequest;
import com.bcxin.tenant.open.rest.apis.utils.JwtUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.ResponseEntity;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;

@Tag(name = "RollCallController", description = "开始点名--人工点名和自动点名")
@RestController
@RequestMapping("/roll-calls")
public class RollCallController extends ControllerAbstract {
    private static final Logger logger = LoggerFactory.getLogger(RollCallController.class);
    private final RollCallWriterRpcProvider rollCallWriterRpcProvider;

    public RollCallController(RollCallWriterRpcProvider rollCallWriterRpcProvider) {
        this.rollCallWriterRpcProvider = rollCallWriterRpcProvider;
    }

    @Operation(
            summary = "开始点名", description = "根据条件进行开始点名操作",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回调度系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping
    public ResponseEntity<CreateRollCallResponse> post(@RequestBody DoCreateRollCallRequest request) {
        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException();
        }

        Collection<String> selectedSuperviseDepartIds = new ArrayList<>();
        if (userModel.getAssignedSuperviseDepartIds() != null) {
            selectedSuperviseDepartIds =
                    Arrays.stream(userModel.getAssignedSuperviseDepartIds())
                            .toList();
        }

        CreateRollCallRequest.DoActionOperatorInfo operator =
                CreateRollCallRequest.DoActionOperatorInfo.create(
                        userModel.getOrganizationId(),
                        userModel.getOrgInstitutional(),
                        userModel.getOrgName(),
                        userModel.getEmployeeId(),
                        userModel.getName()
                );

        CreateRollCallResponse data = this.rollCallWriterRpcProvider.create(
                CreateRollCallRequest.create(
                        selectedSuperviseDepartIds,
                        request.getSecurityStationIds(),
                        request.getStationTypes(),
                        operator
                ));

        return this.ok(data);
    }

    @Operation(
            summary = "回复点名和督导操作", description = "手机端点击回复点名/督导操作",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回调度系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PutMapping("/{id}/reply")
    public ResponseEntity doReply(@PathVariable Long id,
                                  @RequestHeader("accessToken") String accessToken,
                                  @RequestBody RollCallDoReplyRequest request
    ) {
        String employeeId = JwtUtil.getUserIdFromToken(accessToken);
        if (!StringUtils.hasLength(employeeId)) {
            throw new UnAuthorizedTenantException();
        }

        if (CollectionUtils.isEmpty(request.getEmployeeIds())) {
            throw new ArgumentTenantException("职员Id(employeeIds)不能为空");
        }

        this.rollCallWriterRpcProvider.dispatch(
                AppReplyRollCallEmployeeStatusRequest.create(id, request.getEmployeeIds()));

        return this.ok();
    }


    @Operation(
            summary = "回复点名和督导操作", description = "手机端点击回复点名/督导操作",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回调度系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/status")
    public ResponseEntity<Collection<RollCallProgressStatusResponse>> getStatus(@RequestBody GetBatchRollCallStatusRequest request) {
        Collection<RollCallProgressStatusResponse> rcSr =
                this.rollCallWriterRpcProvider.getProgressStatus(request.getRollCallIds());

        return this.ok(rcSr);
    }
}
