package com.bcxin.tenant.open.rest.apis.controllers;

import com.bcxin.tenant.open.infrastructures.EntityCollection;
import com.bcxin.tenant.open.jdks.ProjectReaderRpcProvider;
import com.bcxin.tenant.open.jdks.ProjectWriterRpcProvider;
import com.bcxin.tenant.open.jdks.requests.ProjectAdvanceSearchRequest;
import com.bcxin.tenant.open.jdks.requests.ProjectSearchRequest;
import com.bcxin.tenant.open.jdks.responses.ProjectResponse;
import com.bcxin.tenant.open.jdks.responses.ProjectOverviewResponse;
import com.bcxin.tenant.open.jdks.responses.ProjectTeamResponse;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.Collection;

@Tag(name = "ProjectController", description = "临保项目")
@RestController
@RequestMapping("/projects")
public class ProjectController extends ControllerAbstract {
    private final ProjectReaderRpcProvider projectReaderRpcProvider;
    private final ProjectWriterRpcProvider projectWriterRpcProvider;

    public ProjectController(ProjectReaderRpcProvider projectReaderRpcProvider,
                             ProjectWriterRpcProvider projectWriterRpcProvider) {
        this.projectReaderRpcProvider = projectReaderRpcProvider;
        this.projectWriterRpcProvider = projectWriterRpcProvider;
    }

    @Operation(summary = "查询项目的详情", description = "查询项目的详情",
            requestBody = @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "查询项目的详情")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @GetMapping("/{id}")
    public ResponseEntity<ProjectResponse> get(@PathVariable String id) {
        var data = this.projectWriterRpcProvider.get(id);
        if (data == null) {
            return this.notFound();
        }

        return this.ok(data);
    }

    @Operation(summary = "搜索临保项目", description = "搜索临保项目; 只返回集合数据不返回总行数",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "搜索临保项目")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/search")
    public ResponseEntity<Collection<ProjectOverviewResponse>> post(@RequestBody ProjectSearchRequest request) {
        Collection<ProjectOverviewResponse> data =
                this.projectReaderRpcProvider.search(request);

        return this.ok(data);
    }

    @Operation(summary = "搜索临保项目", description = "搜索临保项目; 只返回集合数据不返回总行数",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "搜索临保项目")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/advance-search")
    public ResponseEntity<EntityCollection<ProjectResponse>> post(@RequestBody ProjectAdvanceSearchRequest request) {
        EntityCollection<ProjectResponse> data =
                this.projectWriterRpcProvider.search(request);

        return this.ok(data);
    }


    @Operation(summary = "getTeams", description = "临保项目底下的项目信息--树形结构",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "临保项目底下的项目信息--树形结构")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/{id}/tree-teams")
    public ResponseEntity<Collection<ProjectTeamResponse>> getTeams(@PathVariable String id) {
        var data = this.projectWriterRpcProvider.getTeams(id);

        return this.ok(data);
    }
}
