package com.bcxin.tenant.open.rest.apis.controllers;

import com.bcxin.tenant.open.infrastructures.EntityCollection;
import com.bcxin.tenant.open.infrastructures.TenantContext;
import com.bcxin.tenant.open.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.open.infrastructures.components.IdWorker;
import com.bcxin.tenant.open.infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.tenant.open.infrastructures.utils.StringUtil;
import com.bcxin.tenant.open.infrastructures.criterias.CriteriaAbstract;
import com.bcxin.tenant.open.infrastructures.enums.PoliceEventCategory;
import com.bcxin.tenant.open.infrastructures.enums.PoliceEventLevel;
import com.bcxin.tenant.open.infrastructures.enums.PoliceEventStatus;
import com.bcxin.tenant.open.infrastructures.exceptions.BadTenantException;
import com.bcxin.tenant.open.infrastructures.utils.StringUtil;
import com.bcxin.tenant.open.jdks.PoliceIncidentsRpcProvider;
import com.bcxin.tenant.open.jdks.QueueRpcProvider;
import com.bcxin.tenant.open.jdks.SecurityStationReaderRpcProvider;
import com.bcxin.tenant.open.jdks.requests.CreatePoliceIncidentsRequest;
import com.bcxin.tenant.open.jdks.requests.PoliceIncidentStatisticsRequest;
import com.bcxin.tenant.open.jdks.requests.CreateSosRequest;
import com.bcxin.tenant.open.jdks.requests.PoliceIncidentsSearchRequest;
import com.bcxin.tenant.open.jdks.requests.UpdatePoliceIncidentsRequest;
import com.bcxin.tenant.open.jdks.requests.*;
import com.bcxin.tenant.open.jdks.responses.*;
import com.bcxin.tenant.open.infrastructures.models.DataItemOption;
import com.bcxin.tenant.open.infrastructures.utils.PoliceEventCategoryUtils;
import com.bcxin.tenant.open.jdks.responses.exports.PoliceIncidentOverviewExportResponse;
import com.bcxin.tenant.open.jdks.responses.exports.PoliceIncidentSsoOverviewExportResponse;
import com.bcxin.tenant.open.rest.apis.utils.ExcelExportUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.servlet.http.HttpServletResponse;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;


@Tag(name = "PoliceIncidentsController", description = "警情上报接口;针对晋城(/identity/{}/{}/auto-login 返回的数据regionCode为1405则为晋城)的话,则为线索上报; ")
@RestController
@RequestMapping("/policeIncidents")
public class PoliceIncidentsController extends ControllerAbstract {
    private final PoliceIncidentsRpcProvider policeIncidentsRpcProvider;
    private final QueueRpcProvider queueRpcProvider;
    private final SecurityStationReaderRpcProvider securityStationReaderRpcProvider;
    private final IdWorker idWorker;

    public PoliceIncidentsController(PoliceIncidentsRpcProvider policeIncidentsRpcProvider,
                                     QueueRpcProvider queueRpcProvider,
                                     SecurityStationReaderRpcProvider securityStationReaderRpcProvider,
                                     IdWorker idWorker) {
        this.policeIncidentsRpcProvider = policeIncidentsRpcProvider;
        this.queueRpcProvider = queueRpcProvider;
        this.securityStationReaderRpcProvider = securityStationReaderRpcProvider;
        this.idWorker = idWorker;
    }

    @Operation(
            summary = "新增警情信息", description = "根据原型提交相应信息，不需要额外上送ID",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok."),
                    @ApiResponse(responseCode = "404", description = "找不到保安人员.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping
    public ResponseEntity post(@RequestBody CreatePoliceIncidentsRequest request) {
        if (request.getPoliceEventLevel() == null) {
            request.setPoliceEventLevel(PoliceEventLevel.Comon);
        }

        if (request.getCategory() == null) {
            request.setCategory(PoliceEventCategory.QT);
            request.setPoliceEventType(0);
        }

        if(StringUtil.isEmpty(request.getAddress())) {
            request.setAddress("");
        }

        if(StringUtil.isEmpty(request.getLocation())) {
            request.setLocation("");
        }

        if(StringUtil.isEmpty(request.getReason())) {
            request.setReason("");
        }

        this.policeIncidentsRpcProvider.create(request);

        return this.ok();
    }

    @Operation(
            summary = "补充警情/线索信息", description = "通过ID匹配到对应记录自动将记录更新警官相关处理意见",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok."),
                    @ApiResponse(responseCode = "404", description = "找不到保安人员.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PutMapping("/info")
    public ResponseEntity put(@RequestBody UpdatePoliceIncidentsInfoRequest request) {
        this.policeIncidentsRpcProvider.updateInfo(request);

        return this.ok();
    }

    @Operation(
            summary = "修改警情信息", description = "通过ID匹配到对应记录自动将记录更新警官相关处理意见",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok."),
                    @ApiResponse(responseCode = "404", description = "找不到保安人员.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PutMapping
    public ResponseEntity put(@RequestBody UpdatePoliceIncidentsRequest request) {
        this.policeIncidentsRpcProvider.update(request);

        return this.ok();
    }

    @Operation(
            summary = "修改警情信息状态为已处理", description = "通过ID匹配到对应记录自动将记录更新为已处理，不需要额外上送处理状态",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok."),
                    @ApiResponse(responseCode = "404", description = "找不到保安人员.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken"),
                    @Parameter(in = ParameterIn.DEFAULT, required = true, name = "id", description = "警情记录的id")
            }
    )
    @PutMapping("/{id}")
    public ResponseEntity updateStatus(@PathVariable Long id) {
        this.policeIncidentsRpcProvider.updateStatus(id);

        return this.ok();
    }

    @Operation(
            summary = "删除警情/线索信息", description = "通过ID删除警情/线索信息",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok."),
                    @ApiResponse(responseCode = "404", description = "找不到保安人员.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken"),
                    @Parameter(in = ParameterIn.DEFAULT, required = true, name = "id", description = "警情记录的id")
            }
    )
    @DeleteMapping("/{id}")
    public ResponseEntity delete(@PathVariable Long id) {
        this.policeIncidentsRpcProvider.delete(id);

        return this.ok();
    }

    @Operation(
            summary = "获取警情上报详情信息", description = "获取警情上报详情信息",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回警情上报详情信息."),
                    @ApiResponse(responseCode = "404", description = "找不到保安人员.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken"),
                    @Parameter(in = ParameterIn.DEFAULT, required = true, name = "id", description = "警情记录的id")
            }
    )
    @GetMapping("/{id}")
    public ResponseEntity<PoliceIncidentOverviewResponse> get(@PathVariable Long id) {
        PoliceIncidentOverviewResponse data =
                this.policeIncidentsRpcProvider.get(id);
        if (data == null) {
            return this.notFound();
        }

        return this.ok(data);
    }

    @Operation(
            summary = "获取警情上报详情信息并且更新处理状态", description = "在警情上报图层中，获取警情上报详情信息时将状态更新为“处理中”",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回警情上报详情信息."),
                    @ApiResponse(responseCode = "404", description = "找不到保安人员.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken"),
                    @Parameter(in = ParameterIn.DEFAULT, required = true, name = "id", description = "警情记录的id")
            }
    )
    @GetMapping("layerGet/{id}")
    public ResponseEntity<PoliceIncidentOverviewResponse> layerGet(@PathVariable Long id) {
        PoliceIncidentOverviewResponse data =
                this.policeIncidentsRpcProvider.layerGet(id);
        if (data == null) {
            return this.notFound();
        }

        return this.ok(data);
    }

    @Operation(summary = "业务管理-高级搜索警情上报", description = "按页面显示警情上报信息;APP端、WEB调度、WEB警情管理等都可以使用（上送不同的筛选条件） ",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回警情上报信息列表")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/search")
    public ResponseEntity<EntityCollection<PoliceIncidentOverviewResponse>> search(@RequestBody PoliceIncidentsSearchRequest request) {
        EntityCollection<PoliceIncidentOverviewResponse> data =
                this.policeIncidentsRpcProvider.search(request);

        return this.ok(data);
    }

    @Operation(summary = "图层搜索警情上报", description = "按页面显示警情上报信息;接口限定只查询非已处理的上报记录 ",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回警情上报信息列表")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/searchNeOver")
    public ResponseEntity<EntityCollection<PoliceIncidentOverviewResponse>> searchNeOver(@RequestBody PoliceIncidentGetPendingRequest request) {
        EntityCollection<PoliceIncidentOverviewResponse> data =
                this.policeIncidentsRpcProvider.getPendingByLevel(request);

        return this.ok(data);
    }

    @Operation(summary = "统计权限下各等级的警情案件数", description = "WEB调度列表页用到 ",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回各等级的警情案件数")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/levelCount")
    public ResponseEntity<PoliceIncidentsLevelCountResponse> levelCount(
            @RequestParam(value = "fromRedis", defaultValue = "false") boolean fromRedis
    ) {
        PoliceIncidentsLevelCountResponse data =
                this.policeIncidentsRpcProvider.getPendingTotal(
                        PoliceIncidentStatisticsRequest.create(fromRedis)
                );

        return this.ok(data);
    }

    @Operation(summary = "新增一键报警", description = "新增一键报警 ",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/sos")
    public ResponseEntity doSosPost(@RequestBody CreateSosRequest request) {
        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException();
        }

        String organizationName = "#";
        String stationAddress = "#";
        if (!StringUtil.isEmpty(userModel.getSecurityStationId())) {
            SecurityStationDetailResponse detailResponse =
                    this.securityStationReaderRpcProvider.get(userModel.getSecurityStationId());
            if (detailResponse != null) {
                stationAddress = detailResponse.getAddress();
                organizationName = detailResponse.getCompanyName();
            }
        }

        CreateSosPoliceIncidentQueueRequest queueRequest =
                CreateSosPoliceIncidentQueueRequest.create(
                        idWorker,
                        userModel, request, organizationName, stationAddress);

        this.queueRpcProvider.dispatch(queueRequest);

        return this.ok();
    }

    @Operation(summary = "调度人员之后进行一键报警处理", description = "调度人员之后进行一键报警处理",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = false),
            parameters = {
                    @Parameter(in = ParameterIn.PATH, required = true, name = "id", description = "sos的Id"),
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PutMapping("/sos/{id}")
    public ResponseEntity updateSosStatus(@PathVariable long id) {
        this.policeIncidentsRpcProvider.update(UpdateSosPoliceIncidentStatusRequest.create(id));

        return this.ok();
    }

    @Operation(summary = "业务管理-导出-高级搜索警情上报", description = "按页面显示警情上报信息;APP端、WEB调度、WEB警情管理等都可以使用（上送不同的筛选条件） ",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回警情上报信息列表")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/export")
    public void export(@RequestBody PoliceIncidentsSearchRequest request,
                       HttpServletResponse response) throws IOException {
        if(request.getCreatedStarTime()==null || request.getCreatedEndTime()==null) {
            throw new BadTenantException("时间范围不能为空");
        }

        request.setForExport(true);
        request.setPageSize(1000);
        String fileName = String.format("警情上报-%s", new SimpleDateFormat("yyyy-MM-dd").format(new Date()));
        Class tClass = PoliceIncidentOverviewExportResponse.class;
        if (request.isFromSos()) {
            fileName = String.format("sos记录-%s", new SimpleDateFormat("yyyy-MM-dd").format(new Date()));
            tClass = PoliceIncidentSsoOverviewExportResponse.class;
        }

        ExcelExportUtil.export(response, tClass, fileName, (excelWriter, sheet) -> {
            int pageIndex = 1;
            EntityCollection<PoliceIncidentOverviewResponse> data = null;
            do {
                request.setPageIndex(pageIndex);
                data = this.policeIncidentsRpcProvider.search(request);
                if (request.isFromSos()) {
                    excelWriter.write(data.getData()
                            .stream().map(ii -> PoliceIncidentSsoOverviewExportResponse.create(ii)).collect(Collectors.toList()), sheet);
                } else {
                    excelWriter.write(data.getData()
                            .stream().map(ii -> PoliceIncidentOverviewExportResponse.create(ii)).collect(Collectors.toList()), sheet);
                }

                pageIndex++;
            } while (data != null && !data.endOfData());
        });
    }

    @Operation(summary = "区域报表", description = "针对晋城的区域报表",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回警情上报信息列表")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/region-report")
    public ResponseEntity<Collection<PoliceIncidentRegionReportResponse>> getRegionReport(@RequestBody PoliceIncidentRegionReportRequest request) {
        Collection<PoliceIncidentRegionReportResponse> data =
                this.policeIncidentsRpcProvider.getReport(request);

        return this.ok(data);
    }
}
