package com.bcxin.tenant.open.rest.apis.controllers;
import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.configs.TenantSystemConfig;
import com.bcxin.tenant.open.infrastructures.enums.DispatchAccountType;
import com.bcxin.tenant.open.infrastructures.enums.ResourceType;
import com.bcxin.tenant.open.infrastructures.utils.BusinessUtil;
import com.bcxin.tenant.open.jdks.CompanyReaderRpcProvider;
import com.bcxin.tenant.open.jdks.IdentityRpcProvider;
import com.bcxin.tenant.open.jdks.OrgPurseRpcProvider;
import com.bcxin.tenant.open.jdks.responses.CompanyDetailResponse;
import com.bcxin.tenant.open.jdks.responses.OrgPurseValidateResponse;
import com.bcxin.tenant.open.jdks.responses.RdDeviceDeskPermissionResponse;
import com.bcxin.tenant.open.infrastructures.UserDetailResponse;
import com.bcxin.tenant.open.rest.apis.caches.CacheKeyManage;
import com.bcxin.tenant.open.rest.apis.caches.CompanyBasicCache;
import com.bcxin.tenant.open.rest.apis.components.HotCacheProvider;
import com.bcxin.tenant.open.rest.apis.controllers.responses.DispatchResponse;
import com.bcxin.tenant.open.rest.apis.controllers.responses.ResponseBuilder;
import com.bcxin.tenant.open.rest.apis.utils.JwtUtil;
import com.bcxin.tenant.open.rest.apis.utils.OrgPurseValidationUtil;
import com.bcxin.tenant.open.rest.apis.utils.UserUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

@Tag(name = "IdentityController", description = "身份认证")
@RestController
@RequestMapping("/identity")
public class IdentityController extends ControllerAbstract {
    private final IdentityRpcProvider identityRpcProvider;
    private final JsonProvider jsonProvider;

    private final OrgPurseRpcProvider orgPurseRpcProvider;
    private final CompanyReaderRpcProvider companyReaderRpcProvider;

    private final HotCacheProvider hotCacheProvider;

    private final TenantSystemConfig systemConfig;

    public IdentityController(
            IdentityRpcProvider identityRpcProvider,
            JsonProvider jsonProvider,
            OrgPurseRpcProvider orgPurseRpcProvider,
            CompanyReaderRpcProvider companyReaderRpcProvider,
            HotCacheProvider hotCacheProvider,
            TenantSystemConfig systemConfig) {
        this.identityRpcProvider = identityRpcProvider;
        this.jsonProvider = jsonProvider;
        this.orgPurseRpcProvider = orgPurseRpcProvider;
        this.companyReaderRpcProvider = companyReaderRpcProvider;
        this.hotCacheProvider = hotCacheProvider;
        this.systemConfig = systemConfig;
    }

    @Operation(
            summary = "单点登入调度系统", description = "单点登入调度系统-获取调度系统 dispatchToken",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回调度系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "accessToken",
                            description = "来自V5的accessToken"),
                    @Parameter(in = ParameterIn.PATH, required = true, name = "accountType", description = "账号类型: 使用百保盾APP的时候使用APP(现有方式); 集成第三方设备的时候的使用Device;进入GIS的调度台(WEB及后期APP的调度界面)进行调度的时候使用Desk")
            }
    )
    @PostMapping("/{accountType}/auto-login")
    public ResponseEntity<DispatchResponse> autoLogin(@RequestHeader("accessToken") String accessToken,
                                                              @PathVariable("accountType") DispatchAccountType accountType) {
        String employeeId = JwtUtil.getUserIdFromToken(accessToken);

        if (!StringUtils.hasLength(employeeId)) {
            return this.status(HttpStatus.UNAUTHORIZED, "未授权用户");
        }

        if (BusinessUtil.isOrgMember(employeeId)) {
            employeeId = BusinessUtil.extractMemberId(employeeId);
        }

        UserDetailResponse response = UserUtil.get(hotCacheProvider, identityRpcProvider, employeeId, accountType);

        if (response == null) {
            return this.status(HttpStatus.UNAUTHORIZED, "未授权用户");
        }

        boolean isSupervise = false;
        if (BusinessUtil.checkIfDesk(accountType)) {
            RdDeviceDeskPermissionResponse permissionResponse =
                    this.identityRpcProvider.getDeviceDeskPermission(
                            response.getEmployeeId(),
                            response.getOrganizationId(),
                            response.getAccountType(),
                            response.getInstitutional(),
                            response.getOrgName(),
                            response.getOrganizationId()
                    );

            if (permissionResponse == null || CollectionUtils.isEmpty(permissionResponse.getServiceScopes())) {
                return this.status(HttpStatus.UNAUTHORIZED,
                        String.format("您(%s)尚未购买当前应用商品，请前往商城应用市场购买“调度台”商品或者绑定调度台后，方可进行调度操作", response.getOrgName())

                );
            }

            if (permissionResponse.getServiceScopes().stream().allMatch(ii -> !ii.isActive())) {
                return this.status(HttpStatus.UNAUTHORIZED,
                        String.format("您(%s)的调度台已经过期，请前往商城应用市场购买“调度台”商品或者绑定调度台后，方可进行调度操作", response.getOrgName())
                );
            }

            isSupervise = permissionResponse.isSuperviseDesk();
        }

        CompanyBasicCache companyBasicCache = hotCacheProvider.get(
                CacheKeyManage.getOrgBasicKey(response.getOrganizationId()), CompanyBasicCache.class);
        if (companyBasicCache == null) {
            CompanyDetailResponse detailResponse =
                    this.companyReaderRpcProvider.get(response.getOrganizationId(), false);

            if (detailResponse != null) {
                companyBasicCache = CompanyBasicCache.create(detailResponse.getLatitude(), detailResponse.getLongitude(), detailResponse.getInstitutional());

                hotCacheProvider.put(
                        CacheKeyManage.getOrgBasicKey(response.getOrganizationId()), 2 * 60 * 60, companyBasicCache, false);
            }
        }

        if (companyBasicCache != null) {
            response.assignOrgLatLon(companyBasicCache.getLatitude(), companyBasicCache.getLongitude());
            response.assignIsSupervise(BusinessUtil.isSupervise(companyBasicCache.getInstitutional()));

            /**
             * 非监管调度台的话, 那么取企业的机构类型进行判断
             */
            isSupervise = BusinessUtil.isSupervise(companyBasicCache.getInstitutional());
        }

        if (!isSupervise) {
            /**
             * 001 临保项目; 暂时允许没有联动值也可以进行签到.
             */
            if (!BusinessUtil.checkIfDesk(response.getAccountType()) &&
                    ResourceType.hasResourceType(response.getRts(), ResourceType.TemporarySecurity)) {
                //如果当前用户是非调度台且为临保APP用户, 那么不进行联动值的查询验证
            } else {
                ResponseEntity<ResponseBuilder.ResponseBody<OrgPurseValidateResponse>> validateResponse =
                        OrgPurseValidationUtil.doValidate(
                                this.hotCacheProvider,
                                this.orgPurseRpcProvider, response.getAccountType(),
                                BusinessUtil.getPaymentOrgId(response.getOrganizationId(),response.getPaymentOrgId())
                        );
                if (validateResponse.getStatusCode() != HttpStatus.OK) {
                    ResponseBuilder.ResponseBody<OrgPurseValidateResponse> body = validateResponse.getBody();

                    return this.status(body.getHttpStatus(), body.getData(), body.getMessage());
                }
            }
        }

        String content = this.jsonProvider.getJson(response);
        String dispatchToken = JwtUtil.getTokenFromIdentityData(content);

        return this.ok(DispatchResponse.create(dispatchToken, isSupervise, "140500"));
    }

    @Operation(
            summary = "单点登入调度系统", description = "单点登入调度系统-获取调度系统 dispatchToken",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回调度系统的 dispatchToken."),
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "accessToken",
                            description = "来自V5的accessToken"),
                    @Parameter(in = ParameterIn.PATH, required = true, name = "accountType", description = "账号类型: 使用百保盾APP的时候使用APP(现有方式); 集成第三方设备的时候的使用Device;进入GIS的调度台(WEB及后期APP的调度界面)进行调度的时候使用Desk")
            }
    )
    @PostMapping("/{accountType}/{employeeId}/auto-login")
    public ResponseEntity<DispatchResponse> autoLoginByEmployeeId(
            @PathVariable("employeeId") String employeeId,
                                                      @PathVariable("accountType") DispatchAccountType accountType) {
        return autoLogin(JwtUtil.getMobileToken(employeeId), accountType);
    }
}