package com.bcxin.tenant.open.rest.apis.controllers;

import com.bcxin.tenant.open.infrastructures.TenantContext;
import com.bcxin.tenant.open.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.constants.BusinessConstants;
import com.bcxin.tenant.open.infrastructures.enums.CompanyLevel;
import com.bcxin.tenant.open.infrastructures.enums.DeskType;
import com.bcxin.tenant.open.infrastructures.enums.DispatchAccountType;
import com.bcxin.tenant.open.infrastructures.enums.ResourceType;
import com.bcxin.tenant.open.infrastructures.exceptions.ArgumentTenantException;
import com.bcxin.tenant.open.infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.tenant.open.infrastructures.utils.BusinessUtil;
import com.bcxin.tenant.open.jdks.*;
import com.bcxin.tenant.open.jdks.requests.SyncUpdateGeoRequest;
import com.bcxin.tenant.open.jdks.requests.UpdateGeoRequest;
import com.bcxin.tenant.open.jdks.requests.LogoSettingRequest;
import com.bcxin.tenant.open.jdks.requests.SosSettingRequest;
import com.bcxin.tenant.open.jdks.responses.*;
import com.bcxin.tenant.open.rest.apis.caches.CacheKeyManage;
import com.bcxin.tenant.open.rest.apis.caches.EmployeeDutyStatusCache;
import com.bcxin.tenant.open.rest.apis.components.HotCacheProvider;
import com.bcxin.tenant.open.rest.apis.controllers.requests.OrgPurseValidateRequest;
import com.bcxin.tenant.open.rest.apis.controllers.responses.ResponseBuilder;
import com.bcxin.tenant.open.rest.apis.controllers.responses.UserInfoResponse;
import com.bcxin.tenant.open.rest.apis.controllers.responses.WorkstationResponseContainerWrapper;
import com.bcxin.tenant.open.rest.apis.utils.JwtUtil;
import com.bcxin.tenant.open.rest.apis.utils.OrgPurseValidationUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import jakarta.servlet.http.HttpServletRequest;
import org.springframework.http.HttpStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.ResponseEntity;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.stream.Collectors;

@RestController
@RequestMapping("/tenant/users/current")
public class CurrentUserController  extends ControllerAbstract {
    private static final Logger logger = LoggerFactory.getLogger(CurrentUserController.class);
    private final JsonProvider jsonProvider;
    private final QueueRpcProvider queueRpcProvider;

    private final TenantAccountWriterRpcProvider accountWriterRpcProvider;

    private final EmployeeReaderRpcProvider employeeReaderRpcProvider;

    private final RdSecurityStationRailWriterRpcProvider securityStationRailWriterRpcProvider;

    private final HotCacheProvider hotCacheProvider;

    private final OrgPurseRpcProvider orgPurseRpcProvider;

    private final WorkstationRpcProvider workstationRpcProvider;

    private final UserWriterRpcProvider userWriterRpcProvider;

    public CurrentUserController(JsonProvider jsonProvider,
                                 QueueRpcProvider queueRpcProvider,
                                 TenantAccountWriterRpcProvider accountWriterRpcProvider,
                                 EmployeeReaderRpcProvider employeeReaderRpcProvider,
                                 RdSecurityStationRailWriterRpcProvider securityStationRailWriterRpcProvider,
                                 HotCacheProvider hotCacheProvider,
                                 OrgPurseRpcProvider orgPurseRpcProvider,
                                 WorkstationRpcProvider workstationRpcProvider, UserWriterRpcProvider userWriterRpcProvider) {
        this.jsonProvider = jsonProvider;
        this.queueRpcProvider = queueRpcProvider;
        this.accountWriterRpcProvider = accountWriterRpcProvider;
        this.employeeReaderRpcProvider = employeeReaderRpcProvider;
        this.securityStationRailWriterRpcProvider = securityStationRailWriterRpcProvider;
        this.hotCacheProvider = hotCacheProvider;
        this.orgPurseRpcProvider = orgPurseRpcProvider;
        this.workstationRpcProvider = workstationRpcProvider;
        this.userWriterRpcProvider = userWriterRpcProvider;
    }

    @Operation(summary = "获取当前用户信息", description = "获取当前用户信息",
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            },
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功获取当前用户信息"),
                    @ApiResponse(responseCode = "404", description = "找不到该用户信息")
            }
    )
    @GetMapping
    public ResponseEntity<UserInfoResponse> get() {
        TenantEmployeeContext.TenantUserModel userModel =
                TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            return this.notFound();
        }

        UserInfoResponse response =
                UserInfoResponse.create(
                        userModel.getId(),
                        userModel.getEmployeeId(),
                        userModel.getName(),
                        userModel.getOrganizationId(),
                        userModel.getOrgInstitutional(),
                        userModel.getTencentUserId(),
                        userModel.getAccountType(),
                        userModel.isSupervise(),
                        userModel.getLimitedPermissionStations(),
                        userModel.getSuperviseRegionCode(),
                        userModel.getNoSuperviseScopePermissions(),
                        CompanyLevel.getLevels(userModel.getCLevel())
                );

        response.assignOrgLatLon(userModel.getOrgLatitude(), userModel.getOrgLongitude());

        if (BusinessUtil.checkIfDesk(userModel.getAccountType())) {
            response.assignRole(
                    userModel.getOrgInstitutional(), userModel.getOrgName()
            );

            if (!CollectionUtils.isEmpty(userModel.getDeviceDeskServiceScopes())) {
                Collection<UserInfoResponse.TenantDeviceDeskServiceScopeResponse>
                        deviceDeskServiceScopeResponses =
                        userModel.getDeviceDeskServiceScopes()
                                /**
                                 * 针对当前的企业; 这边只返回当前的调度台的监管列表
                                 */
                                .stream()
                                .map(ix -> UserInfoResponse.TenantDeviceDeskServiceScopeResponse.create(
                                                ix.getId(),
                                                ix.getName(),
                                                ix.getNumber(),
                                                ix.getBeginDate(),
                                                ix.getEndDate(),
                                                ix.getBoundEmployeeId(),
                                                ix.getAssignedSuperviseDepartId(),
                                                ix.getIpAddress(),
                                                ix.getOrganizationId(),
                                                ix.getIr(),
                                                ResourceType.TemporarySecurity,
                                        ix.getDesc()
                                        )
                                ).collect(Collectors.toList());

                response.assignDeviceDeskServiceScopes(deviceDeskServiceScopeResponses);
            }
        } else {
            if (!BusinessUtil.checkIfDesk(userModel.getAccountType())) {
                EmployeeDutyStatusCache statusCache =
                        this.hotCacheProvider.get(CacheKeyManage.getUserCurrentDutyStatus(userModel.getEmployeeId()), EmployeeDutyStatusCache.class);
                if (statusCache == null) {
                    EmployeeDetailResponse employeeDetailResponse =
                            this.employeeReaderRpcProvider.get(userModel.getEmployeeId(), DeskType.Normal);
                    if (employeeDetailResponse != null) {
                        statusCache = EmployeeDutyStatusCache.create(employeeDetailResponse.getDutyStatus(), employeeDetailResponse.getSecurityStationId());

                        hotCacheProvider.put(CacheKeyManage.getUserCurrentDutyStatus(userModel.getEmployeeId()), 30 * 60, statusCache,false);
                    }
                }

                if (statusCache != null) {
                    response.assignDutySignInType(statusCache.getDutyStatus());
                    response.assignSecurityStationId(statusCache.getSecurityStationId());
                }
            }
        }

        response.setPermission(userModel.getPermission());

        return this.ok(response);
    }

    /**
     * 针对坐标信息; 我们只验证token是否有效, 不再次获取用户信息
     *
     * @param request
     * @return
     */
    @Operation(summary = "实时更新当前用户坐标", description = "实时更新当前用户坐标",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功获取当前用户信息"),
                    @ApiResponse(responseCode = "404", description = "找不到该用户信息")
            }
    )
    @PostMapping("/real-location")
    public ResponseEntity postLocation(
            @RequestHeader("accessToken") String accessToken,
            @RequestBody UpdateGeoRequest request,
            HttpServletRequest servletRequest) {
        String employeeId = JwtUtil.getUserIdFromToken(accessToken);

        if (!StringUtils.hasLength(employeeId)) {
            return this.status(HttpStatus.UNAUTHORIZED, "未授权用户");
        }

        Collection<String> selectedEmployeeIds = request.getEmployeeIds();
        if (selectedEmployeeIds == null) {
            selectedEmployeeIds = new ArrayList<>();
        }
        selectedEmployeeIds.add(employeeId);

        SyncUpdateGeoRequest syncUpdateGeoRequest = SyncUpdateGeoRequest.create(
                request, selectedEmployeeIds
        );
        this.queueRpcProvider.syncUpdatedLonLat(syncUpdateGeoRequest);

        if (request.isDirectlyCheck()) {
            this.securityStationRailWriterRpcProvider.doCheck(Collections.singleton(syncUpdateGeoRequest));
        }

        return this.ok();
    }

    @Operation(summary = "设置SOS配置", description = "设置SOS配置",
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            },
            responses = {
                    @ApiResponse(responseCode = "200", description = "SOS配置设置成功"),
                    @ApiResponse(responseCode = "404", description = "找不到该用户信息")
            }
    )
    @PutMapping("/sos")
    public ResponseEntity doSosSettingPatch(@RequestBody SosSettingRequest request) {
        accountWriterRpcProvider.doSosSettingPatch(request);

        TenantEmployeeContext.TenantUserModel userModel =
                TenantContext.getInstance().getUserContext().get();
        if (userModel != null) {
            hotCacheProvider.delete(CacheKeyManage.getUserSetting(userModel.getAccountType(), userModel.getTencentUserId()));
        }

        return this.ok();
    }

    @Operation(summary = "设置读取APP操作指引", description = "设置读取APP操作指引",
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "accessToken", description = "可以取来自V5的AccessToken")
            },
            responses = {
                    @ApiResponse(responseCode = "200", description = "SOS配置设置成功"),
                    @ApiResponse(responseCode = "404", description = "找不到该用户信息")
            }
    )
    @PostMapping("/done-operation-instruction")
    public ResponseEntity doReadOperationInstruction() {
        accountWriterRpcProvider.doReadOperationInstruction();
        TenantEmployeeContext.TenantUserModel userModel =
                TenantContext.getInstance().getUserContext().get();
        if (userModel != null) {
            hotCacheProvider.delete(CacheKeyManage.getUserSetting(userModel.getAccountType(), userModel.getTencentUserId()));
        }

        return this.ok();
    }


    @Operation(summary = "设置系统图标及系统名称", description = "获取当前用户信息",
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            },
            responses = {
                    @ApiResponse(responseCode = "200", description = "SOS配置设置成功"),
                    @ApiResponse(responseCode = "404", description = "找不到该用户信息")
            }
    )
    @PutMapping("/info")
    public ResponseEntity doLogoSettingPatch(@RequestBody LogoSettingRequest request) {
        accountWriterRpcProvider.doLogoSettingPatch(request);
        TenantEmployeeContext.TenantUserModel userModel =
                TenantContext.getInstance().getUserContext().get();
        if (userModel != null) {
            hotCacheProvider.delete(CacheKeyManage.getUserSetting(userModel.getAccountType(), userModel.getTencentUserId()));
        }

        return this.ok();
    }

    @Operation(
            summary = "获取当前账号的配置信息", description = "当前的信息通过dispatchToken(调度台/硬件设备)或者app(才有accessToken)来获取当前的配置信息",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回调度系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = false,
                            name = "accessToken", description = "来自V5的accessToken"),
                    @Parameter(in = ParameterIn.HEADER, required = false, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @GetMapping("/setting")
    public ResponseEntity<TenantAccountSettingResponse> getAccountSetting(
            @RequestHeader(value = "accessToken", required = false) String accessToken,
            @RequestHeader(value = "dispatchToken", required = false) String dispatchToken

    ) {
        /**
         * 信息解析来自于AuthFilter中进行
         */
        TenantEmployeeContext.TenantUserModel userModel =
                TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException();
        }

        String cacheKey = CacheKeyManage.getUserSetting(userModel.getAccountType(), userModel.getTencentUserId());

        TenantAccountSettingResponse data = hotCacheProvider.get(cacheKey, TenantAccountSettingResponse.class);
        if (data == null) {
            data = this.accountWriterRpcProvider.getById(
                    userModel.getAccountType(), userModel.getTencentUserId());

            hotCacheProvider.put(cacheKey, 10 * 60, data, false);
        }

        return this.ok(data);
    }


    @Operation(
            summary = "验证当前的用户是否有权限进入调度台/APP", description = "当前用户登入之后通过dispatchToken来验证是否允许打开",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回调度系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = false, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/do-validate")
    public ResponseEntity doValidate(@RequestBody OrgPurseValidateRequest request,
                                     @RequestHeader(value = "bcx-institutional", required = false) String institutional,
                                     @RequestHeader(value = "bcx-account-type", required = false) String accountType) {
        /**
         * 暂时临保的企业不验证联动值
         * todo: 20250403
         */
        try {
            if (BusinessUtil.isDaHuoManagedOrg(institutional)) {
                DispatchAccountType dispatchAccountType = DispatchAccountType.valueOf(accountType);
                if (!BusinessUtil.checkIfDesk(dispatchAccountType)) {
                    return this.ok();
                } else if (dispatchAccountType == DispatchAccountType.AppDesk) {
                    return this.ok();
                }
            }
        } catch (Exception ex) {
            logger.error("当前联动值验证发生异常", ex);

            throw new ArgumentTenantException(String.format("参数异常institutional=%s;accountType=%s", institutional, accountType));
        }

        String paymentOrgId = request.getOrganizationId();
        try {
            TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
            if (userModel == null) {
                throw new UnAuthorizedTenantException();
            }

            paymentOrgId = BusinessUtil.getPaymentOrgId(userModel.getOrganizationId(), userModel.getPaymentOrgId());
        } catch (Exception ex) {
            logger.error("联动值检测异常-验证联动值余额的API未收到Token信息", ex);
        }

        ResponseEntity<ResponseBuilder.ResponseBody<OrgPurseValidateResponse>> validateResponse =
                OrgPurseValidationUtil.doValidate(
                        this.hotCacheProvider,
                        this.orgPurseRpcProvider, request.getAccountType(), paymentOrgId
                );

        return validateResponse;
    }


    @Operation(
            summary = "获取当前用户的点位/驻勤/岗点信息", description = "当前的信息通过dispatchToken(调度台/硬件设备)或者app(才有accessToken)来获取当前用户的点位/驻勤/岗点信息",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回调度系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = false,
                            name = "accessToken", description = "来自V5的accessToken"),
                    @Parameter(in = ParameterIn.HEADER, required = false, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @GetMapping("/work-stations")
    public ResponseEntity<Collection<WorkstationResponse>> getWorkstationsByEmployeeId() {
        TenantEmployeeContext.TenantUserModel userModel =
                TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException();
        }

        WorkstationResponseContainerWrapper rpsWrapper =
                this.hotCacheProvider.get(
                        CacheKeyManage.getHotWorkstationByEmployeeId(userModel.getEmployeeId()),
                        () -> {
                            Collection<WorkstationResponse> data =
                                    this.workstationRpcProvider.getWorkstationsByEmployeeId(userModel.getEmployeeId());

                            return WorkstationResponseContainerWrapper.create(
                                    data
                            );
                        }, 5 * 30, WorkstationResponseContainerWrapper.class, false);


        return this.ok(rpsWrapper.getWorkstations());
    }


    @Operation(
            summary = "获取当前用户的积分信息", description = "当前的用户的积分信息",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回调度系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = false,
                            name = "accessToken", description = "来自V5的accessToken"),
                    @Parameter(in = ParameterIn.HEADER, required = false, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @Deprecated
    @GetMapping("/purse")
    public ResponseEntity<UserPurseResponse> getUserPurseResponse() {
        TenantEmployeeContext.TenantUserModel userModel =
                TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException();
        }

        UserPurseResponse purseResponse =
                this.userWriterRpcProvider.getById(userModel.getId());


        return this.ok(purseResponse);
    }
}
