package com.bcxin.tenant.open.rest.apis.controllers;

import com.bcxin.tenant.open.infrastructures.EntityCollection;
import com.bcxin.tenant.open.jdks.CompanyReaderRpcProvider;
import com.bcxin.tenant.open.jdks.CompanyWriterRpcProvider;
import com.bcxin.tenant.open.jdks.requests.CompanyOverviewRequest;
import com.bcxin.tenant.open.jdks.requests.CompanySearchRequest;
import com.bcxin.tenant.open.jdks.requests.EmployeeAdvanceRequest;
import com.bcxin.tenant.open.jdks.responses.CompanyDetailResponse;
import com.bcxin.tenant.open.jdks.responses.CompanyAdvanceResponse;
import com.bcxin.tenant.open.jdks.responses.CompanySearchResponse;
import com.bcxin.tenant.open.jdks.responses.EmployeeAdvanceResponse;
import com.bcxin.tenant.open.rest.apis.utils.ExcelExportUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import jakarta.servlet.http.HttpServletResponse;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;

@RestController
@RequestMapping("/companies")
public class CompanyController extends ControllerAbstract {
    private final CompanyReaderRpcProvider companyReaderRpcProvider;
    private final CompanyWriterRpcProvider companyWriterRpcProvider;

    public CompanyController(CompanyReaderRpcProvider companyReaderRpcProvider,
                             CompanyWriterRpcProvider companyWriterRpcProvider) {
        this.companyReaderRpcProvider = companyReaderRpcProvider;
        this.companyWriterRpcProvider = companyWriterRpcProvider;
    }

    @Operation(summary = "获取企业详情信息", description = "获取企业详情信息",
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            },
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回企业详情信息"),
                    @ApiResponse(responseCode = "404", description = "找不到该企业信息")
            }
    )
    @GetMapping("/{id}")
    public ResponseEntity<CompanyDetailResponse> get(@PathVariable String id,
                                                     @RequestParam(value = "includeForExamSiteRoom",defaultValue = "false")
                                                     Boolean includeForExamSiteRoom) {
        CompanyDetailResponse data =
                this.companyReaderRpcProvider.get(id, (includeForExamSiteRoom != null ? includeForExamSiteRoom.booleanValue() : false));
        if (data == null) {
            return this.notFound();
        }

        return this.ok(data);
    }

    @Operation(summary = "高级搜索企业信息", description = "按页面显示企业信息; 只返回集合数据不返回总行数",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回公司信息列表")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/search")
    public ResponseEntity<Collection<CompanySearchResponse>> post(@RequestBody CompanySearchRequest request) {
        Collection<CompanySearchResponse> data =
                this.companyReaderRpcProvider.search(request);

        return this.ok(data);
    }

    @Operation(summary = "业务管理的企业搜索", description = "业务管理的企业搜索 advance-search",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回公司信息列表")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/advance-search")
    public ResponseEntity<EntityCollection<CompanyAdvanceResponse>> doBusinessSearch(@RequestBody CompanyOverviewRequest request) {
        EntityCollection<CompanyAdvanceResponse> data =
                this.companyWriterRpcProvider.search(request);

        return this.ok(data);
    }
    @Operation(summary = "导出-业务管理-企业信息", description = "导出-根据过滤条件查询企业信息列表 ",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "企业信息列表")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/export")
    public void export(@RequestBody CompanyOverviewRequest request,
                       HttpServletResponse response) throws IOException {
        String fileName = String.format("企业信息-%s", new SimpleDateFormat("yyyy-MM-dd").format(new Date()));
        request.setForExport(true);
        request.setPageSize(1000);
        ExcelExportUtil.export(response, CompanyAdvanceResponse.class, fileName, (excelWriter, sheet) -> {
            int pageIndex = 1;
            EntityCollection<CompanyAdvanceResponse> data = null;
            do {
                request.setPageIndex(pageIndex);
                data = this.companyWriterRpcProvider.search(request);

                excelWriter.write(data.getData(), sheet);
                pageIndex++;
            } while (data != null && !data.endOfData());
        });
    }
}
