package com.bcxin.tenant.open.rest.apis.controllers;

import com.bcxin.tenant.open.infrastructures.EntityCollection;
import com.bcxin.tenant.open.infrastructures.TenantContext;
import com.bcxin.tenant.open.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.open.infrastructures.UserDetailResponse;
import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.constants.BusinessConstants;
import com.bcxin.tenant.open.infrastructures.enums.DeskType;
import com.bcxin.tenant.open.infrastructures.exceptions.BadTenantException;
import com.bcxin.tenant.open.infrastructures.exceptions.NoFoundTenantException;
import com.bcxin.tenant.open.infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.tenant.open.infrastructures.models.AgentHeaderOptionValue;
import com.bcxin.tenant.open.infrastructures.utils.BusinessUtil;
import com.bcxin.tenant.open.infrastructures.utils.StringUtil;
import com.bcxin.tenant.open.jdks.AttendanceRpcProvider;
import com.bcxin.tenant.open.jdks.EmployeeReaderRpcProvider;
import com.bcxin.tenant.open.jdks.RdSyncRpcWriterProvider;
import com.bcxin.tenant.open.jdks.WorkstationRpcProvider;
import com.bcxin.tenant.open.jdks.requests.*;
import com.bcxin.tenant.open.jdks.requests.wrappers.WrappedAttendanceWriterRequest;
import com.bcxin.tenant.open.jdks.responses.*;
import com.bcxin.tenant.open.rest.apis.caches.CacheKeyManage;
import com.bcxin.tenant.open.rest.apis.caches.EmployeeDutyStatusCache;
import com.bcxin.tenant.open.rest.apis.components.HotCacheProvider;
import com.bcxin.tenant.open.rest.apis.controllers.requests.AttendanceAppSearchRequest;
import com.bcxin.tenant.open.rest.apis.controllers.requests.AttendanceValidationRestRequest;
import com.bcxin.tenant.open.rest.apis.controllers.responses.EmployeeDefaultStationResponse;
import com.bcxin.tenant.open.rest.apis.utils.ExcelExportUtil;
import com.bcxin.tenant.open.rest.apis.utils.JwtUtil;
import com.bcxin.tenant.open.rest.apis.utils.ServletRequestUtil;
import com.bcxin.tenant.open.rest.apis.utils.UserUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.servlet.ServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.ResponseEntity;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;
import java.util.stream.Collectors;

@Tag(name = "AttendanceController", description = "签到接口")
@RestController
@RequestMapping("/attendance")
public class AttendanceController extends ControllerAbstract {
    private static final Logger logger = LoggerFactory.getLogger(AttendanceController.class);
    private final AttendanceRpcProvider attendanceRpcProvider;
    private final EmployeeReaderRpcProvider employeeReaderRpcProvider;
    private final HotCacheProvider hotCacheProvider;
    private final RdSyncRpcWriterProvider syncRpcWriterProvider;
    private final JsonProvider jsonProvider;

    private final WorkstationRpcProvider workstationRpcProvider;

    public AttendanceController(AttendanceRpcProvider attendanceRpcProvider,
                                EmployeeReaderRpcProvider employeeReaderRpcProvider,
                                HotCacheProvider hotCacheProvider, RdSyncRpcWriterProvider syncRpcWriterProvider,
                                JsonProvider jsonProvider, WorkstationRpcProvider workstationRpcProvider) {
        this.attendanceRpcProvider = attendanceRpcProvider;
        this.employeeReaderRpcProvider = employeeReaderRpcProvider;
        this.hotCacheProvider = hotCacheProvider;
        this.syncRpcWriterProvider = syncRpcWriterProvider;
        this.jsonProvider = jsonProvider;
        this.workstationRpcProvider = workstationRpcProvider;
    }

    @Operation(
            summary = "验证是否允许签到", description = "验证是否允许签到; 结果会返回是否进行人脸识别",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/pre-validation")
    public ResponseEntity<AttendanceValidationResponse> doValidate(
            @RequestBody AttendanceValidationRestRequest restRequest,
                                                                   @RequestHeader(name = "bcx-agent",defaultValue = "",required = false) String hAgent,
                                                                   @RequestHeader(name = "bcx-device",defaultValue = "",required = false) String hDevice,
                                                                   @RequestHeader(name = "bcx-emp-id",defaultValue = "",required = false) String hEmpId,
                                                                   @RequestHeader(name = "bcx-com-id",defaultValue = "",required = false) String hComId,
                                                                   ServletRequest servletRequest) {
        String employeeId = hEmpId;
        EmployeeDefaultStationResponse employeeDefaultStationResponse  = null;
        String securityStationId = null;
        try {
            if (!StringUtils.hasLength(employeeId)) {
                String accessToken = ServletRequestUtil.getAccessToken(servletRequest);
                employeeId = JwtUtil.getUserIdFromToken(accessToken);
            }

            employeeDefaultStationResponse =
                    UserUtil.getSecurityStationId(hotCacheProvider, employeeReaderRpcProvider, syncRpcWriterProvider, employeeId);
            securityStationId = employeeDefaultStationResponse.getReferenceNumber();
        } catch (Exception ex) {
            String dispatchToken = ServletRequestUtil.getDispatchToken(servletRequest);
            String content = JwtUtil.getUserIdFromToken(dispatchToken);

            /**
             * 如果通过token的方式来保持权限
             *
             */
            UserDetailResponse userDetailResponse =
                    this.jsonProvider.toObject(UserDetailResponse.class, content);
            if (userDetailResponse == null) {
                throw new UnAuthorizedTenantException();
            }

            employeeId = userDetailResponse.getEmployeeId();
        } finally {
            if (securityStationId == null || BusinessConstants.DefaultEmptyValue.equalsIgnoreCase(securityStationId)) {
                employeeDefaultStationResponse =
                        UserUtil.getSecurityStationId(hotCacheProvider, employeeReaderRpcProvider, syncRpcWriterProvider, employeeId);
                try {
                    securityStationId = employeeDefaultStationResponse.getReferenceNumber();
                } catch (NullPointerException ex) {
                    throw new NoFoundTenantException(String.format("该职员(%s)未安排驻勤点/岗点", employeeId));
                }
            }
        }

        AttendanceValidationRequest request =
                AttendanceValidationRequest.create(
                        employeeId,
                        restRequest.getRecordStatus(),
                        restRequest.getReferenceType(),
                        securityStationId,
                        restRequest.getNote(),
                        restRequest.getAddress(),
                        restRequest.getLatitude(),
                        restRequest.getLongitude(),
                        AgentHeaderOptionValue.create(
                                hAgent,
                                hDevice,
                                hEmpId,
                                hComId,
                                null
                        )
                );

        AttendanceValidationResponse validate = null;
        try {
            if (!employeeDefaultStationResponse.isSwitch2OtherStations()) {
                validate = doAttendanceValidation(securityStationId, employeeId, request);
            }
        } catch (NoFoundTenantException ex) {
            /**
             * 后备操作
             */
            this.hotCacheProvider.delete(CacheKeyManage.getHotAttStationBasicKey(securityStationId));

            validate = doAttendanceValidation(securityStationId, employeeId, request);

            logger.error("非预期情况: 该人员的缓存信息无效:securityStationId={};employeeId={}", securityStationId, employeeId, ex);

            throw ex;
        }

        if(employeeDefaultStationResponse.isCheckOtherStations()) {
            Collection<WorkstationResponse> workstationResponses =
                    this.workstationRpcProvider.getWorkstationsByEmployeeId(employeeId);
            if (employeeDefaultStationResponse.isSwitch2OtherStations()) {
                WorkstationResponse firstWorkStation = workstationResponses.stream().findFirst().orElse(null);
                if (firstWorkStation == null) {
                    throw new NoFoundTenantException(String.format("该职员(%s)未安排驻勤点/岗点/点位任务", employeeId));
                }

                validate = AttendanceValidationResponse.create(
                        firstWorkStation.getId(),
                        firstWorkStation.getName(),
                        firstWorkStation.getAddress(),
                        firstWorkStation.getLatitude(),
                        firstWorkStation.getLongitude(),
                        firstWorkStation.getPerformRange(),
                        this.attendanceRpcProvider.checkIfEnableFaceValidation(),
                        false
                );
                validate.generateActionCode(employeeId, firstWorkStation.getId());
            }

            boolean ignoredPerformRangeLimited = validate.isIgnoredPerformRangeLimited();
            String finalEmployeeId = employeeId;
            Collection<AttendanceValidationResponse.WorkstationValidationResponse> matchedWorkstations =
                    workstationResponses.stream()
                            .map(ii -> AttendanceValidationResponse.WorkstationValidationResponse.create(
                                    finalEmployeeId,
                                    ii, ignoredPerformRangeLimited
                            )).collect(Collectors.toList());
            validate.assignWorkstations(matchedWorkstations);
        }

        return this.ok(validate);
    }

    @Operation(
            summary = "新增签到记录", description = "类型默认驻勤点，状态根据签到或签退传",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken"),
                    @Parameter(in = ParameterIn.QUERY, required = true, name = "permission",
                            description = "固定传:advance 表示根据用户当前的权限而非依赖于调度台来实现权限管理; 不支持监管方")
            }
    )
    @PostMapping
    public ResponseEntity post(@RequestBody AttendanceWriterRequest request,
                               @RequestHeader(name = "bcx-agent",defaultValue = "",required = false) String hAgent,
                               @RequestHeader(name = "bcx-device",defaultValue = "",required = false) String hDevice,
                               @RequestHeader(name = "bcx-emp-id",defaultValue = "",required = false) String hEmpId,
                               @RequestHeader(name = "bcx-com-id",defaultValue = "",required = false) String hComId,
                               ServletRequest servletRequest) {

        String employeeId = getCurrentEmployeeId();
        String securityStationId = request.getReferenceNumber();
        EmployeeDefaultStationResponse employeeDefaultStationResponse = null;
        if(BusinessConstants.DefaultEmptyValue.equalsIgnoreCase(securityStationId)) {
            throw new BadTenantException(String.format("非预期行为, 系统参数(stationId=%s)异常", securityStationId));
        }

        /**
         * 直接以客户端传入的参数作为驻勤点/岗位/点位信息进行签到
         */
        if(false) {
            /**
             * 跟踪情况
             */
            if (false) {
                try {
                    securityStationId = securityStationId.toString();

                    if (BusinessConstants.DefaultEmptyValue.equalsIgnoreCase(securityStationId)) {
                        /**
                         * 为了兼容APP端一旦没传出过来的情况
                         */
                        employeeDefaultStationResponse =
                                UserUtil.getSecurityStationId(hotCacheProvider, employeeReaderRpcProvider, syncRpcWriterProvider, employeeId);
                        securityStationId = employeeDefaultStationResponse.getReferenceNumber();
                    }
                } catch (Exception ex) {
                    /**
                     * 为了兼容APP端一旦没传出过来的情况
                     */
                    employeeDefaultStationResponse =
                            UserUtil.getSecurityStationId(hotCacheProvider, employeeReaderRpcProvider, syncRpcWriterProvider, employeeId);
                    securityStationId = employeeDefaultStationResponse.getReferenceNumber();
                }
            } else {
                employeeDefaultStationResponse =
                        UserUtil.getSecurityStationId(hotCacheProvider, employeeReaderRpcProvider, syncRpcWriterProvider, employeeId);
                String stationId = employeeDefaultStationResponse.getReferenceNumber();
                try {
                    if (!StringUtil.isEqual(stationId, securityStationId)) {
                        logger.error("异常情况: APP传入的驻勤点({})与实际人员的驻勤点的Id({})不一致", securityStationId, stationId);
                    }
                } catch (Exception ex) {
                    //todo
                    ex.printStackTrace();
                }


                securityStationId = stationId;
            }
        }

        request.setHeaderOption(
                AgentHeaderOptionValue.create(
                        hAgent,
                        hDevice,
                        hEmpId,
                        hComId,
                        request.getActionCode()
                )
        );

        WrappedAttendanceWriterRequest wrappedAttendanceWriterRequest = WrappedAttendanceWriterRequest.create(
                employeeId,
                securityStationId,
                request
        );

        this.attendanceRpcProvider.create(wrappedAttendanceWriterRequest);

        return this.ok();
    }

    @Operation(summary = "获取最后一次签到状态", description = "自动获取当前用户的信息",
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken"),
                    @Parameter(in = ParameterIn.QUERY, required = true, name = "permission",
                            description = "固定传:advance 表示根据用户当前的权限而非依赖于调度台来实现权限管理; 不支持监管方")
            },
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功驻勤点信息")
            }
    )
    @GetMapping("/lastStatus")
    public ResponseEntity get() {
        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException();
        }

        EmployeeDutyStatusCache statusCache =
                this.hotCacheProvider.get(CacheKeyManage.getUserCurrentDutyStatus(userModel.getEmployeeId()), EmployeeDutyStatusCache.class);
        if (statusCache == null) {
            EmployeeDetailResponse employeeDetailResponse =
                    this.employeeReaderRpcProvider.get(userModel.getEmployeeId(), DeskType.Normal);
            if (employeeDetailResponse != null) {
                statusCache =
                        EmployeeDutyStatusCache.create(employeeDetailResponse.getDutyStatus(), employeeDetailResponse.getSecurityStationId());

                this.hotCacheProvider.put(CacheKeyManage.getUserCurrentDutyStatus(userModel.getEmployeeId()), 10 * 60,statusCache,false);
            }
        }


        /**
         * 未签到
         */
        int status = 1;

        if (statusCache != null && statusCache.getDutyStatus() != null) {
            status = statusCache.getDutyStatus().getValue();
        }

        return this.ok(status);
    }

    @Operation(summary = "根据日期和雇员ID查询当日签到记录", description = "签到日期必填，WEB端监管查询时雇员ID必填，按签到时间降序",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回雇员当日签到记录.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken"),
                    @Parameter(in = ParameterIn.QUERY, required = true, name = "permission",
                            description = "固定传:advance 表示根据用户当前的权限而非依赖于调度台来实现权限管理; 不支持监管方")
            }
    )
    @PostMapping("/searchDay")
    public ResponseEntity<Collection<AttendanceDetailReaderResponse>> getByDay(@RequestBody AttendanceDaySearchRequest request) {
        Collection<AttendanceDetailReaderResponse> data =
                this.attendanceRpcProvider.getByDay(request.getTenantEmployeeId(),request.getCreatedDate());
        if (data == null) {
            return this.notFound();
        }

        return this.ok(data);
    }


    @Operation(summary = "业务管理/手机端-搜索签到列表", description = "根据过滤条件查询签到信息列表，按最新签到日期降序 ",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回签到信息列表及驻勤点信息列表")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken"),
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "permission",
                            description = "固定传:advance 表示根据用户当前的权限而非依赖于调度台来实现权限管理; 不支持监管方")
            }
    )
    @PostMapping("/search")
    public ResponseEntity<EntityCollection<AttendanceEmpReaderResponse>> search(@RequestBody AttendanceSearchRequest request) throws ParseException {
        EntityCollection<AttendanceEmpReaderResponse> data =
                this.attendanceRpcProvider.search(request);

        return this.ok(data);
    }

    @Operation(summary = "手机端-我的记录信息", description = "过滤我的签到信息 ",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回签到信息列表及驻勤点信息列表")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken"),
                    @Parameter(in = ParameterIn.QUERY, required = true, name = "permission",
                            description = "固定传:advance 表示根据用户当前的权限而非依赖于调度台来实现权限管理; 不支持监管方")
            }
    )
    @PostMapping("/my-records")
    public ResponseEntity<EntityCollection<MyAttendanceRecordResponse>> postMy_records(@RequestBody AttendanceAppSearchRequest request) {
        String employeeId = getCurrentEmployeeId();

        Collection<MyAttendanceRecordResponse> data =
                this.attendanceRpcProvider.searchMyRecords(MyAttendanceSearchRequest.create(request.getSearchModel(),
                        employeeId, request.getPageIndex(), request.getPageSize()));

        return this.ok(EntityCollection.create(data, request.getPageSize(), (long) data.size()));
    }

    @Operation(summary = "业务管理-分组搜索签到列表", description = "根据过滤条件查询签到信息列表，按照人员进行分组查询 ",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回签到信息列表及驻勤点信息列表")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken"),
                    @Parameter(in = ParameterIn.QUERY, required = true, name = "permission",
                            description = "固定传:advance 表示根据用户当前的权限而非依赖于调度台来实现权限管理; 不支持监管方")
            }
    )
    @PostMapping("/search-group")
    public ResponseEntity<EntityCollection<AttendanceGroupSearchResponse>>
    searchByGroup(@RequestBody AttendanceGroupSearchRequest request) {
        EntityCollection<AttendanceGroupSearchResponse> data =
                this.attendanceRpcProvider.search(request);

        return this.ok(data);
    }

    @Operation(summary = "业务管理-查询指定人的签到/签退明细信息", description = "查询指定人的签到/签退明细信息",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回签到信息列表及驻勤点信息列表")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken"),
                    @Parameter(in = ParameterIn.QUERY, required = true, name = "permission",
                            description = "固定传:advance 表示根据用户当前的权限而非依赖于调度台来实现权限管理; 不支持监管方")
            }
    )
    @PostMapping("/search-group/{employeeId}")
    public ResponseEntity<EntityCollection<AttendanceGroupDetailSearchResponse>> searchSpecialUserDetail(
            @PathVariable String employeeId,
            @RequestBody AttendanceGroupDetailSearchRequest request){

        request.setEmployeeId(employeeId);
        EntityCollection<AttendanceGroupDetailSearchResponse> data =
                this.attendanceRpcProvider.search(request);

        return this.ok(data);
    }

    @Operation(summary = "导出-业务管理-分组搜索签到列表", description = "导出-根据过滤条件查询签到信息列表，按照人员进行分组查询-根据签到/签退数量降序 ",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回签到信息列表及驻勤点信息列表")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken"),
                    @Parameter(in = ParameterIn.QUERY, required = true, name = "permission",
                            description = "固定传:advance 表示根据用户当前的权限而非依赖于调度台来实现权限管理; 不支持监管方")
            }
    )
    @PostMapping("/export")
    public void export(@RequestBody AttendanceGroupSearchRequest request,
                       HttpServletResponse response) throws IOException {
        if (request.getCreatedStarDate() == null || request.getCreatedEndDate() == null) {
            throw new BadTenantException("调度时间范围不能为空");
        }

        String fileName = String.format("签到管理-%s", new SimpleDateFormat("yyyy-MM-dd").format(new Date()));
        request.setForExport(true);
        request.setPageSize(1000);
        ExcelExportUtil.export(response, AttendanceGroupSearchResponse.class, fileName, (excelWriter, sheet) -> {
            int pageIndex = 1;
            EntityCollection<AttendanceGroupSearchResponse> data = null;
            do {
                request.setPageIndex(pageIndex);
                data = this.attendanceRpcProvider.search(request);

                excelWriter.write(data.getData(), sheet);
                pageIndex++;
            } while (data != null && !data.endOfData());
        });
    }

    @Operation(summary = "导出-业务管理-查询指定人的签到/签退明细信息", description = "导出-查询指定人的签到/签退明细信息",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回签到信息列表及驻勤点信息列表")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken"),
                    @Parameter(in = ParameterIn.QUERY, required = true, name = "permission",
                            description = "固定传:advance 表示根据用户当前的权限而非依赖于调度台来实现权限管理; 不支持监管方")
            }
    )
    @PostMapping("/export-detail")
    public void export_detail(@RequestBody AttendanceGroupDetailSearchRequest request,
                       HttpServletResponse response) throws IOException {
        if (request.getCreatedStarDate() == null || request.getCreatedEndDate() == null) {
            throw new BadTenantException("调度时间范围不能为空");
        }

        String fileName = String.format("签到明细-%s", new SimpleDateFormat("yyyy-MM-dd").format(new Date()));
        request.setForExport(true);
        request.setPageSize(1000);
        ExcelExportUtil.export(response, AttendanceGroupDetailSearchResponse.class, fileName, (excelWriter, sheet) -> {
            int pageIndex = 1;
            EntityCollection<AttendanceGroupDetailSearchResponse> data = null;
            do {
                request.setPageIndex(pageIndex);
                data = this.attendanceRpcProvider.search(request);

                excelWriter.write(data.getData(), sheet);
                pageIndex++;
            } while (data != null && !data.endOfData());
        });
    }

    @Operation(summary = "根据业务类型进行搜索（根据任务类型和Id）", description = "根据业务类型进行搜索（根据任务类型和Id）",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回相关的签到信息")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken"),
                    @Parameter(in = ParameterIn.QUERY, required = true, name = "permission",
                            description = "固定传:advance 表示根据用户当前的权限而非依赖于调度台来实现权限管理; 不支持监管方")
            }
    )
    @Deprecated
    @PostMapping("/search-by-references")
    public ResponseEntity<Collection<AttendanceResponse>> searchByAttendanceType(@RequestBody AttendanceSearchByTypeRequest request) {
        Collection<AttendanceResponse> data = this.attendanceRpcProvider.search(request);

        return this.ok(data);
    }


    private AttendanceValidationResponse doAttendanceValidation(String securityStationId,String employeeId,AttendanceValidationRequest request) {
        AttendanceValidationResponse validate =
                hotCacheProvider.get(CacheKeyManage.getHotAttStationBasicKey(securityStationId), AttendanceValidationResponse.class);
        if (validate == null) {
            validate = this.attendanceRpcProvider.validate(request);

            hotCacheProvider.put(CacheKeyManage.getHotAttStationBasicKey(securityStationId), 24 * 60 * 60, validate, true);
        }

        /*
        validate = AttendanceValidationResponse.create(
                validate.getId(),
                validate.getName(),
                validate.getAddress(),
                validate.getLatitude(),
                validate.getLongitude(),
                validate.getPerformRange(),
                false,
                validate.isIgnoredPerformRangeLimited()
        );
         */

        validate.generateActionCode(employeeId, securityStationId);

        return validate;
    }

    private String getCurrentEmployeeId() {
        String employeeId = null;
        try {
            String accessToken = ServletRequestUtil.getAccessToken(servletRequest);
            employeeId = JwtUtil.getUserIdFromToken(accessToken);
        } catch (Exception ex) {
            String dispatchToken = ServletRequestUtil.getDispatchToken(servletRequest);
            String content = JwtUtil.getUserIdFromToken(dispatchToken);
            /**
             * 如果通过token的方式来保持权限
             *
             */
            UserDetailResponse userDetailResponse =
                    this.jsonProvider.toObject(UserDetailResponse.class, content);
            if (userDetailResponse == null) {
                throw new UnAuthorizedTenantException();
            }

            employeeId = userDetailResponse.getEmployeeId();
        }

        if(BusinessUtil.isOrgMember(employeeId)) {
            employeeId = BusinessUtil.extractMemberId(employeeId);
        }

        return employeeId;
    }
}
