package com.bcxin.tenant.open.rest.apis.components;

import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.exceptions.*;
import com.bcxin.tenant.open.infrastructures.utils.ExceptionUtil;
import com.bcxin.tenant.open.rest.apis.controllers.responses.ResponseBuilder;
import org.springframework.context.support.DefaultMessageSourceResolvable;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.HttpStatusCode;
import org.springframework.http.ResponseEntity;
import org.springframework.http.converter.HttpMessageConversionException;
import org.springframework.validation.FieldError;
import org.springframework.validation.ObjectError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.context.request.WebRequest;
import org.springframework.web.servlet.mvc.method.annotation.ResponseEntityExceptionHandler;

import java.util.Arrays;
import java.util.Collection;
import java.util.stream.Collectors;

@ControllerAdvice
public class RestResponseEntityExceptionHandler extends ResponseEntityExceptionHandler {
    private final JsonProvider jsonProvider;

    public RestResponseEntityExceptionHandler(JsonProvider jsonProvider) {
        this.jsonProvider = jsonProvider;
    }

    @ExceptionHandler
    public ResponseEntity exceptionHandle(Exception ex, WebRequest request) {
        boolean recordLog = false;
        try {
            MethodArgumentNotValidException invalidException = ExceptionUtil.cast(ex, MethodArgumentNotValidException.class);
            if (invalidException != null) {
                Collection<String> errors = invalidException.getBindingResult()
                        .getAllErrors().stream()
                        .map(ii -> {
                            StringBuilder msg = new StringBuilder();
                            Object[] arguments = ii.getArguments();
                            if (arguments != null && arguments.length > 0) {
                                String fieldName = Arrays.stream(arguments).map(arg -> {
                                    DefaultMessageSourceResolvable dmsr = (DefaultMessageSourceResolvable) arg;
                                    if (dmsr != null) {
                                        return dmsr.getDefaultMessage();
                                    }

                                    return null;
                                }).filter(arg -> arg != null).collect(Collectors.joining(","));

                                msg.append(fieldName);
                                msg.append(" ");
                            }

                            msg.append(ii.getDefaultMessage());

                            return msg.toString();
                        }).collect(Collectors.toList());

                recordLog = true;
                return ResponseBuilder.build(HttpStatus.BAD_REQUEST, errors);
            }

            DataOperationTenantException dtx = ExceptionUtil.cast(ex, DataOperationTenantException.class);
            if (dtx != null) {
                return ResponseBuilder.build(HttpStatus.METHOD_NOT_ALLOWED, dtx.getData(), dtx.getCode(), dtx.getMessage(), null);
            }

            UnAuthorizedTenantException authorizedException = ExceptionUtil.cast(ex, UnAuthorizedTenantException.class);
            if (authorizedException != null) {
                return ResponseBuilder.build(HttpStatus.UNAUTHORIZED, null, ex.getMessage(), null);
            }

            ForbidTenantException forbidException = ExceptionUtil.cast(ex, ForbidTenantException.class);
            if (forbidException != null) {
                return ResponseBuilder.build(HttpStatus.FORBIDDEN, null, forbidException.getMessage(), null);
            }

            NoFoundTenantException notFoundException = ExceptionUtil.cast(ex, NoFoundTenantException.class);
            if (notFoundException != null) {
                return ResponseBuilder.build(HttpStatus.NOT_FOUND, null, notFoundException.getMessage(), null);
            }

            NoAllowedTenantException notAllowedException = ExceptionUtil.cast(ex, NoAllowedTenantException.class);
            if (notAllowedException != null) {
                return ResponseBuilder.build(HttpStatus.METHOD_NOT_ALLOWED, null, notAllowedException.getMessage(), null);
            }

            ConflictTenantException conflictTenantException = ExceptionUtil.cast(ex, ConflictTenantException.class);
            if (conflictTenantException != null) {
                return ResponseBuilder.build(HttpStatus.BAD_REQUEST, null, conflictTenantException.getMessage(), null);
            }

            TenantExceptionAbstract tenantException = ExceptionUtil.cast(ex, TenantExceptionAbstract.class);
            if (tenantException != null) {
                return ResponseBuilder.build(HttpStatus.BAD_REQUEST, ExceptionUtil.getStackMessage(ex), tenantException.getMessage(), null);
            }

            String detailException = ExceptionUtil.getStackMessage(ex);
            if (detailException.contains("IdentityNotFoundException") || detailException.contains("该账号未注册")) {
                return ResponseBuilder.build(HttpStatus.BAD_REQUEST, detailException, "该账号未注册", null);
            }
            if (detailException.contains("IdentityNotMatchedException") || detailException.contains("账号/密码不正确")) {
                return ResponseBuilder.build(HttpStatus.BAD_REQUEST, detailException, "账号/密码不正确", null);
            }

            recordLog = true;
            if (detailException.contains("TenantException:") && detailException.contains("\n")) {
                int msgIndex = detailException.indexOf("TenantException:") + 16;
                int lastIndex = detailException.indexOf("\r");
                if (lastIndex > msgIndex && msgIndex >= 0) {
                    String msg = detailException.substring(msgIndex, lastIndex);
                    return ResponseBuilder.build(HttpStatus.BAD_REQUEST, null, String.format("2-%s-%s", msg, detailException), null);
                }
            }

            return ResponseBuilder.build(HttpStatus.BAD_REQUEST, ex.getMessage(), "系统异常, 请联系管理员", null);
        } finally {
            if (recordLog) {
                String detail = ExceptionUtil.getStackMessage(ex);
                logError("RestResponseEntityExceptionHandler.exceptionHandle", detail, request);
            }
        }
    }

    @Override
    protected ResponseEntity<Object> handleMethodArgumentNotValid(MethodArgumentNotValidException ex, HttpHeaders headers, HttpStatusCode status, WebRequest request) {
        ValidationErrorResponse validationErrorResponse = new ValidationErrorResponse("请求参数不符合要求");
        for (ObjectError error : ex.getBindingResult().getAllErrors()) {
            if (error instanceof FieldError) {
                FieldError fieldError = (FieldError) error;
                validationErrorResponse.addFieldMessage(fieldError.getField(), fieldError.getDefaultMessage(), fieldError.getRejectedValue());
            } else {
                validationErrorResponse.addFieldMessage(error.getObjectName(), error.getDefaultMessage());
            }
        }

        return ResponseBuilder.build(HttpStatus.BAD_REQUEST, validationErrorResponse);
    }



    @Override
    protected ResponseEntity<Object> handleExceptionInternal(Exception ex, Object body, HttpHeaders headers, HttpStatusCode status, WebRequest request) {
        ResponseEntity<Object> responseEntity = super.handleExceptionInternal(ex, body, headers, status, request);
        if (status == HttpStatus.BAD_REQUEST || status == HttpStatus.INTERNAL_SERVER_ERROR) {
            String detail = String.format("body=%s;exception=%s",
                    this.jsonProvider.getJson(body),
                    ExceptionUtil.getStackMessage(ex)
            );
            try {
                String friendMessage = "系统异常, 请联系管理员";
                if (ex instanceof HttpMessageConversionException) {
                    friendMessage = "参数格式不正确, 请联系管理员进行确认(02)";
                }

                return ResponseBuilder.build(HttpStatus.BAD_REQUEST, detail, friendMessage, null);
            } finally {
                logError("RestResponseEntityExceptionHandler.handleExceptionInternal", detail, request);
            }
        }

        return responseEntity;
    }

    private void logError(String title, String detail, WebRequest request) {

    }
}
