package com.bcxin.tenant.open.backend.tasks.configs;

import com.bcxin.tenant.open.infrastructures.enums.BillType;
import com.bcxin.tenant.open.infrastructures.enums.ContentType;
import com.bcxin.tenant.open.infrastructures.enums.DispatchDataType;
import com.bcxin.tenant.open.infrastructures.utils.ExceptionUtil;
import com.bcxin.tenant.open.jdks.*;
import com.bcxin.tenant.open.jdks.requests.DailyBillWriteRequest;
import com.bcxin.tenant.open.jdks.requests.MonthlyBillWriteRequest;
import com.bcxin.tenant.open.jdks.requests.SyncParameterWrapperRequest;
import com.bcxin.tenant.open.jdks.requests.TencentCalculateCallLogRequest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.BeanFactoryAware;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Configuration;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;

@Configuration
@EnableScheduling
public class ScheduleConfig  implements BeanFactoryAware {
    private static final Logger logger = LoggerFactory.getLogger(ScheduleConfig.class);
    private static final int SCHEDULE_PAGE_SIZE = 800;

    private BeanFactory beanFactory;

    @Override
    public void setBeanFactory(BeanFactory beanFactory) throws BeansException {
        this.beanFactory = beanFactory;
    }

    /**
     * SELECT
     * ( CASE WHEN l.event_type = 101 THEN '创建房间' WHEN l.event_type = 102 THEN '解散房间'
     * when l.event_type = 103 THEN '进入房间' when l.event_type = 104 THEN '退出房间'
     * when l.event_type = 105 THEN '切换角色'	when l.event_type = 201 THEN '开始推送视频数据'
     * when l.event_type = 202 THEN '停止推送视频数据'	when l.event_type = 203 THEN '开始推送音频数据'
     * when l.event_type = 204 THEN '停止推送音频数据'	when l.event_type = 205 THEN '开始推送辅路数据'
     * when l.event_type = 206 THEN '停止推送辅路数据'	end) as xx,
     * l.event_info,l.event_group_id,l.callback_ts,l.processed_status,l.last_processed_result,
     * l.room_id
     * FROM
     * tencent_callback_logs l order by l.room_id desc,l.created_time desc limit 100;
     */
    @Scheduled(fixedDelay = 1 * 30 * 1000)
    public void doCalculateTencentCost() {
        try {
            TencentCallbackLogRpcProvider tencentCallbackLogRpcProvider =
                    this.beanFactory.getBean(TencentCallbackLogRpcProvider.class);
            RoomRpcProvider roomRpcProvider = this.beanFactory.getBean(RoomRpcProvider.class);
            Collection<Long> roomIds = roomRpcProvider.getReadyForCalculatedRoomIds(100);

            if (!CollectionUtils.isEmpty(roomIds)) {
                TencentCalculateCallLogRequest
                        request = new TencentCalculateCallLogRequest();
                request.setRoomIds(roomIds);

                tencentCallbackLogRpcProvider.execute(request);

                /*
                logger.error("2.定时执行doCalculateTencentCost 成功:{}",
                        roomIds.stream().map(ii -> String.valueOf(ii)).collect(Collectors.joining(";")));

                 */
            }
        } catch (Exception ex) {
            logger.error("2.定时执行doCalculateTencentCost异常", ex);
        }
    }

    @Scheduled(cron = "${scheduled.cron.stationCheck}")
    public void updateStationStatus() {
        if (!isAllowed()) {
            logger.error("not allow to auto refresh station");
            return;
        }

        RdSyncRpcWriterProvider syncRpcWriterProvider = this.beanFactory.getBean(RdSyncRpcWriterProvider.class);

        Collection<String> ids = syncRpcWriterProvider.getRelativeIds(ContentType.Station, 0, SCHEDULE_PAGE_SIZE);

        long totalCount = ids.size();
        /**
         * 每次请求; 刷新检查最多10万次
         */
        for (int index = 1; index <= 200; index++) {
            if (!CollectionUtils.isEmpty(ids)) {
                try {
                    SyncParameterWrapperRequest wrapperRequest = SyncParameterWrapperRequest.create(DispatchDataType.Station,ids);
                    wrapperRequest.assignFromAutoExpiredCheck();
                    wrapperRequest.setAdditionalParameter(ids);
                    syncRpcWriterProvider.sync(wrapperRequest);

                    logger.warn("double refresh station with size={}", ids.size());
                } catch (Exception ex) {
                    logger.error("更新驻勤点(pageIndex={})发生异常:{}",
                            index, ids.stream().collect(Collectors.joining(",")), ex);
                } finally {
                    if (ids.size() < SCHEDULE_PAGE_SIZE) {
                        ids.clear();
                        break;
                    }
                    ids.clear();
                }

                ids = syncRpcWriterProvider.getRelativeIds(ContentType.Station, index, SCHEDULE_PAGE_SIZE);
                totalCount += ids.size();
            } else {
                break;
            }
        }

        logger.error("refresh the station status-total count-{}", totalCount);
    }

    @Scheduled(cron = "${scheduled.cron.stationCheck}")
    public void updateEmployeeStatus() {
        RdSyncRpcWriterProvider syncRpcWriterProvider = this.beanFactory.getBean(RdSyncRpcWriterProvider.class);

        Collection<String> ids = syncRpcWriterProvider.getRelativeIds(ContentType.Employee, 0, SCHEDULE_PAGE_SIZE);
        Collection<String> memberIds = syncRpcWriterProvider.getRelativeIds(ContentType.Member, 0, SCHEDULE_PAGE_SIZE);
        long totalCount = ids.size();
        /**
         * 每天晚上再次核对一下人员数据; 刷新检查最多100万次
         */
        for (int index = 1; index <= 2000; index++) {
            if (!isAllowed()) {
                logger.error("not allow to auto refresh employee");
                return;
            }
            if (!CollectionUtils.isEmpty(ids)) {
                boolean isEmployeeEnd = false;
                boolean isMemberEnd = false;
                try {
                    syncRpcWriterProvider.sync(SyncParameterWrapperRequest.create(DispatchDataType.Employee,ids));
                    syncRpcWriterProvider.sync(SyncParameterWrapperRequest.create(DispatchDataType.Member,memberIds));
                    logger.warn("double refresh employee-member with size={}", ids.size());
                } catch (Exception ex) {
                    logger.error("Failed-updateEmployeeStatus-更新无效人员(pageIndex={})发生异常:{}", index,
                            ids.stream().collect(Collectors.joining(",")), ex);
                } finally {
                    isMemberEnd = memberIds.size() < SCHEDULE_PAGE_SIZE;
                    isEmployeeEnd = ids.size() < SCHEDULE_PAGE_SIZE;
                    if (isEmployeeEnd && isMemberEnd) {
                        ids.clear();
                        memberIds.clear();
                        break;
                    }
                    ids.clear();
                    memberIds.clear();
                }

                if(!isEmployeeEnd) {
                    ids = syncRpcWriterProvider.getRelativeIds(ContentType.Employee, index, SCHEDULE_PAGE_SIZE);
                }

                if(!isMemberEnd) {
                    memberIds = syncRpcWriterProvider.getRelativeIds(ContentType.Member, index, SCHEDULE_PAGE_SIZE);
                }

                totalCount += ids.size();
            } else {
                logger.error("updateEmployeeStatus-未找到无效人员数据-自动结束");
                break;
            }
        }

        logger.error("refresh the employee-member-total count-{}", totalCount);
    }

    @Scheduled(cron = "${scheduled.cron.stationCheck}")
    public void updateCompanyStatus() {
        RdSyncRpcWriterProvider syncRpcWriterProvider = this.beanFactory.getBean(RdSyncRpcWriterProvider.class);

        Collection<String> ids = syncRpcWriterProvider.getRelativeIds(ContentType.Organization, 0, SCHEDULE_PAGE_SIZE);
        long totalCount = ids.size();
        /**
         * 每次请求; 刷新检查最多10万次
         */
        for (int index = 1; index <= 200; index++) {
            if (!isAllowed()) {
                logger.error("not allow to auto refresh company");
                return;
            }

            if (!CollectionUtils.isEmpty(ids)) {
                try {
                    syncRpcWriterProvider.sync(SyncParameterWrapperRequest.create(DispatchDataType.Company,ids));
                } catch (Exception ex) {
                    logger.error("update organization (pageIndex={})发生异常:{}",
                            index, ids.stream().collect(Collectors.joining(",")),
                            ex);
                } finally {
                    if (ids.size() < SCHEDULE_PAGE_SIZE) {
                        ids.clear();
                        break;
                    }
                    ids.clear();
                }

                ids = syncRpcWriterProvider.getRelativeIds(ContentType.Organization, index, SCHEDULE_PAGE_SIZE);
                totalCount += ids.size();
            } else {
                break;
            }
        }

        logger.error("refresh the organization status-total count-{}", totalCount);
    }

    @Scheduled(cron = "${scheduled.cron.stationCheck}")
    public void updateProjectsStatus() {
        if (!isAllowed()) {
            logger.error("not allow to auto refresh station");
            return;
        }

        RdSyncRpcWriterProvider syncRpcWriterProvider = this.beanFactory.getBean(RdSyncRpcWriterProvider.class);

        Collection<String> ids = syncRpcWriterProvider.getRelativeIds(ContentType.EventProject, 0, SCHEDULE_PAGE_SIZE);

        long totalCount = ids.size();
        /**
         * 每次请求; 刷新检查最多10万次
         */
        for (int index = 1; index <= 200; index++) {
            if (!CollectionUtils.isEmpty(ids)) {
                try {
                    SyncParameterWrapperRequest wrapperRequest =
                            SyncParameterWrapperRequest.create(DispatchDataType.TemporaryProtectionProject,ids);
                    wrapperRequest.assignFromAutoExpiredCheck();
                    wrapperRequest.setAdditionalParameter(ids);
                    syncRpcWriterProvider.sync(wrapperRequest);

                    logger.warn("double refresh event project with size={}", ids.size());
                } catch (Exception ex) {
                    logger.error("更新临保/赛演项目(pageIndex={})发生异常:{}, 异常={}", index, ids.stream().collect(Collectors.joining(",")), ExceptionUtil.getStackMessage(ex));
                } finally {
                    if (ids.size() < SCHEDULE_PAGE_SIZE) {
                        ids.clear();
                        break;
                    }
                    ids.clear();
                }

                ids = syncRpcWriterProvider.getRelativeIds(ContentType.EventProject, index, SCHEDULE_PAGE_SIZE);
                totalCount += ids.size();
            } else {
                break;
            }
        }

        logger.error("refresh the event project status-total count-{}", totalCount);
    }

    /**
     * 仅在22点到7点之间执行
     *
     * @return
     */
    private boolean isAllowed() {
        int hour = Calendar.getInstance().get(Calendar.HOUR_OF_DAY);

        return hour >= 23 || hour <= 5;
    }

    /**
     * 每天凌晨00：00：01，获取前一天的日账单
     */
    @Scheduled(cron = "0 0 * * * ?")
    public void generateDailyBills() {
        Date date = new Date();
        //Date date = new Date(123,8,11);
        logger.error("Begin daily bill calculation, date= {}", date);
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.DAY_OF_MONTH, -1);

        DailyBillRpcProvider dailyBillRpcProvider = this.beanFactory.getBean(DailyBillRpcProvider.class);
        try {
            dailyBillRpcProvider.billGenerated(DailyBillWriteRequest.create(calendar.getTime(),
                    Arrays.asList(BillType.Attendance, BillType.RollCall,BillType.Track, BillType.Fence)));
        } catch (Exception e) {
            logger.error("Exception occurs when calculating daily bill. date= {}, exception={}", new Date(), e);
        }

        logger.error("End daily bill calculation, date= {}", new Date());
    }

    /**
     * 每天凌晨03：00：00，获取月度账单
     */
    @Scheduled(cron = "0 30 * * * ?")
    public void generateMonthlyBills() {
        Date date = new Date();
        //Date date = new Date(124,1,26);
        logger.error("Begin monthly bill calculation, date= {}", date);
        MonthlyBillRpcProvider dailyBillRpcProvider = this.beanFactory.getBean(MonthlyBillRpcProvider.class);
        try {
            dailyBillRpcProvider.billsGenerate(MonthlyBillWriteRequest.create(date,
                    Arrays.asList(BillType.Attendance, BillType.RollCall, BillType.Track, BillType.Fence)));
        } catch (Exception e) {
            logger.error("Exception occurs when calculating monthly bill. date= {}, exception={}", new Date(), e);
        }
        logger.error("End monthly bill calculation, date= {}", new Date());
    }

    @Scheduled(cron = "${scheduled.cron.examCheck}")
    public void updateExamSite() {
        RdSyncRpcWriterProvider syncRpcWriterProvider = this.beanFactory.getBean(RdSyncRpcWriterProvider.class);
        executeDoubleCheckFlow((pageIndex) -> {
            return syncRpcWriterProvider.getRelativeIds(ContentType.ExamSite, pageIndex, SCHEDULE_PAGE_SIZE);
        }, ids -> {
            syncRpcWriterProvider.sync(SyncParameterWrapperRequest.create(DispatchDataType.ExamSite,ids));
        }, "company");
    }

    @Scheduled(cron = "${scheduled.cron.examCheck}")
    public void updateExamSiteRoom() {
        RdSyncRpcWriterProvider syncRpcWriterProvider = this.beanFactory.getBean(RdSyncRpcWriterProvider.class);
        executeDoubleCheckFlow((pageIndex) -> {
            return syncRpcWriterProvider.getRelativeIds(ContentType.ExamRoom, pageIndex, SCHEDULE_PAGE_SIZE);
        }, ids -> {
            syncRpcWriterProvider.sync(SyncParameterWrapperRequest.create(DispatchDataType.ExamRoom,ids));
        }, "company");
    }


    @Scheduled(cron = "${scheduled.cron.examCheck}")
    public void updateSystemExamInfo() {
        RdSyncRpcWriterProvider syncRpcWriterProvider = this.beanFactory.getBean(RdSyncRpcWriterProvider.class);
        executeDoubleCheckFlow((pageIndex) -> {
            return syncRpcWriterProvider.getRelativeIds(ContentType.SystemExamInfo, pageIndex, SCHEDULE_PAGE_SIZE);
        }, ids -> {
            syncRpcWriterProvider.sync(SyncParameterWrapperRequest.create(DispatchDataType.SystemExamInfo,ids));
        }, "company");
    }

    private void executeDoubleCheckFlow(
            Function<Integer,Collection<String>> extractIds,
            Consumer<Collection<String>> consumer, String caption) {
        logger.error("begin to auto refresh {}", caption);

        Collection<String> ids = extractIds.apply(0);
        long totalCount = ids.size();
        /**
         * 每次请求; 刷新检查最多100万次
         */
        for (int index = 1; index <= 2000; index++) {
            if (!isAllowed()) {
                logger.error("not allow to auto refresh company");
                return;
            }

            if (!CollectionUtils.isEmpty(ids)) {
                try {
                    consumer.accept(ids);
                    logger.error("double refresh {} with size={}", caption, ids.size());
                } catch (Exception ex) {
                    logger.error("update {} (pageIndex={})发生异常:{},ids={}, 异常={}",
                            caption,
                            index, ids.stream().collect(Collectors.joining(",")),
                            ExceptionUtil.getStackMessage(ex));
                    ex.printStackTrace();
                } finally {
                    /**
                     * 如果数量小于页码, 则直接跳出
                     */
                    if (ids.size() < SCHEDULE_PAGE_SIZE) {
                        ids.clear();
                        break;
                    }
                    ids.clear();
                }

                ids = extractIds.apply(index);
                totalCount += ids.size();
            } else {
                break;
            }
        }

        logger.error("refresh the {} status-total count-{}", caption, totalCount);
    }
}