package com.bcxin.tenant.open.backend.tasks.configs;

import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.constants.KafkaConstants;
import com.bcxin.tenant.open.infrastructures.enums.DispatchDataType;
import com.bcxin.tenant.open.infrastructures.exceptions.BadTenantException;
import com.bcxin.tenant.open.infrastructures.exceptions.NotSupportTenantException;
import com.bcxin.tenant.open.jdks.DispatchDataScopeRpcProvider;
import com.bcxin.tenant.open.jdks.RdSyncRpcWriterProvider;
import com.bcxin.tenant.open.jdks.SecurityStationReaderRpcProvider;
import com.bcxin.tenant.open.jdks.requests.SyncParameterWrapperRequest;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.kafka.annotation.KafkaListener;
import org.springframework.kafka.support.Acknowledgment;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

@Component
public class KafkaComponent_EventProject extends KafkaComponentAbstract{
    private static final Logger logger = LoggerFactory.getLogger(KafkaComponent_EventProject.class);
    protected KafkaComponent_EventProject(JsonProvider jsonProvider,
                                          SecurityStationReaderRpcProvider securityStationReaderRpcProvider,
                                          DispatchDataScopeRpcProvider dispatchDataScopeRpcProvider,
                                          RdSyncRpcWriterProvider syncRpcWriterProvider) {
        super(jsonProvider, securityStationReaderRpcProvider, dispatchDataScopeRpcProvider, syncRpcWriterProvider);
    }

    /**
     * 赛演的项目
     * @param records
     * @param ack
     */
    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-TOPIC_BAIBAODUNFLOW_TLK_EVENT_PROJECT",
            topics = {
                    KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_EVENT_PROJECT
            }, groupId = "${spring.kafka.consumer.group-id}-TOPIC_BAIBAODUNFLOW_TLK_EVENT_PROJECT")
    public void ackEventProjectListener(List<ConsumerRecord<String, String>> records,
                                                                                    Acknowledgment ack) {
        executeConsumeData(syncRpcWriterProvider, records, ack);
    }

    /**
     * 赛演的管理单位
     * @param records
     * @param ack
     */
    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-TOPIC_BAIBAODUNFLOW_TLK_EVENT_TEAM",
            topics = {
                    KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_EVENT_TEAM
            }, groupId = "${spring.kafka.consumer.group-id}-TOPIC_BAIBAODUNFLOW_TLK_EVENT_TEAM")
    public void ackEventTeamListener(List<ConsumerRecord<String, String>> records,
                                                                                    Acknowledgment ack) {
        executeConsumeData(syncRpcWriterProvider, records, ack);
    }

    /**
     * 赛演的管理单位所管理的岗点信息
     * @param records
     * @param ack
     */
    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-TOPIC_BAIBAODUNFLOW_TLK_EVENT_TEAM_POWER",
            topics = {
                    KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_EVENT_TEAM_POWER
            }, groupId = "${spring.kafka.consumer.group-id}-TOPIC_BAIBAODUNFLOW_TLK_EVENT_TEAM_POWER")
    public void ackEventTeamPowerListener(List<ConsumerRecord<String, String>> records,
                                                              Acknowledgment ack) {
        executeConsumeData(syncRpcWriterProvider, records, ack);
    }

    /**
     * 赛演项目的负责人信息
     * @param records
     * @param ack
     */
    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-TOPIC_BAIBAODUNFLOW_TLK_EVENT_MANAGER",
            topics = {
                    KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_EVENT_MANAGER
            }, groupId = "${spring.kafka.consumer.group-id}-TOPIC_BAIBAODUNFLOW_TLK_EVENT_MANAGER")
    public void ackEventManagerPowerListener(List<ConsumerRecord<String, String>> records,
                                          Acknowledgment ack) {
        logger.error("赛眼项目负责人信息:{}", records.stream().map(ii -> ii.value()).collect(Collectors.joining(",")));
        executeConsumeData(syncRpcWriterProvider, records, ack);
    }

    /**
     * 赛演项目的岗点负责人
     * @param records
     * @param ack
     */
    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-TOPIC_BAIBAODUNFLOW_TLK_EVENT_POST_MANAGER",
            topics = {
                    KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_EVENT_POST_MANAGER
            }, groupId = "${spring.kafka.consumer.group-id}-TOPIC_BAIBAODUNFLOW_TLK_EVENT_POST_MANAGER")
    public void ackEventPostManagerPowerListener(List<ConsumerRecord<String, String>> records,
                                             Acknowledgment ack) {
        executeConsumeData(syncRpcWriterProvider, records, ack);
    }

    /**
     * 赛演的人员
     * @param records
     * @param ack
     */
    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-TOPIC_OBPM2_EXTERNAL_MEMBERS",
            topics = {
                    KafkaConstants.TOPIC_OBPM2_EXTERNAL_MEMBERS
            }, groupId = "${spring.kafka.consumer.group-id}-TOPIC_OBPM2_EXTERNAL_MEMBERS")
    public void ackExternalManagerPowerListener(List<ConsumerRecord<String, String>> records,
                                                Acknowledgment ack) {
        executeConsumeData(syncRpcWriterProvider, records, ack);
    }

    /**
     * 协会项目管理
     * @param records
     * @param ack
     */
    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-TOPIC_BAIBAODUNFLOW_TLK_ADD_WORK_DEMAND",
            topics = {
                    KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_ADD_WORK_DEMAND
            }, groupId = "${spring.kafka.consumer.group-id}-TOPIC_BAIBAODUNFLOW_TLK_ADD_WORK_DEMAND")
    public void ackExternalADD_WORK_DEMANDListener(List<ConsumerRecord<String, String>> records,
                                                 Acknowledgment ack) {
        executeConsumeData(syncRpcWriterProvider, records, ack);
    }

    @Override
    protected void executeConsumeData(RdSyncRpcWriterProvider syncRpcWriterProvider,
                                      List<ConsumerRecord<String, String>> records,
                                      Acknowledgment ack) {
        Collection<String> topics = records.stream().map(ix -> ix.topic()).distinct()
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(topics)) {
            return;
        }

        DispatchDataType dataType = DispatchDataType.TemporaryProtectionProject;
        boolean allowedToAck = true;
        Exception lastException = null;
        try {
            for (String topic : topics) {
                Collection<String> ids = new ArrayList<>();
                try {
                    List<ConsumerRecord<String, String>> selectedRecords = getMatchRecords(records,topic);
                    if (CollectionUtils.isEmpty(selectedRecords)) {
                        continue;
                    }

                    ids = extractIdsFromRecords(selectedRecords);

                    Object additionalParameter = null;
                    switch (topic) {
                        case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_EVENT_PROJECT:
                            dataType = DispatchDataType.TemporaryProtectionProject;
                            break;
                        case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_EVENT_TEAM:
                            dataType = DispatchDataType.EventOrganizerJoinProject;
                            ids = selectedRecords.stream().flatMap(ii->{
                                Collection<String> personIds = extractFieldFromJson(ii.value(),"ITEM_TEAM_ID");
                                if(personIds==null) {
                                    personIds = new ArrayList<>();
                                }

                                return personIds.stream();
                            }).collect(Collectors.toList());
                            break;
                        case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_ADD_WORK_DEMAND:
                            dataType = DispatchDataType.EventOrganizerJoinProject;
                            ids = selectedRecords.stream().flatMap(ii->{
                                Collection<String> projectIds = extractFieldFromJson(ii.value(),"ITEM_PROJECT_ID");
                                if(projectIds == null) {
                                    projectIds = new ArrayList<>();
                                }

                                return projectIds.stream();
                            }).collect(Collectors.toList());
                            break;
                        case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_EVENT_TEAM_POWER:
                            dataType = DispatchDataType.EventOrganizerLimitedResource;
                            ids = selectedRecords.stream().flatMap(ii->{
                                Collection<String> teamIds = extractFieldFromJson(ii.value(),"ITEM_TEAM_ID");
                                if(teamIds==null) {
                                    teamIds = new ArrayList<>();
                                }

                                return teamIds.stream();
                            }).collect(Collectors.toList());
                            break;
                        case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_EVENT_MANAGER:
                        case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_EVENT_POST_MANAGER:
                            dataType = DispatchDataType.EventOrganizerStationManager;
                            ids = selectedRecords.stream().flatMap(ii->{
                                Collection<String> personIds = extractFieldFromJson(ii.value(),"ITEM_person_id");
                                if(personIds==null) {
                                    personIds = new ArrayList<>();
                                }

                                return personIds.stream();
                            }).collect(Collectors.toList());
                            break;
                        case KafkaConstants.TOPIC_OBPM2_EXTERNAL_MEMBERS:
                            dataType = DispatchDataType.Member;
                            break;
                        case KafkaConstants.TOPIC_DEAD_LETTER_DISPATCH: {
                            /**
                             * 针对死信; 不支持如下操作后直接返回
                             */
                            handleDeadLetterQueueData(syncRpcWriterProvider, jsonProvider, selectedRecords);
                            return;
                        }

                        default:
                            throw new NotSupportTenantException(String.format("1.不支持该主题(%s)信息", topic));
                    }

                    Collection<String> finalIds = ids;
                    DispatchDataType finalDataType = dataType;
                    if(finalDataType==DispatchDataType.EventOrganizerJoinProject ||
                            finalDataType==DispatchDataType.DispatchDataSource ||
                            finalDataType==DispatchDataType.EventOrganizerStationManager ||
                            finalDataType == DispatchDataType.TemporaryProtectionProjectPersonType) {
                        additionalParameter = finalIds;
                    }

                    syncRpcWriterProvider.sync(SyncParameterWrapperRequest.create(finalDataType, finalIds, additionalParameter));
                } catch (Exception ex) {
                    lastException = ex;
                    allowedToAck = false;
                    logger.error(
                            "KafkaComponent_EventProject: failed to consumer data with topic={};data={}",
                            topic,
                            (ids == null ? "NULL" : ids.stream().map(ii -> String.format("'%s'", ii))
                                    .collect(Collectors.joining(","))),
                            ex);
                }
            }
        } catch (Exception ex) {
            lastException = ex;
            allowedToAck = false;
            logger.error("刷新赛演项目信息异常",ex);
        } finally {
            if (allowedToAck) {
                ack.acknowledge();
            } else {
                if (lastException != null) {
                    throw new BadTenantException(
                            String.format("failed to cost topic=%s", topics.stream().collect(Collectors.joining(";"))),
                            lastException);
                }
            }
        }
    }
}
