package com.bcxin.tenant.open.backend.tasks.configs;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bcxin.tenant.open.backend.tasks.properties.RtcDeviceProperty;
import com.bcxin.tenant.open.backend.tasks.utils.DebeziumUtil;
import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.constants.KafkaConstants;
import com.bcxin.tenant.open.infrastructures.enums.DispatchDataType;
import com.bcxin.tenant.open.infrastructures.exceptions.BadTenantException;
import com.bcxin.tenant.open.jdks.DeviceWriterRpcProvider;
import com.bcxin.tenant.open.jdks.RdSyncRpcWriterProvider;
import com.bcxin.tenant.open.jdks.ThirdDeviceRpcProvider;
import com.bcxin.tenant.open.jdks.requests.SyncParameterWrapperRequest;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.kafka.annotation.KafkaListener;
import org.springframework.kafka.support.Acknowledgment;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

@Component
public class KafkaComponent_Device {
    private static Logger logger = LoggerFactory.getLogger(KafkaComponent_Dispatch.class);

    private final RtcDeviceProperty deviceProperty;
    private final JsonProvider jsonProvider;
    private final CloseableHttpClient httpClient;
    private final DeviceWriterRpcProvider deviceWriterRpcProvider;
    private final RdSyncRpcWriterProvider syncRpcWriterProvider;

    private final ThirdDeviceRpcProvider thirdDeviceRpcProvider;

    public KafkaComponent_Device(RtcDeviceProperty deviceProperty, JsonProvider jsonProvider,
                                 CloseableHttpClient httpClient, DeviceWriterRpcProvider deviceWriterRpcProvider,
                                 RdSyncRpcWriterProvider syncRpcWriterProvider,
                                 ThirdDeviceRpcProvider thirdDeviceRpcProvider) {
        this.deviceProperty = deviceProperty;
        this.jsonProvider = jsonProvider;
        this.httpClient = httpClient;
        this.deviceWriterRpcProvider = deviceWriterRpcProvider;
        this.syncRpcWriterProvider = syncRpcWriterProvider;
        this.thirdDeviceRpcProvider = thirdDeviceRpcProvider;

    }

    /**
     * 驻勤点的设备硬件信息
     * @param records
     * @param ack
     */
    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-rtc-TLK_HARDWARE",
            topics = {
                    KafkaConstants.TOPIC_DISPATCH_TLK_HARDWARE
            }, groupId = "${spring.kafka.consumer.group-id}-rtc-TLK_HARDWARE")
    public void ackHardWareDevice(List<ConsumerRecord<String, String>> records,
                             Acknowledgment ack) {
        if (CollectionUtils.isEmpty(records)) {
            return;
        }
        Collection<String> ids = null;
        try{
            ids =
                    records.stream().map(ii->ii.key())
                            .distinct().collect(Collectors.toList());

            this.syncRpcWriterProvider.sync(
                    SyncParameterWrapperRequest.create(DispatchDataType.HareWareDevice,ids,""));
        }
        catch (Exception ex) {
            logger.error("刷新硬件设备数据:{}", ids, ex);
        }

        boolean allowed2CommitAtFinial = true;
        try {
            Collection<DebeziumUtil.ExtractDebeziumItem> debeziumItems =
                    extractDebeziumItems(records, "ID", "ITEM_DEVICE_NO", "ITEM_DEVICE_TYPE", "ITEM_BINDED_USER_ATTID");
            Collection<DebeziumUtil.ExtractDebeziumItem> deviceNoDebeziumItems =
                    debeziumItems.stream()
                            .filter(ii -> ii.getName().equalsIgnoreCase("ITEM_DEVICE_TYPE") &&
                                    /**
                                     * 4是华为云摄像头
                                     */
                                    ii.getValues() == null || !ii.getValues().contains("4"))
                            .filter(ii ->
                                    ii.getName().equalsIgnoreCase("ITEM_DEVICE_NO")
                            ).collect(Collectors.toList());

            Collection<String> deviceIds = debeziumItems.stream().filter(ii ->
                    ii.getName().equalsIgnoreCase("ID") && ii.getValues() != null
            ).flatMap(ii -> ii.getValues().stream()).collect(Collectors.toList());
            deviceWriterRpcProvider.batchSync(deviceIds);

            Collection<DebeziumUtil.ExtractDebeziumItem> videoDevices =
                    debeziumItems.stream()
                            .filter(ii -> ii.getName().equalsIgnoreCase("ITEM_DEVICE_TYPE") &&
                                    /**
                                     * 4是华为云摄像头
                                     */
                                    ii.getValues() != null && ii.getValues().contains("4"))
                            .collect(Collectors.toList());

            if(!CollectionUtils.isEmpty(videoDevices)) {
                Collection<String> videoDeviceIds =
                        videoDevices.stream()
                                .filter(ii -> ii.getRelations() != null)
                                .flatMap(ii -> ii.getRelations().stream())
                                .filter(ii -> ii.getName().equalsIgnoreCase("ID") && ii.getValues() != null)
                                .flatMap(ii -> ii.getValues().stream())
                                .distinct().collect(Collectors.toList());
                Collection<String> bindAttIds =
                        videoDevices.stream()
                                .filter(ii -> ii.getRelations() != null)
                                .flatMap(ii -> ii.getRelations().stream())
                                .filter(ii -> ii.getName().equalsIgnoreCase("ITEM_BINDED_USER_ATTID") && ii.getValues() != null)
                                .flatMap(ii -> ii.getValues().stream())
                                .distinct().collect(Collectors.toList());
                logger.error("ackHardWareDevice: 即将刷新视频摄像头信息:id={}, bindAttIds={}",
                        videoDeviceIds.stream().collect(Collectors.joining()),
                        bindAttIds.stream().collect(Collectors.joining())
                );
                this.syncRpcWriterProvider.sync(SyncParameterWrapperRequest.create(DispatchDataType.VideoDevice, videoDeviceIds, bindAttIds));
            }

            Collection<String> deviceNos =
                    deviceNoDebeziumItems.stream()
                            .flatMap(ii -> ii.getValues().stream())
                            .collect(Collectors.toSet());

            this.thirdDeviceRpcProvider.refresh(deviceNos);
        } catch (Exception ex) {
            logger.error("消费设备出场的基本信息发生异常", ex);
            allowed2CommitAtFinial = false;
        } finally {
            if (allowed2CommitAtFinial) {
                ack.acknowledge();
            }
        }
    }

    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-rtc-device",
            topics = {
                    KafkaConstants.TOPIC_DISPATCH_RD_DEVICE_SUMMARY
            }, groupId = "${spring.kafka.consumer.group-id}-rtc-device")
    public void ackNewDevice(List<ConsumerRecord<String, String>> records,
                                   Acknowledgment ack) {
        if (CollectionUtils.isEmpty(records)) {
            return;
        }

        boolean allowed2CommitAtFinial = true;
        try {
            Collection<DebeziumUtil.ExtractDebeziumItem> debeziumItems = extractDebeziumItems(records,"dispatch_no", "organization_id","organization_name");
            Collection<String> dispatchNos = debeziumItems
                    .stream()
                    .filter(ii -> ii.getStatus() == DebeziumUtil.DebeziumItemStatus.Created && ii.getName()
                            .equalsIgnoreCase("dispatch_no"))
                    .flatMap(ii -> ii.getValues().stream())
                    .collect(Collectors.toSet());

            if (CollectionUtils.isEmpty(dispatchNos)) {
                return;
            }

            HttpPost post = new HttpPost(String.format("%s/sapi/org/manage/update/createBy", deviceProperty.getUrl()));
            post.setHeader("auth_code", deviceProperty.getAuthCode());
            post.addHeader("Content-Type", "application/json");
            Map bodyMap = new HashMap();
            bodyMap.put("newAdminName", deviceProperty.getAdminName());
            bodyMap.put("accountList", dispatchNos);

            String body = this.jsonProvider.getJson(bodyMap);
            post.setEntity(new StringEntity(body));

            try (CloseableHttpResponse response = this.httpClient.execute(post)) {
                logger.error("转移组织: 推送url={}; 请求内容:{}, 状态:{};响应信息:{}",
                        post.getURI(), body,
                        response.getStatusLine(),
                        EntityUtils.toString(response.getEntity())
                );
            }

            /**
             * 加入默认的企业群
             */

            Collection<String> organizationIds = debeziumItems
                    .stream()
                    .filter(ii -> ii.getName().equalsIgnoreCase("organization_id"))
                    .flatMap(ii -> ii.getValues().stream())
                    .collect(Collectors.toSet());

            logger.error("准备初始化企业的默认群:{}", organizationIds.stream().collect(Collectors.joining(",")));
            for (String orgId : organizationIds) {
                Collection<DebeziumUtil.ExtractDebeziumItem> selectedDebeziumnItems
                        = debeziumItems.stream().filter(ii -> ii.getName().equalsIgnoreCase("organization_id") && ii.getValues().contains(orgId))
                        .collect(Collectors.toSet());

                if (!CollectionUtils.isEmpty(selectedDebeziumnItems)) {
                    String orgName =
                            selectedDebeziumnItems.stream()
                                    .filter(ii -> !CollectionUtils.isEmpty(ii.getRelations()))
                                    .flatMap(ii -> ii.getRelations().stream())
                                    .filter(ii -> ii.getName().equalsIgnoreCase("organization_name"))
                                    .flatMap(ii -> ii.getValues().stream()).findFirst().orElse(null);
                    String message = "";
                    if (!StringUtils.hasLength(orgName)) {
                        orgName = orgId;

                        message = "当前的消息得到了组织名称为空的异常情况; 队列数据=" + orgId;
                    }

                    String rtcGroupId = ensureSpecialGroup(getGroupUUId(false, orgId), orgName);
                    logger.error("获取到的群信息为:orgId={};name={};rtcGroupId={};message={}", orgId, orgName, rtcGroupId, message);

                    Collection<String> accountList = selectedDebeziumnItems.stream().flatMap(ii -> ii.getValues().stream()).collect(Collectors.toSet());
                    joinGroup(rtcGroupId, accountList);

                    logger.error("转移设备的时候, 自动将如下设备({})加入如下的群{}({})", accountList.stream().collect(Collectors.joining(",")), rtcGroupId, orgName);
                }
            }
        } catch (Exception ex) {
            logger.error("消费设备出场的基本信息发生异常", ex);
            allowed2CommitAtFinial = false;
        } finally {
            if (allowed2CommitAtFinial) {
                ack.acknowledge();
            }
        }
    }

    /**
     * 根据驻勤点来查找群信息
     * @param records
     * @param ack
     */
    @KafkaListener(
            id = "${spring.kafka.consumer.group-id}-new-station-rtc-device",
            topics = {
                    KafkaConstants.TOPIC_DISPATCH_RD_DEVICE_SUMMARY
            }, groupId = "${spring.kafka.consumer.group-id}-new-station-rtc-device")
    public void ackNewStatsionDevice(List<ConsumerRecord<String, String>> records,
                                   Acknowledgment ack) {
        if (CollectionUtils.isEmpty(records)) {
            return;
        }

        boolean allowed2CommitAtFinial = true;
        try {
            Collection<DebeziumUtil.ExtractDebeziumItem> selectedDebeziumItem =
                    extractDebeziumItems(records,"dispatch_no", "security_station_id", "security_station_name")
                    .stream()
                    .filter(ii -> ii.getStatus() == DebeziumUtil.DebeziumItemStatus.Created)
                    .collect(Collectors.toList());

            if (CollectionUtils.isEmpty(selectedDebeziumItem)) {
                return;
            }

            Collection<DebeziumUtil.ExtractDebeziumItem> selectedStationItems = selectedDebeziumItem.stream()
                    .filter(ii->ii.getName().equalsIgnoreCase("security_station_id"))
                    .collect(Collectors.toSet());

            /**
             * 移除的驻勤点信息
             */
            Collection<String> removedFromStationIds = selectedStationItems.stream()
                    .filter(ii->StringUtils.hasLength(ii.getBeforeValue()) && !ii.getBeforeValue().equalsIgnoreCase(ii.getAfterValue()))
                    .map(ii->ii.getBeforeValue())
                    .filter(ii->StringUtils.hasLength(ii))
                    .collect(Collectors.toSet());
            logger.error("准备从以下群({})中移除人员", removedFromStationIds.stream().collect(Collectors.joining(",")));
            for(String rmStationId:removedFromStationIds) {
                String rtcGroupId = getRtcGroupIdBySpecialId(getGroupUUId(true, rmStationId));
                /**
                 * 第一步: 获取驻勤点对应的设备信息
                 */
                Collection<DebeziumUtil.ExtractDebeziumItem> selectedStation =
                        selectedStationItems.stream().filter(ix -> ix.getValues().contains(rmStationId))
                                .flatMap(ix -> ix.getRelations().stream()).collect(Collectors.toSet());

                /**
                 * 第二部: 从群里
                 */
                Collection<String> accountList =
                        selectedStation.stream().filter(ii -> ii.getName().equalsIgnoreCase("dispatch_no"))
                                .flatMap(ix -> ix.getValues().stream())
                                .collect(Collectors.toSet());

                removeFromGroup(rtcGroupId, accountList);
                logger.error("成功将人员({})从如下驻勤点群中移除设备群{}({})",
                        accountList.stream().collect(Collectors.joining(",")),
                        rtcGroupId, rmStationId);
            }

            Collection<String> stationIds = selectedStationItems.stream().map(ii->ii.getAfterValue())
                    .filter(ii->StringUtils.hasLength(ii))
                    .collect(Collectors.toSet());
            logger.error("准备初始化企业的驻勤点信息:{}", stationIds.stream().collect(Collectors.joining(",")));
            for(String stationId :stationIds) {
                /**
                 * 第一步: 根据驻勤点;创建无限群信息
                 */
                Collection<DebeziumUtil.ExtractDebeziumItem> selectedStation =
                        selectedStationItems.stream().filter(ix -> ix.getValues().contains(stationId))
                                .flatMap(ix -> ix.getRelations().stream()).collect(Collectors.toSet());

                String rtcGroupId = null;
                String stationName = null;
                if (selectedStation != null) {
                    stationName =
                            selectedStation.stream().filter(ii -> ii.getName().equalsIgnoreCase("security_station_name"))
                                    .flatMap(ii -> ii.getValues().stream())
                                    .findFirst().orElse(null);

                    rtcGroupId = ensureSpecialGroup(getGroupUUId(true, stationId), stationName);
                    logger.error("获取到的群信息为:stationId={};name={};rtcGroupId={}", stationId, stationName, rtcGroupId);
                }

                if (!StringUtils.hasLength(rtcGroupId)) {
                    throw new BadTenantException(String.format("找不到对应的群组(id=%s;name=%s)信息", stationId, stationName));
                }

                /**
                 * 第二部: 根据群Id; 将人加入群中
                 */
                Collection<String> accountList =
                        selectedStation.stream().filter(ii -> ii.getName().equalsIgnoreCase("dispatch_no"))
                                .flatMap(ix -> ix.getValues().stream())
                                .collect(Collectors.toSet());

                joinGroup(rtcGroupId, accountList);

                logger.error("成功将人员加入驻勤点设备群, 自动将如下设备({})加入如下的群{}({})",
                        accountList.stream().collect(Collectors.joining(",")),
                        rtcGroupId, stationName);
            }

        } catch (Exception ex) {
            logger.error("自动根据驻勤点创建群组发生异常", ex);
            allowed2CommitAtFinial = false;
        } finally {
            if (allowed2CommitAtFinial) {
                ack.acknowledge();
            }
        }
    }

    private Collection<DebeziumUtil.ExtractDebeziumItem> extractDebeziumItems(List<ConsumerRecord<String, String>> records,String ...fields) {
        Collection<DebeziumUtil.ExtractDebeziumItem> data =
                records.stream().flatMap(ii -> {
                    Collection<DebeziumUtil.ExtractDebeziumItem> nos =
                            DebeziumUtil.extractItems(jsonProvider, ii.value(), fields);

                    nos.forEach(ix -> {
                        ix.assignRelations(nos);
                    });

                    return nos.stream();
                }).collect(Collectors.toList());

        return data;
    }

    private void joinGroup(String rtcGroupId,Collection<String> accountLists) throws IOException {
        if(!CollectionUtils.isEmpty(accountLists)) {
            HttpPost post = new HttpPost(String.format("%s/sapi/dispatch/users_add", deviceProperty.getUrl()));
            post.setHeader("auth_code", deviceProperty.getAuthCode());
            post.addHeader("Content-Type", "application/json");
            Map bodyMap = new HashMap();
            bodyMap.put("adminName", deviceProperty.getAdminName());
            bodyMap.put("accountLists", accountLists);
            bodyMap.put("gid", rtcGroupId);

            String body = this.jsonProvider.getJson(bodyMap);
            post.setEntity(new StringEntity(body));

            try (CloseableHttpResponse response = this.httpClient.execute(post)) {
                logger.error("加入群组: 推送url={}; 请求内容:{}, 状态:{};响应信息:{}",
                        post.getURI(), body,
                        response.getStatusLine(),
                        EntityUtils.toString(response.getEntity())
                );
            }
        }
    }

    private void removeFromGroup(String rtcGroupId,Collection<String> accountLists) throws IOException {
        if(!CollectionUtils.isEmpty(accountLists)) {
            HttpPost post = new HttpPost(String.format("%s/sapi/org/manage/groupMoveUsers", deviceProperty.getUrl()));
            post.setHeader("auth_code", deviceProperty.getAuthCode());
            post.addHeader("Content-Type", "application/json");
            Map bodyMap = new HashMap();
            bodyMap.put("adminName", deviceProperty.getAdminName());
            bodyMap.put("accountLists", accountLists);
            bodyMap.put("gid", rtcGroupId);

            String body = this.jsonProvider.getJson(bodyMap);
            post.setEntity(new StringEntity(body));

            try (CloseableHttpResponse response = this.httpClient.execute(post)) {
                logger.error("从群组({})移除: 推送url={}; 请求内容:{}, 状态:{};响应信息:{}",
                        rtcGroupId,
                        post.getURI(), body,
                        response.getStatusLine(),
                        EntityUtils.toString(response.getEntity())
                );
            }
        }
    }

    /**
     * 根据驻勤点或者企业名称创建群组
     * @param id
     * @param name
     * @return
     * @throws IOException
     */
    private static final Map<String,String> _localSpecialGroupMap = new HashMap<>();
    private String ensureSpecialGroup(String id, String name) throws IOException {

        /**
         * 本地如果已经有该群, 则去本地
         */
        if (_localSpecialGroupMap.containsKey(id)) {
            return _localSpecialGroupMap.get(id);
        }

        /**
         * 找不到群, 则创建群
         */
        HttpPost post = new HttpPost(String.format("%s/sapi/org/manage/create/group", deviceProperty.getUrl()));
        post.setHeader("auth_code", deviceProperty.getAuthCode());
        post.addHeader("Content-Type", "application/json");
        Map bodyMap = new HashMap();
        bodyMap.put("adminName", deviceProperty.getAdminName());
        bodyMap.put("groupName", name);
        bodyMap.put("strangers", id);

        String body = this.jsonProvider.getJson(bodyMap);
        post.setEntity(new StringEntity(body));

        String groupId = null;
        try (CloseableHttpResponse response = this.httpClient.execute(post)) {
            String content = EntityUtils.toString(response.getEntity());
            logger.error("创建群组: 推送url={}; 请求内容:{}, 状态:{};响应信息:{}",
                    post.getURI(), body,
                    response.getStatusLine(),
                    content
            );

            String responseData = this.jsonProvider.toObject(JSONObject.class, content).getString("data");
            /**
             * 获取该组的Id(组为驻勤点或者企业)
             */
            groupId = this.jsonProvider.toObject(JSONObject.class, responseData).getString("gid");
        }

        if (_localSpecialGroupMap.size() > 1000) {
            _localSpecialGroupMap.clear();
        }

        _localSpecialGroupMap.put(id, groupId);


        return groupId;
    }

    private String getRtcGroupIdBySpecialId(String id) throws IOException {
        /**
         * 本地如果已经有该群, 则去本地
         */
        if (_localSpecialGroupMap.containsKey(id)) {
            return _localSpecialGroupMap.get(id);
        }

        /**
         * 找不到群, 则创建群
         */
        HttpPost post = new HttpPost(String.format("%s/sapi/misc/groups/list", deviceProperty.getUrl()));
        post.setHeader("auth_code", deviceProperty.getAuthCode());
        post.addHeader("Content-Type", "application/json");
        Map bodyMap = new HashMap();
        bodyMap.put("adminName", deviceProperty.getAdminName());
        bodyMap.put("strangers", id);

        String body = this.jsonProvider.getJson(bodyMap);
        post.setEntity(new StringEntity(body));

        String groupId = null;
        try (CloseableHttpResponse response = this.httpClient.execute(post)) {
            String content = EntityUtils.toString(response.getEntity());
            logger.error("查询群信息: 推送url={}; 请求内容:{}, 状态:{};响应信息:{}",
                    post.getURI(), body,
                    response.getStatusLine(),
                    content
            );

            JSONArray jsonArray = this.jsonProvider.toObject(JSONObject.class, content).getJSONArray("records");
            groupId =
                    jsonArray.stream().map(ii -> ((JSONObject) ii)
                                    .getString("gid")).collect(Collectors.toSet()).stream()
                            .findFirst().orElse(null);
        }

        if(!StringUtils.hasLength(groupId)) {
            return null;
        }

        _localSpecialGroupMap.put(id, groupId);

        return groupId;
    }

    /**
     * 返回驻勤点或者企业的群信息
     * @param station
     * @param id
     * @return
     */
    private static String getGroupUUId(boolean station,String id) {
        if (station) {
            return String.format("s_%s", id);
        }

        return String.format("o_%s", id);
    }
}
