package com.bcxin.tenant.open.backend.tasks.configs;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.nacos.spring.util.Tuple;
import com.bcxin.tenant.open.backend.tasks.utils.DebeziumUtil;
import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.constants.KafkaConstants;
import com.bcxin.tenant.open.infrastructures.enums.DispatchDataScopeType;
import com.bcxin.tenant.open.infrastructures.enums.DispatchDataType;
import com.bcxin.tenant.open.infrastructures.exceptions.BadTenantException;
import com.bcxin.tenant.open.infrastructures.exceptions.NotSupportTenantException;
import com.bcxin.tenant.open.infrastructures.utils.StringUtil;
import com.bcxin.tenant.open.jdks.DispatchDataScopeRpcProvider;
import com.bcxin.tenant.open.jdks.HotCacheRpcProvider;
import com.bcxin.tenant.open.jdks.RdSyncRpcWriterProvider;
import com.bcxin.tenant.open.jdks.SecurityStationReaderRpcProvider;
import com.bcxin.tenant.open.jdks.requests.HotCacheRequest;
import com.bcxin.tenant.open.jdks.requests.SyncParameterWrapperRequest;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.common.header.Header;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.kafka.support.Acknowledgment;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

public abstract class KafkaComponentAbstract {
    private static final Logger logger = LoggerFactory.getLogger(KafkaComponent.class);
    protected final JsonProvider jsonProvider;
    protected final SecurityStationReaderRpcProvider securityStationReaderRpcProvider;
    protected final DispatchDataScopeRpcProvider dispatchDataScopeRpcProvider;
    protected final RdSyncRpcWriterProvider syncRpcWriterProvider;

    protected KafkaComponentAbstract(JsonProvider jsonProvider, SecurityStationReaderRpcProvider securityStationReaderRpcProvider,
                                     DispatchDataScopeRpcProvider dispatchDataScopeRpcProvider,
                                     RdSyncRpcWriterProvider syncRpcWriterProvider) {
        this.jsonProvider = jsonProvider;
        this.securityStationReaderRpcProvider = securityStationReaderRpcProvider;
        this.dispatchDataScopeRpcProvider = dispatchDataScopeRpcProvider;
        this.syncRpcWriterProvider = syncRpcWriterProvider;
    }

    protected void executeConsumeData(
            RdSyncRpcWriterProvider syncRpcWriterProvider,
            List<ConsumerRecord<String, String>> records,
            Acknowledgment ack) {
        if (CollectionUtils.isEmpty(records)) {
            logger.error("无效订阅数据:{}", records);
            return;
        }

        Collection<String> topics = records.stream().map(ix -> ix.topic()).distinct()
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(topics)) {
            return;
        }

        DispatchDataType dataType = DispatchDataType.Employee;
        boolean allowedToAck = true;
        Exception lastException = null;
        try {
            for (String topic : topics) {
                Collection<String> ids = new ArrayList<>();
                try {
                    List<ConsumerRecord<String, String>> selectedRecords = getMatchRecords(records,topic);
                    if (CollectionUtils.isEmpty(selectedRecords)) {
                        continue;
                    }

                    ids = extractIdsFromRecords(selectedRecords);
                    Object additionalParameter = null;
                    switch (topic) {
                        case KafkaConstants.TOPIC_OBPM2_TENANT_USERS:
                            dataType = DispatchDataType.User;
                            break;
                        case KafkaConstants.TOPIC_OBPM2_TENANT_EMPLOYEES:
                            dataType = DispatchDataType.Employee;
                            break;
                        case KafkaConstants.TOPIC_OBPM2_TENANT_ORGANIZATIONS:
                            dataType = DispatchDataType.Company;

                            additionalParameter = extractOrgFromJson(records, jsonProvider);
                            break;
                        case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_STATION:
                            dataType = DispatchDataType.Station;
                            break;
                        case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_PT_SECURITYMAN:
                            dataType = DispatchDataType.CommunityUser;
                            break;
                        case KafkaConstants.TOPIC_DISPATCH_TLK_DEVICE:
                            dataType = DispatchDataType.RefreshCompanyDesks;
                            break;
                        case KafkaConstants.TOPIC_DISPATCH_TLK_ORG_PURSE:
                            dataType = DispatchDataType.RefreshCompanyPoints;
                            break;
                        case KafkaConstants.TOPIC_OBPM2_T_SUPERVISE_DEPARTMENT:
                            ids = selectedRecords.stream()
                                    .filter(ix -> ix.topic().equalsIgnoreCase(topic))
                                    .flatMap(ix -> extractFieldFromJson(ix.value(), "id").stream())
                                    .collect(Collectors.toList());
                            dataType = DispatchDataType.SuperviseDeparts;
                            break;
                        case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_N_SECURITY:
                            dataType = DispatchDataType.ProprietorStations;
                            break;
                        case KafkaConstants.TOPIC_OBPM2_TENANT_DATA_PERMISSIONS:
                            dataType = DispatchDataType.CompanyPermissions;
                            break;
                        case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_TEMPORARY_PROTECTION_PROJECT:
                            dataType = DispatchDataType.TemporaryProtectionProject;
                            break;
                        case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_TEMPORARY_PROTECTION_GROUP:
                            dataType = DispatchDataType.TemporaryProtectionProjectGroup;
                            break;
                        case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_POST_PERSON:
                            dataType = DispatchDataType.TemporaryProtectionProjectMember;
                            break;
                        case KafkaConstants.TOPIC_DISPATCH_PROPRIETOR_COMPANY_CHANGED:
                            dataType = DispatchDataType.ProprietorCompanyChanged;
                            /**
                             * 关注的组织Id列表
                             */
                            ids = selectedRecords.stream()
                                    .filter(ix -> ix.topic().equalsIgnoreCase(topic))
                                    .flatMap(ix -> {
                                        if (StringUtil.isEmpty(ix.key())) {
                                            return null;
                                        }

                                        Collection<String> selectedKeys = jsonProvider.toObjects(String.class, ix.key());
                                        if (selectedKeys == null) {
                                            return null;
                                        }

                                        return selectedKeys.stream();
                                    }).filter(ix -> ix != null)
                                    .collect(Collectors.toList());
                            break;
                        case KafkaConstants.TOPIC_TMS_TLK_EXAM_ROOM:
                            dataType = DispatchDataType.ExamRoom;
                            break;
                        case KafkaConstants.TOPIC_TMS_TLK_EXAM_SITE:
                            dataType = DispatchDataType.ExamSite;
                            break;
                        case KafkaConstants.TOPIC_TMS_TLK_SYSTEM_EXAM_INFO:
                            dataType = DispatchDataType.SystemExamInfo;
                            break;
                        case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_PROJECT_PERSON_TYPE:
                            dataType = DispatchDataType.TemporaryProtectionProjectPersonType;
                            break;
                        case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_PROJECT_JOIN:
                            dataType = DispatchDataType.EventOrganizerJoinProject;

                            Collection<String> domainIds =
                                    records.stream().flatMap(ii -> extractFieldFromJson(ii.value(),"DOMAINID").stream())
                                            .filter(ii->StringUtils.hasLength(ii))
                                            .collect(Collectors.toList());
                            if(CollectionUtils.isEmpty(domainIds)) {
                                logger.error("TOPIC_BAIBAODUNFLOW_TLK_PROJECT_JOIN-异常数据: domainId不应该为空; 无法同步:{}",
                                        records.stream().map(ii -> ii.value()).collect(Collectors.joining(";"))
                                );

                                return;
                            }
                            additionalParameter = domainIds;
                            break;
                        case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_MANAGE_POST:
                            dataType = DispatchDataType.EventOrganizerStationManager;

                            Collection<String> empIds =
                                    records.stream().flatMap(ii -> extractFieldFromJson(ii.value(),"ITEM_PERSON_ID").stream())
                                            .filter(ii->StringUtils.hasLength(ii))
                                            .collect(Collectors.toList());
                            if(CollectionUtils.isEmpty(empIds)) {
                                logger.error("TOPIC_BAIBAODUNFLOW_TLK_MANAGE_POST-异常数据: empIds不应该为空; 无法同步:{}",
                                        records.stream().map(ii -> ii.value()).collect(Collectors.joining(";"))
                                );

                                return;
                            }
                            additionalParameter = empIds;
                            break;
                        case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_PROJECT_PERSON_SOURCE:
                            dataType = DispatchDataType.DispatchDataSource;
                            break;
                        case KafkaConstants.TOPIC_DUTY_TLK_PATROL_TASKS:
                            additionalParameter=0;
                            dataType = DispatchDataType.Workstation;
                            break;
                        case KafkaConstants.TOPIC_DUTY_TLK_PATROL_POINTS:
                            additionalParameter=2;
                            dataType = DispatchDataType.Workstation;
                            break;
                        case KafkaConstants.TOPIC_DUTY_TLK_PATROL_PATH_POINT_RELATIONS:
                            additionalParameter=3;
                            dataType = DispatchDataType.Workstation;
                            break;
                        case KafkaConstants.TOPIC_DUTY_TLK_PATROL_TASK_PATH_RELATIONS:
                            additionalParameter=1;
                            dataType = DispatchDataType.Workstation;
                            break;
                        case KafkaConstants.TOPIC_DUTY_TLK_PATROL_TASK_EMPLOYEE_RELATIONS:
                            ids = selectedRecords.stream()
                                    .flatMap(ii->extractFieldFromJson(ii.value(),"ITEM_EMPLOYEEID").stream())
                                    .collect(Collectors.toList());
                            dataType = DispatchDataType.WorkStationUsers;
                            break;
                        case KafkaConstants.TOPIC_DEAD_LETTER_DISPATCH: {
                            /**
                             * 针对死信; 不支持如下操作后直接返回
                             */
                            handleDeadLetterQueueData(syncRpcWriterProvider, jsonProvider, selectedRecords);
                            return;
                        }

                        default:
                            throw new NotSupportTenantException(String.format("1.不支持该主题(%s)信息", topic));
                    }

                    if(dataType==DispatchDataType.RefreshCompanyDesks ||
                            dataType==DispatchDataType.RefreshCompanyPoints) {
                        DispatchDataType finalDataType1 = dataType;
                        ids = selectedRecords.stream()
                                .filter(ix -> ix.topic().equalsIgnoreCase(topic))
                                .flatMap(ix -> extractOrgIdFromJson(ix.value(), finalDataType1).stream())
                                .collect(Collectors.toList());
                    }else if(dataType==DispatchDataType.ProprietorStations) {
                        DispatchDataType finalDataType1 = dataType;
                        ids = selectedRecords.stream()
                                .filter(ix -> ix.topic().equalsIgnoreCase(topic))
                                .flatMap(ix -> extractOrgIdFromJson(ix.value(), finalDataType1).stream())
                                .collect(Collectors.toList());
                        additionalParameter = ids;

                        logger.error("正在处理内保关注的驻勤信息:{}-驻勤={}",
                                ids.stream().collect(Collectors.joining(";")),
                                ((Collection) additionalParameter).stream().collect(Collectors.joining(";")));

                        /**
                         * 内保单位的话; 同时需要刷新保卫管理员负责的驻勤点
                         */
                        Collection<String> empIds = selectedRecords.stream()
                                .filter(ix -> ix.topic().equalsIgnoreCase(topic))
                                .flatMap(ix -> extractFieldFromJson(ix.value(), "ITEM_EMPID").stream())
                                .filter(ii -> ii != null)
                                .collect(Collectors.toList());
                        if (!CollectionUtils.isEmpty(empIds)) {
                            syncRpcWriterProvider.sync(
                                    SyncParameterWrapperRequest.create(DispatchDataType.Employee, empIds, null)
                            );
                        }
                    }

                    if(dataType!=DispatchDataType.ProprietorCompanyChanged) {
                        Collection<String> finalIds = ids;
                        DispatchDataType finalDataType = dataType;
                        syncRpcWriterProvider.sync(SyncParameterWrapperRequest.create(finalDataType, finalIds, additionalParameter));
                    }else {
                        if (!CollectionUtils.isEmpty(ids)) {
                            /**
                             * 只需要刷新驻勤对应的监管信息即可
                             */
                            Collection<String> securityStationIds =
                                    this.securityStationReaderRpcProvider.getSecurityStationIdsByOrganizationIds(ids);

                            logger.error("即将实时重新计划的驻勤-获取得到的企业={}; 驻勤点={}",
                                    ids.stream().collect(Collectors.joining(";")),
                                    CollectionUtils.isEmpty(securityStationIds) ? "无" :
                                            securityStationIds.stream().collect(Collectors.joining(";"))
                            );

                            if (!CollectionUtils.isEmpty(securityStationIds)) {
                                this.dispatchDataScopeRpcProvider.flush2Redis(securityStationIds, DispatchDataScopeType.Proprietor);
                            }
                        } else {
                            logger.error("无效未取得企业Id信息的数据:{}", ids);
                        }
                    }
                } catch (Exception ex) {
                    lastException = ex;
                    allowedToAck = false;
                    ex.printStackTrace();
                    logger.error(
                            "KafkaComponent: failed to consumer data with topic={};data={}",
                            topic,
                            (ids == null ? "NULL" : ids.stream().map(ii -> String.format("'%s'", ii)).collect(Collectors.joining(","))),
                            ex);
                }
            }
        } catch (Exception ex) {
            lastException = ex;
            allowedToAck = false;
        } finally {
            if (allowedToAck) {
                ack.acknowledge();
            } else {
                if (lastException != null) {
                    throw new BadTenantException(
                            String.format("failed to cost topic=%s", topics.stream().collect(Collectors.joining(";"))),
                            lastException);
                }
            }
        }
    }

    protected static void handleDeadLetterQueueData(RdSyncRpcWriterProvider syncRpcWriterProvider,
                                                  JsonProvider jsonProvider,
                                                  List<ConsumerRecord<String, String>> selectedRecords) throws Exception {
        try {
            if (CollectionUtils.isEmpty(selectedRecords)) {
                return;
            }

            Collection<Tuple<String, DispatchDataType>> dataIds =
                    selectedRecords.stream().flatMap(ii -> {
                        Collection<Tuple<String, DispatchDataType>> ids = new ArrayList<>();
                        if(ii.headers()!=null) {
                            Iterable<Header> selectedHeaders = ii.headers().headers(KafkaConstants.HEADER_ORIGINAL_TOPIC_NAME);
                            if (selectedHeaders != null) {
                                Iterator<Header> iterator = selectedHeaders.iterator();
                                if (iterator.hasNext()) {
                                    byte[] headerValue = iterator.next().value();
                                    if (headerValue != null) {
                                        DispatchDataType dataType = DispatchDataType.Employee;
                                        String originalTopic = new String(headerValue);
                                        switch (originalTopic) {
                                            case KafkaConstants.TOPIC_OBPM2_TENANT_USERS:
                                                dataType = DispatchDataType.User;
                                                break;
                                            case KafkaConstants.TOPIC_OBPM2_TENANT_EMPLOYEES:
                                                dataType = DispatchDataType.Employee;
                                                break;
                                            case KafkaConstants.TOPIC_OBPM2_TENANT_ORGANIZATIONS:
                                                dataType = DispatchDataType.Company;
                                                break;
                                            case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_STATION:
                                                dataType = DispatchDataType.Station;
                                                break;
                                            case KafkaConstants.TOPIC_DISPATCH_TLK_DEVICE:
                                                dataType = DispatchDataType.RefreshCompanyDesks;
                                                break;
                                            case KafkaConstants.TOPIC_DISPATCH_TLK_ORG_PURSE:
                                                dataType = DispatchDataType.RefreshCompanyPoints;
                                                break;
                                            case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_PT_SECURITYMAN:
                                                dataType = DispatchDataType.CommunityUser;
                                                break;
                                            case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_N_SECURITY:
                                                dataType = DispatchDataType.ProprietorStations;
                                                break;
                                            case KafkaConstants.TOPIC_TMS_TLK_EXAM_ROOM:
                                                dataType = DispatchDataType.ExamRoom;
                                                break;
                                            case KafkaConstants.TOPIC_TMS_TLK_EXAM_SITE:
                                                dataType = DispatchDataType.ExamSite;
                                                break;
                                            case KafkaConstants.TOPIC_TMS_TLK_SYSTEM_EXAM_INFO:
                                                dataType = DispatchDataType.SystemExamInfo;
                                                break;
                                            case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_MANAGE_POST:
                                                dataType = DispatchDataType.EventOrganizerStationManager;
                                                break;
                                            case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_PROJECT_PERSON_SOURCE:
                                                dataType = DispatchDataType.DispatchDataSource;
                                                break;
                                            case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_PROJECT_JOIN:
                                                dataType = DispatchDataType.EventOrganizerJoinProject;
                                                break;
                                            case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_PROJECT_PERSON_TYPE:
                                                dataType = DispatchDataType.TemporaryProtectionProjectPersonType;
                                                break;
                                            case KafkaConstants.TOPIC_BAIBAODUNFLOW_TLK_EVENT_PROJECT:
                                                dataType = DispatchDataType.TemporaryProtectionProject;
                                                break;
                                            default:
                                                throw new NotSupportTenantException(String.format("2.不支持该主题(%s)信息", originalTopic));
                                        }

                                        ids.add(Tuple.of(ii.key(), dataType));
                                    }
                                }
                            }
                        }
                        return ids.stream();
                    }).collect(Collectors.toList());

            if (!CollectionUtils.isEmpty(dataIds)) {
                Exception lastException = null;
                Collection<DispatchDataType> dataTypes =
                        dataIds.stream().map(ii -> ii.getSecond()).distinct().collect(Collectors.toList());
                for (DispatchDataType dt : dataTypes) {
                    try {
                        Collection<String> finalIds =
                                dataIds.stream().filter(ii -> ii.getSecond() == dt)
                                        .map(ii -> ii.getFirst())
                                        .collect(Collectors.toList());

                        DispatchDataType finalDataType = dt;
                        SyncParameterWrapperRequest req = SyncParameterWrapperRequest.create(finalDataType, finalIds);
                        if (finalDataType == DispatchDataType.ProprietorStations) {
                            req.setAdditionalParameter(finalIds);
                        }else if(finalDataType==DispatchDataType.EventOrganizerJoinProject || finalDataType==DispatchDataType.DispatchDataSource ||
                                finalDataType==DispatchDataType.EventOrganizerStationManager ||  finalDataType == DispatchDataType.TemporaryProtectionProjectPersonType){
                            req.setAdditionalParameter(finalIds);
                        }

                        syncRpcWriterProvider.sync(req);
                    } catch (Exception ex) {
                        lastException = ex;
                        logger.error("Failed to handle dead letter message", ex);
                    }
                }

                if (lastException != null) {
                    throw lastException;
                }
            }
        } catch (Exception ex) {
            logger.error("Failed to handle dead letter message:{}", jsonProvider.getJson(selectedRecords), ex);
        }
    }

    protected  Collection<String> extractOrgIdFromJson(String json,DispatchDataType dataType) {
        if (!StringUtils.hasLength(json)) {
            return Collections.EMPTY_LIST;
        }

        JSONObject data =
                this.jsonProvider.toObject(JSONObject.class, json);
        if (data == null) {
            return Collections.EMPTY_LIST;
        }

        JSONObject beforeNode = data.getJSONObject("before");
        JSONObject afterNode = data.getJSONObject("after");

        JSONObject node = afterNode == null ? beforeNode : afterNode;

        if (node == null) {
            return Collections.EMPTY_LIST;
        }

        if (dataType == DispatchDataType.RefreshCompanyDesks) {
            /**
             * 来自: tlk_device
             */
            return Collections.singleton(node.getString(
                    node.keySet().stream().filter(ix -> ix.equalsIgnoreCase("ITEM_DOMAIN_ID")).findFirst().orElse(null)));
        } else if (dataType == DispatchDataType.ProprietorStations) {
            Collection<String> ids = new HashSet<>();
            String key = node.keySet().stream().filter(ix -> ix.equalsIgnoreCase("item_attendanceSiteId")).findFirst().orElse(null);

            if (beforeNode != null) {
                ids.add(beforeNode.getString(key));
            }
            if (afterNode != null) {
                ids.add(afterNode.getString(key));
            }

            return ids;
        } else {
            /**
             * 来自： tlk_order_purse
             */
            ;
            return Collections.singleton(node.getString(
                    node.keySet().stream().filter(ix -> ix.equalsIgnoreCase("ITEM_ORGANIZATION_ID")).findFirst().orElse(null))
            );
        }
    }

    protected static Collection<String> extractOrgFromJson(List<ConsumerRecord<String, String>> records, JsonProvider jsonProvider) {
        if (CollectionUtils.isEmpty(records)) {
            return null;
        }

        Collection<String> contents =
                records.stream().map(ii -> {
                    JSONObject data =
                            jsonProvider.toObject(JSONObject.class, ii.value());
                    if (data == null) {
                        return null;
                    }
                    JSONObject beforeNode = data.getJSONObject("before");
                    JSONObject afterNode = data.getJSONObject("after");

                    JSONObject node = afterNode == null ? beforeNode : afterNode;
                    if (node == null) {
                        return null;
                    }

                    String superviseDepartIdFieldName = "supervise_depart_id";
                    String beforeSuperviseDepartId = null;
                    String afterSuperviseDepartId = null;
                    Optional<String> superviseDepartIdFieldOptional = node.keySet().stream().filter(ix -> ix.equalsIgnoreCase(superviseDepartIdFieldName))
                            .findFirst();
                    if (!superviseDepartIdFieldOptional.isPresent()) {
                        return null;
                    }

                    if (beforeNode != null) {
                        beforeSuperviseDepartId = beforeNode.getString(superviseDepartIdFieldOptional.get());
                    }
                    if (afterNode != null) {
                        afterSuperviseDepartId = afterNode.getString(superviseDepartIdFieldOptional.get());
                    }

                    if (StringUtil.isEqual(beforeSuperviseDepartId, afterSuperviseDepartId)) {
                        return null;
                    }

                    /*
                    return ProprietorOrgWrapContent.ProprietorOrgContent.create(
                            node.getString("id"),
                            node.getString("name"),
                            node.getString("supervise_depart_id"),
                            node.getString("supervise_depart_name")
                    );
                     */

                    return node.getString("id");
                }).filter(ii -> ii != null).collect(Collectors.toList());

        if (CollectionUtils.isEmpty(contents)) {
            return null;
        }


        return contents;
    }

    protected Collection<String> extractFieldFromJson(String json,String field) {
        return DebeziumUtil.extract(jsonProvider, json, field);
    }

    protected String extractFieldFromJson(String json,String parent, String field) {
        return DebeziumUtil.extract(jsonProvider, json,parent, field);
    }

        
    protected void doExecute(String topic,
                           Runnable action,
                           Collection<String> data,
                           Acknowledgment acknowledgment) {
        action.run();

        logger.warn("完成执行变更消费主题({})-数据为:{}",
                topic,
                data.size()
        );

        acknowledgment.acknowledge();
    }

    protected Collection<String> extractTopicsFromRecords(List<ConsumerRecord<String, String>> records) {
        if (CollectionUtils.isEmpty(records)) {
            return Collections.EMPTY_LIST;
        }

        Collection<String> topics = records.stream()
                .map(ix -> ix.topic()).distinct()
                .collect(Collectors.toList());

        return topics;
    }

    protected void refreshHotCache(HotCacheRpcProvider hotCacheRpcProvider, HotCacheRequest.CacheCategory category, Collection<String> ids) {
        try {
            hotCacheRpcProvider.refresh(HotCacheRequest.create(category, ids));
        } catch (Exception ex) {
            logger.error("刷新缓存数据发生异常:{}:数据={}", category, this.jsonProvider.getJson(ids), ex);
        }
    }

    protected List<ConsumerRecord<String, String>> getMatchRecords(List<ConsumerRecord<String, String>> records, String topic) {
        return records.stream()
                .filter(ii -> ii.topic().equalsIgnoreCase(topic))
                .collect(Collectors.toList());
    }

    protected Collection<String> extractIdsFromRecords(List<ConsumerRecord<String, String>> records) {
        return records.stream()
                .map(ix -> ix.key()).distinct()
                .collect(Collectors.toList());
    }
}
