export class TimeLimitManager {
  private static readonly TIME_LIMIT = 3 * 60 * 1000; // 3分钟（毫秒）
  private static readonly START_TIME_KEY = 'purchase_start_time';

  /**
   * 开始计时（在商品详情页面调用）
   */
  static startTimer(): void {
    // 只有在没有开始时间的情况下才设置新的开始时间
    if (!this.getStartTime()) {
      const startTime = Date.now();
      sessionStorage.setItem(this.START_TIME_KEY, startTime.toString());
    }
  }

  /**
   * 检查是否超时
   * @returns {boolean} true表示超时，false表示未超时
   */
  static isTimeout(): boolean {
    const startTime = this.getStartTime();
    if (!startTime) {
      return false; // 如果没有开始时间，认为未超时
    }
    
    const currentTime = Date.now();
    const elapsedTime = currentTime - startTime;
    return elapsedTime > this.TIME_LIMIT;
  }

  /**
   * 获取剩余时间（秒）
   * @returns {number} 剩余时间（秒），如果已超时返回0
   */
  static getRemainingTime(): number {
    const startTime = this.getStartTime();
    if (!startTime) {
      return this.TIME_LIMIT / 1000; // 如果没有开始时间，返回完整时间
    }
    
    const currentTime = Date.now();
    const elapsedTime = currentTime - startTime;
    const remainingTime = this.TIME_LIMIT - elapsedTime;
    
    return remainingTime > 0 ? Math.ceil(remainingTime / 1000) : 0;
  }

  /**
   * 获取开始时间
   * @returns {number | null} 开始时间戳
   */
  static getStartTime(): number | null {
    const startTimeStr = sessionStorage.getItem(this.START_TIME_KEY);
    return startTimeStr ? parseInt(startTimeStr, 10) : null;
  }

  /**
   * 清除计时器
   */
  static clearTimer(): void {
    sessionStorage.removeItem(this.START_TIME_KEY);
  }

  /**
   * 格式化剩余时间显示
   * @returns {string} 格式化的时间字符串（如：02:30）
   */
  static formatRemainingTime(): string {
    const remainingSeconds = this.getRemainingTime();
    if (remainingSeconds <= 0) {
      return '00分钟00秒';
    }
    
    const minutes = Math.floor(remainingSeconds / 60);
    const seconds = remainingSeconds % 60;
    
    return `${minutes.toString().padStart(2, '0')}分钟${seconds.toString().padStart(2, '0')}秒`;
  }

  /**
   * 获取超时提示信息
   * @returns {string} 超时提示信息
   */
  static getTimeoutMessage(): string {
    return '操作时间已超时，请重新购买';
  }
}