package com.zbkj.front.controller;


import cn.hutool.core.date.DateUtil;
import com.github.pagehelper.util.StringUtil;
import com.google.common.collect.Lists;
import com.zbkj.common.constants.SysConfigConstants;
import com.zbkj.common.dto.*;
import com.zbkj.common.exception.CrmebException;
import com.zbkj.common.exception.CrmebWalletException;
import com.zbkj.common.model.bcx.BcxPlatformFee;
import com.zbkj.common.model.huifu.*;
import com.zbkj.common.page.CommonPage;
import com.zbkj.common.result.CommonResult;
import com.zbkj.common.utils.RedisUtil;
import com.zbkj.service.config.PaymentChannel;
import com.zbkj.service.factory.PaymentFactory;
import com.zbkj.service.service.*;
import com.zbkj.service.service.bcx.BcxPlatformFeeService;
import com.zbkj.service.wangshang.handle.CommomRequstHandle;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.Base64;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * 首页控制器
 * +----------------------------------------------------------------------
 * | CRMEB [ CRMEB赋能开发者，助力企业发展 ]
 * +----------------------------------------------------------------------
 * | Copyright (c) 2016~2023 https://www.crmeb.com All rights reserved.
 * +----------------------------------------------------------------------
 * | Licensed CRMEB并不是自由软件，未经许可不能去掉CRMEB相关版权
 * +----------------------------------------------------------------------
 * | Author: CRMEB Team <admin@crmeb.com>
 * +----------------------------------------------------------------------
 */
@Slf4j
@RestController
@RequestMapping("api/front/wallet")
@Api(tags = "企业钱包服务")
public class HuifuWalletController {


    @Autowired
    private CommonWalletService commonWalletService;

    @Autowired
    private SystemConfigService systemConfigService;
    @Autowired
    private PaymentFactory paymentFactory;
    @Autowired
    private PayService payService;
    @Resource
    private BcxPlatformFeeService bcxPlatformFeeService;
    @Autowired
    private CommonConfigService commonConfigService;

    @Autowired
    private CommomRequstHandle commomRequstHandle;
    @Autowired
    private RedissonClient redisson;

    @Autowired
    private RedisUtil redisUtil;

    private final static String TRANSFERKEY = "Transfer:";

    /*static {
        huifuWalletService = paymentFactory.getPaymentChannel(channel);
    }*/

    /**
     * 查询银行列表信息
     *
     * @param bankConfig
     * @author llc
     * @date 2019-06-21
     */
    @ApiOperation(value = "查询银行列表信息", httpMethod = "GET", notes = "查询银行列表信息", response = CommonResult.class)
    @PostMapping(value = "find-banks")
    public CommonResult<List<BankConfig>> findBank(@RequestBody BankConfig bankConfig) throws CrmebWalletException {
        commonConfigService.getAccoutCode();
        return CommonResult.success(commonWalletService.findBank(bankConfig));
    }

    /**
     * 获取不同支付渠道企业钱包服务开通情况
     *
     * @param huifuWallet
     * @author llc
     * @date 2019-06-21
     */
    @ApiOperation(value = "获取不同支付渠道企业钱包服务开通情况", httpMethod = "GET", notes = "获取企业钱包服务开通情况", response = CommonResult.class)
    @GetMapping(value = "find-wallet-service-open-info-by-channel")
    public CommonResult<List> getWalletServiceOpenInfoByChannel(HuifuWallet huifuWallet) throws CrmebWalletException {
        huifuWallet.setUniId(commonConfigService.getAccoutCode());
        List<Map<String, String>> list = Lists.newArrayList();
        Map<String, String> map;
        for (PaymentChannel pc : PaymentChannel.values()){
            map = paymentFactory.getPaymentChannel(pc.getService()).getHuifuWalletServiceOpenInfo(huifuWallet);
            map.put("channel", pc.getName());
            list.add(map);
        }
        return CommonResult.success(list);
    }

    /**
     * 设置当前使用的支付通道
     *
     * @param requestDto 包含 channel 字段
     * @author llc
     * @date 2025-07-01
     */
    @ApiOperation(value = "设置当前使用的支付通道", httpMethod = "POST", notes = "设置当前使用的支付通道", response = CommonResult.class)
    @PostMapping(value = "set-current-payment-channel")
    public CommonResult<String> setCurrentPaymentChannel(@RequestBody Map<String, String> requestDto) {
        String channel = requestDto.get("channel");
        if (channel == null || PaymentChannel.getByName(channel) == null) {
            return CommonResult.failed("无效的支付通道");
        }

        // 将 channel 存储到 session 或 redis 中，以用户为维度保存
        String uniId = commonConfigService.getAccoutCode();
        String cacheKey = "user_payment_channel:" + uniId;
        redisUtil.set(cacheKey, channel, 30 * 60L);
        return CommonResult.success("支付通道设置成功");
    }

    private HuifuWalletService getCurrentPaymentChannelService() {
        String uniId = commonConfigService.getAccoutCode();
        String cacheKey = "user_payment_channel:" + uniId;
        String selectedChannel = (String) redisUtil.get(cacheKey);

        if (selectedChannel == null) {
            throw new CrmebException("未设置支付通道，请刷新页面后重新进入操作");
        }

        PaymentChannel pc = PaymentChannel.getByName(selectedChannel);
        if (pc == null) {
            throw new CrmebException("无效的支付通道");
        }

        HuifuWalletService service;
        try {
            service = paymentFactory.getPaymentChannel(pc.getService());
        } catch (CrmebWalletException e) {
            throw new CrmebException("支付通道获取异常");
        }
        if (service == null) {
            throw new CrmebException("不支持的支付通道服务");
        }

        return service;
    }

    /**
     * 获取企业钱包服务开通情况
     * 【百课堂在调用，勿加权限校验】
     *
     * @param huifuWallet
     * @author llc
     * @date 2019-06-21
     */
    @ApiOperation(value = "获取企业钱包服务开通情况", httpMethod = "GET", notes = "获取企业钱包服务开通情况", response = CommonResult.class)
    @GetMapping(value = "get-huifu-wallet-service-open-info")
    public CommonResult<Map<String, String>> getHuifuWalletServiceOpenInfo(HuifuWallet huifuWallet) {
        huifuWallet.setUniId(commonConfigService.getAccoutCode());
        HuifuWalletService service = getCurrentPaymentChannelService();
        return CommonResult.success(service.getHuifuWalletServiceOpenInfo(huifuWallet));
    }

    /**
     * 获取企业钱包服务当前开通步骤(因富民银行开户流程调整作废)
     *
     * @param huifuWalletProcess
     * @author llc
     * @date 2019-06-21
     */
    @ApiOperation(value = "获取企业钱包服务当前开通步骤", httpMethod = "GET", notes = "获取企业钱包服务当前开通步骤", response = CommonResult.class)
    @GetMapping(value = "get-huifu-wallet-current-process")
    public CommonResult<HuifuWalletProcess> getHuifuWalletCurrentProcess(HuifuWalletProcess huifuWalletProcess) {
        huifuWalletProcess.setUniId(commonConfigService.getAccoutCode());
        HuifuWalletService service = getCurrentPaymentChannelService();
        return CommonResult.success(service.getHuifuWalletCurrentProcess(huifuWalletProcess));
    }

    /**
     * 保存开户资料
     *
     * @param dto
     * @author llc
     * @date 2019-06-21
     * @update by llc 2020-09-10 保存开户资料第一步、第二步整合成保存开户资料
     */
    @ApiOperation(value = "保存开户资料", httpMethod = "POST", notes = "保存开户资料", response = CommonResult.class)
    @PostMapping(value = "save-account-info")
    public CommonResult<String> saveAccountInfo(@RequestBody HuifuWalletAccountDto dto) {
        dto.setUniId(commonConfigService.getAccoutCode());
        String lockId = TRANSFERKEY+dto.getUniId();
        RLock lock = redisson.getLock(lockId);
        try {
            if(lock.tryLock(3,10, TimeUnit.SECONDS)){
                log.debug(lockId+":getLock: " + Thread.currentThread().getId());
                HuifuWalletService service = getCurrentPaymentChannelService();
                return CommonResult.success(service.saveAccountInfo(dto));
            }else{
                return CommonResult.failed("资源繁忙，请稍后重试");
            }
        } catch (Exception e) {
            log.error("saveAccountInfo error", e);
            return CommonResult.failed(e.getMessage());
        }finally {
            if (lock.isHeldByCurrentThread()) {
                log.debug(lockId+":unLock: " + Thread.currentThread().getId());
                lock.unlock();
            }
        }
    }


    /**
     * 保存开户资料第二步（作废）
     *
     * @param dto
     * @author llc
     * @date 2019-06-21
     */
    @ApiOperation(value = "保存开户资料第二步", httpMethod = "POST", notes = "保存开户资料第二步", response = CommonResult.class)
    @PostMapping(value = "save-account-info-step-two")
    public CommonResult<String> saveAccountInfoStepTwo(@RequestBody HuifuWalletBusiDto dto) {
        dto.setUniId(commonConfigService.getAccoutCode());
        String lockId = TRANSFERKEY+dto.getUniId();
        RLock lock = redisson.getLock(lockId);
        try {
            if(lock.tryLock(3,10, TimeUnit.SECONDS)){
                log.debug(lockId+":getLock: " + Thread.currentThread().getId());
                HuifuWalletService service = getCurrentPaymentChannelService();
                return CommonResult.success(service.saveAccountInfoStepTwo(dto));
            }else{
                return CommonResult.failed("资源繁忙，请稍后重试");
            }
        } catch (Exception e) {
            log.error("saveAccountInfoStepTwo error", e);
            return CommonResult.failed(e.getMessage());
        }finally {
            if (lock.isHeldByCurrentThread()) {
                log.debug(lockId+":unLock: " + Thread.currentThread().getId());
                lock.unlock();
            }
        }
    }

    /**
     * <b> 获取入驻状态 </b>
     * @author ZXF
     * @create 2025/06/14 0014 11:56
     * @version
     * @注意事项 </b>
     */
    @ApiOperation(value = "获取入驻状态", httpMethod = "GET", notes = "获取入驻状态", response = CommonResult.class)
    @GetMapping(value = "get-huifu-wallet-account-status")
    public CommonResult<HuifuWalletAccountDto> getHuifuWalletAccountStatus() throws CrmebWalletException {
        HuifuWalletAccount huifuWalletAccount = new HuifuWalletAccount();
        huifuWalletAccount.setUniId(commonConfigService.getAccoutCode());
        HuifuWalletService service = getCurrentPaymentChannelService();
        return CommonResult.success(service.getHuifuWalletAccountStatus(huifuWalletAccount));
    }

    /**
     * <b> 获取商户修改状态 </b>
     * @author ZXF
     * @create 2025/06/14 0014 11:56
     * @version
     * @注意事项 </b>
     */
    @ApiOperation(value = "获取商户修改状态", httpMethod = "GET", notes = "获取商户修改状态", response = CommonResult.class)
    @GetMapping(value = "huifu-wallet-account-update-status")
    public CommonResult<HuifuWalletAccountDto> huifuWalletAccountUpdateStatus() throws CrmebWalletException {
        HuifuWalletAccount huifuWalletAccount = new HuifuWalletAccount();
        huifuWalletAccount.setUniId(commonConfigService.getAccoutCode());
        HuifuWalletService service = getCurrentPaymentChannelService();
        return CommonResult.success(service.huifuWalletAccountUpdateStatus(huifuWalletAccount));
    }

    /**
     * <b> 激活状态查询 </b>
     * @author ZXF
     * @create 2025/06/14 0014 11:56
     * @version
     * @注意事项 </b>
     */
    @ApiOperation(value = "激活状态查询", httpMethod = "GET", notes = "激活状态查询", response = CommonResult.class)
    @GetMapping(value = "huifu-wallet-account-active-status")
    public CommonResult<HuifuWalletAccountDto> huifuWalletAccountActiveStatus() throws CrmebWalletException {
        HuifuWalletAccount huifuWalletAccount = new HuifuWalletAccount();
        huifuWalletAccount.setUniId(commonConfigService.getAccoutCode());
        HuifuWalletService service = getCurrentPaymentChannelService();
        return CommonResult.success(service.huifuWalletAccountActiveStatus(huifuWalletAccount));
    }

    /**
     * <b> 获取商户信息 </b>
     * @author ZXF
     * @create 2025/06/14 0014 11:56
     * @version
     * @注意事项 </b>
     */
    @ApiOperation(value = "获取商户信息", httpMethod = "GET", notes = "获取商户信息", response = CommonResult.class)
    @GetMapping(value = "get-huifu-wallet-account-detail")
    public CommonResult<Object> getHuifuWalletAccountDetail() throws CrmebWalletException {
        HuifuWalletAccount huifuWalletAccount = new HuifuWalletAccount();
        huifuWalletAccount.setUniId(commonConfigService.getAccoutCode());
        HuifuWalletService service = getCurrentPaymentChannelService();
        return CommonResult.success(service.getHuifuWalletAccountDetail(huifuWalletAccount));
    }

    /**
     * <b> 解约申请审核确认 </b>
     * @author ZXF
     * @create 2025/06/14 0014 11:56
     * @version
     * @注意事项 </b>
     */
    @ApiOperation(value = "解约申请审核确认", httpMethod = "POST", notes = "解约申请审核确认", response = CommonResult.class)
    @PostMapping(value = "check-arrangement-audit")
    public CommonResult<String> checkArrangementAudit(@RequestBody HuifuWalletAccountDto dto) throws CrmebWalletException {
        dto.setUniId(commonConfigService.getAccoutCode());
        HuifuWalletService service = getCurrentPaymentChannelService();
        return CommonResult.success(service.checkArrangementAudit(dto));
    }

    /**
     * <b> 获取入驻激活地址 </b>
     * @author ZXF
     * @create 2025/06/14 0014 12:23
     * @version
     * @注意事项 </b>
     */
    @ApiOperation(value = "获取入驻激活地址", httpMethod = "GET", notes = "获取入驻激活地址", response = CommonResult.class)
    @GetMapping(value = "get-active-url")
    public CommonResult<String> getActiveUrl() throws CrmebWalletException {
        HuifuWalletAccount huifuWalletAccount = new HuifuWalletAccount();
        huifuWalletAccount.setUniId(commonConfigService.getAccoutCode());
        HuifuWalletService service = getCurrentPaymentChannelService();
        return CommonResult.success(service.getActiveUrl(huifuWalletAccount));
    }
    /**
     * <b> 组装登录小程序二维码 </b>
     * @author ZXF
     * @create 2025/06/14 0014 12:23
     * @version
     * @注意事项 </b>
     */
    @ApiOperation(value = "组装登录小程序二维码", httpMethod = "GET", notes = "组装登录小程序二维码", response = CommonResult.class)
    @GetMapping(value = "get-xcx-login-url")
    public CommonResult<String> getXCXLoginUrl() throws CrmebWalletException {
        HuifuWalletAccount huifuWalletAccount = new HuifuWalletAccount();
        huifuWalletAccount.setUniId(commonConfigService.getAccoutCode());
        HuifuWalletService service = getCurrentPaymentChannelService();
        return CommonResult.success(service.getACXLoginUrl(huifuWalletAccount));
    }

    /**
     * 获取企业钱包账户
     *
     * @author llc
     * @date 2019-06-21
     */
    @ApiOperation(value = "获取企业钱包账户", httpMethod = "GET", notes = "获取企业钱包账户", response = CommonResult.class)
    @GetMapping(value = "get-huifu-wallet-account")
    public CommonResult<HuifuWalletAccountDto> getHuifuWalletAccount() throws CrmebWalletException {
        HuifuWalletAccount huifuWalletAccount = new HuifuWalletAccount();
        huifuWalletAccount.setUniId(commonConfigService.getAccoutCode());
        HuifuWalletService service = getCurrentPaymentChannelService();
        return CommonResult.success(service.getHuifuWalletAccount(huifuWalletAccount));
    }

    /**
     * <b> 商户余额接口 </b>
     * @author ZXF
     * @create 2025/06/23 0023 11:48
     * @version
     * @注意事项 </b>
     */
    @ApiOperation(value = "商户余额接口", httpMethod = "GET", notes = "商户余额接口", response = CommonResult.class)
    @GetMapping(value = "get-huifu-wallet-account-amount")
    public CommonResult<Map<String, String>> getHuifuWalletAccountAmount() throws CrmebWalletException {
        HuifuWalletAccount huifuWalletAccount = new HuifuWalletAccount();
        huifuWalletAccount.setUniId(commonConfigService.getAccoutCode());
        HuifuWalletService service = getCurrentPaymentChannelService();
        return CommonResult.success(service.getHuifuWalletAccountAmount(huifuWalletAccount));
    }

    /**
     * <b> 获取企业钱包账户余额 </b>
     * @author ZXF
     * @create 2025/06/23 0023 11:48
     * @version
     * @注意事项 </b>
     */
    @ApiOperation(value = "获取企业钱包账户余额", httpMethod = "GET", notes = "获取企业钱包账户余额", response = CommonResult.class)
    @GetMapping(value = "get-account-blanance")
    public CommonResult<Map<String, String>> getAccountBlanance() throws CrmebWalletException {
        HuifuWalletAccount huifuWalletAccount = new HuifuWalletAccount();
        huifuWalletAccount.setUniId(commonConfigService.getAccoutCode());
        HuifuWalletService service = getCurrentPaymentChannelService();
        return CommonResult.success(service.getAccountBlanance(huifuWalletAccount));
    }


    /**
     * 获取企业钱包账户交易明细
     *
     * @param dto
     * @author llc
     * @date 2019-06-21
     */
    @ApiOperation(value = "获取企业钱包账户交易明细", httpMethod = "GET", notes = "获取企业钱包账户交易明细", response = CommonResult.class)
    @PostMapping(value = "get-huifu-wallet-account-trade-detail")
    public CommonPage<HuifuWalletAccountDetail> getHuifuWalletAccountTradeDetail(@RequestBody HuifuWalletAccountDetailDto dto) throws CrmebWalletException {
        dto.setUniId(commonConfigService.getAccoutCode());
        return CommonPage.restPage(commonWalletService.thisHuifuWalletAccountTradeDetail(dto));
    }


    /**
     * 根据交易明细记录生成收支明细-任务
     *
     * @author lp
     * @date 2019-12-09
     */
    @ApiOperation(value = "根据交易明细记录生成收支明细", httpMethod = "POST", notes = "根据交易明细记录生成收支明细", response = CommonResult.class)
    @PostMapping(value = "huifu-wallet-account-trade-detail-task")
    public CommonResult<String> huifuWalletAccountTradeDetailTask(HuifuWalletAccountDto dto) {
        commonConfigService.getAccoutCode();
        HuifuWalletService service = getCurrentPaymentChannelService();
        return CommonResult.success(service.huifuWalletAccountTradeDetailTask(dto.getStartDate(), dto.getEndDate()));
    }

    /**
     * 将钱包收支明细保存到本地-任务 余额支付
     *
     * @author lp
     * @date 2019-12-09
     */
    @ApiOperation(value = "将钱包收支明细保存到本地 余额支付", httpMethod = "POST", notes = "将钱包收支明细保存到本地 余额支付", response = CommonResult.class)
    @PostMapping(value = "yepay-detail-task")
    public CommonResult<String> yepayDetailTask() throws CrmebWalletException {
        commonConfigService.getAccoutCode();
        paymentFactory.getPaymentChannel(PaymentChannel.HUIFU.getService()).payTradeDetailTaskSch();
        paymentFactory.getPaymentChannel(PaymentChannel.WANGSHANG.getService()).payTradeDetailTaskSch();
        return CommonResult.success();
    }


    /**
     * 获取企业钱包充值token
     *
     * @param dto
     * @author llc
     * @date 2019-06-25
     */
    @ApiOperation(value = "获取企业钱包充值token", httpMethod = "POST", notes = "获取企业钱包充值token", response = CommonResult.class)
    @PostMapping(value = "get-wallet-recharge-token")
    public CommonResult<String> getWalletRechargeToken(@RequestBody HuifuWalletAccountDto dto) throws CrmebWalletException {
        dto.setUniId(commonConfigService.getAccoutCode());
        HuifuWalletService service = getCurrentPaymentChannelService();
        return CommonResult.success(service.getWalletRechargeToken(dto));
    }

    /**
     * 企业钱包充值回调
     *
     * @param dto
     * @author llc
     * @date 2019-06-25
     */
    @ApiOperation("企业钱包充值回调")
    @PostMapping(value = "huifuWalletRechargeCallback")
    public String huifuWalletRechargeCallback(@RequestBody HuifuRequestDataDto dto) {
        return commonWalletService.huifuWalletRechargeCallback(dto);
    }

    /**
     * 企业钱包提现回调
     *
     * @author llc
     * @date 2019-06-25
     */
    @ApiOperation("企业钱包提现回调")
    @PostMapping(value = "huifuWalletExtractCallback")
    public String huifuWalletExtractCallback(HttpServletRequest request) {
        String messageBody = (String)request.getAttribute("requestBodyMessage");
        return commonWalletService.huifuWalletExtractCallback(messageBody);
    }

    /**
     * <b> C端聚合正扫回调 </b>
     * @author ZXF
     * @create 2023/08/25 0025 16:13
     * @version
     * @注意事项 </b>
     */
    @ApiOperation("C端聚合正扫回调")
    @PostMapping(value = "huifuJspayCallback")
    public String huifuJspayCallback(HttpServletRequest request) {
        String messageBody = (String)request.getAttribute("requestBodyMessage");
        return payService.huifuJspayCallback(messageBody);
    }

    /**
     * <b> C端聚合正扫退款回调 </b>
     * @author ZXF
     * @create 2023/08/25 0025 16:13
     * @version
     * @注意事项 </b>
     */
    @ApiOperation("C端聚合正扫退款回调")
    @PostMapping(value = "huifuJspayRefundCallback")
    public String huifuJspayRefundCallback(HttpServletRequest request) {
        String messageBody = (String)request.getAttribute("requestBodyMessage");
        return payService.huifuJspayRefundCallback(messageBody);
    }

    /**
     * <b> 根据业务编码或业务交易流水号查询交易信息 </b>
     * @author ZXF
     * @create 2023/03/23 0023 10:23
     * @version
     * @注意事项 </b>
     */
    @ApiOperation(value = "根据业务编码或业务交易流水号查询交易信息", httpMethod = "POST", notes = "根据业务编码或业务交易流水号查询交易信息", response = CommonResult.class)
    @PostMapping(value = "find-wallet-trade-info-by-business")
    public CommonResult<List<HuifuWalletTrade>> getHuifuWalletTradeBySeqAndCode(@RequestBody HuifuWalletTrade huifuWalletTrade) throws CrmebWalletException {
        commonConfigService.getAccoutCode();
        return CommonResult.success(commonWalletService.getHuifuWalletTradeBySeqAndCode(huifuWalletTrade));
    }

    /**
     * 获取企业充值交易状态
     *
     * @param huifuWalletTrade
     * @author llc
     * @date 2019-06-28
     */
    @ApiOperation(value = "获取企业充值交易状态", httpMethod = "GET", notes = "获取企业充值交易状态", response = CommonResult.class)
    @GetMapping(value = "get-wallet-trade-info")
    public CommonResult<HuifuWalletTrade> getWalletTradeInfo( HuifuWalletTrade huifuWalletTrade) throws CrmebWalletException {
        huifuWalletTrade.setUniId(commonConfigService.getAccoutCode());
        return CommonResult.success(commonWalletService.getWalletTradeInfo(huifuWalletTrade));
    }


    /**
     * 企业钱包账户提现
     *
     * @param dto
     * @author llc
     * @date 2019-06-25
     */
    @ApiOperation(value = "企业钱包账户提现", httpMethod = "POST", notes = "企业钱包账户提现", response = CommonResult.class)
    @PostMapping(value = "huifu-wallet-account-extract")
    public CommonResult<String> huifuWalletAccountExtract(@RequestBody HuifuWalletAccountDto dto) throws CrmebWalletException {
        dto.setUniId(commonConfigService.getAccoutCode());
        String lockId = TRANSFERKEY+dto.getUniId();
        RLock lock = redisson.getLock(lockId);
        try {
            if(lock.tryLock(3,10, TimeUnit.SECONDS)){
                log.debug(lockId+":getLock: " + Thread.currentThread().getId());
                HuifuWalletService service = getCurrentPaymentChannelService();
                return CommonResult.success(service.huifuWalletAccountExtract(dto));
            }else{
                return CommonResult.failed("资源繁忙，请稍后重试");
            }
        } catch (InterruptedException e) {
            log.error("doCatchRecommendUser error", e);
            return CommonResult.failed("操作失败，请稍后重试");
        }finally {
            if (lock.isHeldByCurrentThread()) {
                log.debug(lockId+":unLock: " + Thread.currentThread().getId());
                lock.unlock();
            }
        }
    }

    /**
     * <b> 企业钱包账户提现确认（传上一步的订单号和验证码） </b>
     * @author ZXF
     * @create 2025/06/14 0014 15:01
     * @version
     * @注意事项 </b>
     */
    @ApiOperation(value = "企业钱包账户提现确认", httpMethod = "POST", notes = "企业钱包账户提现确认", response = CommonResult.class)
    @PostMapping(value = "huifu-wallet-account-extract-confirm")
    public CommonResult<String> huifuWalletAccountExtractConfirm(@RequestBody HuifuWalletAccountDto dto) throws CrmebWalletException {
        dto.setUniId(commonConfigService.getAccoutCode());
        String lockId = TRANSFERKEY+dto.getUniId();
        RLock lock = redisson.getLock(lockId);
        try {
            if(lock.tryLock(3,10, TimeUnit.SECONDS)){
                log.debug(lockId+":getLock: " + Thread.currentThread().getId());
                HuifuWalletService service = getCurrentPaymentChannelService();
                return CommonResult.success(service.huifuWalletAccountExtractConfirm(dto));
            }else{
                return CommonResult.failed("资源繁忙，请稍后重试");
            }
        } catch (InterruptedException e) {
            log.error("doCatchRecommendUser error", e);
            return CommonResult.failed("操作失败，请稍后重试");
        }finally {
            if (lock.isHeldByCurrentThread()) {
                log.debug(lockId+":unLock: " + Thread.currentThread().getId());
                lock.unlock();
            }
        }
    }

    /**
     * <b> 企业钱包账户可退回列表查询 </b>
     * @author ZXF
     * @create 2025/10/11 0011 9:37
     * @version
     * @注意事项 </b>
     */
    @ApiOperation(value = "企业钱包账户可退回列表查询", httpMethod = "POST", notes = "企业钱包账户可退回列表查询", response = CommonResult.class)
    @PostMapping(value = "batch-query")
    public CommonResult<String> batchQuery(@RequestBody HuifuWalletAccountDto dto) throws CrmebWalletException {
        dto.setUniId(commonConfigService.getAccoutCode());
        HuifuWalletService service = getCurrentPaymentChannelService();
        return CommonResult.success(service.batchQuery(dto));
    }

    /**
     * <b> 企业钱包账户支付退回查询 </b>
     * @author ZXF
     * @create 2025/10/11 0011 9:37
     * @version
     * @注意事项 </b>
     */
    @ApiOperation(value = "企业钱包账户支付退回查询", httpMethod = "POST", notes = "企业钱包账户支付退回查询", response = CommonResult.class)
    @PostMapping(value = "refund-query")
    public CommonResult<String> refundQuery(@RequestBody HuifuWalletAccountDto dto) throws CrmebWalletException {
        dto.setUniId(commonConfigService.getAccoutCode());
        HuifuWalletService service = getCurrentPaymentChannelService();
        return CommonResult.success(service.refundQuery(dto));
    }

    /**
     * <b> 企业钱包账户订单退回 </b>
     * @author ZXF
     * @create 2025/10/11 0011 9:38
     * @version
     * @注意事项 </b>
     */
    @ApiOperation(value = "企业钱包账户订单退回", httpMethod = "POST", notes = "企业钱包账户订单退回", response = CommonResult.class)
    @PostMapping(value = "refund-apply")
    public CommonResult<String> refundApply(@RequestBody HuifuWalletAccountDto dto) throws CrmebWalletException {
        dto.setUniId(commonConfigService.getAccoutCode());
        String lockId = TRANSFERKEY+dto.getUniId();
        RLock lock = redisson.getLock(lockId);
        try {
            if(lock.tryLock(3,10, TimeUnit.SECONDS)){
                log.debug(lockId+":getLock: " + Thread.currentThread().getId());
                HuifuWalletService service = getCurrentPaymentChannelService();
                return CommonResult.success(service.refundApply(dto));
            }else{
                return CommonResult.failed("资源繁忙，请稍后重试");
            }
        } catch (InterruptedException e) {
            log.error("doCatchRecommendUser error", e);
            return CommonResult.failed("操作失败，请稍后重试");
        }finally {
            if (lock.isHeldByCurrentThread()) {
                log.debug(lockId+":unLock: " + Thread.currentThread().getId());
                lock.unlock();
            }
        }
    }

    /**
     * <b> 测试离线对账文件查询 </b>
     * @author ZXF
     * @create 2025/06/14 0014 15:01
     * @version
     * @注意事项 </b>
     */
    @ApiOperation(value = "测试离线对账文件查询", httpMethod = "GET", notes = "测试离线对账文件查询", response = CommonResult.class)
    @GetMapping(value = "executeReconcile")
    public CommonResult<Map> executeReconcile(@RequestBody HuifuWalletAccountDto dto) throws CrmebWalletException {
        return CommonResult.success(paymentFactory.getPaymentChannel(PaymentChannel.WANGSHANG.getService()).executeReconcile("20250724"));
    }

    @ResponseBody
    @PostMapping(value = "/notify", produces = MediaType.APPLICATION_XML_VALUE)
    public void mybankNotifyManage(HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse) throws Exception {
        //获取请求报文
        String xmlContext = commomRequstHandle.getXmlContextString(httpServletRequest);

        //校验请求数据验签 及 校验appId; 校验不过会抛出异常
        //commomRequstHandle.checkNotifyRequest(xmlContext);

        //处理通知
//        commomRequstHandle.notifyRequstHandle(xmlContext);
        httpServletResponse.setStatus(200);
        httpServletResponse.setContentType("application/xml");
        httpServletResponse.setCharacterEncoding("utf-8");
        try {
            httpServletResponse.getWriter().print(commomRequstHandle.notifyRequstHandle(xmlContext));
        } catch (IOException ioException) {
            ioException.printStackTrace();
        }

    }

    /**
     * <b> 企业账户金额冻结/解冻 </b>
     * @author ZXF
     * @create 2023/03/29 0029 17:20
     * @version
     * @注意事项 </b>
     */
    @ApiOperation(value = "企业消费转账接口", httpMethod = "POST", notes = "企业账户金额冻结/解冻", response = CommonResult.class)
    @PostMapping(value = "huifu-wallet-account-lock")
    public CommonResult<String> huifuWalletAccountLock(@RequestBody HuifuWalletTransferDto dto) {
        commonConfigService.getAccoutCode();
        if (StringUtil.isEmpty(dto.getTradeType())) {
            throw new CrmebException("交易类型不能为空");
        }
        if (dto.getOutId() == null) {
            throw new CrmebException("企业出账ID不能为空");
        }
        if (dto.getInId() == null) {
            throw new CrmebException("企业入账ID不能为空");
        }
        String comId = "05".equals(dto.getTradeType())?dto.getInId():dto.getOutId();
        String lockId = TRANSFERKEY+comId;
        RLock lock = redisson.getLock(lockId);
        try {
            if(lock.tryLock(3,10, TimeUnit.SECONDS)){
                log.debug(lockId+":getLock: " + Thread.currentThread().getId());
                HuifuWalletService service = getCurrentPaymentChannelService();
                return CommonResult.success(service.huifuWalletAccountLock(dto));
            }else{
                return CommonResult.failed("资源繁忙，请稍后重试");
            }
        } catch (Exception e) {
            log.error("doCatchRecommendUser error", e);
            return CommonResult.failed(e.getMessage());
        }finally {
            if (lock.isHeldByCurrentThread()) {
                log.debug(lockId+":unLock: " + Thread.currentThread().getId());
                lock.unlock();
            }
        }
    }

    /**
     * 企业消费转账接口
     * 【百课堂在调用，勿加权限校验】
     *
     * @author lp
     * @date 2019-08-20
     */
    @ApiOperation(value = "企业消费转账接口", httpMethod = "POST", notes = "企业消费转账接口", response = CommonResult.class)
    @PostMapping(value = "huifu-wallet-account-transfer")
    public CommonResult<String> huifuWalletAccountTransfer(@RequestBody HuifuWalletTransferDto dto) {
        commonConfigService.getAccoutCode();
        if (dto.getOutId() == null) {
            throw new CrmebException("企业出账ID不能为空");
        }
        if (dto.getInId() == null) {
            throw new CrmebException("企业入账ID不能为空");
        }
        if (StringUtil.isEmpty(dto.getTradeType())) {
            throw new CrmebException("交易类型不能为空");
        }

        String lockId = TRANSFERKEY+dto.getOutId();
        RLock lock = redisson.getLock(lockId);
        try {
            if(lock.tryLock(3,10, TimeUnit.SECONDS)){
                log.debug(lockId+":getLock: " + Thread.currentThread().getId());
                HuifuWalletService service = getCurrentPaymentChannelService();
                return CommonResult.success(service.huifuWalletAccountTransfer(dto));
            }else{
                return CommonResult.failed("资源繁忙，请稍后重试");
            }
        } catch (Exception e) {
            log.error("doCatchRecommendUser error", e);
            return CommonResult.failed(e.getMessage());
        }finally {
            if (lock.isHeldByCurrentThread()) {
                log.debug(lockId+":unLock: " + Thread.currentThread().getId());
                lock.unlock();
            }
        }
    }

    /**
     * <b> 订单退款申请 </b>
     * @author ZXF
     * @create 2025/06/23 0023 20:55
     * @version
     * @注意事项 </b>
     */
    @ApiOperation(value = "订单退款申请", httpMethod = "POST", notes = "订单退款申请", response = CommonResult.class)
    @PostMapping(value = "huifu-wallet-account-refund-apply")
    public CommonResult<String> huifuWalletAccountRefundApply(@RequestBody HuifuWalletAccountDto dto) throws CrmebWalletException {
        dto.setUniId(commonConfigService.getAccoutCode());
        return CommonResult.success(paymentFactory.getPaymentChannel(PaymentChannel.WANGSHANG.getService()).huifuWalletAccountRefundApply(dto));
    }

    /**
     * <b> 订单退款结果查询 </b>
     * @author ZXF
     * @create 2025/06/23 0023 20:55
     * @version
     * @注意事项 </b>
     */
    @ApiOperation(value = "订单退款结果查询", httpMethod = "POST", notes = "订单退款结果查询", response = CommonResult.class)
    @PostMapping(value = "huifu-wallet-account-refund-query")
    public CommonResult<Map<String, Object>> huifuWalletAccountRefundQuery(@RequestBody HuifuWalletAccountDto dto) throws CrmebWalletException {
        dto.setUniId(commonConfigService.getAccoutCode());
        return CommonResult.success(paymentFactory.getPaymentChannel(PaymentChannel.WANGSHANG.getService()).huifuWalletAccountRefundQuery(dto));
    }

    /**
     * <b> 查询并生成交易明细记录 </b>
     * @author ZXF
     * @create 2025/06/22 0022 15:27
     * @version
     * @注意事项 </b>
     */
    @ApiOperation(value = "查询并生成交易明细记录", httpMethod = "POST", notes = "查询并生成交易明细记录", response = CommonResult.class)
    @PostMapping(value = "get-pay-trade-detail-task")
    public CommonResult<String> getPayTradeDetailTask(@RequestBody HuifuWalletTransferDto dto) throws CrmebWalletException {
        commonConfigService.getAccoutCode();
        if (dto.getHfqSeqId() == null) {
            throw new CrmebException("全局流水号不能为空");
        }
        paymentFactory.getPaymentChannel(PaymentChannel.WANGSHANG.getService()).getPayTradeDetailTask(commonConfigService.getAccoutCode(),"",dto.getHfqSeqId(),"");
        return CommonResult.success("1");
    }

    /**
     * 企业退款接口
     *
     * @author lp
     * @date 2019-08-20
     */
    @ApiOperation(value = "企业退款接口", httpMethod = "POST", notes = "企业退款接口", response = CommonResult.class)
    @PostMapping(value = "huifu-wallet-account-refund")
    public CommonResult<String> huifuWalletAccountRefund(@RequestBody HuifuWalletTransferDto dto) {
        commonConfigService.getAccoutCode();
        if (dto.getOutId() == null) {
            throw new CrmebException("企业出账ID不能为空");
        }
        if (dto.getInId() == null) {
            throw new CrmebException("企业入账ID不能为空");
        }
        if (StringUtil.isEmpty(dto.getTradeType())) {
            throw new CrmebException("交易类型不能为空");
        }

        String lockId = TRANSFERKEY+dto.getOutId();
        RLock lock = redisson.getLock(lockId);
        try {
            if(lock.tryLock(3,10, TimeUnit.SECONDS)){
                log.debug(lockId+":getLock: " + Thread.currentThread().getId());
                HuifuWalletService service = getCurrentPaymentChannelService();
                return CommonResult.success(service.huifuWalletAccountRefund(dto));
            }else{
                return CommonResult.failed("资源繁忙，请稍后重试");
            }
        } catch (Exception e) {
            log.error("doCatchRecommendUser error", e);
            return CommonResult.failed(e.getMessage());
        }finally {
            if (lock.isHeldByCurrentThread()) {
                log.debug(lockId+":unLock: " + Thread.currentThread().getId());
                lock.unlock();
            }
        }
    }

    /**
     * 平台分销转账接口
     *
     * @author lp
     * @date 2019-08-20
     */
    @ApiOperation(value = "平台分销转账接口", httpMethod = "POST", notes = "平台分销转账接口", response = CommonResult.class)
    @PostMapping(value = "plat-account-retail-transfer")
    public CommonResult<String> platAccountRetailTransfer(@RequestBody HuifuWalletTransferDto dto) {
        commonConfigService.getAccoutCode();
        if (dto.getInId() == null) {
            throw new CrmebException("企业入账ID不能为空");
        }
        if (StringUtil.isEmpty(dto.getTradeType())) {
            throw new CrmebException("交易类型不能为空");
        }

        String lockId = TRANSFERKEY+dto.getOutId();
        RLock lock = redisson.getLock(lockId);
        try {
            if(lock.tryLock(5,60, TimeUnit.MILLISECONDS)){
                log.debug(lockId+":getLock: " + Thread.currentThread().getId());
                HuifuWalletService service = getCurrentPaymentChannelService();
                return CommonResult.success(service.platAccountRetailTransfer(dto));
            }else{
                return CommonResult.failed("资源繁忙，请稍后重试");
            }
        } catch (Exception e) {
            log.error("doCatchRecommendUser error", e);
            return CommonResult.failed(e.getMessage());
        }finally {
            if (lock.isHeldByCurrentThread()) {
                log.debug(lockId+":unLock: " + Thread.currentThread().getId());
                lock.unlock();
            }
        }
    }


    /**
     * 电子回单
     *
     * @param dto
     * @author lp
     * @date 2019-07-08
     */
    @ApiOperation(value = "电子回单", httpMethod = "POST", notes = "电子回单", response = CommonResult.class)
    @PostMapping(value = "huifu-account-trade-voucher")
    public CommonResult<String> getHuifuAccountTradeVoucher(@RequestBody HuifuWalletTradeDto dto, HttpServletResponse response) throws CrmebWalletException {
        dto.setUniId(commonConfigService.getAccoutCode());
        HuifuWalletService service = getCurrentPaymentChannelService();
        return CommonResult.success(service.getHuifuAccountTradeVoucher(dto));
    }

    /**
     * 测试汇总电子回单查询
     *
     * @param dto
     * @author lp
     * @date 2019-07-08
     */
    @ApiOperation(value = "测试汇总电子回单查询", httpMethod = "POST", notes = "测试汇总电子回单查询", response = CommonResult.class)
    @PostMapping(value = "get-batch-trade-voucher")
    public CommonResult<String> getBatchTradeVoucher(@RequestBody HuifuWalletTradeDto dto, HttpServletResponse response) throws CrmebWalletException {
        dto.setUniId(commonConfigService.getAccoutCode());
        return CommonResult.success(paymentFactory.getPaymentChannel(PaymentChannel.WANGSHANG.getService()).getBatchTradeVoucher(dto));
    }


    private static void downloadBase64(HttpServletResponse response, String base64Code) throws IOException {
        try (ServletOutputStream out = response.getOutputStream()) {
            response.setContentType("application/octet-stream");
            response.setHeader("Content-Disposition", "attachment;filename=" + DateUtil.current(true) + ".pdf");
            Base64.Decoder decoder = Base64.getDecoder();
            byte[] bytes = decoder.decode(base64Code);
            try (ByteArrayInputStream in = new ByteArrayInputStream(bytes)) {
                byte[] buffer = new byte[1024];
                int byteRead = 0;
                while ((byteRead = in.read(buffer)) != -1) {
                    out.write(buffer, 0, byteRead);
                }
                out.flush();
            }
        }
    }

    /**
     * 初始生成化对账单配置
     *
     * @author llc
     * @date 2019-09-16
     */
    @ApiOperation(value = "初始生成化对账单配置", httpMethod = "POST", notes = "初始生成化对账单配置", response = CommonResult.class)
    @PostMapping(value = "set-reconciliation-config")
    public CommonResult<String> setReconciliationConfig() throws CrmebWalletException {
        commonConfigService.getAccoutCode();
        HuifuWalletService service = getCurrentPaymentChannelService();
        return CommonResult.success(service.setReconciliationConfig());
    }

    /**
     * 交易结算对账文件重新生成
     *
     * @author llc
     * @date 2019-09-16
     */
    @ApiOperation(value = "交易结算对账文件重新生成", httpMethod = "POST", notes = "交易结算对账文件重新生成", response = CommonResult.class)
    @PostMapping(value = "reconciliation-replay/{tradeDate}")
    public CommonResult<String> reconciliationReplay(@PathVariable("tradeDate") String tradeDate) {
        commonConfigService.getAccoutCode();
        HuifuWalletService service = getCurrentPaymentChannelService();
        return CommonResult.success(service.reconciliationReplay(tradeDate));
    }

    /**
     * 企业钱包账户对账数据生成
     *
     * @author llc
     * @date 2019-09-16
     */
    @ApiOperation(value = "企业钱包账户对账数据生成", httpMethod = "POST", notes = "企业钱包账户对账数据生成", response = CommonResult.class)
    @PostMapping(value = "create-huifu-wallet-reconciliation")
    public CommonResult<String> createHuifuWalletReconciliation() {
        commonConfigService.getAccoutCode();
        return CommonResult.success(commonWalletService.createHuifuWalletReconciliation());
    }

    /**
     * 获取企业钱包账户提现记录
     *
     * @author llc
     * @date 2019-09-16
     */
    @ApiOperation(value = "获取企业钱包账户提现记录", httpMethod = "POST", notes = "获取企业钱包账户提现记录", response = CommonResult.class)
    @PostMapping(value = "huifu-wallet-account-translog-task")
    public CommonResult<String> huifuWalletAccountTranslogTask() throws CrmebWalletException {
        commonConfigService.getAccoutCode();
        HuifuWalletService service = getCurrentPaymentChannelService();
        return CommonResult.success(service.huifuWalletAccountTranslogTask("2025-07-25","2025-07-25"));
    }

    /**
     * <b> 获取平台下所以收费账单记录 </b>
     * @author ZXF
     * @create 2025/06/24 0024 10:25
     * @version
     * @注意事项 </b>
     */
    @ApiOperation(value = "获取平台下所以收费账单记录", httpMethod = "GET", notes = "获取平台下所以收费账单记录", response = CommonResult.class)
    @GetMapping(value = "huifu-Wallet-Account-Fee-File-Apply")
    public CommonResult<String> huifuWalletAccountFeeFileApply(HttpServletRequest request) throws CrmebWalletException {
        String stmtType = request.getParameter("stmtType");
        String startDate = request.getParameter("startDate");
        String endDate = request.getParameter("endDate");
        return CommonResult.success(paymentFactory.getPaymentChannel(PaymentChannel.WANGSHANG.getService()).huifuWalletAccountFeeFileApply(stmtType,startDate,endDate));
    }

    /**
     * <b> 获取平台下所以收费账单记录 </b>
     * @author ZXF
     * @create 2025/06/24 0024 10:25
     * @version
     * @注意事项 </b>
     */
    @ApiOperation(value = "获取平台下所以收费账单记录", httpMethod = "GET", notes = "获取平台下所以收费账单记录", response = CommonResult.class)
    @GetMapping(value = "huifu-Wallet-Account-Fee-File-query")
    public CommonResult<Map<String, Object>> huifuWalletAccountFeeFileQuery(HttpServletRequest request) throws CrmebWalletException {
        String stmtType = request.getParameter("stmtType");
        String outRequestNo = request.getParameter("outRequestNo");
        String backOrderNo = request.getParameter("backOrderNo");
        return CommonResult.success(paymentFactory.getPaymentChannel(PaymentChannel.WANGSHANG.getService()).huifuWalletAccountFeeFileQuery(stmtType,outRequestNo,backOrderNo));
    }


    /**
     * 获取企业钱包账户对账数据
     *
     * @author llc
     * @date 2019-09-19
     */
    @ApiOperation(value = "获取企业钱包账户对账数据", httpMethod = "POST", notes = "获取企业钱包账户对账数据", response = CommonResult.class)
    @PostMapping(value = "get-huifu-wallet-reconciliation")
    public CommonResult<Map<String, Object>> getHuifuWalletReconciliation(@RequestBody HuifuWalletTradeDto dto) throws CrmebWalletException {
        dto.setUniId(commonConfigService.getAccoutCode());
        return CommonResult.success(commonWalletService.getHuifuWalletReconciliation(dto));
    }

    /**
     * 导出企业钱包账户对账数据
     *
     * @author llc
     * @date 2019-09-20
     */
    @ApiOperation(value = "导出企业钱包账户对账数据", httpMethod = "POST")
    @PostMapping(value = "export-huifu-wallet-reconciliation")
    public void exprokHuifuWalletReconciliation(@RequestBody HuifuWalletTradeDto dto,HttpServletResponse response) throws CrmebWalletException {
        dto.setUniId(commonConfigService.getAccoutCode());
        commonWalletService.exportHuifuWalletReconciliation(dto, response);
    }

    /**
     * <b> 导出企业钱包账户交易明细数据 </b>
     * @author ZXF
     * @create 2023/12/12 0012 14:02
     * @version
     * @注意事项 </b>
     */
    @ApiOperation(value = "导出企业钱包账户交易明细数据", httpMethod = "POST")
    @PostMapping(value = "export-huifu-wallet-account-trade-detail")
    public void exportHuifuWalletAccountTradeDetail(@RequestBody HuifuWalletAccountDetailDto dto,HttpServletResponse response) throws CrmebWalletException {
        dto.setUniId(commonConfigService.getAccoutCode());
        commonWalletService.exportHuifuWalletAccountTradeDetail(dto, response);
    }

    /**
     * <b> 获取企业钱包账户提现信息 </b>
     * @author ZXF
     * @create 2023/08/06 0006 13:21
     * @version
     * @注意事项 </b>
     */
    @ApiOperation(value = "获取企业钱包账户提现信息", httpMethod = "POST", notes = "获取企业钱包账户提现信息", response = CommonResult.class)
    @PostMapping(value = "get-huifu-wallet-translog")
    public CommonPage<HuifuWalletTranslog> getHuifuWalletSettlement(@RequestBody HuifuWalletTranslogDto dto) throws CrmebWalletException {
        dto.setUniId(commonConfigService.getAccoutCode());
        return CommonPage.restPage(commonWalletService.thisHuifuWalletTranslog(dto));
    }
    
    /**
     * <b> 保存钱包银行卡信息变更 </b>
     *
     * @author ZXF
     * @create 2019/09/17 0017 15:45
     * @version
     * @注意事项 </b>
     */
//    @ApiOperation(value = "保存钱包银行卡信息变更", httpMethod = "POST", notes = "保存钱包银行卡信息变更", response = CommonResult.class)
//    @PostMapping(value = "save-bank-card-change-event")
//    public CommonResult<> saveBankCardChangeEvent(HuifuWalletChangeDto dto) {
//        Integer uid = commonConfigService.getAccoutCode();
//        if (uid == null) {
//            dto.setUniId(uid);
//        }
//        return CommonResult.success(huifuWalletChangeService.saveBankCardChangeEvent(dto));
//    }

    /**
     * <b> C端转账接口 - 聚合正扫 </b>
     * @author ZXF
     * @create 2023/08/25 0025 16:12
     * @version
     * @注意事项 </b>
     */
    @ApiOperation(value = "C端转账接口 - 聚合正扫", httpMethod = "POST", notes = "C端转账接口 - 聚合正扫", response = CommonResult.class)
    @PostMapping(value = "huifu-jspay-transfer")
    public CommonResult<String> huifuJspayTransfer(@RequestBody JsPayTransferDto dto) {
        dto.setUniId(commonConfigService.getAccoutCode()+"");
        if (StringUtil.isEmpty(dto.getBusinessPartnerSeq())) {
            throw new CrmebException("业务交易流水号不能为空");
        }
        BcxPlatformFee platFee = bcxPlatformFeeService.queryByOrderNo(dto.getBusinessPartnerSeq(), 0);
        dto.setSellFeePriority(platFee.getTransPriority());
        dto.setPlatAmount(platFee.getPlatFee() != null ? platFee.getPlatFee().toPlainString() : "0");
        Map<String,String> map = payService.getIdentifyNoByOrderNo(dto.getBusinessPartnerSeq());
        dto.setInId(map.get("inId"));
        dto.setAmount(map.get("amount"));
        if (StringUtil.isEmpty(map.get("inId"))) {
            throw new CrmebException("企业入账ID不存在");
        }
//        String inId = payService.getMerchantIdentifyNoByOrderNo(dto.getBusinessPartnerSeq());
//        dto.setInId(inId);
//        if (StringUtil.isEmpty(inId)) {
//            throw new CrmebException("企业入账ID不存在");
//        }
        //TODO C端支付当前固定走04冻结流程
        dto.setTradeType("04");
        if (StringUtil.isEmpty(dto.getTradeType())) {
            throw new CrmebException("交易类型不能为空");
        }

        String lockId = TRANSFERKEY+dto.getUniId();
        RLock lock = redisson.getLock(lockId);
        try {
            if(lock.tryLock(5,60, TimeUnit.MILLISECONDS)){
                log.debug(lockId+":getLock: " + Thread.currentThread().getId());
                //TODO 临时让订单通过支付
//                payService.setOrderPaySuccess(dto.getBusinessPartnerSeq());
                HuifuWalletService service = paymentFactory.getPaymentChannel(PaymentChannel.getByName(systemConfigService.getValueByKey(SysConfigConstants.CONFIG_PAY_CHANNEL)).getService());
                return CommonResult.success(service.huifuJspayTransfer(dto));
            }else{
                return CommonResult.failed("资源繁忙，请稍后重试");
            }
        } catch (Exception e) {
            log.error("doCatchRecommendUser error", e);
            return CommonResult.failed(e.getMessage());
        }finally {
            if (lock.isHeldByCurrentThread()) {
                log.debug(lockId+":unLock: " + Thread.currentThread().getId());
                lock.unlock();
            }
        }
    }

    /**
     * <b> C端退款接口 - 扫码交易退款 </b>
     * @author ZXF
     * @create 2023/08/25 0025 16:12
     * @version
     * @注意事项 </b>
     */
    @ApiOperation(value = "C端退款接口 - 扫码交易退款", httpMethod = "POST", notes = "C端退款接口 - 扫码交易退款", response = CommonResult.class)
    @PostMapping(value = "huifu-jspay-refund")
    public CommonResult<String> huifuJspayRefund(@RequestBody JsPayTransferDto dto) {
        dto.setUniId(commonConfigService.getAccoutCode()+"");

        String lockId = TRANSFERKEY+dto.getUniId();
        RLock lock = redisson.getLock(lockId);
        try {
            if(lock.tryLock(5,60, TimeUnit.MILLISECONDS)){
                log.debug(lockId+":getLock: " + Thread.currentThread().getId());
                HuifuWalletService service = paymentFactory.getPaymentChannel(PaymentChannel.getByName(systemConfigService.getValueByKey(SysConfigConstants.CONFIG_PAY_CHANNEL)).getService());
                return CommonResult.success(service.huifuJspayRefund(dto));
            }else{
                return CommonResult.failed("资源繁忙，请稍后重试");
            }
        } catch (Exception e) {
            log.error("doCatchRecommendUser error", e);
            return CommonResult.failed(e.getMessage());
        }finally {
            if (lock.isHeldByCurrentThread()) {
                log.debug(lockId+":unLock: " + Thread.currentThread().getId());
                lock.unlock();
            }
        }
    }
}



