package com.bcxin.ars.service.util;

import cn.hutool.core.thread.ThreadUtil;
import com.abcxin.smart.validator.annotation.DataSyncAnnotation;
import com.abcxin.smart.validator.annotation.DataSyncOutAnnotation;
import com.abcxin.smart.validator.enums.SyncType;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bcxin.ars.dao.FilerecordDao;
import com.bcxin.ars.dao.FilerecordOutDao;
import com.bcxin.ars.dao.datasync.FileRecordDetailDao;
import com.bcxin.ars.dto.DataSycDto;
import com.bcxin.ars.dto.DataSynchronizationSearchDto;
import com.bcxin.ars.enums.ImportType;
import com.bcxin.ars.exception.ArsException;
import com.bcxin.ars.model.BaseModel;
import com.bcxin.ars.model.Config;
import com.bcxin.ars.model.Filerecord;
import com.bcxin.ars.model.FilerecordOut;
import com.bcxin.ars.model.datasync.FileRecordDetail;
import com.bcxin.ars.model.sb.Personcertificate;
import com.bcxin.ars.util.*;
import com.bcxin.ars.util.spring.util.SpringUtils;
import org.apache.poi.ss.formula.functions.T;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.ReflectionUtils;

import javax.crypto.BadPaddingException;
import java.io.*;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.*;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

/**
 * 同步数据导出导入
 */
@Component
public class DataSyncUtil {

    Logger logger = LoggerFactory.getLogger(DataSyncUtil.class);

    @Autowired
    private ConfigUtils configUtils;

    @Autowired
    private FilerecordDao fileRecordDao;

    @Autowired
    private FilerecordOutDao fileRecordOutDao;

    @Autowired
    private FileRecordDetailDao fileRecordDetailDao;

    @Autowired
    private EmailUtil emailUtil;

    @Autowired
    private IdGeneratorUtil idGeneratorUtil;

    /**
     * 创建导出同步数据文件（内到外）
     * @param startDate 开始日期
     * @return  同步数据文件路径
     */
    public String createDSFile(String startDate,String createFilesPath){
        if(StringUtil.isEmpty(createFilesPath)){
            createFilesPath = configUtils.tempfolder;
        }

        File filePath = new File(createFilesPath);
        if(!filePath.exists()){
            filePath.mkdirs();
        }

        String resultPath = StringUtil.EMPTY;
        List<String> filePaths = new Vector<>();
        //获取所有需要同步操作的Service bean
        Map<String, Object> serviceBeans = SpringUtils.getBeansByAnnotation(DataSyncOutAnnotation.class);

        logger.info("开始生成同步文件：");
        long time1=System.currentTimeMillis();
        //使用线程池
        CountDownLatch countDownLatch = ThreadUtil.newCountDownLatch(serviceBeans.size());
        ThreadPoolExecutor threadPoolExecutor = ThreadUtil.newExecutor(15, 15);


        for (String key : serviceBeans.keySet()) {
            //获取service Bean
            Object entityService = serviceBeans.get(key);
            threadPoolExecutor.execute(new Runnable(){
                @Override
                public void run() {
                    try {
                        createDSFile(entityService,filePaths,startDate);
                    } catch (Exception e) {
                        logger.error(e.getMessage(),e);
                        e.printStackTrace();
                    } finally {
                        countDownLatch.countDown();
                    }
                }
            });
        }
        try {
            //线程等待全部完成
            countDownLatch.await();
        } catch (InterruptedException e) {
            logger.error(e.getMessage(),e);
            Thread.currentThread().interrupt();
        }finally {
            threadPoolExecutor.shutdown();
        }

        if(filePaths.size() > 0) {
            //生成同步数据文件压缩包
            String fileName = FileUtil.generatingFileNameByDate(configUtils.getDataSyncStartFileName(DataSyncOutAnnotation.class)) + Constants.ZIP;
            resultPath = FileUtil.zipFiles(filePaths, createFilesPath + fileName);
            //删除临时文件
            filePaths.forEach(tempPath->{
                if(tempPath.indexOf(Constants.TEMP)>0){
                    FileUtil.deleteFile(tempPath);
                }
            });
        }
        long time2=System.currentTimeMillis();
        logger.info("生成同步文件完成。耗时："+(time2-time1)+"ms");
        return resultPath;
    }


    /**
     * 创建导出同步数据文件（内到外）
     * @param entityService
     * @param filePaths 存放导出文件路径列表
     */
    public void createDSFile(Object entityService,List<String> filePaths,String startDate){
        try {
            DataSycDto dataSycDto = new DataSycDto();
            //service bean的Class
            Class clazz = entityService.getClass();
            //获取是上面的注解
            DataSyncOutAnnotation dataSyncAnnotation = (DataSyncOutAnnotation) clazz.getAnnotation(DataSyncOutAnnotation.class);
            //得到注解上的关联实体类名
            String fileName = dataSyncAnnotation.getClazz().getSimpleName();
            dataSycDto.setDataType(fileName);
            String methodName = dataSyncAnnotation.getSearchMethodName();
            ImportType importType = dataSyncAnnotation.getImportType();
            //获取查询数据的方法
            Method method = null;
            switch (importType){
                case BATCH_FILE:
                case FOREACH_FILE:
                    method = ReflectionUtils.findMethod(entityService.getClass(), methodName, List.class, String.class);
                    break;
                default:
                    method = ReflectionUtils.findMethod(entityService.getClass(), methodName, String.class);
            }
            //根据反射调用查询方法查询数据
            if(method != null) {
                List<T> entityList = null;
                switch (importType){
                    case BATCH_FILE:
                    case FOREACH_FILE:
                        entityList = (List<T>) ReflectionUtils.invokeMethod(method, entityService, filePaths,startDate);
                        break;
                    default:
                        entityList = (List<T>) ReflectionUtils.invokeMethod(method, entityService, startDate);
                }
                //将数据转成json字符串保存到txt文件中
                if (entityList != null && entityList.size() > 0) {
                    Date now = new Date();
                    String dirDate = DateUtil.systemDate.format(now);
                    File dir = new File(configUtils.tempfolder + dirDate);
                    if (!dir.exists()) {
                        dir.mkdirs();
                    }
                    //根据数据数量情况，切分不同的大小
                    int limit = configUtils.limitSize;
                    //计算拆分次数
                    int count = new Double(Math.ceil(entityList.size() * 1.0 / limit)).intValue();
                    //存放拆分数据
                    List<T> subList = null;
                    //循环拆分次数 生成相应数据文件
                    for (int i = 0; i < count; i++) {
                        subList = entityList.stream().skip(i * limit).limit(limit).collect(Collectors.toList());
                        dataSycDto.setData(JSONArray.toJSONString(subList));
                        //将数据转成json字符串
                        String listStr = JSONObject.toJSONString(dataSycDto);
                        //将数据加密
                        String content = CipherDESUtil.parseByte2HexStr(CipherDESUtil.encrypt(listStr.getBytes(Constants.CHARSETUTF8), Constants.APPROVAL_KEY));
                        //保存到txt文件中
                        String filepath = FileUtil.writeTxtFile(content, fileName, configUtils.tempfolder,i);
                        filePaths.add(filepath);
                    }
                }
            }else{
                logger.error(entityService.getClass().getSimpleName() + "类没有生成同步数据文件的方法"+methodName+"或者参数有误！");
            }
        } catch (BadPaddingException badPaddingException) {
            logger.error(badPaddingException.getMessage(), badPaddingException);
            throw new ArsException("导出失败，非该区域数据\n！");
        } catch (Exception e){
            logger.error(e.getMessage(),e);
        }
    }

    /**
     * 创建同步数据文件（外到内）
     * @param AnnotationClazz 数据同步注解类名
     * @param searchDto 查询条件 开始日期
     * @param dataFilePath 同步数据文件存放的路径
     * @return  同步数据文件压缩包路径
     */
    public String createDSFile(Class AnnotationClazz, DataSynchronizationSearchDto searchDto, String dataFilePath){
        //如果同步数据文件存放的路径为空，则使用临时目录
        if(StringUtil.isEmpty(dataFilePath)){
            dataFilePath = configUtils.tempfolder;
        }
        //目录不存在，则创建目录
        File filePath = new File(dataFilePath);
        if(!filePath.exists()){
            filePath.mkdirs();
        }
        //最后生成压缩包文件的路径
        String zipPath = StringUtil.EMPTY;
        //存放数据文件路径列表
        List<String> filePaths = new Vector<>();
        //1.获取需要生成的service列表信息
        Map<String, Object> serviceBeans = SpringUtils.getBeansByAnnotation(AnnotationClazz);
        //2.循环service列表 多线程异步生成数据文件
        logger.info("开始生成同步文件：");
        long time1=System.currentTimeMillis();
        //使用线程池
        CountDownLatch workCount = ThreadUtil.newCountDownLatch(serviceBeans.size());
        ThreadPoolExecutor threadPoolExecutor = ThreadUtil.newExecutor(15, 15);
        for (String key : serviceBeans.keySet()) {
            //获取service Bean
            Object entityService = serviceBeans.get(key);
            threadPoolExecutor.execute(new Runnable(){
                @Override
                public void run() {
                    try {
                        createDSFile(AnnotationClazz,entityService,filePaths,searchDto);
                    } catch (Exception e) {
                        logger.error(e.getMessage(),e);
                        e.printStackTrace();
                    } finally {
                        workCount.countDown();
                    }
                }
            });
        }
        try {
            //线程等待全部数据文件生成完成
            workCount.await();
        } catch (InterruptedException e) {
            logger.error(e.getMessage(),e);
            Thread.currentThread().interrupt();
        }finally {
            threadPoolExecutor.shutdown();
        }

        if(filePaths.size() > 0) {
            String fileName = FileUtil.generatingFileNameByDate(configUtils.getDataSyncStartFileName(DataSyncAnnotation.class)) + Constants.ZIP;
            //生成同步数据文件压缩包
            zipPath = FileUtil.zipFiles(filePaths, dataFilePath + fileName);
            //删除临时文件
            filePaths.forEach(tempPath->{
                if(tempPath.indexOf(Constants.TEMP)>0){
                    FileUtil.deleteFile(tempPath);
                }
            });
        }
        long time2=System.currentTimeMillis();
        logger.info("生成同步文件完成。耗时："+(time2-time1)+"ms");
        return zipPath;
    }

    /**
     * 生成数据同步文件(外到内)
     * @param AnnotationClazz 数据同步注解类名
     * @param entityService 处理类bean
     * @param filePaths 存放生成同步文件路径列表
     * @param searchDto 查询条件 开始日期 由此日期到当然前时间内的数据
     */
    private void createDSFile(Class AnnotationClazz, Object entityService, List<String> filePaths, DataSynchronizationSearchDto searchDto) {
        try {
            DataSycDto dataSycDto = new DataSycDto();

            //得到注解上的关联实体类名做为文件名
            String fileName = StringUtil.EMPTY;
            //获取同步数据列表的方法
            String methodName = StringUtil.EMPTY;
            //导入方式
            ImportType importType = null;

            //service bean的Class
            Class serviceClazz = entityService.getClass();

            //获取是上面的注解
            Annotation annotation = serviceClazz.getAnnotation(AnnotationClazz);

            if(annotation instanceof DataSyncAnnotation){
                DataSyncAnnotation dataSyncAnnotation = (DataSyncAnnotation) annotation;
                //得到注解上的关联实体类名做为文件名
                fileName = dataSyncAnnotation.getClazz().getSimpleName();
                //获取同步数据列表的方法
                methodName = dataSyncAnnotation.getSearchMethodName();
                //导入方式
                importType = dataSyncAnnotation.getImportType();
            }else if(annotation instanceof DataSyncOutAnnotation){
                DataSyncOutAnnotation dataSyncAnnotation = (DataSyncOutAnnotation) annotation;
                //得到注解上的关联实体类名做为文件名
                fileName = dataSyncAnnotation.getClazz().getSimpleName();
                //获取同步数据列表的方法
                methodName = dataSyncAnnotation.getSearchMethodName();
                //导入方式
                importType = dataSyncAnnotation.getImportType();
            }

            //关联实体类名做为数据类型
            dataSycDto.setDataType(fileName);

            //获取查询数据的方法
            Method method = null;
            switch (importType){
                case BATCH_FILE:
                case FOREACH_FILE:
                    method = ReflectionUtils.findMethod(entityService.getClass(), methodName, List.class, DataSynchronizationSearchDto.class);
                    break;
                default:
                    method = ReflectionUtils.findMethod(entityService.getClass(), methodName, DataSynchronizationSearchDto.class);
            }
            //根据反射调用查询方法查询数据
            //获取要生成的数据，根据updateflag+日期
            if(method != null) {
                List<T> entityList = null;
                switch (importType){
                    case BATCH_FILE:
                    case FOREACH_FILE:
                        entityList = (List<T>) ReflectionUtils.invokeMethod(method, entityService, filePaths,searchDto);
                        break;
                    default:
                        entityList = (List<T>) ReflectionUtils.invokeMethod(method, entityService, searchDto);
                }
                //将数据转成json字符串保存到txt文件中
                if (entityList != null && entityList.size() > 0) {
                    //当前日期做为文件夹名
                    String dirDate = DateUtil.getCurrentDate();
                    //根据数据数量情况，切分不同的大小
                    int limit = configUtils.limitSize;
                    //计算拆分次数
                    int count = new Double(Math.ceil(entityList.size()*1.0/limit)).intValue();
                    //存放拆分生成文件名
                    String _fileName = "";
                    //存放拆分数据
                    List<T> subList = null;
                    //循环拆分次数 生成相应数据文件
                    for (int i = 0; i < count; i++) {
                        _fileName = fileName +"_"+ i;
                        subList = entityList.stream().skip(i*limit).limit(limit).collect(Collectors.toList());
                        dataSycDto.setData(JSONArray.toJSONString(subList));
                        File dir = new File(configUtils.tempfolder + dirDate);
                        if (!dir.exists()) {
                            dir.mkdirs();
                        }
                        //将数据转成json字符串
                        String listStr = JSONObject.toJSONString(dataSycDto);
                        //将数据加密
                        String content = CipherDESUtil.parseByte2HexStr(CipherDESUtil.encrypt(listStr.getBytes(Constants.CHARSETUTF8), Constants.APPROVAL_KEY));
                        //保存到txt文件中
                        String filepath = FileUtil.writeTxtFile(content, _fileName, configUtils.tempfolder);
                        filePaths.add(filepath);

                        this.saveFileRecordOut(FileUtil.getFileNameByPath(filepath),filepath,subList.size());
                    }
                }
            }else{
                logger.error(entityService.getClass().getSimpleName() + "类没有生成同步数据文件的方法"+methodName+"或者参数有误！");
            }
        } catch (BadPaddingException badPaddingException) {
            logger.error(badPaddingException.getMessage(), badPaddingException);
            throw new ArsException("导出失败，非该区域数据\n！");
        } catch (Exception e){
            logger.error(e.getMessage(),e);
        }
    }

    /**
     * 读取同步文件
     * 解压缩
     * @param AnnotationClazz 数据同步注解类名
     * @param zipFile
     * @throws Exception
     */
    public void readDSFile(Class AnnotationClazz,File zipFile) throws Exception {
        long startTime = System.currentTimeMillis();
        Date startDate = new Date();
        try {
            List<Filerecord> fileRecordList = fileRecordDao.findByFileName(zipFile.getName(),configUtils.isIntranet());
            if(fileRecordList == null  || fileRecordList.size() == 0){
                List<String> filePaths = new ArrayList<String>();
                //存放附件目录
                Map<String, String> pathMap = new HashMap<>();
                //输入源zip路径
                ZipInputStream Zin = new ZipInputStream(new FileInputStream(zipFile));
                BufferedInputStream Bin = new BufferedInputStream(Zin);
                //输出路径（文件夹目录）
                String tempFilePath = configUtils.tempfolder + DateUtil.getCurrentDate() + File.separatorChar;
                Map<String,List<File>> dataFiles = new HashMap<>();
                File localFile = null;
                ZipEntry entry;
                try {
                    while ((entry = Zin.getNextEntry()) != null && !entry.isDirectory()) {
                        localFile = new File(tempFilePath, entry.getName());
                        if (!localFile.exists()) {
                            (new File(localFile.getParent())).mkdirs();
                        }
                        FileOutputStream out = new FileOutputStream(localFile);
                        BufferedOutputStream Bout = new BufferedOutputStream(out);
                        int b;
                        while ((b = Bin.read()) != -1) {
                            Bout.write(b);
                        }
                        Bout.close();
                        out.close();
                        if (localFile.getName().endsWith(".data")) {
                            String[] splitFiles = localFile.getName().replace(".data", "").split("_");
                            if (splitFiles.length > 0) {
                                String filename = splitFiles[0];
                                List<File> localFiles = dataFiles.get(filename);
                                if(localFiles == null){
                                    localFiles = new ArrayList<>();
                                }

                                localFiles.add(localFile);
                                dataFiles.put(filename, localFiles);
                            }
                        } else {
                            pathMap.put(localFile.getName(), localFile.getPath());
                        }
                        filePaths.add(localFile.getPath());
                        logger.info(localFile + "解压成功");
                    }
                    Bin.close();
                    Zin.close();
                } catch (IOException e) {
                    logger.error(e.getMessage());
                    throw e;
                }
                long endTime = System.currentTimeMillis();
                logger.info("耗费时间： " + (endTime - startTime) + " ms");

                this.readDSFile(AnnotationClazz,dataFiles,pathMap);

                for (String filePath : filePaths) {
                    FileUtil.deleteFolder(filePath);
                }
                this.saveFileRecord(startDate,null,zipFile.getName(),zipFile.getAbsolutePath(),null);
            }
        } catch (FileNotFoundException e) {
            logger.error(e.getMessage());
            throw e;
        }
    }

    /**
     * 读取同步文件
     * 分成异步操作跟同步操作
     * 先用线程池执行异步操作的
     * 再操作同步
     * @param AnnotationClazz 数据同步注解类名
     * @param dataFiles 同步文件列表
     * @param pathMap 附件路径
     */
    private void readDSFile(Class AnnotationClazz,Map<String,List<File>> dataFiles,Map<String, String> pathMap){
        logger.info("开始读取同步文件：");
        //获取所有异步操作的Service bean
        List<Map.Entry<String, Object>> serviceBeans_async = this.getDataSyncBeanList_async(AnnotationClazz);
        long time1=System.currentTimeMillis();
        //使用线程池
        CountDownLatch workCount = ThreadUtil.newCountDownLatch(serviceBeans_async.size());
        ThreadPoolExecutor threadPoolExecutor = ThreadUtil.newExecutor(15, 15);
        for (Map.Entry<String, Object> map : serviceBeans_async) {
            //获取service Bean
            Object entityService = map.getValue();
            threadPoolExecutor.execute(new Runnable(){
                @Override
                public void run() {
                    try {
                        readDSFile(AnnotationClazz,entityService,dataFiles,pathMap);
                    } catch (Exception e) {
                        logger.error(e.getMessage(),e);
                        e.printStackTrace();
                    } finally {
                        workCount.countDown();
                    }
                }
            });
        }

        try {
            //线程等待全部数据文件生成完成
            workCount.await();
        } catch (InterruptedException e) {
            logger.error(e.getMessage(),e);
            Thread.currentThread().interrupt();
        }finally {
            threadPoolExecutor.shutdown();
        }

        //获取所有同步操作的Service bean
        List<Map.Entry<String, Object>> serviceBeans_sync = this.getDataSyncBeanList_sync(AnnotationClazz);

        for (Map.Entry<String, Object> map : serviceBeans_sync) {
            //获取service Bean
            Object entityService = map.getValue();
            readDSFile(AnnotationClazz,entityService,dataFiles,pathMap);
        }

        long time2=System.currentTimeMillis();
        logger.info("同步文件读取完成。耗时："+(time2-time1)+"ms");
    }

    /**
     * 读取同步文件
     * @param AnnotationClazz 数据同步注解类名
     * @param entityService 处理类bean
     * @param dataFiles 同步文件列表
     * @param pathMap 附件路径
     */
    private void readDSFile(Class AnnotationClazz,Object entityService, Map<String, List<File>> dataFiles,Map<String, String> pathMap){
        Date startDate = new Date();
        //得到注解上的关联实体类名做为文件名
        String fileName = StringUtil.EMPTY;
        //获取同步数据列表的方法
        String methodName = StringUtil.EMPTY;
        //导入方式
        ImportType importType = null;
        //得到注解上的关联实体类
        Class entityClazz = null;

        //service bean的Class
        Class serviceClazz = entityService.getClass();


        //获取是上面的注解
        Annotation annotation = serviceClazz.getAnnotation(AnnotationClazz);

        if(annotation instanceof DataSyncAnnotation){
            DataSyncAnnotation dataSyncAnnotation = (DataSyncAnnotation) annotation;
            //得到注解上的关联实体类
            entityClazz = dataSyncAnnotation.getClazz();
            //得到注解上的关联实体类名做为文件名
            fileName = entityClazz.getSimpleName();
            //获取同步数据列表的方法
            methodName = dataSyncAnnotation.getMethodName();
            //导入方式
            importType = dataSyncAnnotation.getImportType();
        }else if(annotation instanceof DataSyncOutAnnotation){
            DataSyncOutAnnotation dataSyncAnnotation = (DataSyncOutAnnotation) annotation;
            //得到注解上的关联实体类
            entityClazz = dataSyncAnnotation.getClazz();
            //得到注解上的关联实体类名做为文件名
            fileName = entityClazz.getSimpleName();
            //获取同步数据列表的方法
            methodName = dataSyncAnnotation.getMethodName();
            //导入方式
            importType = dataSyncAnnotation.getImportType();
        }

        if(entityClazz == null){
            return;
        }
        List<File> files = dataFiles.get(fileName);
        if(files != null && files.size() >0){
            for (File dataFile : files) {
                try {
                    List<Filerecord> fileRecordList = fileRecordDao.findByFileName(dataFile.getName(),configUtils.isIntranet());
                    if (fileRecordList == null || fileRecordList.size() == 0) {
                        BufferedReader br = new BufferedReader(new FileReader(dataFile));
                        StringBuffer content = new StringBuffer();
                        // 获取文件内容
                        String lineTxt = null;
                        while ((lineTxt = br.readLine()) != null) {
                            // 使用readLine方法，一次读一行
                            content.append(lineTxt);
                        }
                        br.close();

                        //文件内容解密
                        String jsonStr = new String(CipherDESUtil.decrypt(Encodes.decodeHex(content.toString()), Constants.APPROVAL_KEY), Constants.CHARSETUTF8);
                        //json字符串转实体
                        DataSycDto dataSycDto = JSONObject.parseObject(jsonStr, DataSycDto.class);
                        if (dataSycDto != null) {
                            List<FileRecordDetail> errorDetailList = new ArrayList<>();
                            List list = JSONArray.parseArray(dataSycDto.getData(), entityClazz);
                            //获取保存数据的方法
                            Method method = null;
                            switch (importType) {
                                case BATCH:
                                    method = ReflectionUtils.findMethod(entityService.getClass(), methodName, List.class);
                                    if (method != null) {
                                        try {
                                            ReflectionUtils.invokeMethod(method, entityService, list);
                                        } catch (Exception e) {
                                            logger.error(e.getMessage(), e);
                                            this.saveError(errorDetailList,null, jsonStr, e.getMessage(), dataFile.getAbsolutePath());
                                        }
                                    }
                                    break;
                                case BATCH_FILE:
                                    method = ReflectionUtils.findMethod(entityService.getClass(), methodName, List.class, Map.class);
                                    if (method != null) {
                                        try {
                                            ReflectionUtils.invokeMethod(method, entityService, list, pathMap);
                                        } catch (Exception e) {
                                            logger.error(e.getMessage(), e);
                                            this.saveError(errorDetailList,null, jsonStr, e.getMessage(), dataFile.getAbsolutePath());
                                        }
                                    }
                                    break;
                                case FOREACH:
                                    method = ReflectionUtils.findMethod(entityService.getClass(), methodName, entityClazz);
                                    if (method != null) {
                                        for (Object detail : list) {
                                            try {
                                                ReflectionUtils.invokeMethod(method, entityService, detail);
                                            } catch (Exception e) {
                                                logger.error(e.getMessage(), e);
                                                BaseModel baseModel = (BaseModel)detail;
                                                this.saveError(errorDetailList,baseModel.getId(),JSONObject.toJSONString(detail), e.getMessage(), dataFile.getAbsolutePath());
                                            }
                                        }
                                    }
                                    break;
                                case FOREACH_FILE:
                                    method = ReflectionUtils.findMethod(entityService.getClass(), methodName, entityClazz,Map.class);
                                    if (method != null) {
                                        for (Object detail : list) {
                                            try {
                                                ReflectionUtils.invokeMethod(method, entityService, detail, pathMap);
                                            } catch (Exception e) {
                                                String errorMsg = e.toString();
                                                logger.error(errorMsg, e);
                                                if(StringUtil.isNotEmpty(e.getMessage())){
                                                    errorMsg = errorMsg + e.getMessage();
                                                }
                                                BaseModel baseModel = (BaseModel)detail;
                                                this.saveError(errorDetailList,baseModel.getId(),JSONObject.toJSONString(detail), errorMsg, dataFile.getAbsolutePath());
                                            }
                                        }
                                    }
                                    break;
                                default:
                                    break;
                            }

                            if (method == null) {
                                logger.error(entityService.getClass().getSimpleName() + "类没有保存同步数据的方法" + methodName + "或者参数有误！");
                            }
                            this.saveFileRecord(startDate,errorDetailList, dataFile.getName(),dataFile.getAbsolutePath(),list.size());
                        }
                    }
                } catch (BadPaddingException badPaddingException) {
                    logger.error(badPaddingException.getMessage(), badPaddingException);
                    throw new ArsException("导入失败，非该区域数据\n！");
                } catch (Exception e) {
                    logger.error(e.getMessage(), e);
                }
            }
        }

    }

    /**
     * 获取异步执行的Service Bean 列表
     * @param clazz
     * @return
     */
    private List<Map.Entry<String,Object>> getDataSyncBeanList_async(Class clazz){
        Map<String, Object> beans = SpringUtils.getBeansByAnnotation(clazz);
        List<Map.Entry<String,Object>> list = new ArrayList<Map.Entry<String,Object>>(beans.entrySet());

        return list.stream().filter((map)-> SyncType.ASYNC.equals(getSyncType(map.getValue(),clazz))).collect(Collectors.toList());
    }

    /**
     * 获取数据同步执行方式
     * @param entityService
     * @param clazz
     * @return
     */
    private SyncType getSyncType(Object entityService,Class clazz){
        Class entityServiceClazz = entityService.getClass();
        Annotation annotation = entityServiceClazz.getAnnotation(clazz);

        if(annotation instanceof DataSyncAnnotation){
            DataSyncAnnotation dataSyncAnnotation = (DataSyncAnnotation) annotation;
            return dataSyncAnnotation.getDataSyncType();
        }else if(annotation instanceof DataSyncOutAnnotation){
            DataSyncOutAnnotation dataSyncAnnotation = (DataSyncOutAnnotation) annotation;
            return dataSyncAnnotation.getDataSyncType();
        }
        return null;
    }

    /**
     * 获取同步执行排序后的数据同步Service Bean 列表
     * @param clazz
     * @return
     */
    private List<Map.Entry<String,Object>> getDataSyncBeanList_sync(Class clazz){
        Map<String, Object> beans = SpringUtils.getBeansByAnnotation(clazz);

        List<Map.Entry<String,Object>> list = new ArrayList<Map.Entry<String,Object>>(beans.entrySet());

        //获取同步执行的数据同步Service Bean 列表
        List<Map.Entry<String,Object>> maps = list.stream().filter((map)->SyncType.SYNC.equals(getSyncType(map.getValue(),clazz))).collect(Collectors.toList());

        //排序
        Collections.sort(maps,new Comparator<Map.Entry<String,Object>>() {
            //升序排序 比较器
            @Override
            public int compare(Map.Entry<String, Object> map1, Map.Entry<String, Object> map2) {
                //compare（a,b）方法:根据第一个参数小于、等于或大于第二个参数分别返回负整数、零或正整数。
                Object o1 = map1.getValue();
                Object o2 = map2.getValue();
                //根据seq排序
                return getDataSyncSeq(map1.getValue(),clazz) - getDataSyncSeq(map2.getValue(),clazz);
            }
        });
        return maps;
    }

    /**
     * 获取 数据同步Service Bean 的seq
     * @param entityService
     * @return
     */
    private int getDataSyncSeq(Object entityService, Class clazz){
        Class entityServiceClazz = entityService.getClass();
        Annotation annotation = entityServiceClazz.getAnnotation(clazz);

        if(annotation instanceof DataSyncAnnotation){
            DataSyncAnnotation dataSyncAnnotation = (DataSyncAnnotation) annotation;
            return dataSyncAnnotation.getSeq();
        }else if(annotation instanceof DataSyncOutAnnotation){
            DataSyncOutAnnotation dataSyncAnnotation = (DataSyncOutAnnotation) annotation;
            return dataSyncAnnotation.getSeq();
        }
        return 0;
    }


    private void sendFailEmail(String data,String errorMsg,String fileName){
        Config receiveUserConfig = configUtils.getConfigByKey(Constants.DATASYNC_RECEIVEUSER);
        Config currentNativeName = configUtils.getConfigByKey(Constants.CURRENT_NATIVE_NAME);
        if(receiveUserConfig != null && currentNativeName != null) {
            new Thread(()->{
                String sendHtml = Constants.DATASYNC_SENDHTML_IN2OUT.replace("{fileName}",fileName)
                        .replace("{data}",data)
                        .replace("{errorMsg}",errorMsg)
                        .replace("{currentNativeName}", currentNativeName.getValue())
                        .replace("url",configUtils.getWebHost());

                String receiveUser = receiveUserConfig.getValue();
                emailUtil.sendEmail(Constants.DATASYNC_SUBJECT, sendHtml, receiveUser);
            }).start();
        }
    }

    private void saveFileRecord(Date startDate,List<FileRecordDetail> errorDetailList, String fileName,String filePath,Integer recordSize){
        Filerecord fileRecord = new Filerecord();
        fileRecord.setId(idGeneratorUtil.createID());
        fileRecord.setFilename(fileName);
        fileRecord.setFilepath(filePath);
        fileRecord.setRecordsize(recordSize);
        fileRecord.setCreateTime(startDate);
        fileRecord.setUpdateflag(true);
        fileRecord.setUpdateTime(new Date());
        fileRecord.setSynchtype(Constants.DATASYNC);
        if(configUtils.isIntranet()){
            fileRecord.setUpdateflag(true);
        }
        fileRecord.setIntranet(configUtils.isIntranet());
        fileRecordDao.save(fileRecord);

        if(errorDetailList != null && errorDetailList.size() > 0){
            errorDetailList.forEach(detail->{
                detail.setRecordId(fileRecord.getId());
            });
            fileRecordDetailDao.saveBatch(errorDetailList);
        }
    }

    private void saveFileRecordOut(String fileName, String filePath,Integer recordSize) {
        FilerecordOut fileRecordOut = new FilerecordOut();
        fileRecordOut.setId(idGeneratorUtil.createID());
        fileRecordOut.setFilename(fileName);
        fileRecordOut.setFilepath(filePath);
        fileRecordOut.setRecordsize(recordSize);
        fileRecordOut.setCreateTime(new Date());
        fileRecordOut.setUpdateTime(new Date());
        fileRecordOut.setSynchtype(Constants.DATASYNC);
        fileRecordOutDao.save(fileRecordOut);
    }

    private void saveError(List<FileRecordDetail> errorDetailList, Long businessId,String data,String errorMsg,String fileName){

        FileRecordDetail fileRecordDetail = new FileRecordDetail();
        fileRecordDetail.setData(data);
        fileRecordDetail.setError(errorMsg);
        fileRecordDetail.setBusinessId(businessId);

        fileRecordDetail.setId(idGeneratorUtil.createID());
        fileRecordDetail.setCreateTime(new Date());
        fileRecordDetail.setUpdateTime(new Date());
        fileRecordDetail.setIntranet(configUtils.isIntranet());
        if(configUtils.isIntranet()){
            fileRecordDetail.setUpdateflag(true);
        }

        fileRecordDetailDao.save(fileRecordDetail);
        errorDetailList.add(fileRecordDetail);

        if(!configUtils.isIntranet()) {
            this.sendFailEmail(data, errorMsg, fileName);
        }
    }

    public static void main(String[] args) {

        try {
            BufferedReader br = new BufferedReader(new FileReader("C:\\Users\\Administrator\\Desktop\\Personcertificate_0_20191010160005.data"));
            StringBuffer content = new StringBuffer();
            // 获取文件内容
            String lineTxt = null;
            while ((lineTxt = br.readLine()) != null) {
                // 使用readLine方法，一次读一行
                content.append(lineTxt);
            }
            br.close();

            //System.out.println(content.toString());
            //文件内容解密
            String jsonStr = new String(CipherDESUtil.decrypt(Encodes.decodeHex(content.toString()), "11000000"), Constants.CHARSETUTF8);
            //System.out.println(jsonStr);
            //json字符串转实体
            DataSycDto dataSycDto = JSONObject.parseObject(jsonStr, DataSycDto.class);


            List<Personcertificate> list = JSONArray.parseArray(dataSycDto.getData(), Personcertificate.class);
            list.forEach(person->{
                if(person.getId() == 631874724065968128L){
                    System.out.println(JSONObject.toJSONString(person));
                }
            });
        }catch (Exception e){

        }




    }
}



