/**
 * @licstart The following is the entire license notice for the
 * JavaScript code in this page
 *
 * Copyright 2022 Mozilla Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @licend The above is the entire license notice for the
 * JavaScript code in this page
 */
"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.JpegStream = void 0;

var _decode_stream = require("./decode_stream.js");

var _primitives = require("./primitives.js");

var _jpg = require("./jpg.js");

var _util = require("../shared/util.js");

class JpegStream extends _decode_stream.DecodeStream {
  constructor(stream, maybeLength, params) {
    let ch;

    while ((ch = stream.getByte()) !== -1) {
      if (ch === 0xff) {
        stream.skip(-1);
        break;
      }
    }

    super(maybeLength);
    this.stream = stream;
    this.dict = stream.dict;
    this.maybeLength = maybeLength;
    this.params = params;
  }

  get bytes() {
    return (0, _util.shadow)(this, "bytes", this.stream.getBytes(this.maybeLength));
  }

  ensureBuffer(requested) {}

  readBlock() {
    if (this.eof) {
      return;
    }

    const jpegOptions = {
      decodeTransform: undefined,
      colorTransform: undefined
    };
    const decodeArr = this.dict.getArray("D", "Decode");

    if (this.forceRGB && Array.isArray(decodeArr)) {
      const bitsPerComponent = this.dict.get("BPC", "BitsPerComponent") || 8;
      const decodeArrLength = decodeArr.length;
      const transform = new Int32Array(decodeArrLength);
      let transformNeeded = false;
      const maxValue = (1 << bitsPerComponent) - 1;

      for (let i = 0; i < decodeArrLength; i += 2) {
        transform[i] = (decodeArr[i + 1] - decodeArr[i]) * 256 | 0;
        transform[i + 1] = decodeArr[i] * maxValue | 0;

        if (transform[i] !== 256 || transform[i + 1] !== 0) {
          transformNeeded = true;
        }
      }

      if (transformNeeded) {
        jpegOptions.decodeTransform = transform;
      }
    }

    if (this.params instanceof _primitives.Dict) {
      const colorTransform = this.params.get("ColorTransform");

      if (Number.isInteger(colorTransform)) {
        jpegOptions.colorTransform = colorTransform;
      }
    }

    const jpegImage = new _jpg.JpegImage(jpegOptions);
    jpegImage.parse(this.bytes);
    const data = jpegImage.getData({
      width: this.drawWidth,
      height: this.drawHeight,
      forceRGB: this.forceRGB,
      isSourcePDF: true
    });
    this.buffer = data;
    this.bufferLength = data.length;
    this.eof = true;
  }

}

exports.JpegStream = JpegStream;