package com.bcxin.runtime.apis.controllers;

import com.bcxin.runtime.apis.configs.RegionConfig;
import com.bcxin.runtime.apis.requests.ChangeLogMappingRequest;
import com.bcxin.runtime.apis.responses.ChangeLogMappingResponse;
import com.bcxin.saas.core.exceptions.SaasForbidException;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.scheduling.annotation.Async;
import org.springframework.web.bind.annotation.*;

import java.util.Collection;
import java.util.concurrent.CompletableFuture;

@RestController
@RequestMapping("/v3/extends/change-mapping")
public class ChangeLogMappingController extends SecurityControllerAbstract {
    private final String INSERT_CHANGE_LOG_MAPPING = "INSERT INTO dynamic_data_map_templates(mapkey,template,note,createdTime)values(?,?,?,CURRENT_TIMESTAMP )";
    private final String UPDATE_CHANGE_LOG_MAPPING = "update dynamic_data_map_templates set mapkey=?,template=?,note=? where id=?";
    private final String GET_CHANGE_LOG_MAPPING = "select * from dynamic_data_map_templates where id=?";
    private final JdbcTemplate jdbcTemplate;
    private final RegionConfig regionConfig;

    public ChangeLogMappingController(JdbcTemplate jdbcTemplate, RegionConfig regionConfig) {
        this.jdbcTemplate = jdbcTemplate;
        this.regionConfig = regionConfig;
    }

    /**
     * ChangeLog的Mapping配置
     *
     * @return
     */
    @GetMapping
    @Async
    public CompletableFuture<ResponseEntity> get() {
        Collection<ChangeLogMappingResponse> logMappingResponses =
                this.jdbcTemplate.query("SELECT * FROM dynamic_data_map_templates", (rs, rsNum) -> {
                    String mapKey = rs.getString("mapKey");
                    String template = rs.getString("template");
                    String note = rs.getString("note");
                    boolean isOnline = rs.getBoolean("is_online");
                    String id = rs.getString("id");

                    return ChangeLogMappingResponse.create(id, mapKey, template, note, isOnline);
                });

        return CompletableFuture.completedFuture(ResponseEntity.ok(logMappingResponses));
    }

    @GetMapping("/{id}")
    public CompletableFuture<ResponseEntity> get(@PathVariable String id) {
        ChangeLogMappingResponse response =
                this.jdbcTemplate.queryForObject(GET_CHANGE_LOG_MAPPING, new Object[]{id}, (rs, rowNum) -> {
                    String mapKey = rs.getString("mapKey");
                    String template = rs.getString("template");
                    String note = rs.getString("note");
                    boolean isOnline = rs.getBoolean("is_online");
                    String rid = rs.getString("id");

                    return ChangeLogMappingResponse.create(rid, mapKey, template, note, isOnline);
                });

        return CompletableFuture.completedFuture(ResponseEntity.ok(response));
    }

    @PostMapping
    @Async
    public CompletableFuture<ResponseEntity> post(@RequestBody ChangeLogMappingRequest request) {
        if(!checkIfAllowedAccess()) {
            return CompletableFuture.completedFuture(ResponseEntity.status(HttpStatus.FORBIDDEN).body("该环境无法访问该服务！"));
        }

        Object[] param = new Object[3];
        param[0] = request.getMapKey();
        param[1] = request.getTemplate();
        param[2] = request.getNote();

        this.jdbcTemplate.update(INSERT_CHANGE_LOG_MAPPING, param);

        return get();
    }

    @PutMapping("/{id}")
    @Async
    public CompletableFuture<ResponseEntity> put(@PathVariable String id, @RequestBody ChangeLogMappingRequest request) {
        if(!checkIfAllowedAccess()) {
            return CompletableFuture.completedFuture(ResponseEntity.status(HttpStatus.FORBIDDEN).body("该环境无法访问该服务！"));
        }

        Object[] param = new Object[4];
        param[0] = request.getMapKey();
        param[1] = request.getTemplate();
        param[2] = request.getNote();
        param[3] = id;

        this.jdbcTemplate.update(UPDATE_CHANGE_LOG_MAPPING, param);

        return get(id);
    }

    private boolean checkIfAllowedAccess() {
        return !this.regionConfig.isDisabled();
    }
}
