package com.bcxin.rbac.apis.impls;

import com.bcxin.api.interfaces.ApiConstant;
import com.bcxin.api.interfaces.rbacs.RbacCategoryRpcProvider;
import com.bcxin.api.interfaces.rbacs.requests.BatchCategoryResourceRequest;
import com.bcxin.api.interfaces.rbacs.requests.CategoryRequest;
import com.bcxin.api.interfaces.rbacs.responses.CategoryDetailGetResponse;
import com.bcxin.api.interfaces.rbacs.responses.CategoryGetResponse;
import com.bcxin.api.interfaces.rbacs.responses.ResourceGetResponse;
import com.bcxin.rbac.domain.RbacDbReader;
import com.bcxin.rbac.domain.dtos.CategoryTreeDto;
import com.bcxin.rbac.domain.dtos.ResourceDto;
import com.bcxin.rbac.domain.entities.CategoryEntity;
import com.bcxin.rbac.domain.repositories.CategoryRepository;
import com.bcxin.rbac.domain.services.CategoryService;
import com.bcxin.rbac.domain.services.commands.categories.BatchCategoryResourceCommand;
import com.bcxin.rbac.domain.services.commands.categories.CreateCategoryCommand;
import com.bcxin.rbac.domain.services.commands.categories.DeleteCategoryCommand;
import com.bcxin.rbac.domain.services.commands.categories.UpdateCategoryCommand;
import org.apache.dubbo.config.annotation.DubboService;

import java.util.Collection;
import java.util.stream.Collectors;

@DubboService(version = ApiConstant.VERSION,validation = "true",timeout = 30 *1000,retries = 0)
public class RbacCategoryRpcProviderImpl implements RbacCategoryRpcProvider {
    private final CategoryService categoryService;
    private final CategoryRepository categoryRepository;
    private final RbacDbReader dbReader;

    public RbacCategoryRpcProviderImpl(CategoryService categoryService,
                                       CategoryRepository categoryRepository,
                                       RbacDbReader dbReader) {
        this.categoryService = categoryService;
        this.categoryRepository = categoryRepository;
        this.dbReader = dbReader;
    }

    @Override
    public void create(CategoryRequest request) {
        this.categoryService.dispatch(CreateCategoryCommand.create(
                request.getCode(), request.getName(), request.getNote()));
    }

    @Override
    public void update(String id, CategoryRequest request) {
        this.categoryService.dispatch(UpdateCategoryCommand.create(id, request.getCode(), request.getName(), request.getNote()));
    }

    @Override
    public void delete(String id) {
        this.categoryService.dispatch(DeleteCategoryCommand.create(id));
    }

    @Override
    public Collection<CategoryGetResponse> getAll() {
        Collection<CategoryEntity> categories = this.categoryRepository.getAll();
        return categories.stream().map(ii ->
                CategoryGetResponse.create(ii.getId(), ii.getCode(), ii.getName(), ii.getNote(), ii.getCreator(), ii.getLastModifier())
        ).collect(Collectors.toList());
    }

    @Override
    public void batch(BatchCategoryResourceRequest request) {
        Collection<BatchCategoryResourceCommand.CategoryCommand> categoryCommands =
                request.getCategories().stream().map(ix -> {
                    Collection<BatchCategoryResourceCommand.CategoryResourceCommand> resources =
                            ix.getResources().stream().map(it -> BatchCategoryResourceCommand.CategoryResourceCommand.create(it.getId(), it.getName()))
                                    .collect(Collectors.toList());

                    return BatchCategoryResourceCommand.CategoryCommand.create(ix.getId(), ix.getName(), resources);
                }).collect(Collectors.toList());

        this.categoryService.dispatch(BatchCategoryResourceCommand.create(categoryCommands));
    }

    @Override
    public Collection<CategoryDetailGetResponse> getByIds(Collection<String> ids) {
        Collection<CategoryTreeDto> categoryTreeDts = dbReader.getTreeCategories(ids);

        return translate2CategoryDetailGetResponses(categoryTreeDts);
    }

    @Override
    public Collection<CategoryDetailGetResponse> getAllDetails() {
        Collection<CategoryTreeDto> categoryTreeDts = dbReader.getAllTreeCategories();

        return translate2CategoryDetailGetResponses(categoryTreeDts);
    }

    private static Collection<CategoryDetailGetResponse> translate2CategoryDetailGetResponses(Collection<CategoryTreeDto> categoryTreeDts)
    {
        return categoryTreeDts.stream().map(ii -> {
            Collection<ResourceGetResponse> resourceGetResponses =
                    ii.getResources().stream().map(ir -> ResourceGetResponse.create(
                                    ir.getId(),
                                    ir.getCategoryId(), ir.getName(), ir.getData(), ir.getDataType(),
                                    ir.getMetaId(), ir.getMetaConfig(), ir.getMetaData(), ir.getCreator(), ir.getLastModifier()))
                            .collect(Collectors.toList());

            return CategoryDetailGetResponse.create(ii.getId(), ii.getName(), resourceGetResponses);
        }).collect(Collectors.toList());
    }
}
