package com.bcxin.rbac.apis.custom;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSONObject;
import com.bcxin.api.interfaces.ApiConstant;
import com.bcxin.api.interfaces.rbacs.custom.RbacPermitManagerProvider;
import com.bcxin.api.interfaces.rbacs.custom.request.RbacPermitAppAuthRequest;
import com.bcxin.api.interfaces.rbacs.custom.request.RbacPermitAuthRequest;
import com.bcxin.api.interfaces.rbacs.custom.request.RbacPermitOptionAuthRequest;
import com.bcxin.api.interfaces.rbacs.custom.request.RbacPermitSearchRequest;
import com.bcxin.api.interfaces.rbacs.custom.response.*;
import com.bcxin.rbac.domain.RbacDbReader;
import com.bcxin.rbac.domain.dtos.custom.*;
import com.bcxin.rbac.domain.services.custom.RbacPermitAppService;
import lombok.AllArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.apache.dubbo.config.annotation.DubboService;
import org.springframework.beans.BeanUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@DubboService(version = ApiConstant.VERSION, validation = "true", timeout = 30 *1000, retries = 0)
@AllArgsConstructor
public class RbacPermitManagerProviderImpl implements RbacPermitManagerProvider {
    private final RbacPermitAppService rbacPermitAppService;
    private final RbacDbReader dbReader;

    /**
     * description：查询企业授权的app列表
     * author：linchunpeng
     * date：2024/2/3
     */
    @Override
    public List<RbacPermitAppResponse> findOrgAppList(String organizationId) {
        List<RbacPermitAppResponse> resultList = new ArrayList<>();

        List<RbacPermitAppResponse> appList = new ArrayList<>();
        if (StringUtils.isNotBlank(organizationId)) {
            List<RbacPermitAppDto> list = rbacPermitAppService.findOrgAppList(organizationId);
            for (RbacPermitAppDto dto : list) {
                RbacPermitAppResponse response = new RbacPermitAppResponse();
                BeanUtils.copyProperties(dto, response);
                appList.add(response);
            }
        }
        if (CollectionUtil.isNotEmpty(appList)) {
            List<String> categoryList = new ArrayList<>();
            for (RbacPermitAppResponse response : appList) {
                if (!categoryList.contains(response.getCategory())) {
                    categoryList.add(response.getCategory());
                }
            }
            //根据分类进行分组
            Map<String, List<RbacPermitAppResponse>> listMap = appList.stream().collect(Collectors.groupingBy(RbacPermitAppResponse::getCategory));
            for (String category : categoryList) {
                RbacPermitAppResponse result = new RbacPermitAppResponse();
                result.setCategory(category);
                result.setChildren(listMap.get(category));
                resultList.add(result);
            }
        }

        return resultList;
    }

    /**
     * description：查询应用的操作项列表
     * author：linchunpeng
     * date：2024/2/3
     */
    @Override
    public List<RbacPermitOptionResponse> findAppOptionList(Long appId) {
        List<RbacPermitOptionResponse> resultList = new ArrayList<>();

        List<RbacPermitOptionResponse> optionList = new ArrayList<>();
        if (appId != null) {
            List<RbacPermitOptionDto> list = rbacPermitAppService.findAppOptionList(appId);
            for (RbacPermitOptionDto dto : list) {
                RbacPermitOptionResponse response = new RbacPermitOptionResponse();
                BeanUtils.copyProperties(dto, response);
                optionList.add(response);
            }
        }
        if (CollectionUtil.isNotEmpty(optionList)) {
            List<String> categoryList = new ArrayList<>();
            for (RbacPermitOptionResponse response : optionList) {
                if (!categoryList.contains(response.getCategory())) {
                    categoryList.add(response.getCategory());
                }
            }
            //根据分类进行分组
            Map<String, List<RbacPermitOptionResponse>> listMap = optionList.stream().collect(Collectors.groupingBy(RbacPermitOptionResponse::getCategory));
            for (String category : categoryList) {
                RbacPermitOptionResponse result = new RbacPermitOptionResponse();
                result.setCategory(category);
                result.setChildren(listMap.get(category));
                resultList.add(result);
            }
        }

        return resultList;
    }

    /**
     * description：查询组织授权的应用和操作项，如果有传职员id或角色id，则返回是否授权
     * author：linchunpeng
     * date：2024/3/5
     */
    @Override
    public List<RbacPermitAppResponse> findPermitList(RbacPermitSearchRequest request) {
        String organizationId = request.getOrganizationId();
        List<RbacPermitAppResponse> resultList = new ArrayList<>();
        List<RbacPermitAppResponse> appList = new ArrayList<>();

        Map<Long, RbacPermitAppResponse> appMap = new HashMap<>();
        Map<Long, RbacPermitOptionResponse> optionMap = new HashMap<>();

        if (StringUtils.isNotBlank(organizationId)) {
            List<RbacPermitAppDto> list = rbacPermitAppService.findOrgAppList(organizationId);
            for (RbacPermitAppDto dto : list) {
                RbacPermitAppResponse response = new RbacPermitAppResponse();
                BeanUtils.copyProperties(dto, response);
                appList.add(response);
            }
        }
        if (CollectionUtil.isNotEmpty(appList)) {
            List<String> categoryList = new ArrayList<>();
            for (RbacPermitAppResponse response : appList) {
                if (!categoryList.contains(response.getCategory())) {
                    categoryList.add(response.getCategory());
                }
                if (response.getAppId() != null) {
                    appMap.put(response.getAppId(), response);
                }
            }
            //根据分类进行分组
            Map<String, List<RbacPermitAppResponse>> listMap = appList.stream().collect(Collectors.groupingBy(RbacPermitAppResponse::getCategory));
            for (String category : categoryList) {
                RbacPermitAppResponse result = new RbacPermitAppResponse();
                result.setCategory(category);
                result.setChildren(listMap.get(category));
                resultList.add(result);
            }
        }
        if (CollectionUtil.isNotEmpty(resultList)) {
            //查询操作项列表
            for (RbacPermitAppResponse response : resultList) {
                for (RbacPermitAppResponse app : response.getChildren()) {
                    List<RbacPermitOptionResponse> optionList = this.findAppOptionList(app.getAppId());
                    app.setOptionList(optionList);
                    if (CollectionUtil.isNotEmpty(optionList)) {
                        for (RbacPermitOptionResponse optionResponse : optionList) {
                            if (optionResponse.getOptionId() != null) {
                                optionMap.put(optionResponse.getOptionId(), optionResponse);
                            }
                            if (CollectionUtil.isNotEmpty(optionResponse.getChildren())) {
                                for (RbacPermitOptionResponse child : optionResponse.getChildren()) {
                                    if (child.getOptionId() != null) {
                                        optionMap.put(child.getOptionId(), child);
                                    }
                                }
                            }
                        }
                    }
                }
            }
            //查询已授权的app和操作项
            RbacPermitAuthResponse permitAuthResponse = null;
            if (StringUtils.isNotBlank(request.getEmployeeId())) {
                if (request.getSearchType() == 1) {
                    //查询单独职员授权
                    permitAuthResponse = this.getEmployeeAuthByEmployeeId(organizationId, request.getEmployeeId());
                } else if (request.getSearchType() == 2) {
                    //查询所有职员授权
                    permitAuthResponse = this.getAllAuthByEmployeeId(organizationId, request.getEmployeeId());
                }
            } else if (request.getRoleId() != null) {
                //查询角色授权
                permitAuthResponse = this.getRoleAuthByRoleId(organizationId, request.getRoleId());
            }
            if (permitAuthResponse != null && CollectionUtil.isNotEmpty(permitAuthResponse.getAppList())) {
                for (RbacPermitAppAuthResponse appAuthResponse : permitAuthResponse.getAppList()) {
                    RbacPermitAppResponse appResponse = appMap.get(appAuthResponse.getAppId());
                    if (appResponse != null) {
                        appResponse.setChecked(1);
                    }
                    if (CollectionUtil.isNotEmpty(appAuthResponse.getOptionList())) {
                        for (RbacPermitOptionAuthResponse optionAuthResponse : appAuthResponse.getOptionList()) {
                            RbacPermitOptionResponse optionResponse = optionMap.get(optionAuthResponse.getOptionId());
                            if (optionResponse != null) {
                                optionResponse.setChecked(1);
                            }
                        }
                    }
                }
            }
        }

        return resultList;
    }

    /**
     * description：角色/职员，功能授权
     * author：linchunpeng
     * date：2024/2/4
     */
    @Override
    public void authPermitOption(RbacPermitAuthRequest request) {
        List<RbacPermitAppAuthRequestDto> appList = new ArrayList<>();
        if (CollectionUtil.isNotEmpty(request.getAppList())) {
            for (RbacPermitAppAuthRequest appAuthRequest : request.getAppList()) {
                List<RbacPermitOptionAuthRequestDto> optionList = new ArrayList<>();
                if (CollectionUtil.isNotEmpty(appAuthRequest.getOptionList())) {
                    for (RbacPermitOptionAuthRequest optionAuthRequest : appAuthRequest.getOptionList()) {
                        optionList.add(new RbacPermitOptionAuthRequestDto(optionAuthRequest.getOptionId()));
                    }
                }
                appList.add(new RbacPermitAppAuthRequestDto(appAuthRequest.getAppId(), optionList));
            }
        }
        rbacPermitAppService.authPermitOption(new RbacPermitAuthRequestDto(request.getOrganizationId(), request.getEmployeeIdList(),
                request.getRoleId(), appList));
    }


    /**
     * description：根据职员id，查询该职员所有的功能授权
     * author：linchunpeng
     * date：2024/2/5
     */
    @Override
    public RbacPermitAuthResponse getEmployeeAuthByEmployeeId(String organizationId, String employeeId) {
        RbacPermitAuthResponse response = new RbacPermitAuthResponse();
        if (StringUtils.isNotBlank(organizationId) && StringUtils.isNotBlank(employeeId)) {
            RbacPermitAuthResponseDto authResponseDto = rbacPermitAppService.getEmployeeAuthByEmployeeId(organizationId, employeeId);
            response = JSONObject.parseObject(JSONObject.toJSONString(authResponseDto), RbacPermitAuthResponse.class);
        }
        return response;
    }

    /**
     * description：根据角色id，查询该角色所有的功能授权
     * author：linchunpeng
     * date：2024/2/5
     */
    @Override
    public RbacPermitAuthResponse getRoleAuthByRoleId(String organizationId, Long roleId) {
        RbacPermitAuthResponse response = new RbacPermitAuthResponse();
        if (StringUtils.isNotBlank(organizationId) && roleId != null) {
            RbacPermitAuthResponseDto authResponseDto = rbacPermitAppService.getRoleAuthByRoleId(organizationId, roleId);
            response = JSONObject.parseObject(JSONObject.toJSONString(authResponseDto), RbacPermitAuthResponse.class);
        }
        return response;
    }

    /**
     * description：根据职员id，查询该职员所有的功能授权，包括个人和角色授权
     * author：linchunpeng
     * date：2024/2/5
     */
    @Override
    public RbacPermitAuthResponse getAllAuthByEmployeeId(String organizationId, String employeeId) {
        RbacPermitAuthResponse response = new RbacPermitAuthResponse();
        if (StringUtils.isNotBlank(organizationId) && StringUtils.isNotBlank(employeeId)) {
            RbacPermitAuthResponseDto authResponseDto = rbacPermitAppService.getAllAuthByEmployeeId(organizationId, employeeId);
            response = JSONObject.parseObject(JSONObject.toJSONString(authResponseDto), RbacPermitAuthResponse.class);
        }
        return response;
    }
}
