package com.bcxin.rbac.apis.custom;

import com.bcxin.Infrastructures.Pageable;
import com.bcxin.api.interfaces.ApiConstant;
import com.bcxin.api.interfaces.rbacs.custom.RbacCustomRoleManagerProvider;
import com.bcxin.api.interfaces.rbacs.custom.request.*;
import com.bcxin.api.interfaces.rbacs.custom.response.RbacCustomRoleResponse;
import com.bcxin.api.interfaces.rbacs.custom.response.RbacUserResponse;
import com.bcxin.api.interfaces.tenants.requests.employees.BatchImportRoleMemberRequest;
import com.bcxin.rbac.domain.RbacDbReader;
import com.bcxin.rbac.domain.dtos.custom.RbacCustomRoleDto;
import com.bcxin.rbac.domain.dtos.custom.RbacCustomRoleUserDto;
import com.bcxin.rbac.domain.dtos.custom.RbacUserDto;
import com.bcxin.rbac.domain.dtos.custom.RbacUserSearchDto;
import com.bcxin.rbac.domain.services.custom.RbacCustomRoleService;
import lombok.AllArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.apache.dubbo.config.annotation.DubboService;
import org.springframework.beans.BeanUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@DubboService(version = ApiConstant.VERSION, validation = "true", timeout = 30 *1000, retries = 0)
@AllArgsConstructor
public class RbacCustomRoleManagerProviderImpl implements RbacCustomRoleManagerProvider {
    private final RbacCustomRoleService rbacCustomRoleService;
    private final RbacDbReader dbReader;

    /**
     * description：根据企业id，查询企业自定义角色列表
     * author：linchunpeng
     * date：2024/2/1
     */
    @Override
    public List<RbacCustomRoleResponse> findOrgRoleList(RbacCustomRoleSearchRequest request) {
        List<RbacCustomRoleResponse> resultList = new ArrayList<>();
        if (StringUtils.isNotBlank(request.getOrganizationId())) {
            List<RbacCustomRoleDto> list = rbacCustomRoleService.findOrgRoleList(request.getOrganizationId(), request.getRoleName(), request.getEmployeeId());
            for (RbacCustomRoleDto dto : list) {
                RbacCustomRoleResponse result = new RbacCustomRoleResponse();
                BeanUtils.copyProperties(dto, result);
                resultList.add(result);
            }
        }
        return resultList;
    }

    /**
     * description：分页查询用户职员列表
     * author：linchunpeng
     * date：2024/2/3
     */
    @Override
    public Pageable<RbacUserResponse> getRbacUserDtoList(RbacUserSearchRequest request) {
        List<RbacUserResponse> resultList = new ArrayList<>();
        int total = 0;
        if (StringUtils.isNotBlank(request.getOrganizationId())) {
            Pageable<RbacUserDto> pageable = rbacCustomRoleService.getRbacUserDtoList(new RbacUserSearchDto(request.getPageIndex(),
                    request.getPageSize(), request.getOrganizationId(), request.getRoleId(), request.getKeyWord(),
                    request.getDepartId(), request.getOccupationType()));
            total = pageable.getTotal();
            for (RbacUserDto dto : pageable.getData()) {
                RbacUserResponse result = new RbacUserResponse(dto.getEmployeeId(), dto.getUserName(), dto.getMobile(),
                        dto.getDepartId(), dto.getDepartName(), dto.getOccupationType(), dto.getAddRoleTime());
                resultList.add(result);
            }
        }
        return Pageable.create(request.getPageIndex(), request.getPageSize(), total, resultList);
    }

    /**
     * description：企业新增角色
     * author：linchunpeng
     * date：2024/2/2
     */
    @Override
    public void addRole(RbacCustomRoleAddRequest request) {
        rbacCustomRoleService.addRole(request.getOrganizationId(), request.getRoleName(), request.getRemark(), request.getSeq());
    }

    /**
     * description：企业编辑角色
     * author：linchunpeng
     * date：2024/2/2
     */
    @Override
    public void editRole(RbacCustomRoleEditRequest request) {
        rbacCustomRoleService.editRole(request.getOrganizationId(), request.getId(), request.getRoleName(), request.getRemark(), request.getSeq());
    }

    /**
     * description：根据id查询角色详情
     * author：linchunpeng
     * date：2024/2/2
     */
    @Override
    public RbacCustomRoleResponse findById(Long id) {
        RbacCustomRoleResponse result = new RbacCustomRoleResponse();
        RbacCustomRoleDto dto = rbacCustomRoleService.findById(id);
        BeanUtils.copyProperties(dto, result);
        return result;
    }

    /**
     * description：根据id复制角色
     * author：linchunpeng
     * date：2024/2/2
     */
    @Override
    public void copyRole(RbacCustomRoleAddRequest request) {
        rbacCustomRoleService.copyRole(request.getSourceId(), request.getRoleName(), request.getRemark(), request.getSeq());
    }

    /**
     * description：根据id列表删除角色
     * author：linchunpeng
     * date：2024/2/2
     */
    @Override
    public void deleteRole(IdListRequest request) {
        rbacCustomRoleService.deleteRole(request.getIdList());
    }

    /**
     * description：根据角色id、职员id列表，批量设置用户关联
     * author：linchunpeng
     * date：2024/2/2
     */
    @Override
    public void addRoleAndUsers(RbacCustomRoleIdEmployeeIdListRequest request) {
        rbacCustomRoleService.addRoleAndUsers(request.getRoleId(), request.getEmployeeIdList());
    }


    /**
     * description：根据职员id、角色id列表，批量设置角色关联
     * author：linchunpeng
     * date：2024/2/2
     */
    @Override
    public void addUserAndRoles(RbacCustomEmployeeIdRoleIdListRequest request) {
        rbacCustomRoleService.addUserAndRoles(request.getEmployeeId(), request.getRoleIdList());
    }

    /**
     * description：根据职员id列表、角色id列表，批量设置角色关联
     * author：linchunpeng
     * date：2024/3/5
     */
    @Override
    public void addUsersAndRoles(RbacCustomRoleIdListEmployeeIdListRequest request) {
        rbacCustomRoleService.addUsersAndRoles(request.getEmployeeIdList(), request.getRoleIdList());
    }

    /**
     * description：根据角色id、职员id列表，批量删除用户关联
     * author：linchunpeng
     * date：2024/2/2
     */
    @Override
    public void deleteRoleAndUsers(RbacCustomRoleIdEmployeeIdListRequest request) {
        rbacCustomRoleService.deleteRoleAndUsers(request.getRoleId(), request.getEmployeeIdList());
    }

    /**
     * description：根据职员id、角色id列表，批量删除用户关联
     * author：linchunpeng
     * date：2024/2/2
     */
    @Override
    public void deleteUserAndRoles(RbacCustomEmployeeIdRoleIdListRequest request) {
        rbacCustomRoleService.deleteUserAndRoles(request.getEmployeeId(), request.getRoleIdList());
    }

    /**
     * description：根据角色id和职员id列表，查询角色用户关联列表
     * author：linchunpeng
     * date：2024/2/2
     */
    @Override
    public List<RbacUserResponse> findRoleUserListByRoleIdAndEmployeeIdList(Long roleId, List<String> employeeIdList) {
        List<RbacUserResponse> resultList = new ArrayList<>();
        List<RbacCustomRoleUserDto> list = rbacCustomRoleService.findRoleUserListByRoleIdAndEmployeeIdList(roleId, employeeIdList);
        for (RbacCustomRoleUserDto dto : list) {
            RbacUserResponse result = new RbacUserResponse();
            BeanUtils.copyProperties(dto, result);
            resultList.add(result);
        }
        return resultList;
    }

    /**
     * description：批量导入角色成员
     * author：linchunpeng
     * date：2024/2/6
     */
    @Override
    public List<Map<String, String>> importMember(String organizationId, Long roleId, Collection<BatchImportRoleMemberRequest> requests) {
        List<Map<String, String>> importDataList = requests.stream().map(BatchImportRoleMemberRequest::toMap).collect(Collectors.toList());
        return rbacCustomRoleService.importMember(organizationId, roleId, importDataList);
    }
}
