package com.zbkj.service.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.github.pagehelper.util.StringUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.huifu.bspay.sdk.opps.core.utils.SequenceTools;
import com.zbkj.common.constants.Constants;
import com.zbkj.common.constants.SysConfigConstants;
import com.zbkj.common.dto.HuifuWalletAccountDetailDto;
import com.zbkj.common.dto.HuifuWalletAccountDto;
import com.zbkj.common.dto.HuifuWalletBusiDto;
import com.zbkj.common.dto.HuifuWalletTransferDto;
import com.zbkj.common.exception.CrmebException;
import com.zbkj.common.model.huifu.HuifuWalletReconciliation;
import com.zbkj.common.model.huifu.HuifuWalletTrade;
import com.zbkj.common.model.huifu.HuifuWalletTranslog;
import com.zbkj.common.utils.DateUtils;
import com.zbkj.common.utils.FileUtil;
import com.zbkj.service.dao.*;
import com.zbkj.service.huifu.constants.HFBusiCode;
import com.zbkj.service.huifu.constants.HFConstants;
import com.zbkj.service.huifu.constants.HFProfileConstants;
import com.zbkj.service.huifu.v2.V2ApiRequest;
import com.zbkj.service.service.*;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.experimental.Accessors;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.support.TransactionTemplate;

import javax.annotation.Resource;
import java.io.File;
import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;

@Service
public class HuifuReconcileServiceImpl implements HuifuReconcileService {
    private static final Logger logger = LoggerFactory.getLogger(HuifuReconcileService.class);

    /**
     * 四种对账类型
     */
    private static final ReconcileType TYPE_ORDER = new ReconcileType().setName("订单/交易对账").setReconType("2").setFileType("1").setFileFix("_order_");
    private static final ReconcileType TYPE_SETTLE = new ReconcileType().setName("结算对账").setReconType("1").setFileType("2").setFileFix("_settle_");
    private static final ReconcileType TYPE_SETTLE_USER = new ReconcileType().setName("结算用户对账").setReconType("1").setFileType("8").setFileFix("_settleuser_");
    private static final ReconcileType TYPE_OUTCASH = new ReconcileType().setName("出金对账").setReconType("1").setFileType("5").setFileFix("_outcash_");
    private static final ReconcileType TYPE_SPLIT = new ReconcileType().setName("分账对账").setReconType("1").setFileType("3").setFileFix("_split_");

    public static final String SUCCESS_CODE = "00000000";
    public static final String CHARSET = "GB18030";
    public static final String BATCH_NO = "1";

    @Resource
    private HuifuWalletReconciliationDao huifuWalletReconciliationDao;

    @Resource
    private HuifuWalletTradeDao huifuWalletTradeDao;

    @Autowired
    private TransactionTemplate transactionTemplate;

    @Autowired
    private CommonConfigService commonConfigService;

    @Resource
    private HuifuWalletAccountDao huifuWalletAccountDao;

    @Autowired
    TransactionLogService transactionLogService;

    @Resource
    private HuifuWalletTranslogDao huifuWalletTranslogDao;
    @Resource
    private HuifuWalletAccountDetailDao huifuWalletAccountDetailDao;
    @Resource
    private HuifuWalletService huifuWalletService;
    @Resource
    private HuifuWalletBusiDao huifuWalletBusiDao;

    @Autowired
    TaskResidualService taskResidualService;
    @Autowired
    private RedissonClient redisson;
    private final static String SCHKEY = "SchTask:";

    /**
     * 交易数据对账单下载、解压、解析、转换、保存
     *
     * @param date yyyy-MM-dd
     */
    @Override
    public void reconcileOrder(String date,String huifuId) {
        executeReconcile(date, TYPE_ORDER, huifuId);
    }

    /**
     * 结算数据对账单下载、解压、解析、转换、保存
     *
     * @param date yyyy-MM-dd
     */
    @Override
    public void reconcileSettle(String date,String huifuId) {
        executeReconcile(date, TYPE_SETTLE, huifuId);
    }

    /**
     * 结算数据对账单下载、解压、解析、转换、保存   （用户结算）
     *
     * @param date yyyy-MM-dd
     */
    @Override
    public void reconcileSettleuser(String date,String huifuId) {
        executeReconcile(date, TYPE_SETTLE_USER, huifuId);
    }

    /**
     * 出金数据对账单下载、解压、解析、转换、保存
     *
     * @param date
     */
    @Override
    public void reconcileOutcash(String date,String huifuId) {
        executeReconcile(date, TYPE_OUTCASH, huifuId);
    }

    /**
     * 分账数据对账单下载、解压、解析、转换、保存
     *
     * @param date
     */
    @Override
    public void reconcileSplit(String date,String huifuId) {
        executeReconcile(date, TYPE_SPLIT, huifuId);
    }

    private void executeReconcile(String tradeDate, ReconcileType type,String huifuId) {
        //默认昨天
        if(StringUtils.isEmpty(tradeDate)){
            throw new CrmebException("查询日期不能为空");
        }
        Map<String, Object> param = new HashMap<>();
        param.put("reqSeqId", SequenceTools.getReqSeqId32());
        param.put("fileDate", tradeDate);  // YYYYMMDD格式
        //文件类型
        param.put("huifuId", huifuId);
        param.put("fileTypeQuery", type.getReconType());

        try {
            Map<String, Object> response = V2ApiRequest.api(HFBusiCode.ZDQUERY, param);
            String isSuf = Constants.N;
            if (SUCCESS_CODE.equals(response.get("resp_code"))) {
                isSuf = Constants.Y;
            }
            transactionLogService.save(HFBusiCode.ZDQUERY.toString(), isSuf, JSONObject.toJSONString(param),JSONObject.toJSONString(response));
            processResult(tradeDate, response, type);
        } catch (Exception e) {
            transactionLogService.save("executeReconcile",Constants.N, JSONObject.toJSONString(param),e.getMessage());
            logger.error(e.getMessage());
//                throw new CrmebException(e.getMessage());
        }
    }

    private void processResult(String tradeDate, Map<String, Object> response, ReconcileType type) {
        if (SUCCESS_CODE.equals(response.get("resp_code"))) {
            if (response.get("file_details") == null) {
                return;
            }
            /*** 解析返回的列表数据 ***/
            ArrayList<Map> recordlist = JSON.parseObject(response.get("file_details").toString(),
                    new TypeReference<ArrayList<Map>>() {
                    });
            for (Map<String, Object> record : recordlist) {
                //zip文件 url
                String downloadUrl = record.get("download_url").toString();
                String fileName = record.get("file_name").toString();
                String fileType = record.get("file_type").toString();
                dealWithEachZipFile(tradeDate, downloadUrl, fileName,type, BATCH_NO);
            }
        } else {
            logger.error(response.get("resp_desc").toString());
        }
    }

    private static Map<String, Object> genParamMap(String tradeDate) {
        Map<String, Object> p = new HashMap<>();
//        p.put("huifuId", HFProfileConstants.HUIFU_ID);
        p.put("reqSeqId", SequenceTools.getReqSeqId32());
        p.put("fileDate", tradeDate);  // YYYYMMDD格式
        return p;
    }

    private static Map<String,HuifuWalletAccountDto> accountMap = Maps.newHashMap();

    /**
     * 下载每个zip文件，并新增对账数据到数据库
     *
     * @param tradeDate
     * @param downloadUrl
     * @param fileName
     */
    private void dealWithEachZipFile(String tradeDate, String downloadUrl, String fileName, ReconcileType reconType, String batchNo) {
        //下载
//        String downloadToFolder = "D:\\data\\upload\\pay";
        String downloadToFolder = commonConfigService.getValueByKey(SysConfigConstants.PAYMENT_RECONCILIATION_DOWNLOAD_FOLDER);
        if (!cn.hutool.core.io.FileUtil.exist(downloadToFolder)) {
            cn.hutool.core.io.FileUtil.mkdir(downloadToFolder);
        }
        String localFile = FileUtil.downloadHttpUrl(downloadUrl, downloadToFolder+"/", fileName);
        if(localFile == null){
            logger.info("对账ZIP文件下载错误：{},{},{}", downloadUrl,downloadToFolder, fileName);
            return;
        }

        //解压到服务器目录
        String upzipFolder = downloadToFolder + File.separator + fileName.substring(0,fileName.length()-4);
        try {
            FileUtil.upzip(localFile, upzipFolder);
        }catch (Exception e){
            logger.info("对账ZIP文件解压错误：{},{}", localFile,upzipFolder);
            return;
        }
        File zipFile = new File(upzipFolder);
        if (!zipFile.exists()) {
            return;
        }
        File[] fi = zipFile.listFiles();
        if (fi.length==0){
            return;
        }
        for(File file : fi){
            //进入目录读取 日对账单【年月日_商户号_remittance_批次号.csv】文件转字符串
            String reconciliationFile = upzipFolder + File.separator + file.getName();
            List<String> lines = FileUtil.readCsv(reconciliationFile, CHARSET);
            if (lines == null) {
                logger.info("对账CSV文件读取错误：{}", reconciliationFile);
                continue;
            }
            lines.remove(0);//去掉行头
            if(lines.isEmpty()){//没有数据
                logger.info("对账CSV内容为空：{}",reconciliationFile);
                continue;
            }

            List<HuifuWalletReconciliation> reconciliationsList = new ArrayList<>();
            for (String line : lines) {
                line = line.replace("=\"\"","").replace("=\"","").replace("\"","");
                logger.info("对账CSV内容行数据信息：{}",line);
                HuifuWalletReconciliation huifuWalletReconciliation = null;
                if (TYPE_ORDER.equals(reconType) && file.getName().contains(reconType.getFileFix())) {
                    constructOrderRecon(line);
                }
                if (TYPE_SETTLE.equals(reconType) && file.getName().contains(reconType.getFileFix())) {
                    huifuWalletReconciliation = constructSettleRecon(line);
                }
                if (TYPE_SETTLE_USER.equals(reconType) && file.getName().contains(reconType.getFileFix())) {
                    huifuWalletReconciliation = constructSettleuserRecon(line);
                }
                if (TYPE_OUTCASH.equals(reconType) && file.getName().contains(reconType.getFileFix())) {
                    huifuWalletReconciliation = constructOutcashRecon(line);
                }
                if (TYPE_SPLIT.equals(reconType) && file.getName().contains(reconType.getFileFix())) {
                    huifuWalletReconciliation = constructSplitRecon(line);
                }
                if (huifuWalletReconciliation != null) {
                    reconciliationsList.add(huifuWalletReconciliation);
                }
            }

            /*** 插入或更新本地数据库 ***/
            if (reconciliationsList.size() > 0) {
                Boolean success = transactionTemplate.execute(e -> {
                    Boolean update = Boolean.TRUE;
                    try {
                        /*** 1.插入临时对帐数据 ***/
                        huifuWalletReconciliationDao.insertTempReconciliationList(reconciliationsList);

                        /*** 2.更新企业钱包对账数据 ***/
                        huifuWalletReconciliationDao.updateHuifuWalletReconciliation(tradeDate);

                        /*** 3.插入企业钱包对账数据 ***/
                        huifuWalletReconciliationDao.insertHuifuWalletReconciliation(tradeDate);

                        /*** 4.清除临时企业钱包对账数据 ***/
                        huifuWalletReconciliationDao.deleteTempWalletReconciliation();
                    } catch (Exception exception) {
                        logger.error(exception.getMessage());
                        logger.error("企业钱包-对帐数据-插入失败，对账日期 = {}", tradeDate);
//                        e.setRollbackOnly();
                        update = false;
                    }
                    return update;
                });
//                if (!success) throw new CrmebException("企业钱包-对帐数据-操作失败");
            }
        }
        //重置
        accountMap = Maps.newHashMap();
    }

    private HuifuWalletReconciliation constructSplitRecon(String line) {
        // 渠道商号	商户号	商户名称	原交易汇付订单号	原交易终端订单号	原交易金额	原交易时间	分账类型	入账方商户号	分账金额	分账手续费	账务状态	交易类型	分账完成时间	手续费出款方	请求流水号	请求日期
        if (StringUtil.isNotEmpty(line)) {
            String[] detailStr = line.split(","); // 交易信息数组
            HuifuWalletReconciliation huifuWalletReconciliation = new HuifuWalletReconciliation();
            huifuWalletReconciliation.setTradeSerialNo(detailStr[3]); // 原交易汇付订单号
            huifuWalletReconciliation.setMerchantNo(detailStr[1]); // 商户号
            huifuWalletReconciliation.setRequestSerialNo(detailStr[15]); // 请求流水号*
            huifuWalletReconciliation.setTradeAmount(StringUtils.isEmpty(detailStr[9]) ? BigDecimal.ZERO : new BigDecimal(detailStr[9])); // 交易金额 *
            huifuWalletReconciliation.setTradeStatus(detailStr[11]); //交易状态*
            huifuWalletReconciliation.setBankTradeType(detailStr[12]); // 银行交易类型*  结算类型
            huifuWalletReconciliation.setTradeStartTime(detailStr[6]); //原交易时间
            huifuWalletReconciliation.setTradeEndTime(detailStr[13]); //分账完成时间

            String outHuifuId = detailStr[17];//出款方商户号
            if (StringUtils.isNotEmpty(outHuifuId)) {
                HuifuWalletAccountDto account = accountMap.get(outHuifuId);
                if(account == null){
                    account = huifuWalletAccountDao.getHuifuWalletAccountByWalletNo(outHuifuId);
                    accountMap.put(outHuifuId,account);
                }
                if (account != null) {
                    huifuWalletReconciliation.setExpendUniId(account.getUniId());
                    huifuWalletReconciliation.setExpendBankCode(account.getBranchCode()); //出账方银行编码* detailStr[23]
                    huifuWalletReconciliation.setExpendAccountNo(outHuifuId); //出账方账号*
                    huifuWalletReconciliation.setExpendAccountName(account.getCardName()); //出账方户名*
                }
            }
            String inHuifuId = detailStr[8];//入账方商户号
            if (StringUtils.isNotEmpty(inHuifuId)) {
                HuifuWalletAccountDto account = accountMap.get(inHuifuId);
                if(account == null){
                    account = huifuWalletAccountDao.getHuifuWalletAccountByWalletNo(inHuifuId);
                    accountMap.put(inHuifuId,account);
                }
                if (account != null) {
                    huifuWalletReconciliation.setIncomeUniId(account.getUniId());
                    huifuWalletReconciliation.setIncomedBankCode(account.getBranchCode()); //入账方银行编码* detailStr[14]
                    huifuWalletReconciliation.setIncomeAccountNo(inHuifuId); //入账方账号*
                    huifuWalletReconciliation.setIncomeAccountName(account.getCardName()); //入账方户名*
                    huifuWalletReconciliation.setIncomeCerNo(account.getComCerNo()); //入账方证件号
                }
            }
            huifuWalletReconciliation.setMerchantRetain(""); //商户保留域
//            huifuWalletReconciliation.setAllotAccountInfo(""); //分账信息
//            huifuWalletReconciliation.setAllotBenefitInfo(""); //分润信息
            huifuWalletReconciliation.setRemark(""); //交易备注
            huifuWalletReconciliation.setServiceCharge(detailStr[10]); //手续费
//            huifuWalletReconciliation.setGuaranteeInfo(""); //担保信息

            return huifuWalletReconciliation;
        }
        return null;
    }

    private HuifuWalletReconciliation constructOutcashRecon(String line) {
        //产品号	渠道商号	商户号	商户名称	汇付订单号	外部订单号	交易类型	交易金额	交易手续费	交易状态	手续费出款方	请求流水号	请求日期	账户	账户名称	银行名称	交易时间
        if (StringUtil.isNotEmpty(line)) {
            String[] detailStr = line.split(","); // 交易信息数组
            HuifuWalletReconciliation huifuWalletReconciliation = new HuifuWalletReconciliation();
            huifuWalletReconciliation.setTradeSerialNo(detailStr[4]); // 汇付订单号
            huifuWalletReconciliation.setMerchantNo(detailStr[2]); // 商户号
            huifuWalletReconciliation.setRequestSerialNo(detailStr[10]); // 请求流水号*
            huifuWalletReconciliation.setTradeAmount(StringUtils.isEmpty(detailStr[6]) ? BigDecimal.ZERO : new BigDecimal(detailStr[6])); // 交易金额 *
            huifuWalletReconciliation.setTradeStatus(detailStr[8]); //交易状态*
            huifuWalletReconciliation.setBankTradeType("ENCHASHMENT".equals(detailStr[5])?"提现":detailStr[5]); // 银行交易类型*  结算类型
            huifuWalletReconciliation.setTradeStartTime(detailStr[11]); //交易时间
            huifuWalletReconciliation.setTradeEndTime(detailStr[16]); //交易时间

            String outHuifuId = detailStr[2];//出款方商户号
            if (StringUtils.isNotEmpty(outHuifuId)) {
                HuifuWalletAccountDto account = accountMap.get(outHuifuId);
                if(account == null){
                    account = huifuWalletAccountDao.getHuifuWalletAccountByWalletNo(outHuifuId);
                    accountMap.put(outHuifuId,account);
                }
                if (account != null) {
                    huifuWalletReconciliation.setExpendUniId(account.getUniId());
                    huifuWalletReconciliation.setExpendBankCode(account.getBranchCode()); //出账方银行编码* detailStr[23]
                    huifuWalletReconciliation.setExpendAccountNo(outHuifuId); //出账方账号*
                    huifuWalletReconciliation.setExpendAccountName(account.getCardName()); //出账方户名*
                }
            }
            String accountNo = detailStr[13];
            if (StringUtils.isNotEmpty(accountNo)) {
                if(accountNo.contains("***")){
                    HuifuWalletBusiDto busiDto = huifuWalletBusiDao.getHuifuWalletBusiByWalletNo(huifuWalletReconciliation.getExpendAccountNo());
                    if(busiDto != null){
                        huifuWalletReconciliation.setIncomeUniId(huifuWalletReconciliation.getExpendUniId());
                        huifuWalletReconciliation.setIncomedBankCode(busiDto.getBranchCode()); //入账方银行编码* detailStr[14]
                        huifuWalletReconciliation.setIncomeAccountNo(accountNo); //入账方账号*
                        huifuWalletReconciliation.setIncomeAccountName(busiDto.getCardName()); //入账方户名*
//                        huifuWalletReconciliation.setIncomeCerNo(); //入账方证件号
                    }
                }else{
                    HuifuWalletAccountDto account = accountMap.get(accountNo);
                    if(account == null){
                        account = huifuWalletAccountDao.getHuifuWalletAccountByAccountNo(accountNo);
                        accountMap.put(accountNo,account);
                    }
                    if (account != null) {
                        huifuWalletReconciliation.setIncomeUniId(account.getUniId());
                        huifuWalletReconciliation.setIncomedBankCode(account.getBranchCode()); //入账方银行编码* detailStr[14]
                        huifuWalletReconciliation.setIncomeAccountNo(detailStr[13]); //入账方账号*
                        huifuWalletReconciliation.setIncomeAccountName(detailStr[14]); //入账方户名*
                        huifuWalletReconciliation.setIncomeCerNo(account.getComCerNo()); //入账方证件号
                    }
                }
            }
            huifuWalletReconciliation.setMerchantRetain(""); //商户保留域
//            huifuWalletReconciliation.setAllotAccountInfo(""); //分账信息
//            huifuWalletReconciliation.setAllotBenefitInfo(""); //分润信息
            huifuWalletReconciliation.setRemark(""); //交易备注
            huifuWalletReconciliation.setServiceCharge(detailStr[7]); //手续费
//            huifuWalletReconciliation.setGuaranteeInfo(""); //担保信息

            return huifuWalletReconciliation;
        }
        return null;
    }

    /**
     * 结算记录解析
     * @param line
     * @return
     */
    private HuifuWalletReconciliation constructSettleRecon(String line) {
        if(line != null || null == line){
            //TODO 暂时提现都由企业发起，走出金对账，结算对账暂时不用
            return null;
        }
        if (StringUtil.isNotEmpty(line)) {
            String[] detailStr = line.split(","); // 交易信息数组
            // TODO 从csv文件中解析 产品号	渠道商号	商户号	商户名称	结算类型	结算日期	账户	账户名称	银行名称	结算交易状态	结算金额	手续费	结算信息
            HuifuWalletReconciliation huifuWalletReconciliation = new HuifuWalletReconciliation();
            huifuWalletReconciliation.setTradeSerialNo(""); // 交易流水号 *
            huifuWalletReconciliation.setMerchantNo(detailStr[2]); // 商户号
            huifuWalletReconciliation.setRequestSerialNo(""); // 请求流水号*
            huifuWalletReconciliation.setTradeAmount(StringUtils.isEmpty(detailStr[10]) ? BigDecimal.ZERO : new BigDecimal(detailStr[10])); // 交易金额 *
            huifuWalletReconciliation.setTradeStatus(detailStr[9]); //交易状态*
            huifuWalletReconciliation.setBankTradeType(detailStr[4]); // 银行交易类型*  结算类型
            huifuWalletReconciliation.setTradeStartTime(detailStr[5]); //结算日期
            huifuWalletReconciliation.setTradeEndTime(detailStr[5]); //结算日期

            String outHuifuId = detailStr[2];//出款方商户号
            if (StringUtils.isNotEmpty(outHuifuId)) {
                HuifuWalletAccountDto account = accountMap.get(outHuifuId);
                if(account == null){
                    account = huifuWalletAccountDao.getHuifuWalletAccountByWalletNo(outHuifuId);
                    accountMap.put(outHuifuId,account);
                }
                if (account != null) {
                    huifuWalletReconciliation.setExpendUniId(account.getUniId());
                    huifuWalletReconciliation.setExpendBankCode(account.getBranchCode()); //出账方银行编码* detailStr[23]
                    huifuWalletReconciliation.setExpendAccountNo(outHuifuId); //出账方账号*
                    huifuWalletReconciliation.setExpendAccountName(account.getCardName()); //出账方户名*
                }
            }
            String inHuifuId = detailStr[6];//入账方银行卡账号
            if (StringUtils.isNotEmpty(inHuifuId)) {
                HuifuWalletAccountDto account = accountMap.get(inHuifuId);
                if(account == null){
                    account = huifuWalletAccountDao.getHuifuWalletAccountByWalletNo(inHuifuId);
                    accountMap.put(inHuifuId,account);
                }
                if (account != null) {
                    huifuWalletReconciliation.setIncomeUniId(account.getUniId());
                    huifuWalletReconciliation.setIncomedBankCode(account.getBranchCode()); //入账方银行编码* detailStr[14]
                    huifuWalletReconciliation.setIncomeAccountNo(inHuifuId); //入账方账号*
                    huifuWalletReconciliation.setIncomeAccountName(account.getCardName()); //入账方户名*
                    huifuWalletReconciliation.setIncomeCerNo(account.getComCerNo()); //入账方证件号
                }
            }
            huifuWalletReconciliation.setMerchantRetain(""); //商户保留域
//            huifuWalletReconciliation.setAllotAccountInfo(""); //分账信息
//            huifuWalletReconciliation.setAllotBenefitInfo(""); //分润信息
            huifuWalletReconciliation.setRemark(""); //交易备注
            huifuWalletReconciliation.setServiceCharge(detailStr[11]); //手续费
//            huifuWalletReconciliation.setGuaranteeInfo(""); //担保信息

            return huifuWalletReconciliation;
        }
        return null;
    }

    /**
     * 结算记录解析 （用户）
     * @param line
     * @return
     */
    private HuifuWalletReconciliation constructSettleuserRecon(String line) {
        if(line != null || null == line){
            //TODO 暂时提现都由企业发起，走出金对账，结算对账暂时不用
            return null;
        }
        if (StringUtil.isNotEmpty(line)) {
            String[] detailStr = line.split(","); // 交易信息数组
            // TODO 从csv文件中解析 产品号	渠道商号	商户号	商户名称	结算类型	结算日期	账户	账户名称	银行名称	结算交易状态	结算金额	手续费	结算信息
            // TODO 从csv文件中解析 产品号,渠道商号,商户号,商户名称,用户名,用户号,结算类型,结算日期,账户,账户名称,银行名称,结算交易状态,结算金额,手续费,结算信息
            HuifuWalletReconciliation huifuWalletReconciliation = new HuifuWalletReconciliation();
            huifuWalletReconciliation.setTradeSerialNo(""); // 交易流水号 *
            huifuWalletReconciliation.setMerchantNo(detailStr[2]); // 商户号
            huifuWalletReconciliation.setRequestSerialNo(""); // 请求流水号*
            huifuWalletReconciliation.setTradeAmount(StringUtils.isEmpty(detailStr[12]) ? BigDecimal.ZERO : new BigDecimal(detailStr[12])); // 交易金额 *
            huifuWalletReconciliation.setTradeStatus(detailStr[11]); //交易状态*
            huifuWalletReconciliation.setBankTradeType(detailStr[6]); // 银行交易类型*  结算类型
            huifuWalletReconciliation.setTradeStartTime(detailStr[7]); //结算日期
            huifuWalletReconciliation.setTradeEndTime(detailStr[7]); //结算日期

            String outHuifuId = detailStr[2];//出款方商户号
            if (StringUtils.isNotEmpty(outHuifuId)) {
                HuifuWalletAccountDto account = accountMap.get(outHuifuId);
                if(account == null){
                    account = huifuWalletAccountDao.getHuifuWalletAccountByWalletNo(outHuifuId);
                    accountMap.put(outHuifuId,account);
                }
                if (account != null) {
                    huifuWalletReconciliation.setExpendUniId(account.getUniId());
                    huifuWalletReconciliation.setExpendBankCode(account.getBranchCode()); //出账方银行编码* detailStr[23]
                    huifuWalletReconciliation.setExpendAccountNo(outHuifuId); //出账方账号*
                    huifuWalletReconciliation.setExpendAccountName(account.getCardName()); //出账方户名*
                }
            }
            String inHuifuId = detailStr[8];//入账方银行卡账号
            if (StringUtils.isNotEmpty(inHuifuId)) {
                HuifuWalletAccountDto account = accountMap.get(inHuifuId);
                if(account == null){
                    account = huifuWalletAccountDao.getHuifuWalletAccountByAccountNo(inHuifuId);
                    accountMap.put(inHuifuId,account);
                }
                if (account != null) {
                    huifuWalletReconciliation.setIncomeUniId(account.getUniId());
                    huifuWalletReconciliation.setIncomedBankCode(account.getBranchCode()); //入账方银行编码* detailStr[14]
                    huifuWalletReconciliation.setIncomeAccountNo(inHuifuId); //入账方账号*
                    huifuWalletReconciliation.setIncomeAccountName(account.getCardName()); //入账方户名*
                    huifuWalletReconciliation.setIncomeCerNo(account.getComCerNo()); //入账方证件号
                }
            }
            huifuWalletReconciliation.setMerchantRetain(""); //商户保留域
//            huifuWalletReconciliation.setAllotAccountInfo(""); //分账信息
//            huifuWalletReconciliation.setAllotBenefitInfo(""); //分润信息
            huifuWalletReconciliation.setRemark(""); //交易备注
            huifuWalletReconciliation.setServiceCharge(detailStr[13]); //手续费
//            huifuWalletReconciliation.setGuaranteeInfo(""); //担保信息

            return huifuWalletReconciliation;
        }
        return null;
    }

    private HuifuWalletReconciliation constructOrderRecon(String line) {
        if (StringUtil.isNotEmpty(line)) {

            String[] detailStr = line.split(","); // 交易信息数组

            HuifuWalletTrade huifuWalletTrade = huifuWalletTradeDao.getHuifuWalletTradeByHfSeqId(detailStr[5]);

            HuifuWalletReconciliation huifuWalletReconciliation = new HuifuWalletReconciliation();
            huifuWalletReconciliation.setTradeSerialNo(detailStr[5]); // 交易流水号 *
            huifuWalletReconciliation.setMerchantNo(detailStr[1]); // 商户号
            huifuWalletReconciliation.setRequestSerialNo(detailStr[6]); // 请求流水号*
            huifuWalletReconciliation.setTradeAmount(StringUtils.isEmpty(detailStr[9]) ? BigDecimal.ZERO : new BigDecimal(detailStr[9])); // 交易金额 *
            huifuWalletReconciliation.setTradeStatus(detailStr[12]); //交易状态*
            huifuWalletReconciliation.setBankTradeType(detailStr[7]); // 银行交易类型*
            huifuWalletReconciliation.setTradeStartTime(detailStr[37]); //交易订单创建时间*35
            huifuWalletReconciliation.setTradeEndTime(detailStr[30]); //交易订单完成时间
//            String yeStr = detailStr[45];//余额支付串 TODO 解析出出账方/入账方
            String outHuifuId = huifuWalletTrade.getExpendAccountNo();//出款方商户号 TODO 从 yeStr 拿值
            if (StringUtils.isNotEmpty(outHuifuId)) {
                HuifuWalletAccountDto account = accountMap.get(outHuifuId);
                if(account == null){
                    account = huifuWalletAccountDao.getHuifuWalletAccountByWalletNo(outHuifuId);
                    accountMap.put(outHuifuId,account);
                }
                if (account != null) {
                    huifuWalletReconciliation.setExpendUniId(account.getUniId());
                    huifuWalletReconciliation.setExpendBankCode(account.getBranchCode()); //出账方银行编码* detailStr[23]
                    huifuWalletReconciliation.setExpendAccountNo(outHuifuId); //出账方账号*
                    huifuWalletReconciliation.setExpendAccountName(account.getCardName()); //出账方户名*
                }
            }
            String inHuifuId = huifuWalletTrade.getIncomeAccountNo();//入账方商户号 TODO 从 yeStr 拿值
            String incomedBankCode = "";
            if (StringUtils.isNotEmpty(inHuifuId)) {
                HuifuWalletAccountDto account = accountMap.get(inHuifuId);
                if(account == null){
                    account = huifuWalletAccountDao.getHuifuWalletAccountByWalletNo(inHuifuId);
                    accountMap.put(inHuifuId,account);
                }
                if (account != null) {
                    incomedBankCode = account.getBankCode();
                    huifuWalletReconciliation.setIncomeUniId(account.getUniId());
                    huifuWalletReconciliation.setIncomedBankCode(account.getBranchCode()); //入账方银行编码* detailStr[14]
                    huifuWalletReconciliation.setIncomeAccountNo(inHuifuId); //入账方账号*
                    huifuWalletReconciliation.setIncomeAccountName(account.getCardName()); //入账方户名*
                    huifuWalletReconciliation.setIncomeCerNo(account.getComCerNo()); //入账方证件号
                }
            }
            huifuWalletReconciliation.setMerchantRetain(detailStr[34]); //商户保留域
//            huifuWalletReconciliation.setAllotAccountInfo(""); //分账信息
//            huifuWalletReconciliation.setAllotBenefitInfo(""); //分润信息
            huifuWalletReconciliation.setRemark(detailStr[39]); //交易备注
            huifuWalletReconciliation.setServiceCharge(detailStr[10]); //手续费
//            huifuWalletReconciliation.setGuaranteeInfo(""); //担保信息
            //如果是入金记录，存入提现记录表，之后该表用来记录出金入金信息
            if(Constants.TRADETYPE_QYZHCZ.equals(huifuWalletTrade.getTradeType())){
                if(huifuWalletTranslogDao.findSizeByTransId(huifuWalletReconciliation.getTradeSerialNo())<1){
                    HuifuWalletTranslog translog = new HuifuWalletTranslog();
                    translog.setUniId(huifuWalletTrade.getUniId());
                    translog.setCreateTime(new Date());
                    translog.setTransId(huifuWalletReconciliation.getTradeSerialNo());
                    translog.setTransDate(huifuWalletReconciliation.getRequestSerialNo());
                    translog.setTransAmt(huifuWalletReconciliation.getTradeAmount().toString());
                    translog.setTransStat(huifuWalletReconciliation.getTradeStatus());
                    translog.setCardNo(huifuWalletReconciliation.getIncomeCerNo());
                    translog.setCardName(huifuWalletReconciliation.getIncomeAccountName());
                    translog.setBankCode(incomedBankCode);
                    translog.setFeeAmt(huifuWalletReconciliation.getServiceCharge());
                    translog.setRemark(huifuWalletReconciliation.getRemark());
                    translog.setTransType(huifuWalletReconciliation.getBankTradeType());
                    huifuWalletTranslogDao.insert(translog);
                }
            }
            return huifuWalletReconciliation;
        }
        return null;
    }

    /**
     * <b> 异步执行平台分账 </b>
     * @author ZXF
     * @create 2023/09/15 0015 17:48
     * @version
     * @注意事项 </b>
     */
    @Async
    @Override
    public void syncTransferToPlat(HuifuWalletTransferDto dto){
        try{
            if(!huifuWalletService.transferToPlat(dto)){
                taskResidualService.saveTaskResidual("transferToPlat","平台分账补偿","com.zbkj.service.service.HuifuWalletService",1,"com.zbkj.common.dto.HuifuWalletTransferDto",JSONObject.toJSONString(dto));
            }
        }catch (Exception e){
            logger.error(">>>>>>>>> 企业分账给平台transferToPlat.执行异常："+e.getMessage());
            taskResidualService.saveTaskResidual("transferToPlat","平台分账补偿","com.zbkj.service.service.HuifuWalletService",1,"com.zbkj.common.dto.HuifuWalletTransferDto",JSONObject.toJSONString(dto));
        }
    }

    /**
     * <b> 异步执行冻结金额计算 </b>
     * @author ZXF
     * @create 2023/09/15 0015 17:48
     * @version
     * @注意事项 </b>
     */
    @Async
    @Override
    public void syncUpdateSumFrozenAmount(String uniId, BigDecimal amount){
        try{
            logger.error(">>>>>>>>> 企业冻结金额计算modifyBalance.开始：");
            if(!huifuWalletService.modifyBalance(uniId,amount)){
                taskResidualService.saveTaskResidual("modifyBalance","冻结金额计算补偿","com.zbkj.service.service.HuifuWalletService",2,"String,BigDecimal",uniId+","+amount);
            }
        }catch (Exception e){
            logger.error(">>>>>>>>> 企业冻结金额计算modifyBalance.执行异常："+e.getMessage());
            taskResidualService.saveTaskResidual("modifyBalance","冻结金额计算补偿","com.zbkj.service.service.HuifuWalletService",1,"String,BigDecimal",uniId+","+amount);
        }
    }

    /**
     * <b> 异步执行平台退款 </b>
     * @author ZXF
     * @create 2023/09/15 0015 17:48
     * @version
     * @注意事项 </b>
     */
    @Async
    @Override
    public void syncRefundToPlat(HuifuWalletTransferDto dto){
        try{
            if(!huifuWalletService.refundToPlat(dto)){
                taskResidualService.saveTaskResidual("refundToPlat","平台退款给企业补偿","com.zbkj.service.service.HuifuWalletService",1,"com.zbkj.common.dto.HuifuWalletTransferDto",JSONObject.toJSONString(dto));
            }
        }catch (Exception e){
            logger.error(">>>>>>>>> 平台退款给企业refundToPlat.执行异常："+e.getMessage());
            taskResidualService.saveTaskResidual("refundToPlat","平台退款给企业补偿","com.zbkj.service.service.HuifuWalletService",1,"com.zbkj.common.dto.HuifuWalletTransferDto",JSONObject.toJSONString(dto));
        }
    }



    /**
     * 余额支付交易分账明细查询接口
     *
     * @param accountNo 发起方商户号
     * @param hfSeqId 全局流水号
     * @param tradeDate 交易日期
     * @return
     * @author lp
     * @date 2019-12-09
     */
    @Async
    @Override
    public void payTradeDetailTask(String accountNo,String hfSeqId,String tradeDate) {

        String lockId = SCHKEY+hfSeqId;
        RLock lock = redisson.getLock(lockId);
        try {
            if(lock.tryLock(3,10, TimeUnit.SECONDS)){
                logger.debug(lockId+":getLock: " + Thread.currentThread().getId());
                HuifuWalletAccountDto aDto = huifuWalletAccountDao.getHuifuWalletAccountByWalletNo(accountNo);
                //如果已经存在就结束
                int cf = huifuWalletAccountDetailDao.checkFlowNo(hfSeqId,aDto.getUniId());
                if(cf>0){
                    return;
                }
                /* 调用支付平台接口获取账户交易明细 ***/
                Map<String, Object> p = new HashMap<>();
                p.put("huifuId", accountNo);
                p.put("hfSeqId", hfSeqId);
                p.put("OrgReqDate", tradeDate);
                Map<String, Object> response;
                HFBusiCode hFBusiCode;
                try {
                    hFBusiCode = HFBusiCode.APQUERY;
                    response = V2ApiRequest.api(hFBusiCode,p);
                }catch (Exception e){
                    transactionLogService.save("payTradeDetailTask",Constants.N, JSONObject.toJSONString(p),e.getMessage());
                    throw new CrmebException(e.getMessage());
                }

                if(HFConstants.REQUEST_SUCC_CODE.equals(RET(response,1))){
                    transactionLogService.save(hFBusiCode.toString(),Constants.Y, JSONObject.toJSONString(p),JSONObject.toJSONString(response));
                    HuifuWalletTrade huifuWalletTrade = huifuWalletTradeDao.getHuifuWalletTradeByHfSeqId(hfSeqId);
                    List<HuifuWalletAccountDetailDto> detailList = Lists.newArrayList();
                    HuifuWalletAccountDetailDto detail = new HuifuWalletAccountDetailDto();
                    detail.setUniId(huifuWalletTrade.getUniId());
                    detail.setWalletAccountNo(accountNo);

                    String splitType = response.get("delay_acct_flag").toString();
                    detail.setTradeDate(response.get("org_req_date").toString()); // 交易时间
                    detail.setTradeFlowNo(response.get("org_hf_seq_id").toString()); // 交易流水号
                    detail.setFlowNo(huifuWalletTrade.getBusinessPartnerSeq()); // 汇付交易订单号
                    detail.setAccountsDealType("内部户处理"); //账务处理类型
                    detail.setAccountsDealDate(response.get("trans_finish_time").toString());// 账务处理时间
                    detail.setAmount(response.get("ord_amt").toString()); // 交易金额
//                    detail.setBalance(response.get("balance_amt").toString()); // 交易后余额,工行没有交易后余额的数据
                    detail.setDirection("出账"); // 资金流向
                    String transStat = response.get("trans_stat").toString();
                    transStat = transStat.equals("S")?"成功":transStat.equals("F")?"失败":"处理中";
                    detail.setHandStatus(transStat); // 处理状态
                    String tradeType = StringUtils.isNotEmpty(huifuWalletTrade.getRemark())&&huifuWalletTrade.getRemark().contains("退款")?"反向交易":"正向交易";
                    detail.setTradeType(tradeType); // 交易类型
                    detail.setCreateDate(DateUtils.formatDate(huifuWalletTrade.getCreateTime(),"yyyy-MM-dd HH:mm:ss")); // 订单创建时间
                    detail.setServiceType("N".equals(splitType)?"实时":"延时"); // 服务类型
                    detail.setAccountIn(huifuWalletTrade.getIncomeAccountNo());// 收款方账户
                    detail.setAccountOut(response.get("out_huifu_id").toString());// 付款方账户
                    detail.setMemo(response.get("remark").toString()); // 备注
                    HuifuWalletAccountDto accountIn = null;
                    HuifuWalletAccountDto accountOut = null;
                    if (StringUtil.isEmpty(detail.getAccountIn())) {
                        detail.setAccountInName("无");// 收款方账户
                    } else {
                        accountIn = huifuWalletAccountDao.getHuifuWalletAccountByWalletNo(detail.getAccountIn());
                        detail.setAccountInName(StringUtils.isEmpty(accountIn.getComName()) ? "无" : accountIn.getComName()); // 收款方账户中文
                    }
                    if (StringUtil.isEmpty(detail.getAccountOut())) {
                        detail.setAccountOutName("无"); // 付款方账户
                    } else {
                        accountOut = huifuWalletAccountDao.getHuifuWalletAccountByWalletNo(detail.getAccountOut());
                        detail.setAccountOutName(StringUtils.isEmpty(accountOut.getComName()) ? "无" : accountOut.getComName()); // 付款方账户中文
                    }

                    if(accountOut!=null && accountIn!=null){//转账
                        int count = huifuWalletAccountDetailDao.checkFlowNo(detail.getTradeFlowNo(),detail.getUniId());
                        if(count==0){
                            detailList.add(detail);
                        }
                        HuifuWalletAccountDetailDto detail2 = new HuifuWalletAccountDetailDto();
                        BeanUtils.copyProperties(detail,detail2);
                        detail2.setUniId(accountIn.getUniId());
                        detail2.setWalletAccountNo(accountIn.getWalletAccountNo());
                        detail2.setDirection("入账"); // 资金流向
                        count = huifuWalletAccountDetailDao.checkFlowNo(detail2.getTradeFlowNo(),detail2.getUniId());
                        if(count==0){
                            detailList.add(detail2);
                        }
                    }else if(accountOut!=null && accountIn==null){//提现
                        int count = huifuWalletAccountDetailDao.checkFlowNo(detail.getTradeFlowNo(),detail.getUniId());
                        if(count==0){
                            detailList.add(detail);
                        }
                    }else if(accountIn!=null && accountOut==null){//充值
                        detail.setDirection("入账"); // 资金流向
                        int count = huifuWalletAccountDetailDao.checkFlowNo(detail.getTradeFlowNo(),detail.getUniId());
                        if(count==0){
                            detailList.add(detail);
                        }
                    }
                    if(detailList.size()>0){
                        /* 批量插入数据 */
                        huifuWalletAccountDetailDao.insertBatch(detailList);
                    }
                }else{
                    transactionLogService.save(hFBusiCode.toString(),Constants.N, JSONObject.toJSONString(p),JSONObject.toJSONString(response));
                    logger.error(RET(response,2));
                }
            }
        } catch (InterruptedException e) {
            logger.error("huifuWalletRechargeCallback error", e);
        }finally {
            if (lock.isHeldByCurrentThread()) {
                logger.debug(lockId+":unLock: " + Thread.currentThread().getId());
                lock.unlock();
            }
        }
    }

    /**
     * <b> 返回常量包装 </b>
     * @param type 1获取code 2获取desc
     * @author ZXF
     * @create 2023/09/12 0012 11:22
     * @version
     * @注意事项 </b>
     */
    private static String RET(Map<String, Object> response,int type){
        if(type == 1){
            return response.get("resp_code").toString();
        }
        return response.get("resp_desc").toString();
    }
}

@Data
@EqualsAndHashCode(callSuper = false)
@Accessors(chain = true)
class ReconcileType {
    /**
     * 类型名称
     */
    private String name;
    /**
     * 1、日结算对账单<br/>2、日交易数据<br/>3、月结算对账单（月对账单取file_date所属月份）<br/>4、月交易数据（月对账单取file_date所属月份）
     */
    private String reconType;
    /**
     * 对账单类型传1、3时必填；<br/>1、交易对账单<br/>2、结算对账单<br/>3、分账对账单<br/>5、出金对账单<br/>6、结算-用户维度对账单
     */
    private String fileType;
    /**
     * 解析文件名用
     */
    private String fileFix;
}
