package com.zbkj.admin.controller.platform;


import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import com.github.pagehelper.Page;
import com.github.pagehelper.PageHelper;
import com.zbkj.common.exception.CrmebException;
import com.zbkj.common.model.admin.SystemAdmin;
import com.zbkj.common.model.user.User;
import com.zbkj.common.page.CommonPage;
import com.zbkj.common.request.*;
import com.zbkj.common.response.UserAdminDetailResponse;
import com.zbkj.common.response.UserResponse;
import com.zbkj.common.response.bcx.BcxV5EmpResponse;
import com.zbkj.common.response.bcx.BcxV5OrgResponse;
import com.zbkj.common.result.CommonResult;
import com.zbkj.service.service.SystemAdminService;
import com.zbkj.service.service.UserService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.constraints.NotNull;
import java.util.ArrayList;
import java.util.List;

/**
 * 平台端用户控制器
 * +----------------------------------------------------------------------
 * | CRMEB [ CRMEB赋能开发者，助力企业发展 ]
 * +----------------------------------------------------------------------
 * | Copyright (c) 2016~2023 https://www.crmeb.com All rights reserved.
 * +----------------------------------------------------------------------
 * | Licensed CRMEB并不是自由软件，未经许可不能去掉CRMEB相关版权
 * +----------------------------------------------------------------------
 * | Author: CRMEB Team <admin@crmeb.com>
 * +----------------------------------------------------------------------
 */
@Slf4j
@RestController
@RequestMapping("api/admin/platform/user")
@Api(tags = "平台端用户控制器")
@Validated
public class PlatformUserController {
    @Autowired
    private UserService userService;
    @Autowired
    private SystemAdminService systemAdminService;

    @PreAuthorize("hasAuthority('platform:user:page:list')")
    @ApiOperation(value = "平台端用户分页列表")
    @RequestMapping(value = "/list", method = RequestMethod.GET)
    public CommonResult<CommonPage<UserResponse>> getList(@ModelAttribute @Validated UserSearchRequest request,
                                                          @ModelAttribute PageParamRequest pageParamRequest) {
        CommonPage<UserResponse> userCommonPage = CommonPage.restPage(userService.getPlatformPage(request, pageParamRequest));
        return CommonResult.success(userCommonPage);
    }

    @PreAuthorize("hasAuthority('platform:user:page:list')")
    @ApiOperation(value = "获取V5企业列表")
    @RequestMapping(value = "/v5org/list", method = RequestMethod.GET)
    public CommonResult<CommonPage<BcxV5OrgResponse>> getV5OrgList(@RequestParam String name, @ModelAttribute PageParamRequest pageParamRequest) {
        ArrayList<BcxV5OrgResponse> result = new ArrayList<>();
        Page<User> pageUser = PageHelper.startPage(pageParamRequest.getPage(), pageParamRequest.getLimit());
        List<User> orgList = userService.getByV5OrgList(name);
        if (orgList == null || orgList.isEmpty()) {
            return CommonResult.success(CommonPage.restPage(CommonPage.copyPageInfo(pageUser, result)));
        }

        orgList.forEach(user -> {
            result.add(BcxV5OrgResponse.builder()
                    .orgAccount(user.getAccount())
                    .orgName(user.getNickname())
                    .build());
        });
        return CommonResult.success(CommonPage.restPage(CommonPage.copyPageInfo(pageUser, result)));
    }

    @PreAuthorize("hasAuthority('platform:user:page:list')")
    @ApiOperation(value = "获取V5职员列表")
    @RequestMapping(value = "/v5emp/list", method = RequestMethod.GET)
    public CommonResult<CommonPage<BcxV5EmpResponse>> getV5EmpList(@RequestParam @NotNull Integer merId, @RequestParam String name, @ModelAttribute PageParamRequest pageParamRequest) {
        ArrayList<BcxV5EmpResponse> result = new ArrayList<>();
        Page<User> pageUser = PageHelper.startPage(pageParamRequest.getPage(), pageParamRequest.getLimit());
        List<SystemAdmin> admins = systemAdminService.query().eq("mer_id", merId).eq("type", 2).list();
        if (CollUtil.isEmpty(admins)) {
            throw new CrmebException("商户未绑定v5企业，请绑定后再获取职员列表");
        }
        if (admins.stream().map(SystemAdmin::getOrgAccount).distinct().count() > 1) {
            throw new CrmebException("商户绑定了多家v5企业，请确认绑定关系后联系研发");
        }
        User org = userService.getByAccount(admins.get(0).getOrgAccount());
        if (ObjectUtil.isNull(org)) {
            throw new CrmebException("找不到org用户：" + admins.get(0).getOrgAccount());
        }
        List<User> orgList = userService.getByV5EmpList(org.getId(), name);
        if (orgList == null || orgList.isEmpty()) {
            return CommonResult.success(CommonPage.restPage(CommonPage.copyPageInfo(pageUser, result)));
        }

        orgList.forEach(user -> {
            result.add(BcxV5EmpResponse.builder()
                    .empAccount(user.getAccount())
                    .empName(user.getNickname())
                    .build());
        });
        return CommonResult.success(CommonPage.restPage(CommonPage.copyPageInfo(pageUser, result)));
    }

    @PreAuthorize("hasAuthority('platform:user:update')")
    @ApiOperation(value = "修改用户信息")
    @RequestMapping(value = "/update", method = RequestMethod.POST)
    public CommonResult<String> update(@RequestBody @Validated UserUpdateRequest userRequest) {
        if (userService.updateUser(userRequest)) {
            return CommonResult.success();
        }
        return CommonResult.failed();
    }

    @PreAuthorize("hasAuthority('platform:user:detail')")
    @ApiOperation(value = "用户详情")
    @RequestMapping(value = "/detail/{id}", method = RequestMethod.GET)
    public CommonResult<UserAdminDetailResponse> detail(@PathVariable(value = "id") Integer id) {
        return CommonResult.success(userService.getAdminDetail(id));
    }

    @PreAuthorize("hasAuthority('platform:user:tag')")
    @ApiOperation(value = "用户分配标签")
    @RequestMapping(value = "/tag", method = RequestMethod.POST)
    public CommonResult<String> tag(@RequestBody @Validated UserAssignTagRequest request) {
        if (userService.tag(request)) {
            return CommonResult.success();
        }
        return CommonResult.failed();
    }

    @PreAuthorize("hasAuthority('platform:user:operate:integer')")
    @ApiOperation(value = "操作用户积分")
    @RequestMapping(value = "/operate/integer", method = RequestMethod.GET)
    public CommonResult<Object> founds(@Validated UserOperateIntegralRequest request) {
        if (userService.operateUserInteger(request)) {
            return CommonResult.success();
        }
        return CommonResult.failed();
    }

    @PreAuthorize("hasAuthority('platform:user:operate:balance')")
    @ApiOperation(value = "操作用户余额")
    @RequestMapping(value = "/operate/balance", method = RequestMethod.GET)
    public CommonResult<Object> founds(@Validated UserOperateBalanceRequest request) {
        if (userService.operateUserBalance(request)) {
            return CommonResult.success();
        }
        return CommonResult.failed();
    }
}



