package cn.myapps.webservice;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;

import cn.myapps.authtime.common.dao.PersistenceUtils;
import cn.myapps.authtime.common.service.AuthTimeServiceManager;
import cn.myapps.authtime.domain.model.DomainVO;
import cn.myapps.authtime.user.model.UserDepartmentRoleSet;
import cn.myapps.authtime.user.model.UserVO;
import cn.myapps.authtime.user.service.UserProcess;
import cn.myapps.common.model.application.Application;
import cn.myapps.common.model.permission.Permission;
import cn.myapps.common.model.resource.ResourceVO;
import cn.myapps.common.model.role.Role;
import cn.myapps.common.util.StringUtil;
import cn.myapps.designtime.common.service.DesignTimeServiceManager;
import cn.myapps.designtime.resource.service.ResourceDesignTimeService;
import cn.myapps.util.json.JsonUtil;
import cn.myapps.util.sequence.Sequence;
import cn.myapps.util.xml.XmlUtil;
import cn.myapps.webservice.fault.RoleServiceFault;
import cn.myapps.webservice.fault.UserServiceFault;
import cn.myapps.webservice.model.SimpleRole;
import cn.myapps.webservice.util.RoleUtil;

/**
 * 提供角色增删改和常用操作查功能接口
 * @author Administrator
 *
 */
public class RoleService {
	
	/**
	 * 传入一个SimpleRole对象创建一个角色
	 * @param role SimpleRole对象
	 * @return -1:失败 ,0:成功
	 * @throws RoleServiceFault
	 */
	public int createRole(SimpleRole role) throws RoleServiceFault {
		int result = -1;
		try {
			cn.myapps.designtime.role.service.RoleDesignTimeService roleService = DesignTimeServiceManager.roleDesignTimeService();

			if (role == null || StringUtil.isBlank(role.getName())) {
				throw new NullPointerException("对象为空或对象的名称为空.");
			}

			Application application = WebServiceUtil.validateApplication(role
					.getApplicationName());
			Role vo = new Role();
			RoleUtil.convertToVO(vo, role);

			Role temp = roleService.doViewByName(role.getName(),
					application.getId());
			if (temp == null) {
				roleService.save(vo);
			} else {
				if (role.getName().equalsIgnoreCase(temp.getName())) {
					throw new Exception("该角色名称" + role.getName() + "已存在.");
				}
			}
			result = 0;
		} catch (Exception e) {
			throw new RoleServiceFault(e.getMessage());
		}finally {
			try {
				PersistenceUtils.closeSessionAndConnection();
			} catch (Exception e) {
				throw new RoleServiceFault(e.getMessage());
			}
		}
		return result;
	}

	/**
	 * 传入一个SimpleRole对象更新一个角色
	 * @param role SimpleRole对象
	 * @return  -1:失败 ,0:成功
	 * @throws RoleServiceFault
	 */
	public int updateRole(SimpleRole role) throws RoleServiceFault {
		int result = -1;
		try {
			cn.myapps.designtime.role.service.RoleDesignTimeService roleService = DesignTimeServiceManager.roleDesignTimeService();
			if (role == null || StringUtil.isBlank(role.getName())) {
				throw new NullPointerException("对象为空或对象名称为空!");
			}
			Application application = WebServiceUtil.validateApplication(role
					.getApplicationName());
			Role vo = (Role) roleService.findById(role.getId());
			if (vo == null)
				throw new Exception("数据库不存在该" + role.getId() + "对象.");

			if (!vo.getName().equals(role.getName())) {
				Role temp = ( roleService).doViewByName(role
						.getName(), application.getId());
				if (temp != null) {
					throw new Exception("该角色名称" + role.getName() + "已存在.");
				}
			}

			RoleUtil.convertToVO(vo, role);
			roleService.update(vo);
			
			result = 0;
		} catch (Exception e) {
			throw new RoleServiceFault(e.getMessage());
		}finally {
			try {
				PersistenceUtils.closeSessionAndConnection();
			} catch (Exception e) {
				throw new RoleServiceFault(e.getMessage());
			}
		}
		return result;
	}

	/**
	 * 传入主键查找一个角色
	 * @param pk 角色主键
	 * @return SimpleRole对象
	 * @throws RoleServiceFault
	 */
	public SimpleRole getRole(String pk) throws RoleServiceFault {
		SimpleRole role = null;
		try {
			cn.myapps.designtime.role.service.RoleDesignTimeService roleService = DesignTimeServiceManager.roleDesignTimeService();
			if (pk == null) {
				throw new NullPointerException("主键为空.");
			}
			Role vo = (Role) roleService.findById(pk);
			if (vo != null) {
				role = new SimpleRole();
				RoleUtil.convertToSimple(role, vo);
			}
		} catch (Exception e) {
			throw new RoleServiceFault(e.getMessage());
		}finally {
			try {
				PersistenceUtils.closeSessionAndConnection();
			} catch (Exception e) {
				throw new RoleServiceFault(e.getMessage());
			}
		}
		return role;
	}

	/**
	 * 传入主键删除一个角色
	 * @param pk 主键
	 * @return  -1:失败 ,0:成功
	 * @throws RoleServiceFault
	 */
	public int deleteRole(String pk) throws RoleServiceFault {
		int result = -1;
		try {
			cn.myapps.designtime.role.service.RoleDesignTimeService roleService = DesignTimeServiceManager.roleDesignTimeService();
			if (pk == null) {
				throw new NullPointerException("主键为空.");
			}
			roleService.delete(pk);
			
			result = 0;
		} catch (Exception e) {
			throw new RoleServiceFault(e.getMessage());
		}finally {
			try {
				PersistenceUtils.closeSessionAndConnection();
			} catch (Exception e) {
				throw new RoleServiceFault(e.getMessage());
			}
		}
		return result;
	}
	
	/**
	 * 传入主键数组删除角色
	 * @param pks 主键数组
	 * @return  -1:失败 ,0:成功
	 * @throws RoleServiceFault
	 */
	public int deleteRole(String[] pks) throws RoleServiceFault {
		int result = -1;
		try {
			if(pks != null){
				for (int i = 0; i < pks.length; i++) {
					String pk = pks[i];
					deleteRole(pk);
				}
			}
			result = 0;
		} catch (Exception e) {
			throw new RoleServiceFault(e.getMessage());
		}
		return result;
	}

	/**
	 * 传入Json格式字符串创建一个角色
	 * @param jsonStr
	 * @return  -1:失败 ,0:成功
	 * @throws RoleServiceFault
	 */
	public int createRoleFromJson (String jsonStr) throws RoleServiceFault {
		int result = -1;
		try {
			if(StringUtil.isBlank(jsonStr))
				throw new RoleServiceFault("Parameter 'jsonStr' can not be null.");
			SimpleRole simpleRole = (SimpleRole) JsonUtil.toBean(jsonStr, SimpleRole.class);
			result = createRole(simpleRole);
		} catch (Exception e) {
			throw new RoleServiceFault(e.getMessage());
		}
		return result;
	}
	
	/**
	 * 传入XML格式字符串创建一个角色
	 * @param xmlStr
	 * @return  -1:失败 ,0:成功
	 * @throws RoleServiceFault
	 */
	public int createRoleFromXML (String xmlStr) throws RoleServiceFault {
		int result = -1;
		try {
			if(StringUtil.isBlank(xmlStr))
				throw new RoleServiceFault("Parameter 'xmlStr' can not be null.");
			SimpleRole simpleRole = (SimpleRole) XmlUtil.toOjbect(xmlStr);
			result = createRole(simpleRole);
		} catch (Exception e) {
			throw new RoleServiceFault(e.getMessage());
		}
		return result;
	}
	
	/**
	 * 传入主键返回角色对象格式化成JSON的字符串
	 * @param pk 主键
	 * @return JSON字符串
	 * @throws RoleServiceFault
	 */
	public String findRoleFormat2Json(String pk) throws RoleServiceFault {
		SimpleRole simpleRole = getRole(pk);
		return JsonUtil.toJson(simpleRole);
	}
	
	/**
	 * 传入主键返回角色对象格式化成XML的字符串
	 * @param pk 主键
	 * @return XML字符串
	 * @throws RoleServiceFault
	 */
	public String findRoleFormat2XML(String pk) throws RoleServiceFault {
		SimpleRole simpleRole = getRole(pk);
		return XmlUtil.toXml(simpleRole);
	}
	
	/**
	 * 传入用户Id查询角色的集合
	 * @param userId 用户id
	 * @return SimpleRole对象集合
	 * @throws RoleServiceFault
	 */
	public Collection<SimpleRole> getRolesByUserId(String userId) throws RoleServiceFault {
		Collection<SimpleRole> simpleRole = new ArrayList<SimpleRole>();
		try {
			cn.myapps.designtime.role.service.RoleDesignTimeService roleService = DesignTimeServiceManager.roleDesignTimeService();
			Collection<Role> roles = roleService.queryByUser(userId);
			for (Iterator<Role> iterator = roles.iterator(); iterator.hasNext();) {
				Role roleVO = (Role) iterator.next();
				SimpleRole role = new SimpleRole();
				RoleUtil.convertToSimple(role, roleVO);
				simpleRole.add(role);
			}
		} catch (Exception e) {
			throw new RoleServiceFault(e.getMessage());
		}finally {
			try {
				PersistenceUtils.closeSessionAndConnection();
			} catch (Exception e) {
				throw new RoleServiceFault(e.getMessage());
			}
		}
		return simpleRole;
	}
	
	/**
	 * 为指定的角色添加一个用户
	 * @param roleId 角色id
	 * @param userId 用户id
	 * @return  -1:失败 ,0:成功
	 * @throws RoleServiceFault
	 */
	public int addUser(String roleId,String userId) throws RoleServiceFault {
		int result = -1;
		try {
			Role role = WebServiceUtil.findRoleWithValidate(roleId);
			UserVO vo = WebServiceUtil.findUserWithValidate(userId);
			
			UserProcess userProcess = AuthTimeServiceManager.userRuntimeService();
			
			DomainVO domain = vo.getDomain();
			boolean flag = true;
			if (role != null) {
				Iterator<?> it = domain.getApplications().iterator();
				while (it.hasNext()) {
					Application temp = (Application) it.next();
					if (temp.getId().equals(role.getApplicationid())) {
						flag = false;
						break;
					}
				}
			}
			if (flag) {
				throw new Exception("该角色" + roleId + "对应的应用"
						+ role.getApplicationid() + "还没应用到该域"
						+ domain.getName());
			}
			
			Collection<UserDepartmentRoleSet> coll = vo.getUserDepartmentRoleSets();
			boolean exist = false;
			for (Iterator<UserDepartmentRoleSet> iterator = coll.iterator(); iterator.hasNext();) {
				UserDepartmentRoleSet userRoleSet = (UserDepartmentRoleSet) iterator.next();
				if(userRoleSet.getRoleId().equals(roleId)){
					exist = true;//角色下已存在该用户
					break;
				}
			}
			if(!exist){
				String [] array = new String[1];
				array[0] = userId;
				userProcess.addUserToRole(array, roleId);
			}
			result = 0;
		} catch (Exception e) {
			throw new RoleServiceFault(e.getMessage());
		}finally {
			try {
				PersistenceUtils.closeSessionAndConnection();
			} catch (Exception e) {
				throw new RoleServiceFault(e.getMessage());
			}
		}
		return result;
	}
	
	/**
	 * 为角色添加用户集合
	 * @param roleId 角色id
	 * @param userIds 用户id数组
	 * @return  -1:失败 ,0:成功
	 * @throws RoleServiceFault
	 */
	public int addUsers(String roleId, String[] userIds) throws RoleServiceFault {
		int result = -1;
		try {
			if(userIds == null)
				throw new RoleServiceFault("用户数组不能为null");
			for (int i = 0; i < userIds.length; i++) {
				addUser(roleId,userIds[i]);
			}
			
			result = 0;
		} catch (Exception e) {
			throw new RoleServiceFault(e.getMessage());
		}finally {
			try {
				PersistenceUtils.closeSessionAndConnection();
			} catch (Exception e) {
				throw new RoleServiceFault(e.getMessage());
			}
		}
		return result;
	}
	
	/**
	 * 为指定的角色移除一个用户
	 * @param id 角色id
	 * @param userId 用户id
	 * @return  -1:失败 ,0:成功
	 * @throws RoleServiceFault
	 */
	public int removeUser(String id,String userId) throws RoleServiceFault {
		int result = -1;
		try {
			new UserService().removeRole(userId, id);
			
			result = 0;
		} catch (UserServiceFault e) {
			throw new RoleServiceFault(e.getMessage());
		}finally {
			try {
				PersistenceUtils.closeSessionAndConnection();
			} catch (Exception e) {
				throw new RoleServiceFault(e.getMessage());
			}
		}
		return result;
	}
	
	/**
	 * 传入所有属性值创建一个角色
	 * @param id 角色id
	 * @param name 角色名
	 * @param engname 角色engname
	 * @param applicationName 软件名称
	 * @return  -1:失败 ,0:成功
	 * @throws RoleServiceFault
	 */
	public int createRole (String id, String name, String engname, String applicationName)
			throws RoleServiceFault {
		int result = -1;
		try {
			SimpleRole role = new SimpleRole();
			role.setId(id);
			role.setName(name);
			role.setEngname(engname);
			role.setApplicationName(applicationName);
			result = createRole(role);
		} catch (Exception e) {
			throw new RoleServiceFault(e.getMessage());
		}
		return result;
	}

}
