package com.bcxin.rbac.domain.services.impls;

import com.bcxin.Infrastructures.UnitWork;
import com.bcxin.Infrastructures.exceptions.BadTenantException;
import com.bcxin.Infrastructures.exceptions.NotFoundTenantException;
import com.bcxin.Infrastructures.utils.ExceptionUtil;
import com.bcxin.rbac.domain.entities.ResourceEntity;
import com.bcxin.rbac.domain.entities.RoleEntity;
import com.bcxin.rbac.domain.entities.SubjectEntity;
import com.bcxin.rbac.domain.repositories.ResourceRepository;
import com.bcxin.rbac.domain.repositories.RoleRepository;
import com.bcxin.rbac.domain.repositories.SubjectRepository;
import com.bcxin.rbac.domain.services.RoleService;
import com.bcxin.rbac.domain.services.commands.roles.CreateRoleCommand;
import com.bcxin.rbac.domain.services.commands.roles.DeleteRoleCommand;
import com.bcxin.rbac.domain.services.commands.roles.ResourceItem;
import com.bcxin.rbac.domain.services.commands.roles.UpdateRoleCommand;
import org.springframework.stereotype.Service;

import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

@Service
public class RoleServiceImpl implements RoleService {
    private final UnitWork unitWork;
    private final RoleRepository roleRepository;
    private final SubjectRepository subjectRepository;
    private final ResourceRepository resourceRepository;

    public RoleServiceImpl(UnitWork unitWork, RoleRepository roleRepository,
                           SubjectRepository subjectRepository,
                           ResourceRepository resourceRepository) {
        this.unitWork = unitWork;
        this.roleRepository = roleRepository;
        this.subjectRepository = subjectRepository;
        this.resourceRepository = resourceRepository;
    }

    @Override
    public void dispatch(CreateRoleCommand command) {
        command.validate();
        Optional<SubjectEntity> subjectOptional = this.subjectRepository.findById(command.getSubjectReferencedId());
        AtomicReference<SubjectEntity> subjectAto = new AtomicReference<>();
        if (subjectOptional.isPresent()) {
            subjectAto.set(subjectOptional.get());
        }

        if (subjectAto.get() == null) {
            try {
                this.unitWork.executeTran(() -> {
                    subjectAto.set(SubjectEntity.create(String.format("组:%s", new SimpleDateFormat("yyyMMdd").format(new Date())),
                            command.getSubjectReferencedId()));
                    this.subjectRepository.save(subjectAto.get());
                });
            }
            catch (Exception ex)
            {
                if(ExceptionUtil.getStackMessage(ex).contains("PRIMARY"))
                {
                    subjectOptional = this.subjectRepository.findById(command.getSubjectReferencedId());
                    if (subjectOptional.isPresent()) {
                        subjectAto.set(subjectOptional.get());
                    }
                }
            }
        }

        if(subjectAto.get()==null) {
            throw new BadTenantException("角色创建失败");
        }

        RoleEntity role = RoleEntity.create(subjectAto.get(), command.getName());

        this.assignRoleResources(role,command.getResources());

        this.unitWork.executeTran(() -> {
            this.roleRepository.save(role);
        });
    }

    @Override
    public void dispatch(UpdateRoleCommand command) {
        Optional<RoleEntity> roleOptional = this.roleRepository.findById(command.getId());
        if (roleOptional.isPresent()) {
            throw new NotFoundTenantException("找不到该角色信息");
        }

        RoleEntity role = roleOptional.get();
        this.assignRoleResources(role, command.getResources());
        role.change(command.getName());

        this.unitWork.executeTran(() -> {
            this.roleRepository.save(role);
        });
    }

    @Override
    public void dispatch(DeleteRoleCommand command) {
        Optional<RoleEntity> roleOptional = this.roleRepository.findById(command.getId());
        if (!roleOptional.isPresent()) {
            throw new NotFoundTenantException("找不到角色信息");
        }

        this.unitWork.executeTran(() -> {
            this.roleRepository.delete(roleOptional.get());
        });
    }

    private void assignRoleResources(RoleEntity role, Collection<ResourceItem> resourceItems) {
        if (resourceItems != null) {
            Collection<String> resourceIds = resourceItems.stream().flatMap(ii -> ii.getResourceIds().stream()).collect(Collectors.toList());
            Collection<ResourceEntity> resources = this.resourceRepository.getByIds(resourceIds);

            Collection<String> notExistsResourceIds = resourceIds.stream().filter(ii -> resources.stream().anyMatch(ix -> ix.getId().equals(ii)))
                    .collect(Collectors.toList());
            if (notExistsResourceIds.size() > 0) {
                throw new NotFoundTenantException(String.format("找不到目标资源(%s)", notExistsResourceIds.stream().collect(Collectors.joining(","))));
            }
            role.assignResources(resources);
        }
    }
}
