package com.bcxin.rbac.domain.services.impls;

import com.bcxin.Infrastructures.UnitWork;
import com.bcxin.Infrastructures.exceptions.ConflictTenantException;
import com.bcxin.Infrastructures.exceptions.NotFoundTenantException;
import com.bcxin.Infrastructures.exceptions.TenantExceptionAbstract;
import com.bcxin.Infrastructures.utils.StringUtil;
import com.bcxin.rbac.domain.entities.CategoryEntity;
import com.bcxin.rbac.domain.entities.ResourceEntity;
import com.bcxin.rbac.domain.repositories.CategoryRepository;
import com.bcxin.rbac.domain.repositories.ResourceRepository;
import com.bcxin.rbac.domain.services.ResourceService;
import com.bcxin.rbac.domain.services.commands.resources.CreateResourceCommand;
import com.bcxin.rbac.domain.services.commands.resources.DeleteResourceCommand;
import com.bcxin.rbac.domain.services.commands.resources.UpdateResourceCommand;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import java.util.Optional;

@Service
public class ResourceServiceImpl implements ResourceService {
    private final CategoryRepository categoryRepository;
    private final ResourceRepository resourceRepository;
    private final UnitWork unitWork;

    public ResourceServiceImpl(CategoryRepository categoryRepository,
                               ResourceRepository resourceRepository,
                               UnitWork unitWork) {
        this.categoryRepository = categoryRepository;
        this.resourceRepository = resourceRepository;
        this.unitWork = unitWork;
    }

    @Override
    public void dispatch(CreateResourceCommand command) {
        command.validate();
        Optional<CategoryEntity> categoryOptional = this.categoryRepository.findById(command.getCategoryId());
        if (!categoryOptional.isPresent()) {
            throw new NotFoundTenantException("找不到分类信息");
        }

        this.unitWork.executeTran(() -> {
            ResourceEntity resource = ResourceEntity.create(categoryOptional.get(), command.getCode(), command.getName(), command.getData(), command.getDataType());
            this.assignParent(resource, command.getCategoryId(), command.getParentId());

            this.resourceRepository.save(resource);
        });
    }

    @Override
    public void dispatch(UpdateResourceCommand command) {
        command.validate();
        Optional<ResourceEntity> resourceOptional = this.resourceRepository.getByCategoryAndId(command.getCategoryId(), command.getId());
        if (!resourceOptional.isPresent()) {
            throw new NotFoundTenantException("找不到资源信息");
        }

        this.unitWork.executeTran(() -> {
            resourceOptional.get().change(command.getCode(), command.getName(), command.getData(), command.getDataType());
            this.assignParent(resourceOptional.get(), command.getCategoryId(), command.getParentId());

            this.resourceRepository.save(resourceOptional.get());
        });
    }

    @Override
    public void dispatch(DeleteResourceCommand command) {
        command.validate();
        Optional<ResourceEntity> resourceOptional = this.resourceRepository.getByCategoryAndId(command.getCategoryId(), command.getId());
        if (!resourceOptional.isPresent()) {
            throw new NotFoundTenantException("找不到资源信息");
        }

        try {
            this.unitWork.executeTran(() -> {
                this.resourceRepository.delete(resourceOptional.get());
            });
        } catch (Exception ex) {
            ex.printStackTrace();
            throw ex;
        }
    }

    private void assignParent(ResourceEntity resource, String categoryId, String parentId) {
        if (StringUtils.hasLength(parentId)) {
            Optional<ResourceEntity> selectedResourceOptional = this.resourceRepository.getByCategoryAndId(categoryId, parentId);
            if (!selectedResourceOptional.isPresent()) {
                throw new NotFoundTenantException("找不到父资源信息");
            }

            resource.assignParent(selectedResourceOptional.get());
        }
    }
}
