package com.bcxin.rbac.domain.services.impls;

import com.bcxin.Infrastructures.UnitWork;
import com.bcxin.Infrastructures.exceptions.NotFoundTenantException;
import com.bcxin.rbac.domain.RbacConstant;
import com.bcxin.rbac.domain.entities.CategoryEntity;
import com.bcxin.rbac.domain.repositories.CategoryRepository;
import com.bcxin.rbac.domain.services.CategoryService;
import com.bcxin.rbac.domain.services.commands.categories.BatchCategoryResourceCommand;
import com.bcxin.rbac.domain.services.commands.categories.CreateCategoryCommand;
import com.bcxin.rbac.domain.services.commands.categories.DeleteCategoryCommand;
import com.bcxin.rbac.domain.services.commands.categories.UpdateCategoryCommand;
import org.springframework.stereotype.Service;

import java.util.Collection;
import java.util.Optional;
import java.util.stream.Collectors;

@Service
public class CategoryServiceImpl implements CategoryService {
    private final CategoryRepository categoryRepository;
    private final UnitWork unitWork;

    public CategoryServiceImpl(CategoryRepository categoryRepository, UnitWork unitWork) {
        this.categoryRepository = categoryRepository;
        this.unitWork = unitWork;
    }

    @Override
    public void dispatch(CreateCategoryCommand command) {
        this.unitWork.executeTran(() -> {
            CategoryEntity category = CategoryEntity.create(command.getCode(), command.getName(), command.getNote());

            this.categoryRepository.save(category);
        });
    }

    @Override
    public void dispatch(UpdateCategoryCommand command) {
        Optional<CategoryEntity> categoryOptional = this.categoryRepository.findById(command.getId());
        if (!categoryOptional.isPresent()) {
            throw new NotFoundTenantException("找不到分类信息");
        }

        this.unitWork.executeTran(() -> {
            categoryOptional.get().change(command.getCode(), command.getName(), command.getNote());

            this.categoryRepository.save(categoryOptional.get());
        });
    }

    @Override
    public void dispatch(DeleteCategoryCommand command) {
        Optional<CategoryEntity> categoryOptional = this.categoryRepository.findById(command.getId());
        if (!categoryOptional.isPresent()) {
            throw new NotFoundTenantException("找不到分类信息");
        }

        this.unitWork.executeTran(() -> {
            this.categoryRepository.delete(categoryOptional.get());
        });
    }

    @Override
    public void dispatch(BatchCategoryResourceCommand command) {
        command.validate();
        Collection<String> categoryIds = command.getCategories().stream().map(ix -> ix.getId())
                .collect(Collectors.toList());

        Collection<CategoryEntity> categories = this.categoryRepository.getByIds(categoryIds);
        Collection<BatchCategoryResourceCommand.CategoryCommand> notExistsCategories = command.getCategories()
                .stream().filter(ix -> !categories.stream().anyMatch(ii -> ii.getId().equals(ix.getId())))
                .collect(Collectors.toList());

        for (BatchCategoryResourceCommand.CategoryCommand item : notExistsCategories) {
            CategoryEntity newCategory =
                    CategoryEntity.create(item.getId(), String.format("RC_%s", item.getId()), item.getName(), "");
            categories.add(newCategory);
        }

        command.getCategories().forEach(ci -> {
            Optional<CategoryEntity> selectedCategoryOptional = categories.stream()
                    .filter(ix -> ix.getId().equals(ci.getId())).findFirst();
            if (!selectedCategoryOptional.isPresent()) {
                throw new NotFoundTenantException("系统无效, 竟然找不到category");
            }
            ci.getResources().forEach(cir -> {
                selectedCategoryOptional.get().assignResource(cir.getId(), String.format("RCR_%s", cir.getId()), cir.getName(), cir.getName(), RbacConstant.RESOURCE_DATATYPE_ROLE);
            });
        });

        this.unitWork.executeTran(() -> {
            categories.forEach(ci -> {
                this.categoryRepository.save(ci);
            });
        });
    }
}
