<?php

declare(strict_types=1);
/**
 *  +----------------------------------------------------------------------
 *  | 陀螺匠 [ 赋能开发者，助力企业发展 ]
 *  +----------------------------------------------------------------------
 *  | Copyright (c) 2016~2024 https://www.tuoluojiang.com All rights reserved.
 *  +----------------------------------------------------------------------
 *  | Licensed 陀螺匠并不是自由软件，未经许可不能去掉陀螺匠相关版权
 *  +----------------------------------------------------------------------
 *  | Author: 陀螺匠 Team <admin@tuoluojiang.com>
 *  +----------------------------------------------------------------------
 */

namespace App\Http\Controller\UniApi\Notice;

use App\Http\Controller\UniApi\AuthController;
use App\Http\Service\Notice\MessageCateService;
use App\Http\Service\Notice\NoticeRecordService;
use Illuminate\Contracts\Container\BindingResolutionException;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;
use Spatie\RouteAttributes\Attributes\Delete;
use Spatie\RouteAttributes\Attributes\Get;
use Spatie\RouteAttributes\Attributes\Middleware;
use Spatie\RouteAttributes\Attributes\Prefix;
use Spatie\RouteAttributes\Attributes\Put;

/**
 * 系统通知控制器.
 */
#[Prefix('uni/message')]
#[Middleware(['auth.admin', 'ent.auth', 'ent.log'])]
class NoticeController extends AuthController
{
    public function __construct(NoticeRecordService $services)
    {
        parent::__construct();
        $this->service = $services;
    }

    /**
     * 通知分类.
     * @return mixed
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    #[Get('cate', '通知分类')]
    public function cate(MessageCateService $service)
    {
        return $this->success($service->getUniNoticeCate(auth('admin')->id()));
    }

    /**
     * 消息列表.
     * @return mixed
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    #[Get('list', '消息列表')]
    public function index()
    {
        [$cateId,$title,$isRead] = $this->request->getMore([
            ['cate_id', 0],
            ['title', ''],
            ['is_read', ''],
            ['cate_id', 0],
        ], true);
        return $this->success($this->service->getMessageList($this->uuid, $this->entId, (int) $cateId, $title, $isRead, reverse: true));
    }

    /**
     * 修改处理状态
     * @return mixed
     */
    #[Put('update/{id}/{isHandle}', '修改处理状态')]
    public function updateHandle($id, $isHandle)
    {
        if (! $id) {
            return $this->fail('common.empty.attrs');
        }
        $messageInfo = $this->service->get($id);
        if (! $messageInfo) {
            return $this->fail('消息不存在');
        }
        $messageInfo->is_handle = $isHandle;
        if ($messageInfo->save()) {
            return $this->success('common.update.succ');
        }
        return $this->fail('common.update.fail');
    }

    /**
     * 按照分类修改处理状态
     * @throws BindingResolutionException
     */
    #[Put('handle/{cate_id}/{isHandle}', '按照分类修改处理状态')]
    public function updateCateHandle($cate_id, $isHandle): mixed
    {
        if (! $cate_id) {
            return $this->fail('common.empty.attrs');
        }
        $messageInfo = $this->service->select(['cate_id' => $cate_id, 'to_uid' => auth('admin')->id()]);
        if (! $messageInfo) {
            return $this->fail('消息不存在');
        }
        $this->service->update(['cate_id' => $cate_id, 'to_uid' => auth('admin')->id()], ['is_read' => $isHandle]);
        return $this->success('common.update.succ');
    }

    /**
     * 批量修改.
     * @throws BindingResolutionException
     */
    #[Put('batch/{isRead}', '批量修改')]
    public function batchUpdate($isRead): mixed
    {
        [$ids, $cateId] = $this->request->postMore([
            ['ids', []],
            ['cate_id', 0],
        ], true);

        if (! $ids && ! $cateId) {
            return $this->fail('缺少参数');
        }
        if ($this->service->batchUpdate( auth('admin')->id(), $this->entId, (int) $isRead, (int) $cateId, $ids)) {
            return $this->success('修改成功');
        }
        return $this->fail('修改失败');
    }

    /**
     * 批量删除.
     */
    #[Delete('batch/delete', '批量删除')]
    public function batchDelete(): mixed
    {
        [$ids, $cateId] = $this->request->postMore([
            ['ids', []],
            ['cate_id', 0],
        ], true);

        if (! $ids && ! $cateId) {
            return $this->fail('缺少参数');
        }
        if ($this->service->batchDelete( auth('admin')->id(), $this->entId, (int) $cateId, $ids)) {
            return $this->success('删除成功');
        }

        return $this->fail('删除失败');
    }

    /**
     * 获取待处理消息数量.
     * @return mixed
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    #[Get('count', '获取待处理消息数量')]
    public function pendingCount()
    {
        return $this->success($this->service->getCount($this->uuid, $this->entId));
    }
}
