<?php

declare(strict_types=1);
/**
 *  +----------------------------------------------------------------------
 *  | 陀螺匠 [ 赋能开发者，助力企业发展 ]
 *  +----------------------------------------------------------------------
 *  | Copyright (c) 2016~2024 https://www.tuoluojiang.com All rights reserved.
 *  +----------------------------------------------------------------------
 *  | Licensed 陀螺匠并不是自由软件，未经许可不能去掉陀螺匠相关版权
 *  +----------------------------------------------------------------------
 *  | Author: 陀螺匠 Team <admin@tuoluojiang.com>
 *  +----------------------------------------------------------------------
 */

namespace App\Http\Controller\UniApi\Client;

use App\Constants\CustomEnum\ContractEnum;
use App\Constants\CustomEnum\CustomEnum;
use App\Constants\UserAgentEnum;
use App\Http\Contract\Client\ClientContractSubscribeInterface;
use App\Http\Controller\UniApi\AuthController;
use App\Http\Service\Client\ContractService;
use App\Http\Service\Config\FormService;
use crmeb\interfaces\ResourceControllerInterface;
use crmeb\traits\ResourceControllerTrait;
use Illuminate\Contracts\Container\BindingResolutionException;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;
use Spatie\RouteAttributes\Attributes\Get;
use Spatie\RouteAttributes\Attributes\Middleware;
use Spatie\RouteAttributes\Attributes\Post;
use Spatie\RouteAttributes\Attributes\Prefix;
use Spatie\RouteAttributes\Attributes\Resource;

/**
 * 合同管理
 * Class ContractController.
 */
#[Prefix('uni/client/contract')]
#[Resource('/', false, except: ['show'], names: [
    'index'   => '获取客户合同列表',
    'create'  => '保存客户合同表单',
    'store'   => '保存客户合同',
    'edit'    => '修改客户合同表单',
    'update'  => '修改客户合同',
    'destroy' => '删除客户合同',
], parameters: ['' => 'id'])]
#[Middleware(['auth.admin', 'ent.auth', 'ent.log'])]
class ContractController extends AuthController implements ResourceControllerInterface
{
    use ResourceControllerTrait;

    public function __construct(ContractService $services)
    {
        parent::__construct();
        $services->setPlatform(UserAgentEnum::UNI_AGENT);
        $this->service = $services;
    }

    /**
     * 列表数据.
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    public function index(): mixed
    {
        $types = (int)$this->request->get('types', 6);
        $where = $this->request->getMore($this->service->getSearchField());
        if ($types != ContractEnum::CONTRACT_CHARGE) {
            $types = ContractEnum::CONTRACT_VIEWER;
        }
        return $this->success($this->service->getUniListByType($types, $where, auth('admin')->id()));
    }

    /**
     * 保存表单.
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    public function create(): mixed
    {
        $service = app()->get(FormService::class);
        return $this->success($service->getFormDataWithType(CustomEnum::CONTRACT, platform: UserAgentEnum::UNI_AGENT));
    }

    /**
     * 保存.
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    public function store(): mixed
    {
        $service = app()->get(FormService::class);
        $data    = $this->request->postMore($service->getRequestFields(CustomEnum::CONTRACT));
        $res     = $this->service->saveContract($data, auth('admin')->id());
        return $this->success('common.insert.succ', ['id' => $res->id]);
    }

    /**
     * 修改.
     * @param mixed $id
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    public function update($id): mixed
    {
        if (!$id) {
            return $this->fail($this->message['update']['emtpy']);
        }
        $service = app()->get(FormService::class);
        $data    = $this->request->postMore($service->getRequestFields(CustomEnum::CONTRACT));
        $this->service->updateContract($data, (int)$id);
        return $this->success(__('common.update.succ'));
    }

    /**
     * 详情.
     * @param mixed $id
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    #[Get('info/{id}', '合同详情')]
    public function info($id): mixed
    {
        if (!$id) {
            return $this->fail($this->message['update']['emtpy']);
        }

        return $this->success($this->service->getUniInfo((int)$id, auth('admin')->id()));
    }

    /**
     * 修改表单.
     * @param mixed $id
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    public function edit($id): mixed
    {
        if (!$id) {
            return $this->fail($this->message['update']['emtpy']);
        }

        return $this->success($this->service->getEditInfo((int)$id, auth('admin')->id()));
    }

    /**
     * 删除.
     * @param mixed $id
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    public function destroy($id): mixed
    {
        if (!$id) {
            return $this->fail('common.empty.attrs');
        }
        $this->service->deleteContract((int)$id, $this->uuid);
        return $this->success('common.delete.succ');
    }

    /**
     * 下拉列表.
     * @param mixed $eid
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    #[Get('select/{eid}', '获取合同选择列表')]
    public function select($eid): mixed
    {
        if (!$eid) {
            return $this->fail('common.empty.attrs');
        }
        return $this->success($this->service->getSelectList((int)$eid, auth('admin')->id()));
    }

    /**
     * 修改关注状态
     * @param mixed $id
     * @param mixed $status
     * @throws BindingResolutionException
     */
    #[Post('subscribe/{id}/{status}', '修改关注状态')]
    public function subscribe(ClientContractSubscribeInterface $clientSubscribeService, $id, $status): mixed
    {
        if (!$id) {
            return $this->fail('common.empty.attrs');
        }
        $clientSubscribeService->subscribe(auth('admin')->id(), (int)$id, (int)$status);
        return $this->success(__('common.operation.succ'));
    }

    /**
     * 转移.
     * @param mixed $id
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    #[Post('shift/{id}', '客户合同转移')]
    public function shift($id): mixed
    {
        [$toUid, $invoice] = $this->request->postMore([
            ['to_uid', 0],
            ['invoice', 0],
        ], true);
        if (!$id) {
            return $this->fail('common.empty.attrs');
        }
        $this->service->shift([(int)$id], (int)$toUid, (int)$invoice);
        return $this->success(__('common.operation.succ'));
    }

    protected function getRequestClassName(): string
    {
        return '';
    }

    protected function getSearchField(): array
    {
        return [];
    }

    protected function getRequestFields(): array
    {
        return [];
    }
}
