<?php

declare(strict_types=1);
/**
 *  +----------------------------------------------------------------------
 *  | 陀螺匠 [ 赋能开发者，助力企业发展 ]
 *  +----------------------------------------------------------------------
 *  | Copyright (c) 2016~2024 https://www.tuoluojiang.com All rights reserved.
 *  +----------------------------------------------------------------------
 *  | Licensed 陀螺匠并不是自由软件，未经许可不能去掉陀螺匠相关版权
 *  +----------------------------------------------------------------------
 *  | Author: 陀螺匠 Team <admin@tuoluojiang.com>
 *  +----------------------------------------------------------------------
 */

namespace App\Http\Service\User;

use App\Http\Dao\User\UserResumeDao;
use App\Http\Model\BaseModel;
use App\Http\Service\Admin\AdminInfoService;
use App\Http\Service\BaseEntService;
use GuzzleHttp\Client;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Database\Concerns\BuildsQueries;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Log;

/**
 * 用户简历.
 */
class UserResumeService extends BaseEntService
{

    /**
     * 个人敏感字段.
     * @var array|string[]
     */
    public array $hiddenField = ['id', 'bank_num', 'bank_name', 'spare_name', 'spare_tel', 'social_num', 'fund_num', 'card_front', 'card_both', 'education_image', 'acad_image'];

    public function __construct(UserResumeDao $dao)
    {
        $this->dao = $dao;
    }

    /**
     * 获取用户个人简历.
     * @return null|BaseModel|BuildsQueries|mixed|Model|object
     * @throws BindingResolutionException
     * @throws \ReflectionException
     */
    public function getInfo(string $uuid)
    {
        if (!$this->dao->exists(['uid' => $uuid])) {
            $this->dao->create(['uid' => $uuid]);
        }
        return $this->dao->get(['uid' => $uuid], ['*'], [
            'works', 'educations',
        ])->toArray();
    }

    /**
     * 保存简历信息.
     * @throws BindingResolutionException
     */
    public function saveInfo($uuid, $data)
    {
        $resume = $this->dao->get(['uid' => $uuid]);
        if (!$resume) {
            throw $this->exception('保存失败,未找到简历信息');
        }
        foreach ($data as $key => $value) {
            isset($resume->$key) && $resume->$key = $value;
        }
        return $resume->save();
    }

    /**
     * 人脸识别.
     * @param string $uid
     * @param string $image
     * @return mixed|string
     * @throws BindingResolutionException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function checkFace(string $uid, string $image)
    {
        $cardImg = $this->dao->value(['uid' => $uid], 'card_front');
        if (!$cardImg){
            $cardImg = app()->get(AdminInfoService::class)->value(['uid' => $uid], 'card_front');
        }
        if (!$cardImg){
            throw $this->exception('未找到身份证图片');
        }
        $client  = new Client();
        try {
            $response = $client->post('https://identify.test.bcxin.com.cn/identify/api/baidu/match-face', [
                'form_params' => [
                    'oneImg' => $cardImg,
                    'twoImg' => $image,
                    'bizFaceMode' => 'UNIAPP',
                ]
            ]);
            $ip_info  = json_decode($response->getBody()->getContents(), true);
            Log::info('人脸识别数据: ' . json_encode($ip_info), compact('cardImg','image'));
            if (!isset($ip_info['data']['response'])){
                throw $this->exception('获取人脸识别数据失败');
            }
            $resData = $ip_info['data']['response'];
            if (is_string($resData)){
                $resData = json_decode($resData, true);
            }
            if ($resData['error_msg'] == 'SUCCESS' && $resData['error_code']  == 0 && $resData['result']['score'] > 80){
                return true;
            }else{
                return false;
            }
        } catch (\Exception $e) {
            throw $this->exception('获取人脸识别数据失败');
        }
    }
}
