<?php

declare(strict_types=1);
/**
 *  +----------------------------------------------------------------------
 *  | 陀螺匠 [ 赋能开发者，助力企业发展 ]
 *  +----------------------------------------------------------------------
 *  | Copyright (c) 2016~2024 https://www.tuoluojiang.com All rights reserved.
 *  +----------------------------------------------------------------------
 *  | Licensed 陀螺匠并不是自由软件，未经许可不能去掉陀螺匠相关版权
 *  +----------------------------------------------------------------------
 *  | Author: 陀螺匠 Team <admin@tuoluojiang.com>
 *  +----------------------------------------------------------------------
 */

namespace App\Http\Service\Position;

use App\Http\Dao\Position\PositionLevelDao;
use App\Http\Model\BaseModel;
use App\Http\Service\BaseEntService;
use crmeb\traits\service\ResourceServiceTrait;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

/**
 * 职级体系.
 */
class PositionLevelService extends BaseEntService
{
    use ResourceServiceTrait;

    public function __construct(PositionLevelDao $dao)
    {
        $this->dao = $dao;
    }

    /**
     * 职位体系图列表数据.
     * @param array|string[] $field
     * @param null $sort
     * @throws BindingResolutionException
     * @throws \ReflectionException
     */
    public function getList(array $where, array $field = ['*'], $sort = null, array $with = []): array
    {
        $tag  = 'Rank';
        $keys = 'RankLevel_' . $this->entId(false);
        if (Cache::tags([$tag])->has($keys)) {
            $list = json_decode(Cache::tags([$tag])->get($keys), true);
        } else {
            $cates = toArray(app()->get(PositionCateService::class)->select([], ['id', 'name']));
            $list  = $this->dao->getList($where, $field, sort: 'id');
            foreach ($list as &$item) {
                $info = [];
                foreach ($cates as $val) {
                    $info[] = [
                        'name' => $val['name'],
                        'id'   => $val['id'],
                        'info' => app()->get(PositionRelationService::class)->get(['level_id' => $item['id'], 'cate_id' => $val['id']], ['id', 'level_id', 'rank_id'], [
                            'rank' => function ($query) {
                                $query->select(['id', 'name', 'alias']);
                            },
                            'job' => function ($query) use ($val) {
                                $query->where('cate_id', $val['id'])->select(['rank_id', 'cate_id', 'name']);
                            },
                        ]),
                    ];
                }
                $item['info'] = $info;
            }
            Cache::tags([$tag])->put($keys, json_encode($list));
        }
        return $list;
    }

    /**
     * 保存数据.
     * @return BaseModel|Model
     * @throws BindingResolutionException
     */
    public function resourceSave(array $data)
    {
        $data['min_level'] = (int) $data['min_level'];
        $data['max_level'] = (int) $data['max_level'];
        if ($data['max_level'] < $data['min_level']) {
            throw $this->exception('请输入正确的职等范围！');
        }
        return $this->transaction(function () use ($data) {
            Cache::tags(['Rank'])->flush();
            return $this->dao->create($data);
        });
    }

    /**
     * 修改数据.
     * @return int
     * @throws BindingResolutionException
     * @throws \ReflectionException
     */
    public function resourceUpdate($id, array $data)
    {
        $data['min_level'] = (int) $data['min_level'];
        $data['max_level'] = (int) $data['max_level'];
        if ($data['max_level'] < $data['min_level']) {
            throw $this->exception('请输入正确的职等范围！');
        }
        $res = $this->transaction(function () use ($id, $data) {
            return $this->dao->update($id, $data);
        });
        $res && Cache::tags(['Rank'])->flush();
        return true;
    }

    /**
     * 删除数据.
     * @return mixed
     * @throws BindingResolutionException
     * @throws \ReflectionException
     */
    public function resourceDelete($id, ?string $key = null)
    {
        $res = $this->transaction(function () use ($id) {
            $res = $this->dao->delete($id);
            app()->get(PositionRelationService::class)->delete($id, 'level_id');
            return $res;
        });
        return $res && Cache::tags(['Rank'])->flush();
    }

    /**
     * 批量设置职等.
     * @return mixed
     * @throws BindingResolutionException
     * @throws \ReflectionException
     */
    public function batchSection(int $batch, $entid)
    {
        if (! $batch || $batch < 0) {
            throw $this->exception('请填写正确的区间！');
        }
        $list = $this->dao->select([], ['id', 'min_level', 'max_level']);
        if ($list) {
            $list = $list->toArray();
            return $this->transaction(function () use ($list, $batch) {
                foreach ($list as $key => $value) {
                    $this->dao->update($value['id'], [
                        'min_level' => 1 + ($key * $batch),
                        'max_level' => ($key + 1) * $batch,
                    ]);
                }
                Cache::tags(['Rank'])->flush();
                return true;
            });
        } else {
            throw $this->exception('暂无职等，无法设置！');
        }
    }
}
