<?php

namespace App\Http\Service\Common;

use App\Constants\UserEnum;
use App\Http\Model\Admin\Admin;
use App\Http\Model\Auth\Role;
use App\Http\Model\Company\Company;
use App\Http\Model\System\Menus;
use App\Http\Model\User\User;
use App\Http\Model\User\UserResume;
use App\Http\Service\Admin\AdminInfoService;
use App\Http\Service\Admin\AdminService;
use App\Http\Service\BaseService;
use App\Http\Service\Company\CompanyService;
use App\Http\Service\Frame\FrameAssistService;
use App\Http\Service\Frame\FrameService;
use App\Http\Service\User\UserService;
use App\Task\frame\FrameCensusTask;
use Hhxsv5\LaravelS\Swoole\Task\Task;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Support\Facades\Log;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;
use Webpatser\Uuid\Uuid;

class CommonService extends BaseService
{
    private string $key = '57e13b9b5c841d1a';

    public function createCompany($data)
    {
        $data['verify'] = 1;
        $data['status'] = 1;
        if (!User::where('uid', $data['uid'])->exists()) {
            throw $this->exception('用户不存在');
        }
        $data['uniqued'] = $data['uniqued'] ?: str_replace('-', '', (string)Uuid::generate(4));
        if (Company::where('uniqued', $data['uniqued'])->exists()) {
            throw $this->exception('企业已存在');
        }
        return Company::create($data)->toArray();
    }

    /**
     * 创建用户.
     * @param array $data
     * @param string $uuid
     * @return mixed
     * @throws \Exception
     */
    public function saveUser(array $data = [], string $uuid = ''): string
    {
        if (!$data) {
            throw $this->exception('缺少必要参数');
        }
        if (!$data['phone']) {
            throw $this->exception('请检查手机号是否正确');
        }
        if (!$data['card_id'] || !$this->validateIDCard($data['card_id'])) {
            throw $this->exception('请检查身份证号是否正确');
        }
        $data['real_name'] = $data['real_name'] ?: substr($data['phone'], 0, 3) . '****' . substr($data['phone'], 7, 4);
        if ($uuid && $user = User::where('uid', $uuid)->first()) {
            $uuid = $this->transaction(function () use ($user, $data, $uuid) {
                foreach ($data as $key => $value) {
                    if (!in_array($key, ['photo'])){
                        $user->$key = $value;
                    }
                }
                $user->save();
                $data['name'] = $data['real_name'];
                unset($data['real_name']);
                UserResume::where('uid', $uuid)->update($data);
                return $uuid;
            });
        } else {
//            if (User::where('phone', $data['phone'])->exists()) {
//                throw $this->exception('存在相同记录，请勿重复操作');
//            }
            if (User::where('card_id', $data['card_id'])->exists()) {
                throw $this->exception('存在相同身份证号，请勿重复添加');
            }
            $data['avatar']   = $data['photo'] ?: ('https://shmily-album.oss-cn-shenzhen.aliyuncs.com/admin_face/face' . rand(1, 10) . '.png');
            $data['password'] = password_hash('888888', PASSWORD_BCRYPT);
            $uuid             = $this->transaction(function () use ($data, $uuid) {
                $data['uid'] = $uuid ?: str_replace('-', '', (string)Uuid::generate(4));
                UserResume::create($data);
                unset($data['photo']);
                User::create($data);
                return $data['uid'];
            });
        }
        return $uuid;
    }

    /**
     * 验证身份证号
     * @param $idCard
     * @return bool|int
     */
    private function validateIDCard($idCard): bool|int
    {
        $pattern = "/^[1-9]\d{5}(19|20)\d{2}(0[1-9]|1[0-2])(0[1-9]|[12]\d|3[01])\d{3}[0-9Xx]$/";
        return preg_match($pattern, $idCard);
    }

    /**
     * 验证手机号
     * @param $phone
     * @return bool|int
     */
    private function validatePhoneNumber($phone): bool|int
    {
        $pattern = "/^1[3-9]\d{9}$/";
        return preg_match($pattern, $phone);
    }

    /**
     * 用户加入企业.
     * @param string $uid
     * @param string $entId
     * @param string $uniqued
     * @return mixed|true
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    public function userJoinCompany(string $uid, string $entId, string $uniqued = '', array $frames = [])
    {
        $user = User::where('uid', $uid)->first();
        if (!$user) {
            throw $this->exception('用户不存在');
        }
        $company = Company::where('uniqued', $entId)->select(['*'])->first();
        if (!$company) {
            throw $this->exception('未找到该企业的信息');
        }
        $frameInfo = [];
        foreach ($frames as $frame) {
            $frameId = app()->get(FrameService::class)->setEntValue($company->id)->value(['uniqued' => $frame['frame_id']], 'id');
            if (!$frameId) {
                continue;
            }
            $frameInfo[] = [
                'id'        => $frameId,
                'uniqued'   => $frame['uniqued'] ?: str_replace('-', '', (string)Uuid::generate(4)),
                'entid'     => $company->id,
                'is_master' => $frame['master'] ?? 0,
                'is_admin'  => $frame['admin'] ?? 0,
            ];
        }
        $admin = Admin::where('uid', $uid)->where('entid', $company->id)->first();
        if ($admin) {
            $user->entid = $company->id;
            $user->save();
            if ($uniqued) {
                $admin->uniqued = $uniqued;
                $admin->saveQuietly();
            }
        } else {
            $admin = $this->transaction(function () use ($user, $entId, $uniqued, $company) {
                $user->entid = $company->id;
                $user->save();
                $user             = $user->toArray();
                $user['entid']    = $company->id;
                $user['name']     = $user['real_name'];
                $user['is_admin'] = 0;
                $user['uniqued']  = $uniqued;
                return app()->get(AdminService::class)->create($user);
            });
        }
        if ($frameInfo) {
            $this->saveFrames($frameInfo, $admin->id, $company->id);
        }
        return true;
    }

    /**
     * 用户接口登录.
     * @param string $uid
     * @param string $entId
     * @param string $cardId
     * @return array
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    public function userLogin(string $uid, string $entId, string $cardId): array
    {
        $user = app()->get(UserService::class)->get(['uid' => $uid, 'card_id' => $cardId]);
        if (!$user) {
            throw $this->exception('未找到用户的信息');
        }
        $company = app()->get(CompanyService::class)->get(['uniqued' => $entId]);
        if (!$company) {
            throw $this->exception('未找到该企业的信息');
        }
        $admin = app()->get(AdminService::class)->setEntValue($company->id)->get(['entid' => $company->id, 'uid' => $uid]);
        if (!$admin) {
            throw $this->exception('未找到该用户的企业信息');
        }
        if ($admin->status == UserEnum::USER_LOCKING || app()->get(AdminInfoService::class)->value($admin->id, 'type') == 4) {
            throw $this->exception('您的账号已被锁定,无法登录!');
        }
        $user->entid = $company->id;
        $user->save();
        $token = auth('admin')->login($admin, true);
        if (!$token) {
            throw $this->exception('token创建失败');
        }
        $admin->last_ip = app('request')->ip();
        ++$admin->login_count;
        $admin->save();
        return [
            'token'      => $token,
            'token_type' => 'bearer',
            'expires_in' => auth()->factory()->getTTL() * 60,
        ];
    }

    /**
     * 用户接口登录.
     * @param string $uid
     * @param string $entId
     * @param string $cardId
     * @return array
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    public function login(string $sign, string $uid, string $cardId, string $entId): array
    {
        $this->checkSign($sign, $uid, $entId);
        $user = app()->get(UserService::class)->get(['uid' => $uid, 'card_id' => $cardId]);
        if (!$user) {
            throw $this->exception('未找到用户的信息');
        }
        $company = app()->get(CompanyService::class)->get(['uniqued' => $entId]);
        if (!$company) {
            throw $this->exception('未找到该企业的信息');
        }
        $admin = app()->get(AdminService::class)->setEntValue($company->id)->get(['entid' => $company->id, 'uid' => $uid]);
        if (!$admin) {
            throw $this->exception('未找到该用户的企业信息');
        }
        if ($admin->status == UserEnum::USER_LOCKING || app()->get(AdminInfoService::class)->setEntValue($company->id)->value($admin->id, 'type') == 4) {
            throw $this->exception('您的账号已被锁定,无法登录!');
        }
        $user->entid = $company->id;
        $user->save();
        $token = auth('admin')->login($admin, true);
        if (!$token) {
            throw $this->exception('token创建失败');
        }
        $admin->last_ip = app('request')->ip();
        ++$admin->login_count;
        $admin->save();
        return [
            'token'      => $token,
            'token_type' => 'bearer',
            'expires_in' => auth()->factory()->getTTL() * 60,
        ];
    }

    private function checkSign(string $sign, string $uid, string $entId)
    {
        $time = time();
        $i    = 0;
        do {
            if (md5($uid . strtoupper($entId) . bcsub((string)$time, (string)$i, 0) . $this->key) === $sign) {
                return true;
            }
            $i++;
        } while ($i < 20);
        Log::error('Uni签名校验失败', compact('uid', 'entId', 'sign', 'time'));
        throw $this->exception('签名校验失败');
    }

    private function saveFrames(array $frames, int $userId, int $entId = 0)
    {
        $service = app()->get(FrameAssistService::class)->setEntValue($entId);
        $service->forceDelete(['not_frame_id' => array_column($frames, 'id'), 'user_id' => $userId]);
        $service->restore(['frame_id' => array_column($frames, 'id'), 'user_id' => $userId]);
        foreach ($frames as $frame) {
            if ($frame['is_admin']) {
                $service->update(['frame_id' => $frame['id'], 'is_admin' => 1], ['is_admin' => 0]);
            }
            $service->updateOrCreate(['frame_id' => $frame['id'], 'user_id' => $userId, 'entid' => $entId], [
                'frame_id'   => $frame['id'],
                'user_id'    => $userId,
                'is_mastart' => $frame['is_master'],
                'is_admin'   => $frame['is_admin'],
                'uniqued'    => $frame['uniqued'],
                'entid'      => $entId,
            ]);
        }
        Task::deliver(new FrameCensusTask());
    }
}
