<?php

declare(strict_types=1);
/**
 *  +----------------------------------------------------------------------
 *  | 陀螺匠 [ 赋能开发者，助力企业发展 ]
 *  +----------------------------------------------------------------------
 *  | Copyright (c) 2016~2024 https://www.tuoluojiang.com All rights reserved.
 *  +----------------------------------------------------------------------
 *  | Licensed 陀螺匠并不是自由软件，未经许可不能去掉陀螺匠相关版权
 *  +----------------------------------------------------------------------
 *  | Author: 陀螺匠 Team <admin@tuoluojiang.com>
 *  +----------------------------------------------------------------------
 */

namespace App\Http\Controller\UniApi\Client;

use App\Constants\CustomEnum\CustomEnum;
use App\Constants\UserAgentEnum;
use App\Http\Controller\UniApi\AuthController;
use App\Http\Service\Client\CustomerLiaisonService;
use App\Http\Service\Config\FormService;
use crmeb\interfaces\ResourceControllerInterface;
use crmeb\traits\ResourceControllerTrait;
use Illuminate\Contracts\Container\BindingResolutionException;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;
use Spatie\RouteAttributes\Attributes\Middleware;
use Spatie\RouteAttributes\Attributes\Prefix;
use Spatie\RouteAttributes\Attributes\Resource;

/**
 * 联系人管理
 * Class CustomerLiaisonController.
 */
#[Prefix('uni/client/liaison')]
#[Resource('/', false, except: ['show'], names: [
    'index'   => '获取客户联系人列表',
    'create'  => '保存客户联系人表单',
    'store'   => '保存客户联系人',
    'edit'    => '修改客户联系人表单',
    'update'  => '修改客户联系人',
    'destroy' => '删除客户联系人',
], parameters: ['' => 'id'])]
#[Middleware(['auth.admin', 'ent.auth', 'ent.log'])]
class CustomerLiaisonController extends AuthController implements ResourceControllerInterface
{
    use ResourceControllerTrait;

    public function __construct(CustomerLiaisonService $services)
    {
        parent::__construct();
        $services->setPlatform(UserAgentEnum::UNI_AGENT);
        $this->service = $services;
    }

    /**
     * 列表数据.
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    public function index(): mixed
    {
        $where = $this->request->getMore($this->service->getSearchField());
        if (isset($where['eid']) && ! $where['eid']) {
            return $this->fail('common.empty.attrs');
        }
        return $this->success($this->service->getUniListByType($where));
    }

    /**
     * 创建表单.
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    public function create(): mixed
    {
        $service = app()->get(FormService::class);
        return $this->success($service->getFormDataWithType(CustomEnum::LIAISON, platform: UserAgentEnum::UNI_AGENT));
    }

    /**
     * 保存.
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    public function store(): mixed
    {
        $eid = (int) $this->request->post('eid', 0);
        if ($eid < 1) {
            return $this->fail('客户数据异常');
        }
        $service = app()->get(FormService::class);
        $data    = $this->request->postMore($service->getRequestFields(CustomEnum::LIAISON));
        $res     = $this->service->saveLiaison($data, $eid, auth('admin')->id());
        return $this->success('common.insert.succ', ['id' => $res->id]);
    }

    /**
     * 修改.
     * @param mixed $id
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    public function update($id): mixed
    {
        if (! $id) {
            return $this->fail($this->message['update']['emtpy']);
        }
        $service = app()->get(FormService::class);
        $data    = $this->request->postMore($service->getRequestFields(CustomEnum::LIAISON));
        $this->service->updateLiaison($data, (int) $id);
        return $this->success(__('common.update.succ'));
    }

    /**
     * 删除.
     * @param mixed $id
     * @throws BindingResolutionException
     * @throws \ReflectionException
     */
    public function destroy($id): mixed
    {
        if (! $id) {
            return $this->fail('common.empty.attrs');
        }
        $this->service->deleteLiaison((int) $id, auth('admin')->id());
        return $this->success('common.delete.succ');
    }

    /**
     * 修改表单.
     * @param mixed $id
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    public function edit($id): mixed
    {
        if (! $id) {
            return $this->fail($this->message['update']['emtpy']);
        }

        return $this->success($this->service->getEditInfo((int) $id, $this->uuid));
    }

    protected function getRequestClassName(): string
    {
        return '';
    }

    protected function getSearchField(): array
    {
        return [];
    }

    protected function getRequestFields(): array
    {
        return [];
    }
}
