<?php

declare(strict_types=1);
/**
 *  +----------------------------------------------------------------------
 *  | 陀螺匠 [ 赋能开发者，助力企业发展 ]
 *  +----------------------------------------------------------------------
 *  | Copyright (c) 2016~2024 https://www.tuoluojiang.com All rights reserved.
 *  +----------------------------------------------------------------------
 *  | Licensed 陀螺匠并不是自由软件，未经许可不能去掉陀螺匠相关版权
 *  +----------------------------------------------------------------------
 *  | Author: 陀螺匠 Team <admin@tuoluojiang.com>
 *  +----------------------------------------------------------------------
 */

namespace App\Http\Controller\UniApi\Client;

use App\Constants\CustomEnum\CustomEnum;
use App\Constants\UserAgentEnum;
use App\Http\Contract\Client\ClientSubscribeInterface;
use App\Http\Controller\UniApi\AuthController;
use App\Http\Service\Client\CustomerService;
use App\Http\Service\Config\FormService;
use App\Http\Service\Config\SystemConfigService;
use crmeb\interfaces\ResourceControllerInterface;
use crmeb\traits\ResourceControllerTrait;
use crmeb\traits\SearchTrait;
use FormBuilder\Exception\FormBuilderException;
use Illuminate\Contracts\Container\BindingResolutionException;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;
use Spatie\RouteAttributes\Attributes\Get;
use Spatie\RouteAttributes\Attributes\Middleware;
use Spatie\RouteAttributes\Attributes\Post;
use Spatie\RouteAttributes\Attributes\Prefix;
use Spatie\RouteAttributes\Attributes\Resource;

/**
 * 客户管理
 * Class CustomerController.
 */
#[Prefix('uni/client/customer')]
#[Resource('/', false, except: ['show'], names: [
    'index'   => '获取客户列表接口',
    'create'  => '保存客户表单接口',
    'store'   => '保存客户接口',
    'edit'    => '修改客户表单接口',
    'update'  => '修改客户接口',
    'destroy' => '删除客户接口',
], parameters: ['' => 'id'])]
#[Middleware(['auth.admin', 'ent.auth', 'ent.log'])]
class CustomerController extends AuthController implements ResourceControllerInterface
{
    use SearchTrait;
    use ResourceControllerTrait;

    public function __construct(CustomerService $services)
    {
        parent::__construct();
        $services->setPlatform(UserAgentEnum::UNI_AGENT);
        $this->service = $services;
    }

    /**
     * 列表数据.
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    public function index(): mixed
    {
        $types = (int) $this->request->get('types', 1);
        if ($types == 3) {
            $this->request->merge([
                'uid' => 0,
            ]);
        } else {
            $scope_frame = $this->request->get('scope_frame', '');
            if (! $scope_frame) {
                switch ($types) {
                    case 1:
                        $this->request->merge([
                            'scope_frame' => 'all',
                        ]);
                        break;
                    case 2:
                        $this->request->merge([
                            'scope_frame' => 'self',
                        ]);
                        break;
                }
            }
            $this->withScopeFrame();
        }
        $where = $this->request->getMore(array_merge($this->service->searchField($types), [
            ['customer_label', ''],
            ['customer_status', ''],
            ['time', '', 'created_at'],
            ['salesman_id', '', 'salesman'],
        ]));

        return $this->success($this->service->getUniListByType((int) $this->request->get('types', 1), $where, auth('admin')->id()));
    }

    /**
     * 保存表单.
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    public function create(): mixed
    {
        $service = app()->get(FormService::class);
        return $this->success($service->getFormDataWithType(CustomEnum::CUSTOMER, platform: UserAgentEnum::UNI_AGENT));
    }

    /**
     * 保存.
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    public function store(): mixed
    {
        $service         = app()->get(FormService::class);
        $data            = $this->request->postMore($service->getRequestFields(CustomEnum::CUSTOMER));
        [$types, $force] = $this->request->postMore([
            ['types', 2],
            ['force', 0],
        ], true);
        $res = $this->service->saveCustomer($data, auth('admin')->id(), (int) $types, (int) $force);
        return $this->success('common.insert.succ', ['id' => $res->id]);
    }

    /**
     * 修改.
     * @param mixed $id
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    public function update($id): mixed
    {
        if (! $id) {
            return $this->fail($this->message['update']['emtpy']);
        }
        $service = app()->get(FormService::class);
        $data    = $this->request->postMore($service->getRequestFields(CustomEnum::CUSTOMER));
        $this->service->updateCustomer($data, (int) $id, (int) $this->request->post('force', 0));
        return $this->success(__('common.update.succ'));
    }

    /**
     * 详情.
     * @param mixed $id
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    #[Get('info/{id}', '客户详情接口')]
    public function info($id): mixed
    {
        if (! $id) {
            return $this->fail($this->message['update']['emtpy']);
        }
        return $this->success($this->service->getUniInfo((int) $id));
    }

    /**
     * 修改表单.
     * @param mixed $id
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    public function edit($id): mixed
    {
        if (! $id) {
            return $this->fail($this->message['update']['emtpy']);
        }

        return $this->success($this->service->getEditInfo((int) $id, auth('admin')->id()));
    }

    /**
     * 删除.
     * @param mixed $id
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    public function destroy($id): mixed
    {
        if (! $id) {
            return $this->fail('common.empty.attrs');
        }
        $this->service->deleteCustomer((int) $id);
        return $this->success('common.delete.succ');
    }

    /**
     * 下拉数据.
     * @throws BindingResolutionException
     * @throws \ReflectionException
     */
    #[Get('select', '客户列表选择')]
    public function select(): mixed
    {
        return $this->success($this->service->getSelectList(auth('admin')->id()));
    }

    /**
     * 修改关注状态
     * @param mixed $id
     * @param mixed $status
     * @throws BindingResolutionException
     */
    #[Post('subscribe/{id}/{status}', '修改客户关注状态')]
    public function subscribe(ClientSubscribeInterface $clientSubscribeService, $id, $status): mixed
    {
        if (! $id) {
            return $this->fail('common.empty.attrs');
        }
        $clientSubscribeService->subscribe(auth('admin')->id(), (int) $id, (int) $status);
        return $this->success(__('common.operation.succ'));
    }

    /**
     * 获取客户相关审批.
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws FormBuilderException
     * @throws \ReflectionException
     */
    #[Get('approve/{form?}', '获取客户相关审批')]
    public function getApprove(SystemConfigService $service, int|string $form = 0): mixed
    {
        $keys = ['contract_disburse_switch', 'contract_refund_switch', 'contract_renew_switch', 'invoicing_switch', 'void_invoice_switch'];
        if ($form) {
            return $this->success($service->getApproveConfig($keys));
        }
        return $this->success($service->getApproveConfigs($keys));
    }

    /**
     * 退回公海.
     * @param mixed $id
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    #[Post('return/{id}', '退回公海')]
    public function return($id): mixed
    {
        if (! $id) {
            return $this->fail(__('common.empty.attrs'));
        }
        $reason = $this->request->post('reason', '');
        $this->service->returnHighSeas([$id], $reason,auth('admin')->id());
        return $this->success(__('common.operation.succ'));
    }

    /**
     * 流失.
     * @param mixed $id
     * @throws BindingResolutionException
     * @throws \ReflectionException
     */
    #[Post('lost/{id}', '客户流失')]
    public function lost($id): mixed
    {
        if (! $id) {
            return $this->fail(__('common.empty.attrs'));
        }

        $this->service->lost([(int) $id],auth('admin')->id());
        return $this->success(__('common.operation.succ'));
    }

    /**
     * 领取.
     * @param mixed $id
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    #[Post('claim/{id}', '客户领取')]
    public function claim($id): mixed
    {
        if (! $id) {
            return $this->fail(__('common.empty.attrs'));
        }
        $this->service->claim([(int) $id], auth('admin')->id());
        return $this->success(__('common.operation.succ'));
    }

    /**
     * 取消流失.
     * @param mixed $id
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws \ReflectionException
     */
    #[Post('cancel_lost/{id}', '取消流失')]
    public function cancelLost($id): mixed
    {
        if (! $id) {
            return $this->fail('common.empty.attrs');
        }
        $this->service->cancelLost((int) $id,auth('admin')->id());
        return $this->success(__('common.operation.succ'));
    }

    /**
     * 设置标签.
     * @param mixed $id
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    #[Post('label/{id}', '设置标签')]
    public function label($id): mixed
    {
        if (! $id) {
            return $this->fail('common.empty.attrs');
        }
        [$label] = $this->request->postMore([
            ['label', []],
        ], true);
        $this->service->label([$id], (array) $label);
        return $this->success(__('common.operation.succ'));
    }

    /**
     * 客户转移.
     * @param mixed $id
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    #[Post('shift/{id}', '客户转移')]
    public function shift($id): mixed
    {
        if (! $id) {
            return $this->fail('common.empty.attrs');
        }

        [$toUid, $invoice, $contract] = $this->request->postMore([
            ['to_uid', 0],
            ['invoice', 0],
            ['contract', 0],
        ], true);
        $this->service->shift([(int) $id], (int) $toUid,auth('admin')->id(), (int) $invoice, (int) $contract);
        return $this->success(__('common.operation.succ'));
    }

    /**
     * 业绩简报接口.
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    #[Get('brief_statistics', '业绩简报')]
    public function briefStatistics(): mixed
    {
        [$time] = $this->request->getMore([
            ['time', ''],
        ], true);
        $userIds = $this->getStatisticsUserIds();
        $data    = $this->service->getBriefStatistics($time, $userIds);
        return $this->success($data);
    }

    /**
     * 业务员业绩排行接口.
     */
    #[Get('salesman_rank', '业务员业绩排行')]
    public function salesmanRank(): mixed
    {
        [$time] = $this->request->getMore([
            ['time', ''],
        ], true);
        $userIds = $this->getStatisticsUserIds();
        $data    = $this->service->getSalesmanRank($time, $userIds);
        return $this->success($data);
    }

    /**
     * 合同类型分析统计
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    #[Get('contract_rank', '合同类型分析统计')]
    public function contractRank(): mixed
    {
        [$time, $categoryId] = $this->request->getMore([
            ['time', ''],
            ['category_id', 0],
        ], true);
        $userIds = $this->getStatisticsUserIds();
        $data    = $this->service->getContractRankWithNotRatio($time, $userIds, (int) $categoryId);
        return $this->success($data);
    }

    /**
     * 业务员.
     * @throws BindingResolutionException
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \ReflectionException
     */
    #[Get('salesman', '业务员列表')]
    public function salesman(): mixed
    {
        return $this->success($this->service->getSalesman(auth('admin')->id()));
    }

    /**
     * 获取数据范围.
     */
    public function getStatisticsUserIds(): array
    {
        [$member, $type] = $this->request->getMore([
            ['member', []],
            ['type', ''],
        ], true);

        if ($type < 1) {
            $this->request->merge(['scope_frame' => 'team']);
            $this->withScopeFrame();
            $userIds = request()->get('uid');
        } else {
            $userIds = $this->service->getIdsByType(auth('admin')->id(), $member, (int) $type);
        }
        return (array) $userIds;
    }

    protected function getRequestClassName(): string
    {
        return '';
    }

    protected function getSearchField(): array
    {
        return [];
    }

    protected function getRequestFields(): array
    {
        return [];
    }
}
