<?php

declare(strict_types=1);
/**
 *  +----------------------------------------------------------------------
 *  | 陀螺匠 [ 赋能开发者，助力企业发展 ]
 *  +----------------------------------------------------------------------
 *  | Copyright (c) 2016~2024 https://www.tuoluojiang.com All rights reserved.
 *  +----------------------------------------------------------------------
 *  | Licensed 陀螺匠并不是自由软件，未经许可不能去掉陀螺匠相关版权
 *  +----------------------------------------------------------------------
 *  | Author: 陀螺匠 Team <admin@tuoluojiang.com>
 *  +----------------------------------------------------------------------
 */

namespace App\Http\Controller\AdminApi\Approve;

use App\Http\Controller\UniApi\AuthController;
use App\Http\Service\Approve\ApproveRoleService;
use crmeb\interfaces\ResourceControllerInterface;
use crmeb\traits\ResourceControllerTrait;
use Illuminate\Contracts\Container\BindingResolutionException;
use Spatie\RouteAttributes\Attributes\Get;
use Spatie\RouteAttributes\Attributes\Middleware;
use Spatie\RouteAttributes\Attributes\Post;
use Spatie\RouteAttributes\Attributes\Prefix;
use Spatie\RouteAttributes\Attributes\Resource;

/**
 * 考勤组
 * Class ApproveRoleController.
 */
#[Prefix('ent/approve/roles')]
#[Resource('/', false, except: ['create', 'edit'], names: [
    'index'   => '角色列表',
    'create'  => '创建角色表单',
    'store'   => '保存角色',
    'edit'    => '修改角色表单',
    'show'    => '修改角色状态',
    'update'  => '修改角色',
    'destroy' => '删除角色',
], parameters: ['' => 'id'])]
#[Middleware(['auth.admin', 'ent.auth', 'ent.log'])]
class ApproveRoleController extends AuthController implements ResourceControllerInterface
{
    use ResourceControllerTrait;

    public function __construct(ApproveRoleService $services)
    {
        parent::__construct();
        $this->service = $services;
    }

    /**
     * 获取角色成员.
     */
    #[Get('user/{id}', '获取角色用户列表')]
    public function getRoleUser($id): mixed
    {
        if (!$id) {
            return $this->fail('缺少角色ID');
        }
        return $this->success($this->service->getRoleUser((int)$id, $this->entId));
    }

    /**
     * 获取用户角色.
     * @param mixed $uid
     */
    #[Get('role/{uid}', '获取用户角色列表')]
    public function getUserRole($uid): mixed
    {
        if (!$uid) {
            return $this->fail('缺少用户ID');
        }
        $res = $this->service->getUserRole($this->entId, (int)$uid);
        return $this->success($res);
    }

    /**
     * 修改用户角色.
     */
    #[Post('user', '修改用户角色')]
    public function updateUserRole(): mixed
    {
        [$userId, $roleIds] = $this->request->postMore([
            ['user_id', 0],
            ['role_id', []],
        ], true);
        $res = $this->service->changeUserRole($this->entId, (int)$userId, $roleIds);
        return $res ? $this->success('修改成功') : $this->fail('修改失败');
    }

    /**
     * 添加角色成员.
     * @throws BindingResolutionException
     */
    #[Post('add_user', '角色新增用户')]
    public function addUser(): mixed
    {
        [$userIds, $frameId, $roleId] = $this->request->postMore([
            ['user_id', []],
            ['frame_id', []],
            ['role_id', 0],
        ], true);

        if (!$userIds && !$frameId) {
            return $this->fail('至少选择一个部门或者一个用户');
        }

        if ($this->service->addRoleUser($this->entId, (int)$roleId, $userIds, $frameId)) {
            return $this->success('添加成员成功');
        }
        return $this->fail('common.insert.fail');
    }

    /**
     * 修改角色成员状态
     */
    #[Post('show_user', '修改角色成员状态')]
    public function showUser(): mixed
    {
        [$userId, $status, $roleId] = $this->request->postMore([
            ['uid', 0],
            ['status', 1],
            ['role_id', 0],
        ], true);
        if ($this->service->changeRoleUser((int)$userId, $this->entId, (int)$roleId, (int)$status)) {
            return $this->success('common.update.succ');
        }
        return $this->fail('common.update.fail');
    }

    /**
     * 删除角色成员.
     */
    #[Post('del_user', '删除角色成员')]
    public function deleteUser(): mixed
    {
        [$uid, $roleId] = $this->request->postMore([
            ['uid', 0],
            ['role_id', 0],
        ], true);
        if (!$uid) {
            return $this->fail('缺少删除角色成员ID');
        }
        if ($this->service->delRoleUser((int)$uid, $this->entId, (int)$roleId)) {
            return $this->success('删除成功');
        }
        return $this->fail('删除失败');
    }

    protected function getRequestClassName(): string
    {
        return '';
    }

    protected function getSearchField(): array
    {
        return [
            ['name', '', 'name_like'],
        ];
    }

    protected function getRequestFields(): array
    {
        return [
            ['name', ''],
            ['uids', []],
            ['status', 1],
            ['entid', $this->entId],
        ];
    }
}
