<?php

declare(strict_types=1);
/**
 *  +----------------------------------------------------------------------
 *  | 陀螺匠 [ 赋能开发者，助力企业发展 ]
 *  +----------------------------------------------------------------------
 *  | Copyright (c) 2016~2024 https://www.tuoluojiang.com All rights reserved.
 *  +----------------------------------------------------------------------
 *  | Licensed 陀螺匠并不是自由软件，未经许可不能去掉陀螺匠相关版权
 *  +----------------------------------------------------------------------
 *  | Author: 陀螺匠 Team <admin@tuoluojiang.com>
 *  +----------------------------------------------------------------------
 */

namespace App\Console\Commands;

use App\Constants\CustomEnum\CustomEnum;
use App\Http\Service\Config\FormService;
use App\Http\Service\Open\OpenapiRuleService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;

class UpgradeCommand extends Command
{
    /**
     * @var string
     */
    protected $signature = 'tl:up';

    /**
     * Create a new command instance.
     */
    public function __construct()
    {
        parent::__construct();
    }

    public function handle(): void
    {
        if ($this->confirm('为了您的数据安全, 更新前请确认是否做好数据备份? (数据库/项目文件)', true)) {
            $this->upgrade();
        }
    }

    /**
     * 读取版本号.
     */
    protected function getVersion(string $key = ''): array|string
    {
        $version_arr    = [];
        $curent_version = @file(base_path('.version'));
        foreach ($curent_version as $val) {
            [$k, $v] = explode('=', $val);
            $version_arr[$k] = $v;
        }
        return $key ? trim($version_arr[$key]) : $version_arr;
    }

    private function upgrade(): void
    {
        $this->info('即将开始执行升级程序...');

        // composer install
        @shell_exec('composer install');
        $this->info('Composer 安装成功!');
        $this->newLine();

        // update database migration files
        $this->info('正在检查是否需要更新数据库...');
        $version = $this->getVersion('version_num');
        $path    = database_path('migrations/v' . $version);
        if (is_dir($path)) {
            $isAllMigrationSucc = true;
            $this->newLine();
            $this->info('开始执行数据迁移...');
            $this->newLine();
            $files = opendir($path);
            while ($file = readdir($files)) {
                if ($file == '.' || $file == '..') {
                    continue;
                }

                $path = 'database/migrations/v' . $version . '/' . $file;
                $this->info('执行文件: ' . $path);
                try {
                    Artisan::call('migrate', ['--path' => $path]);
                } catch (\Throwable $e) {
                    $isAllMigrationSucc = false;
                    $this->error('数据库更新失败: ' . $e->getMessage());
                    $this->error('迁移文件地址: ' . $path);
                    $this->newLine();
                    Log::error($e->getMessage(), ['file path' => $path]);
                }
            }
            $this->newLine();
            if ($isAllMigrationSucc) {
                $this->info('恭喜, 数据库更新成功!');
            } else {
                $this->error('部分数据更新失败, 详情请查看日志');
            }
        } else {
            $this->info('数据库无需更新');
        }

        $handlerFile = database_path('seeders/v' . $version . '/DataUpdateHandler.php');
        if (file_exists($handlerFile)) {
            require_once $handlerFile;
        }

        if (class_exists('DataUpdateHandler')) {
            app()->get('DataUpdateHandler');
        }

        // update system menus
        $sqlPath = database_path('seeders/v' . $version . '.sql');
        if (file_exists($sqlPath)) {
            DB::unprepared(prefix_correction(file_get_contents($sqlPath)));
        }

        // update api rule
        $this->saveApiRule();

        $this->newLine();
        $this->info('更新完成, 请按照文档重新配置守护进程重新登录...');
    }

    /**
     * 保存自定义业务文档
     * @return mixed
     * @throws \Throwable
     */
    public function saveApiRule(): mixed
    {
        $this->info('开始生成接口文档...');
        return DB::transaction(function () {
            $rules       = [
                [
                    'name'    => '对外接口授权',
                    'crud_id' => 0,
                    'method'  => '',
                    'url'     => '',
                    'type'    => 0,
                    'child'   => [
                        [
                            'name'    => '授权登录',
                            'crud_id' => 0,
                            'method'  => 'POST',
                            'url'     => 'open/auth/login',
                            'type'    => 1,
                            'post_prams' => [
                                [
                                    'name'      => 'access_key',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '授权账号'
                                ],
                                [
                                    'name'      => 'secret_key',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '授权秘钥',
                                ],
                            ],
                            'response_data' => [
                                [
                                    'name'      => 'token',
                                    'form_type' => 'string',
                                    'message'   => '授权凭证'
                                ],
                                [
                                    'name'      => 'token_type',
                                    'form_type' => 'string',
                                    'message'   => '凭证类型'
                                ],
                                [
                                    'name'      => 'expires_in',
                                    'form_type' => 'int',
                                    'message'   => '过期时间'
                                ],
                            ]
                        ],
                    ]
                ],
                CustomEnum::CUSTOMER => [
                    'name'    => '客户基本信息',
                    'crud_id' => 0,
                    'method'  => '',
                    'url'     => '',
                    'type'    => 0,
                    'child'   => [
                        [
                            'name'    => '新增客户',
                            'crud_id' => 0,
                            'method'  => 'POST',
                            'url'     => 'open/customer',
                            'type'    => 1
                        ],
                        [
                            'name'    => '修改客户',
                            'crud_id' => 0,
                            'method'  => 'PUT',
                            'url'     => 'open/customer/{id}',
                            'type'    => 1,
                        ],
                        [
                            'name'    => '删除客户',
                            'crud_id' => 0,
                            'method'  => 'DELETE',
                            'url'     => 'open/customer/{id}',
                            'type'    => 1,
                        ],
                    ]
                ],
                CustomEnum::CONTRACT => [
                    'name'    => '合同',
                    'crud_id' => 0,
                    'method'  => '',
                    'url'     => '',
                    'type'    => 0,
                    'child'   => [
                        [
                            'name'    => '新增合同',
                            'crud_id' => 0,
                            'method'  => 'POST',
                            'url'     => 'open/contract',
                            'type'    => 1
                        ],
                        [
                            'name'    => '修改合同',
                            'crud_id' => 0,
                            'method'  => 'PUT',
                            'url'     => 'open/contract/{id}',
                            'type'    => 1,
                        ],
                        [
                            'name'    => '删除合同',
                            'crud_id' => 0,
                            'method'  => 'DELETE',
                            'url'     => 'open/contract/{id}',
                            'type'    => 1,
                        ],
                    ]
                ],
                CustomEnum::LIAISON  => [
                    'name'    => '客户联系人',
                    'crud_id' => 0,
                    'method'  => '',
                    'url'     => '',
                    'type'    => 0,
                    'child'   => [
                        [
                            'name'    => '新增联系人',
                            'crud_id' => 0,
                            'method'  => 'POST',
                            'url'     => 'open/liaison',
                            'type'    => 1
                        ],
                        [
                            'name'    => '修改联系人',
                            'crud_id' => 0,
                            'method'  => 'PUT',
                            'url'     => 'open/liaison/{id}',
                            'type'    => 1,

                        ],
                        [
                            'name'    => '删除联系人',
                            'crud_id' => 0,
                            'method'  => 'DELETE',
                            'url'     => 'open/liaison/{id}',
                            'type'    => 1,
                        ],
                    ]
                ],
                [
                    'name'    => '发票记录',
                    'crud_id' => 0,
                    'method'  => '',
                    'url'     => '',
                    'type'    => 0,
                    'child'   => [
                        [
                            'name'       => '新增发票',
                            'crud_id'    => 0,
                            'method'     => 'POST',
                            'url'        => 'open/invoice',
                            'type'       => 1,
                            'post_prams' => [
                                [
                                    'name'      => 'uid',
                                    'form_type' => 'int',
                                    'is_must'   => false,
                                    'message'   => '关联业务员id'
                                ],
                                [
                                    'name'      => 'bill_id',
                                    'form_type' => 'array',
                                    'is_must'   => false,
                                    'message'   => '关联付款单id',
                                    'symbol'    => 'billId',
                                ],
                                [
                                    'name'      => 'eid',
                                    'form_type' => 'int',
                                    'is_must'   => true,
                                    'message'   => '关联客戶id',
                                    'symbol'    => 'customerList',
                                ],
                                [
                                    'name'      => 'cid',
                                    'form_type' => 'int',
                                    'is_must'   => true,
                                    'message'   => '关联合同id',
                                    'symbol'    => 'contractList',
                                ],
                                [
                                    'name'      => 'bill_date',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '期望开票日期',
                                    'symbol'    => 'desireDate',
                                ],
                                [
                                    'name'      => 'collect_type',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '开票要求',
                                    'symbol'    => 'invoicingMethod',
                                ],
                                [
                                    'name'      => 'collect_name',
                                    'form_type' => 'string',
                                    'is_must'   => false,
                                    'message'   => '联系人',
                                    'symbol'    => 'liaisonMan',
                                ],
                                [
                                    'name'      => 'collect_tel',
                                    'form_type' => 'string',
                                    'is_must'   => false,
                                    'message'   => '联系电话',
                                    'symbol'    => 'telephone',
                                ],
                                [
                                    'name'      => 'collect_email',
                                    'form_type' => 'string',
                                    'is_must'   => false,
                                    'message'   => '邮箱地址',
                                    'symbol'    => 'invoicingEmail',
                                ],
                                [
                                    'name'      => 'mail_address',
                                    'form_type' => 'string',
                                    'is_must'   => false,
                                    'message'   => '邮寄地址',
                                    'symbol'    => 'mailingAddress',
                                ],
                                [
                                    'name'      => 'types',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '发票类型',
                                    'symbol'    => 'invoiceType',
                                ],
                                [
                                    'name'      => 'amount',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '开票金额（元）',
                                    'symbol'    => 'invoiceAmount',
                                ],
                                [
                                    'name'      => 'price',
                                    'form_type' => 'string',
                                    'is_must'   => false,
                                    'message'   => '付款单金额（元）',
                                    'symbol'    => 'billAmount',
                                ],
                                [
                                    'name'      => 'title',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '发票抬头',
                                    'symbol'    => 'invoiceHeader',
                                ],
                                [
                                    'name'      => 'ident',
                                    'form_type' => 'string',
                                    'is_must'   => false,
                                    'message'   => '纳税人识别号',
                                    'symbol'    => 'dutyParagraph',
                                ],
                                [
                                    'name'      => 'mark',
                                    'form_type' => 'string',
                                    'is_must'   => false,
                                    'message'   => '备注',
                                    'symbol'    => 'remark',
                                ],
                            ],
                        ],
                        [
                            'name'       => '发票作废',
                            'crud_id'    => 0,
                            'method'     => 'PUT',
                            'url'        => 'open/invoice/{id}/invalid',
                            'type'       => 1,
                            'post_prams' => [
                                [
                                    'name'      => 'uid',
                                    'form_type' => 'int',
                                    'is_must'   => true,
                                    'message'   => '业务员id',
                                ],
                                [
                                    'name'      => 'remark',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '作废原因',
                                ],
                            ],
                        ],
                    ]
                ], // 发票记录
                [
                    'name'    => '客户账目记录',
                    'crud_id' => 0,
                    'method'  => '',
                    'url'     => '',
                    'type'    => 0,
                    'child'   => [
                        [
                            'name'       => '新增账目回款',
                            'crud_id'    => 0,
                            'method'     => 'POST',
                            'url'        => 'open/bill/payment',
                            'type'       => 1,
                            'post_prams' => [
                                [
                                    'name'      => 'uid',
                                    'form_type' => 'int',
                                    'is_must'   => false,
                                    'message'   => '关联业务员id'
                                ],
                                [
                                    'name'      => 'cid',
                                    'form_type' => 'int',
                                    'is_must'   => true,
                                    'message'   => '关联合同id',
                                    'symbol'    => 'contractList',
                                ],
                                [
                                    'name'      => 'bill_cate_id',
                                    'form_type' => 'array',
                                    'is_must'   => true,
                                    'message'   => '财务收入科目id',
                                    'symbol'    => 'incomeCategories',
                                ],
                                [
                                    'name'      => 'type_id',
                                    'form_type' => 'int',
                                    'is_must'   => true,
                                    'message'   => '支付方式id',
                                    'symbol'    => 'payType',
                                ],
                                [
                                    'name'      => 'num',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '回款金额（元）',
                                    'symbol'    => 'collectionAmount',
                                ],
                                [
                                    'name'      => 'date',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '付款时间',
                                    'symbol'    => 'payTime',
                                ],

                                [
                                    'name'      => 'attach',
                                    'form_type' => 'array',
                                    'is_must'   => false,
                                    'message'   => '付款凭证id',
                                    'symbol'    => 'paymentVoucher',
                                ],
                                [
                                    'name'      => 'mark',
                                    'form_type' => 'string',
                                    'is_must'   => false,
                                    'message'   => '备注',
                                    'symbol'    => 'remark',
                                ],
                            ],
                        ],
                        [
                            'name'       => '新增账目续费',
                            'crud_id'    => 0,
                            'method'     => 'POST',
                            'url'        => 'open/bill/renewal',
                            'type'       => 1,
                            'post_prams' => [
                                [
                                    'name'      => 'uid',
                                    'form_type' => 'int',
                                    'is_must'   => false,
                                    'message'   => '关联业务员id'
                                ],
                                [
                                    'name'      => 'cid',
                                    'form_type' => 'int',
                                    'is_must'   => true,
                                    'message'   => '关联合同id',
                                    'symbol'    => 'contractList',
                                ],
                                [
                                    'name'      => 'bill_cate_id',
                                    'form_type' => 'array',
                                    'is_must'   => true,
                                    'message'   => '财务收入科目id',
                                    'symbol'    => 'incomeCategories',
                                ],
                                [
                                    'name'      => 'cate_id',
                                    'form_type' => 'int',
                                    'is_must'   => true,
                                    'message'   => '续费类型id',
                                    'symbol'    => 'renewalType',
                                ],
                                [
                                    'name'      => 'type_id',
                                    'form_type' => 'int',
                                    'is_must'   => true,
                                    'message'   => '支付方式id',
                                    'symbol'    => 'payType',
                                ],
                                [
                                    'name'      => 'num',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '续费金额（元）',
                                    'symbol'    => 'renewalAmount',
                                ],
                                [
                                    'name'      => 'date',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '付款时间',
                                    'symbol'    => 'payTime',
                                ],
                                [
                                    'name'      => 'end_date',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '续费结束日期',
                                    'symbol'    => 'renewalEndTime',
                                ],

                                [
                                    'name'      => 'attach',
                                    'form_type' => 'array',
                                    'is_must'   => false,
                                    'message'   => '付款凭证id',
                                    'symbol'    => 'paymentVoucher',
                                ],
                                [
                                    'name'      => 'mark',
                                    'form_type' => 'string',
                                    'is_must'   => false,
                                    'message'   => '备注',
                                    'symbol'    => 'remark',
                                ],
                            ],
                        ],
                        [
                            'name'       => '新增账目支出',
                            'crud_id'    => 0,
                            'method'     => 'POST',
                            'url'        => 'open/bill/expend',
                            'type'       => 1,
                            'post_prams' => [
                                [
                                    'name'      => 'uid',
                                    'form_type' => 'int',
                                    'is_must'   => false,
                                    'message'   => '关联业务员id'
                                ],
                                [
                                    'name'      => 'cid',
                                    'form_type' => 'int',
                                    'is_must'   => true,
                                    'message'   => '关联合同id',
                                    'symbol'    => 'contractList',
                                ],
                                [
                                    'name'      => 'bill_cate_id',
                                    'form_type' => 'array',
                                    'is_must'   => true,
                                    'message'   => '财务支出科目id',
                                    'symbol'    => 'expenditureCategories',
                                ],
                                [
                                    'name'      => 'type_id',
                                    'form_type' => 'int',
                                    'is_must'   => true,
                                    'message'   => '支付方式id',
                                    'symbol'    => 'payType',
                                ],
                                [
                                    'name'      => 'num',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '支出金额（元）',
                                    'symbol'    => 'expenditureAmount',
                                ],
                                [
                                    'name'      => 'date',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '付款时间',
                                    'symbol'    => 'payTime',
                                ],
                                [
                                    'name'      => 'attach',
                                    'form_type' => 'array',
                                    'is_must'   => false,
                                    'message'   => '付款凭证id',
                                    'symbol'    => 'paymentVoucher',
                                ],
                                [
                                    'name'      => 'mark',
                                    'form_type' => 'string',
                                    'is_must'   => false,
                                    'message'   => '备注',
                                    'symbol'    => 'remark',
                                ],
                            ],
                        ],
                        [
                            'name'    => '删除账目',
                            'crud_id' => 0,
                            'method'  => 'DELETE',
                            'url'     => 'open/bill/{id}',
                            'type'    => 1,
                        ],
                        [
                            'name'       => '新增付款提醒',
                            'crud_id'    => 0,
                            'method'     => 'POST',
                            'url'        => 'open/bill/remind',
                            'type'       => 1,
                            'post_prams' => [
                                [
                                    'name'      => 'types',
                                    'form_type' => 'int',
                                    'is_must'   => true,
                                    'message'   => '提醒类型'
                                ],
                                [
                                    'name'      => 'eid',
                                    'form_type' => 'int',
                                    'is_must'   => true,
                                    'message'   => '关联客户id'
                                ],
                                [
                                    'name'      => 'cid',
                                    'form_type' => 'int',
                                    'is_must'   => true,
                                    'message'   => '关联合同id'
                                ],
                                [
                                    'name'      => 'num',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '金额(元)'
                                ],
                                [
                                    'name'      => 'time',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '提醒日期'
                                ],
                                [
                                    'name'      => 'mark',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '提醒内容'
                                ],
                            ],
                        ],
                    ]
                ], // 账目记录
                [
                    'name'    => '日程待办',
                    'crud_id' => 0,
                    'method'  => '',
                    'url'     => '',
                    'type'    => 0,
                    'child'   => [
                        [
                            'name'       => '新增日程',
                            'crud_id'    => 0,
                            'method'     => 'POST',
                            'url'        => 'open/schedule',
                            'type'       => 1,
                            'post_prams' => [
                                [
                                    'name'      => 'uid',
                                    'form_type' => 'int',
                                    'is_must'   => false,
                                    'message'   => '关联业务员id'
                                ],
                                [
                                    'name'      => 'title',
                                    'form_type' => 'string',
                                    'is_must'   => false,
                                    'message'   => '待办标题',
                                ],
                                [
                                    'name'      => 'member',
                                    'form_type' => 'array',
                                    'is_must'   => true,
                                    'message'   => '参与人id',
                                ],
                                [
                                    'name'      => 'start_time',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '开始时间',
                                ],
                                [
                                    'name'      => 'end_time',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '结束时间',
                                ],
                                [
                                    'name'      => 'all_day',
                                    'form_type' => 'int',
                                    'is_must'   => false,
                                    'message'   => '是否全天',
                                ],
                                [
                                    'name'      => 'period',
                                    'form_type' => 'int',
                                    'is_must'   => true,
                                    'message'   => '重复周期',
                                ],
                                [
                                    'name'      => 'days',
                                    'form_type' => 'array',
                                    'is_must'   => true,
                                    'message'   => '重复周期为周/月时的重复频率',
                                ],
                                [
                                    'name'      => 'rate',
                                    'form_type' => 'int',
                                    'is_must'   => true,
                                    'message'   => '重复频率',
                                ],
                                [
                                    'name'      => 'remind',
                                    'form_type' => 'int',
                                    'is_must'   => true,
                                    'message'   => '是否提醒',
                                ],
                                [
                                    'name'      => 'type',
                                    'form_type' => 'int',
                                    'is_must'   => true,
                                    'message'   => '操作方式',
                                ],
                                [
                                    'name'      => 'fail_time',
                                    'form_type' => 'string',
                                    'is_must'   => false,
                                    'message'   => '重复截至日期',
                                ],
                            ],
                        ],
                        [
                            'name'       => '修改日程',
                            'crud_id'    => 0,
                            'method'     => 'PUT',
                            'url'        => 'open/schedule/{id}',
                            'type'       => 1,
                            'post_prams' => [
                                [
                                    'name'      => 'title',
                                    'form_type' => 'string',
                                    'is_must'   => false,
                                    'message'   => '待办标题',
                                ],
                                [
                                    'name'      => 'member',
                                    'form_type' => 'array',
                                    'is_must'   => true,
                                    'message'   => '参与人id',
                                ],
                                [
                                    'name'      => 'start_time',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '开始时间',
                                ],
                                [
                                    'name'      => 'end_time',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '结束时间',
                                ],
                                [
                                    'name'      => 'start',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '开始时间',
                                ],
                                [
                                    'name'      => 'end',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '结束时间',
                                ],
                                [
                                    'name'      => 'all_day',
                                    'form_type' => 'int',
                                    'is_must'   => false,
                                    'message'   => '是否全天',
                                ],
                                [
                                    'name'      => 'period',
                                    'form_type' => 'int',
                                    'is_must'   => true,
                                    'message'   => '重复周期',
                                ],
                                [
                                    'name'      => 'days',
                                    'form_type' => 'array',
                                    'is_must'   => true,
                                    'message'   => '重复周期为周/月时的重复频率',
                                ],
                                [
                                    'name'      => 'rate',
                                    'form_type' => 'int',
                                    'is_must'   => true,
                                    'message'   => '重复频率',
                                ],
                                [
                                    'name'      => 'remind',
                                    'form_type' => 'int',
                                    'is_must'   => true,
                                    'message'   => '是否提醒',
                                ],
                                [
                                    'name'      => 'type',
                                    'form_type' => 'int',
                                    'is_must'   => true,
                                    'message'   => '操作方式',
                                ],
                                [
                                    'name'      => 'fail_time',
                                    'form_type' => 'string',
                                    'is_must'   => false,
                                    'message'   => '重复截至日期',
                                ],
                            ],
                        ],
                        [
                            'name'       => '删除日程',
                            'crud_id'    => 0,
                            'method'     => 'DELETE',
                            'url'        => 'open/schedule/{id}',
                            'type'       => 1,
                            'post_prams' => [
                                [
                                    'name'      => 'start',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '开始时间',
                                ],
                                [
                                    'name'      => 'end',
                                    'form_type' => 'string',
                                    'is_must'   => true,
                                    'message'   => '结束时间',
                                ],
                                [
                                    'name'      => 'type',
                                    'form_type' => 'int',
                                    'is_must'   => true,
                                    'message'   => '操作方式',
                                ],
                            ],
                        ],
                    ]
                ] // 日程待办
            ];
            $option      = [];
            $formService = app()->make(FormService::class);
            $ruleService = app()->make(OpenapiRuleService::class);
            $field       = ['id', 'key', 'key_name', 'type', 'input_type', 'required',];
            foreach ($rules as $key => $rule) {
                if (in_array($key, [CustomEnum::CUSTOMER, CustomEnum::CONTRACT, CustomEnum::LIAISON])) {
                    $option = $formService->getCustomDataByTypes($key, $field);
                }
                $pid = $ruleService->value(['name' => $rule['name'], 'crud_id' => 0], 'id');
                if (!$pid) {
                    $parent = $ruleService->create(['name' => $rule['name'], 'crud_id' => 0, 'type' => 0]);
                    $pid    = $parent?->id;
                }

                if (!$pid) {
                    break;
                }

                foreach ($rule['child'] as $child) {
                    $id = $ruleService->value(['name' => $child['name'], 'pid' => $pid], 'id');
                    if (!$id) {
                        $ruleService->create($ruleService->assembleParams(array_merge($child, ['pid' => $pid]), $option));
                    } else {
                        $ruleService->update(['id' => $id], $ruleService->assembleParams($child, $option));
                    }
                }
            }
            return true;
        });
    }
}
