package com.bcxin.sync.service.tuoluojiang;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.conditions.query.LambdaQueryChainWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bcxin.sync.dao.mapper.tuoluojiang.EbFrameMapper;
import com.bcxin.sync.dtos.response.BcxDepartmentResponse;
import com.bcxin.sync.entity.tuoluojiang.EbFrameEntity;
import com.bcxin.sync.service.RedisCache;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * description：陀螺匠-部门服务层
 * author：linchunpeng
 * date：2024/12/31
 */
@Slf4j
@Service
public class EbFrameService extends ServiceImpl<EbFrameMapper, EbFrameEntity> {

    @Autowired
    private RedisCache redisCache;

    /**
     * description：递归创建部门
     * author：linchunpeng
     * date：2025/1/6
     */
    @Transactional
    public Map<String, Long> frameCreateAll(Long entid, List<BcxDepartmentResponse> departmentTree, Long pid, StringBuffer syncResult) {
        log.info("递归创建部门");
        syncResult.append("\n递归创建部门");
        Map<String, Long> resultMap = new HashMap<>();
        //创建下级部门
        for (BcxDepartmentResponse response : departmentTree) {
            log.info("部门信息：{}，pid：{}", JSONObject.toJSONString(response), pid);
            Long frameId;
            if (pid == 0) {
                BcxDepartmentResponse rootDepartment = departmentTree.get(0);
                log.info("修改顶级部门信息");
                syncResult.append("\n修改顶级部门信息");
                frameId = this.updateTopFrame(entid, rootDepartment.getId(), rootDepartment.getDepartmentName());
                resultMap.put(rootDepartment.getId(), frameId);
            } else {
                log.info("创建部门信息");
                syncResult.append("\n创建部门信息");
                frameId = this.frameCreate(entid, response.getId(), response.getDepartmentName(), pid);
                resultMap.put(response.getId(), frameId);
            }
            log.info("frameId：{}", frameId);
            syncResult.append("\nframeId=").append(frameId);
            if (CollectionUtil.isNotEmpty(response.getChildren()) && frameId != null) {
                resultMap.putAll(this.frameCreateAll(entid, response.getChildren(), frameId, syncResult));
            }
        }
        return resultMap;
    }

    /**
     * description：创建部门
     * author：linchunpeng
     * date：2024/12/31
     */
    @Transactional
    public Long frameCreate(Long entid, String departmentId, String name, Long pid) {
        log.info("创建部门信息，name={}，departmentId={}, pid={}", name, departmentId, pid);
        EbFrameEntity entity;
        LambdaQueryChainWrapper<EbFrameEntity> lqw = this.lambdaQuery();
        lqw.eq(EbFrameEntity::getUniqued, departmentId);
        List<EbFrameEntity> list = lqw.list();
        if (CollectionUtil.isNotEmpty(list)) {
            log.info("已存在则更新");
            entity = list.get(0);
        } else {
            log.info("不存在则新增");
            entity = new EbFrameEntity();
            entity.setEntid(entid);
            entity.setUniqued(departmentId);
        }
        entity.setPid(pid);
        entity.setName(name);

        this.saveOrUpdate(entity);
        return entity.getId();
    }

    /**
     * description：删除部门
     * author：linchunpeng
     * date：2024/12/31
     */
    @Transactional
    public void frameDelete(String departmentId, String name) {
        LambdaQueryChainWrapper<EbFrameEntity> lqw = this.lambdaQuery();
        lqw.eq(EbFrameEntity::getUniqued, departmentId);
        List<EbFrameEntity> list = lqw.list();
        if (CollectionUtil.isNotEmpty(list)) {
            //已存在则更新
            EbFrameEntity entity = list.get(0);
            entity.setIsShow(0);
            entity.setName(name);
            this.saveOrUpdate(entity);
        }
    }

    /**
     * description：通过智能人事部门id获取陀螺匠部门id
     * author：linchunpeng
     * date：2024/12/31
     */
    public Long getIdByDepartmentId(String departmentId) {
        String key = "data-sync:department:frame-id:" + departmentId;
        Object value = redisCache.getCacheObject(key);
        if (value != null) {
            return Long.parseLong(value.toString());
        }
        LambdaQueryChainWrapper<EbFrameEntity> lqw = this.lambdaQuery();
        lqw.eq(EbFrameEntity::getUniqued, departmentId);
        EbFrameEntity entity = lqw.one();
        if (entity != null) {
            redisCache.setCacheObject(key, entity.getId(), 1, TimeUnit.DAYS);
            return entity.getId();
        }
        return null;
    }

    /**
     * description：修改顶级部门的信息
     * author：linchunpeng
     * date：2024/12/31
     */
    @Transactional
    public Long updateTopFrame(Long entid, String departmentId, String name) {
        LambdaQueryChainWrapper<EbFrameEntity> lqw = this.lambdaQuery();
        lqw.eq(EbFrameEntity::getEntid, entid);
        lqw.eq(EbFrameEntity::getPid, 0);
        EbFrameEntity entity = lqw.one();
        if (entity != null) {
            log.info("修改顶级部门信息，name={}，departmentId={}", name, departmentId);
            entity.setName(name);
            entity.setUniqued(departmentId);
            this.saveOrUpdate(entity);
            return entity.getId();
        }
        return null;
    }

    /**
     * description：修改顶级部门的信息
     * author：linchunpeng
     * date：2024/12/31
     */
    public String getTopUniquedFrameId(Long entid) {
        String key = "data-sync:department:top-frame-uniqued:" + entid;
        Object value = redisCache.getCacheObject(key);
        if (value != null) {
            return value.toString();
        }
        LambdaQueryChainWrapper<EbFrameEntity> lqw = this.lambdaQuery();
        lqw.eq(EbFrameEntity::getEntid, entid);
        lqw.eq(EbFrameEntity::getPid, 0);
        EbFrameEntity entity = lqw.one();
        if (entity != null) {
            redisCache.setCacheObject(key, entity.getUniqued(), 1, TimeUnit.DAYS);
            return entity.getUniqued();
        }
        return null;
    }

    /**
     * description：修改顶级部门的信息
     * author：linchunpeng
     * date：2024/12/31
     */
    public Long getTopIdFrameId(Long entid) {
        String key = "data-sync:department:top-frame-id:" + entid;
        Object value = redisCache.getCacheObject(key);
        if (value != null) {
            return Long.parseLong(value.toString());
        }
        LambdaQueryChainWrapper<EbFrameEntity> lqw = this.lambdaQuery();
        lqw.eq(EbFrameEntity::getEntid, entid);
        lqw.eq(EbFrameEntity::getPid, 0);
        EbFrameEntity entity = lqw.one();
        if (entity != null) {
            redisCache.setCacheObject(key, entity.getId(), 1, TimeUnit.DAYS);
            return entity.getId();
        }
        return null;
    }

    /**
     * description：根据百川信部门id列表，获取部门id列表的映射map
     * author：linchunpeng
     * date：2025/1/6
     */
    public Map<String, Long> getIdMapByDepartmentId(List<String> departmentIdList) {
        Map<String, Long> resultMap = new HashMap<>();
        LambdaQueryChainWrapper<EbFrameEntity> lqw = this.lambdaQuery();
        lqw.in(EbFrameEntity::getUniqued, departmentIdList);
        List<EbFrameEntity> list = lqw.list();
        if (CollectionUtil.isNotEmpty(list)) {
            for (EbFrameEntity ebFrameEntity : list) {
                resultMap.put(ebFrameEntity.getUniqued(), ebFrameEntity.getId());
            }
        }
        return resultMap;
    }



    /**
     * description：修改部门主管
     * author：linchunpeng
     * date：2024/12/31
     */
    @Transactional
    public void updateFrameUser(String departmentId, Long adminId) {
        LambdaQueryChainWrapper<EbFrameEntity> lqw = this.lambdaQuery();
        lqw.eq(EbFrameEntity::getUniqued, departmentId);
        EbFrameEntity entity = lqw.one();
        if (entity != null) {
            log.info("修改部门主管，name={}，adminId={}", entity.getName(), adminId);
            entity.setUserId(adminId);
            this.updateById(entity);
        }
    }
}