package com.bcxin.sync.service.tenant;

import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.extension.conditions.query.LambdaQueryChainWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bcxin.sync.dao.mapper.tenant.TenantDepartmentsMapper;
import com.bcxin.sync.dtos.response.BcxDepartmentResponse;
import com.bcxin.sync.entity.tenant.TenantDepartmentEmployeeRelationsEntity;
import com.bcxin.sync.entity.tenant.TenantDepartmentsEntity;
import com.bcxin.sync.service.RedisCache;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * description：智能人事-部门表
 * author：linchunpeng
 * date：2024/12/31
 */
@Slf4j
@Service
public class TenantDepartmentsService extends ServiceImpl<TenantDepartmentsMapper, TenantDepartmentsEntity> {

    @Autowired
    private RedisCache redisCache;

    /**
     * description：获取部门树
     * author：linchunpeng
     * date：2025/1/6
     */
    public List<BcxDepartmentResponse> getDepartmentTree(String organizationId, List<String> idList) {
        List<BcxDepartmentResponse> departmentResponseList = this.getDepartmentResponseListByOrganizationId(organizationId, idList);
        //组装成树形结构
        //找到所有的一级分类(也就是树的根节点)
        List<BcxDepartmentResponse> treeList = departmentResponseList.stream()
                .filter(response -> response.getParentId() == null)
                .peek((dept) -> dept.setChildren(getChildrens(dept, departmentResponseList)))
                .collect(Collectors.toList());
        //返回树结构的list
        return treeList;
    }

    /**
     * description：获取组织所有的部门列表
     * author：linchunpeng
     * date：2025/1/6
     */
    public List<BcxDepartmentResponse> getDepartmentResponseListByOrganizationId(String organizationId, List<String> idList) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("organizationId", organizationId);
        paramMap.put("idList", idList);
        List<BcxDepartmentResponse> list = new ArrayList<>();
        list.addAll(baseMapper.selectBcxDepartmentList(paramMap));
        //顶级部门
        TenantDepartmentsEntity rootDept = this.getRootDeptByOrganizationId(organizationId);
        List<BcxDepartmentResponse> externalGroupList = baseMapper.selectExternalGroupList(paramMap);
        if (rootDept != null && CollectionUtil.isNotEmpty(externalGroupList)) {
            for (BcxDepartmentResponse response : externalGroupList) {
                if (StringUtils.isBlank(response.getParentId())) {
                    //顶级外部组，全部挂在顶级部门底下
                    response.setParentId(rootDept.getId());
                }
            }
        }
        list.addAll(externalGroupList);
        return list;
    }

    //递归方法
    private List<BcxDepartmentResponse> getChildrens(BcxDepartmentResponse root, List<BcxDepartmentResponse> all) {
        List<BcxDepartmentResponse> children = all.stream()
                .filter(response -> response.getParentId() != null && response.getParentId().equals(root.getId()))
                .peek(response -> { //找子部门
                    response.setChildren(getChildrens(response, all));
                }).collect(Collectors.toList());
        return children;
    }

    /**
     * description：根据部门id获取组织Id
     * author：linchunpeng
     * date：2024/12/31
     */
    public String getOrgIdByDepartmentId(String departmentId) {
        String key = "data-sync:department:org-id:" + departmentId;
        String value = redisCache.getCacheObject(key);
        if (value != null) {
            return value;
        }
        TenantDepartmentsEntity departmentsEntity = this.getById(departmentId);
        if (departmentsEntity != null) {
            redisCache.setCacheObject(key, departmentsEntity.getOrganizationId(), 1, TimeUnit.DAYS);
            return departmentsEntity.getOrganizationId();
        }
        return null;
    }

    /**
     * description：获取顶级部门
     * author：linchunpeng
     * date：2025/3/27
     */
    public TenantDepartmentsEntity getRootDeptByOrganizationId(String organizationId) {
        LambdaQueryChainWrapper<TenantDepartmentsEntity> lqw = this.lambdaQuery();
        lqw.eq(TenantDepartmentsEntity::getOrganizationId, organizationId);
        lqw.isNull(TenantDepartmentsEntity::getParentId);
        List<TenantDepartmentsEntity> list = lqw.list();
        if (CollectionUtil.isNotEmpty(list)) {
            return list.get(0);
        }
        return null;
    }

}