package com.bcxin.sync.service.tenant;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DatePattern;
import cn.hutool.core.date.DateUnit;
import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.conditions.query.LambdaQueryChainWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bcxin.sync.common.utils.HuaweiIvmUtil;
import com.bcxin.sync.common.utils.IdGeneratorSnowflake;
import com.bcxin.sync.common.utils.ObsUtil;
import com.bcxin.sync.common.utils.PageUtils;
import com.bcxin.sync.configs.SyncConfig;
import com.bcxin.sync.dao.mapper.tenant.DeviceRecordMapper;
import com.bcxin.sync.dtos.request.DeviceRecordListRequest;
import com.bcxin.sync.dtos.response.BindDeviceResponse;
import com.bcxin.sync.entity.tenant.DeviceRecordEntity;
import com.bcxin.sync.service.RedisCache;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.io.File;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * description：设备录像表
 * author：linchunpeng
 * date：2025/5/28
 */
@Slf4j
@Service
public class DeviceRecordService extends ServiceImpl<DeviceRecordMapper, DeviceRecordEntity> {

    @Autowired
    private SyncConfig syncConfig;
    @Autowired
    private RedisCache redisCache;
    @Autowired
    private IdGeneratorSnowflake snowflake;

    public List<DeviceRecordEntity> getList(DeviceRecordListRequest request) {
        LambdaQueryChainWrapper<DeviceRecordEntity> lqw = this.lambdaQuery();
        if (StringUtils.isNotBlank(request.getDeviceNo())) {
            lqw.like(DeviceRecordEntity::getDeviceNo, request.getDeviceNo());
        }
        if (StringUtils.isNotBlank(request.getDeviceName())) {
            lqw.like(DeviceRecordEntity::getDeviceName, request.getDeviceName());
        }
        if (StringUtils.isNotBlank(request.getBindPlace())) {
            lqw.like(DeviceRecordEntity::getBindPlace, request.getBindPlace());
        }
        if (CollectionUtil.isNotEmpty(request.getOrgIdList()) && !"*".equals(request.getOrgIdList().get(0))) {
            lqw.in(DeviceRecordEntity::getBindOrgId, request.getOrgIdList());
        }
        if (CollectionUtil.isNotEmpty(request.getSupDepartIdList()) && !"*".equals(request.getSupDepartIdList().get(0))) {
            lqw.in(DeviceRecordEntity::getBindSupDepartId, request.getSupDepartIdList());
        }
        if (StringUtils.isNotBlank(request.getStartTimeStart())) {
            lqw.ge(DeviceRecordEntity::getStartTime, DateUtil.parseDateTime(request.getStartTimeStart()));
        }
        if (StringUtils.isNotBlank(request.getStartTimeEnd())) {
            lqw.le(DeviceRecordEntity::getStartTime, DateUtil.parseDateTime(request.getStartTimeEnd()));
        }
        PageUtils.startPage(request);//分页
        return lqw.list();
    }

    /**
     * description：查询昨天+今天所有的考试和设备信息
     * author：linchunpeng
     * date：2025/6/11
     */
    public List<BindDeviceResponse> getRecentlyExamDeviceList() {
        log.info("查询6天前+今天所有的考试和设备信息");
        Date now = new Date();
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("startTime", DateUtil.beginOfDay(DateUtil.offsetDay(now, -6)));
        paramMap.put("endTime", DateUtil.endOfDay(now));
        log.info("时间参数：{}", JSONObject.toJSONString(paramMap));
        return this.baseMapper.selectExamDeviceList(paramMap);
    }


    /**
     * description：查询昨天+今天所有的录像记录列表
     * author：linchunpeng
     * date：2025/6/11
     */
    public List<DeviceRecordEntity> getRecentlyDeviceRecordList() {
        log.info("查询6天前+今天所有的录像记录列表");
        Date now = new Date();
        LambdaQueryChainWrapper<DeviceRecordEntity> lqw = this.lambdaQuery();
        lqw.ge(DeviceRecordEntity::getStartTime, DateUtil.beginOfDay(DateUtil.offsetDay(now, -6)));
        lqw.le(DeviceRecordEntity::getStartTime, DateUtil.endOfDay(now));
        return lqw.list();
    }

    /**
     * description：开启华为云设备录像
     * author：linchunpeng
     * date：2025/5/29
     */
//    public void openHwDeviceRecord() {
//        log.info("设置今天的录像计划");
//        Date now = new Date();
//        Map<String, Object> paramMap = new HashMap<>();
//        paramMap.put("startTime", DateUtil.beginOfDay(now));
//        paramMap.put("endTime", DateUtil.endOfDay(now));
//        log.info("时间参数：{}", JSONObject.toJSONString(paramMap));
//        List<ExamDeviceResponse> examDeviceList = this.baseMapper.selectExamDeviceList(paramMap);
//        if (CollectionUtil.isNotEmpty(examDeviceList)) {
//            List<DeviceRecordEntity> entityList = new ArrayList<>();
//            log.info("存在有关联考场的设备，考试数量：{}", examDeviceList.size());
//            String accessToken = HuaweiIvmUtil.getAccessToken(syncConfig.getHuaweiIvm().getHost(), syncConfig.getHuaweiIvm().getUserId(), syncConfig.getHuaweiIvm().getAk(), syncConfig.getHuaweiIvm().getSk(), redisCache);
//
//            Map<String, List<ExamDeviceResponse>> examDeviceListMap = examDeviceList.stream().collect(Collectors.groupingBy(ExamDeviceResponse::getDeviceNo));
//            log.info("存在有关联考场的设备，设备数量：{}", examDeviceListMap.size());
//
//            for (Map.Entry<String, List<ExamDeviceResponse>> entry : examDeviceListMap.entrySet()) {
//                String deviceNo = entry.getKey();
//                log.info("开始设置华为云设备录像计划，设备编号：{}", deviceNo);
//                List<Date> startTimeList = new ArrayList<>();
//                List<Date> endTimeList = new ArrayList<>();
//                for (ExamDeviceResponse examDeviceResponse : entry.getValue()) {
//                    startTimeList.add(examDeviceResponse.getStartTime());
//                    endTimeList.add(examDeviceResponse.getEndTime());
//                }
//                //录像计划的时间点字符串，用来判断路线时间是否有有变化
//                String recordTimeValue = startTimeList.stream().map(DateUtil::formatDateTime).collect(Collectors.joining(";"))
//                        + "|" + endTimeList.stream().map(DateUtil::formatDateTime).collect(Collectors.joining(";"));
//                log.info("开始设置华为云设备录像计划，录像计划的时间点字符串：{}", recordTimeValue);
//                //看看今天的录像计划是否有变，如果没有变，就不需要重新设置了
//                String timeKey = deviceNo.concat("_").concat(DateUtil.formatDate(now));
//                String timeValue = redisCache.getCacheObject(timeKey);
//                if (StringUtils.isBlank(timeValue) || !recordTimeValue.equals(timeValue)) {
//                    log.info("缓存不存在 || 缓存的值和新的值不一致，需要重新设置监控的录像计划");
//                    HuaweiIvmUtil.setDeviceRecordPlan(syncConfig.getHuaweiIvm().getHost(), syncConfig.getHuaweiIvm().getUserId(),
//                            accessToken, deviceNo, startTimeList, endTimeList);
//
//                    log.info("删除录像记录");
//                    LambdaQueryChainWrapper<DeviceRecordEntity> lqw = this.lambdaQuery();
//                    lqw.eq(DeviceRecordEntity::getDeviceNo, deviceNo);
//                    lqw.ge(DeviceRecordEntity::getStartTime, now);
//                    this.baseMapper.delete(lqw);
//
//                    log.info("新增录像记录");
//                    for (ExamDeviceResponse examDeviceResponse : entry.getValue()) {
//                        if (examDeviceResponse.getStartTime().after(now)) {
//                            DeviceRecordEntity entity = new DeviceRecordEntity();
//                            entity.setId(snowflake.snowflakeId());
//                            entity.setDeviceNo(deviceNo);
//                            entity.setDeviceName(examDeviceResponse.getDeviceName());
//                            entity.setDeviceType(examDeviceResponse.getDeviceType());
//                            entity.setBindType(examDeviceResponse.getBindType());
//                            entity.setBindPlace(examDeviceResponse.getBindPlace());
//                            entity.setStartTime(examDeviceResponse.getStartTime());
//                            entity.setEndTime(examDeviceResponse.getEndTime());
//                            entity.setDurationTime((int)DateUtil.between(examDeviceResponse.getStartTime(), examDeviceResponse.getEndTime(), DateUnit.SECOND, true));
//                            entity.setCreateTime(now);
//                            entity.setUpdateTime(now);
//                            entityList.add(entity);
//                        }
//                    }
//                    //放入缓存
//                    redisCache.setCacheObject(timeKey, recordTimeValue, 1, TimeUnit.DAYS);
//                }
//            }
//
//            if (CollectionUtil.isNotEmpty(entityList)) {
//                log.info("保存入库");
//                this.saveBatch(entityList);
//            }
//        }
//    }

//    /**
//     * description：删除华为云设备录像
//     * author：linchunpeng
//     * date：2025/6/10
//     */
//    public void openHwDeviceRecordTask() {
//        Date now = new Date();
//        DateTime todayZero = DateUtil.beginOfDay(now);
//        LambdaQueryChainWrapper<DeviceRecordEntity> lqw = this.lambdaQuery();
//        lqw.lt(DeviceRecordEntity::getEndTime, todayZero);//小于今天0点
//        lqw.ge(DeviceRecordEntity::getEndTime, DateUtil.offsetDay(todayZero, -1) );//大于等于昨天0点
//        List<DeviceRecordEntity> list = lqw.list();
//        Set<String> deleteDeviceNoSet = new HashSet<>();
//        if (CollectionUtil.isNotEmpty(list)) {
//            for (DeviceRecordEntity entity : list) {
//                deleteDeviceNoSet.add(entity.getDeviceNo());
//            }
//        }
//        if (CollectionUtil.isNotEmpty(deleteDeviceNoSet)) {
//            String accessToken = HuaweiIvmUtil.getAccessToken(syncConfig.getHuaweiIvm().getHost(), syncConfig.getHuaweiIvm().getUserId(), syncConfig.getHuaweiIvm().getAk(), syncConfig.getHuaweiIvm().getSk(), redisCache);
//            for (String deviceNo : deleteDeviceNoSet) {
//                HuaweiIvmUtil.deleteDeviceRecordPlan(syncConfig.getHuaweiIvm().getHost(), syncConfig.getHuaweiIvm().getUserId(),
//                        accessToken, deviceNo, deviceNo);
//            }
//        }
//    }


    /**
     * description：异步新增录像记录，并下载录像视频
     * author：linchunpeng
     * date：2025/6/11
     */
    @Async(value = "taskExecutor")
    @Transactional
    public void insertRecord(BindDeviceResponse bindDeviceResponse) {
        log.info("新增录像记录，并下载录像视频，参数：{}", JSONObject.toJSONString(bindDeviceResponse));
        String key = "data-sync:download-hw-device:device_no-start_time:"
                + bindDeviceResponse.getDeviceNo() + "-" + DateUtil.format(bindDeviceResponse.getStartTime(), DatePattern.NORM_DATETIME_FORMAT);
        String value = redisCache.getCacheObject(key);
        if (StringUtils.isBlank(value)) {
            redisCache.setCacheObject(key, key, 24, TimeUnit.HOURS);
            log.info("key：{}，不存在，可以新增和下载", key);
            Date now = new Date();
            DeviceRecordEntity entity = new DeviceRecordEntity();
            entity.setId(snowflake.snowflakeId());
            entity.setDeviceNo(bindDeviceResponse.getDeviceNo());
            entity.setDeviceName(bindDeviceResponse.getDeviceName());
            entity.setDeviceType(bindDeviceResponse.getDeviceType());
            entity.setBindType(bindDeviceResponse.getBindType());
            entity.setBindPlaceId(bindDeviceResponse.getBindPlaceId());
            entity.setBindPlace(bindDeviceResponse.getBindPlace());
            entity.setBindOrgId(bindDeviceResponse.getBindOrgId());
            entity.setBindSupDepartId(bindDeviceResponse.getBindSupDepartId());
            entity.setStartTime(bindDeviceResponse.getStartTime());
            entity.setEndTime(bindDeviceResponse.getEndTime());
            entity.setDurationTime((int)DateUtil.between(bindDeviceResponse.getStartTime(), bindDeviceResponse.getEndTime(), DateUnit.SECOND, true));
            entity.setCreateTime(now);
            entity.setUpdateTime(now);

            //下载信息
            entity.setFileStatus(1);
            entity.setFileName(entity.getId().toString() + ".mp4");
            entity.setDownloadCount(1);
            this.save(entity);

            //启动下载
            this.startDownload(entity);
        } else {
            log.info("key：{}，已经存在，跳过本次任务", key);
        }
    }


    /**
     * description：异步重试下载录像视频
     * author：linchunpeng
     * date：2025/6/11
     */
    @Async(value = "taskExecutor")
    @Transactional
    public void retryDownload(DeviceRecordEntity entity) {
        log.info("重试下载录像视频，参数：{}", JSONObject.toJSONString(entity));
        String key = "data-sync:download-hw-device:device_no-start_time:"
                + entity.getDeviceNo() + "-" + DateUtil.format(entity.getStartTime(), DatePattern.NORM_DATETIME_FORMAT);
        String value = redisCache.getCacheObject(key);
        if (StringUtils.isBlank(value)) {
            redisCache.setCacheObject(key, key, 24, TimeUnit.HOURS);
            log.info("key：{}，不存在，可以重试下载", key);
            Date now = new Date();
            entity.setUpdateTime(now);
            //下载信息
            entity.setFileStatus(1);
            entity.setDownloadCount(entity.getDownloadCount()+1);
            this.updateById(entity);

            //启动下载
            this.startDownload(entity);
        } else {
            log.info("key：{}，已经存在，跳过本次任务", key);
        }
    }

    /**
     * description：启动下载
     * author：linchunpeng
     * date：2025/6/11
     */
    @Transactional
    public void startDownload(DeviceRecordEntity entity) {
        log.info("启动下载，参数：{}", JSONObject.toJSONString(entity));
        String accessToken = HuaweiIvmUtil.getAccessToken(syncConfig.getHuaweiIvm().getHost(), syncConfig.getHuaweiIvm().getUserId(), syncConfig.getHuaweiIvm().getAk(), syncConfig.getHuaweiIvm().getSk(), redisCache);
        //回放视频流地址
        String inputUrl = HuaweiIvmUtil.getPlaybackByDeviceId(syncConfig.getHuaweiIvm().getHost(),
                syncConfig.getHuaweiIvm().getUserId(), accessToken,
                entity.getDeviceNo(), entity.getDeviceNo(),
                DateUtil.formatDateTime(entity.getStartTime()), DateUtil.formatDateTime(entity.getEndTime()));
        //本地保存地址
        File saveDir = new File(syncConfig.getVideoPlaybackSavePath().substring(0, syncConfig.getVideoPlaybackSavePath().length() - 1));
        if (!saveDir.exists()) {
            saveDir.mkdir();
        }
        String outputPath = syncConfig.getVideoPlaybackSavePath().concat(entity.getFileName());
        //下载视频流，并保存
        boolean downloadResult = HuaweiIvmUtil.convertStreamToMp4(inputUrl, outputPath);
        Date now = new Date();
        if (downloadResult) {
            log.info("下载成功");
            String obsFileUrl = syncConfig.getVideoPlaybackSaveObsPath().concat(entity.getFileName());
            File videoFile = new File(outputPath);
            if (videoFile.exists()) {
                //上传视频到obs
                ObsUtil.uploadFile(videoFile, obsFileUrl);
            }
            entity.setFileStatus(2);
            entity.setFileUrl(syncConfig.getObsCdnUrl().concat(obsFileUrl));
            entity.setUpdateTime(now);
            this.updateById(entity);
        } else {
            log.info("下载失败");
            entity.setFileStatus(0);
            entity.setUpdateTime(now);
            this.updateById(entity);
        }
        String key = "data-sync:download-hw-device:device_no-start_time:"
                + entity.getDeviceNo() + "-" + DateUtil.format(entity.getStartTime(), DatePattern.NORM_DATETIME_FORMAT);
        redisCache.deleteObject(key);
    }
}