package com.bcxin.sync.service.datasync;

import com.baomidou.mybatisplus.extension.conditions.query.LambdaQueryChainWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bcxin.sync.common.emus.OpenType;
import com.bcxin.sync.common.utils.IdGeneratorSnowflake;
import com.bcxin.sync.dao.mapper.datasync.OrganizationOpenMapper;
import com.bcxin.sync.dtos.request.OrganizationOpenRequest;
import com.bcxin.sync.entity.datasync.OrganizationOpenEntity;
import com.bcxin.sync.service.RedisCache;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * description：数据同步-企业开通表服务层
 * author：linchunpeng
 * date：2025/1/17
 */
@Slf4j
@Service
public class OrganizationOpenService extends ServiceImpl<OrganizationOpenMapper, OrganizationOpenEntity> {

    @Autowired
    private OrganizationOpenLogService organizationOpenLogService;

    @Autowired
    private IdGeneratorSnowflake snowflake;

    @Autowired
    private RedisCache redisCache;

    /**
     * description：企业开通第三方系统
     * author：linchunpeng
     * date：2025/1/17
     */
    @Transactional
    public void openThird(OrganizationOpenRequest request) {
        OrganizationOpenEntity entity = this.getByOrganizationId(request.getOrganizationId(), request.getOpenType());
        if (entity == null) {
            entity = new OrganizationOpenEntity();
            entity.setId(snowflake.snowflakeId());
            entity.setOrganizationId(request.getOrganizationId());
            entity.setOrganizationName(request.getOrganizationName());
            entity.setOpenType(request.getOpenType());
            entity.setOpenTime(new Date());
            entity.setEndTime(entity.getOpenTime());
            entity.setIsInit(0);
        }
        Date endTime = request.getEndTime();
        if (endTime == null) {
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(entity.getEndTime());
            calendar.set(Calendar.DAY_OF_YEAR, calendar.get(Calendar.DAY_OF_YEAR) + request.getOpenDay());
            endTime = calendar.getTime();
        }
        if (endTime.after(entity.getEndTime())) {
            entity.setEndTime(endTime);
        }

        this.saveOrUpdate(entity);
        //创建开通日志
        organizationOpenLogService.createLog(request);

        //清除缓存
        String key1 = "data-sync:org:open-effective:"+request.getOpenType()+":" + request.getOrganizationId();
        redisCache.deleteObject(key1);
        String key2 = "data-sync:org:open:"+request.getOpenType()+":" + request.getOrganizationId();
        redisCache.deleteObject(key2);
    }

    /**
     * description：根据组织id查询开通记录
     * author：linchunpeng
     * date：2025/1/17
     */
    public OrganizationOpenEntity getByOrganizationId(String organizationId, int openType) {
        LambdaQueryChainWrapper<OrganizationOpenEntity> lqw = this.lambdaQuery();
        lqw.eq(OrganizationOpenEntity::getOrganizationId, organizationId);
        lqw.eq(OrganizationOpenEntity::getOpenType, openType);
        return lqw.one();
    }

    /**
     * description：根据组织id查询开通记录
     * author：linchunpeng
     * date：2025/1/17
     */
    public List<OrganizationOpenEntity> getByOrganizationIdList(List<String> organizationIdList, int openType) {
        LambdaQueryChainWrapper<OrganizationOpenEntity> lqw = this.lambdaQuery();
        lqw.in(OrganizationOpenEntity::getOrganizationId, organizationIdList);
        lqw.eq(OrganizationOpenEntity::getOpenType, openType);
        return lqw.list();
    }

    /**
     * description：获取未初始化数据的列表
     * author：linchunpeng
     * date：2025/1/17
     */
    public List<OrganizationOpenEntity> getNotInitList(int openType) {
        LambdaQueryChainWrapper<OrganizationOpenEntity> lqw = this.lambdaQuery();
        lqw.eq(OrganizationOpenEntity::getIsInit, 0);
        lqw.eq(OrganizationOpenEntity::getOpenType, openType);
        return lqw.list();
    }

    /**
     * description：判断是否开通，并且有效
     * author：linchunpeng
     * date：2025/1/17
     */
    public boolean isOpenAndEffective(String organizationId, int openType) {
        String key = "data-sync:org:open-effective:"+openType+":" + organizationId;
        Object value = redisCache.getCacheObject(key);
        if (value != null) {
            return Long.parseLong(value.toString()) > 0;
        }
        LambdaQueryChainWrapper<OrganizationOpenEntity> lqw = this.lambdaQuery();
        lqw.eq(OrganizationOpenEntity::getOrganizationId, organizationId);
        lqw.eq(OrganizationOpenEntity::getOpenType, openType);
        lqw.gt(OrganizationOpenEntity::getEndTime, new Date());
        Long count = lqw.count();
        redisCache.setCacheObject(key, count, 5, TimeUnit.MINUTES);
        return count > 0;
    }

    /**
     * description：判断是否开通
     * author：linchunpeng
     * date：2025/1/17
     */
    public boolean isOpen(String organizationId, int openType) {
        String key = "data-sync:org:open:"+openType+":" + organizationId;
        Object value = redisCache.getCacheObject(key);
        if (value != null) {
            return Long.parseLong(value.toString()) > 0;
        }
        LambdaQueryChainWrapper<OrganizationOpenEntity> lqw = this.lambdaQuery();
        lqw.eq(OrganizationOpenEntity::getOrganizationId, organizationId);
        lqw.eq(OrganizationOpenEntity::getOpenType, openType);
        Long count = lqw.count();
        redisCache.setCacheObject(key, count, 5, TimeUnit.MINUTES);
        return count > 0;
    }

    /**
     * description：判断是否开通，并且有效
     * author：linchunpeng
     * date：2025/1/17
     */
    public boolean isOpenAndEffective(List<String> organizationIdList, int openType) {
        for (String orgId : organizationIdList) {
            if (this.isOpenAndEffective(orgId, openType)) {
                return true;
            }
        }
        return false;
    }


    /**
     * description：获取jvs的列表
     * author：linchunpeng
     * date：2025/1/17
     */
    public List<OrganizationOpenEntity> getJvsList() {
        LambdaQueryChainWrapper<OrganizationOpenEntity> lqw = this.lambdaQuery();
        lqw.eq(OrganizationOpenEntity::getIsInit, 1);
        lqw.eq(OrganizationOpenEntity::getOpenType, OpenType.JVS.getCode());
        return lqw.list();
    }
}