package com.bcxin.sync.service;

import com.bcxin.sync.common.emus.OpenType;
import com.bcxin.sync.common.utils.AssertUtils;
import com.bcxin.sync.configs.SyncConfig;
import com.bcxin.sync.dtos.request.DepartmentSyncRequest;
import com.bcxin.sync.dtos.request.EmployeeSyncRequest;
import com.bcxin.sync.dtos.request.UserSyncRequest;
import com.bcxin.sync.dtos.response.BcxDepartmentResponse;
import com.bcxin.sync.dtos.response.BcxEmployeeResponse;
import com.bcxin.sync.entity.tenant.*;
import com.bcxin.sync.service.datasync.OrganizationOpenService;
import com.bcxin.sync.service.jvs.*;
import com.bcxin.sync.service.tenant.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.RequestBody;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * description：数据同步JVS服务层
 * author：linchunpeng
 * date：2025/2/11
 */
@Slf4j
@Service
public class DataSyncToJvsService {

    @Autowired
    private TenantOrganizationsService tenantOrganizationsService;
    @Autowired
    private TenantEmployeesService tenantEmployeesService;
    @Autowired
    private TenantDepartmentsService tenantDepartmentsService;
    @Autowired
    private TenantUsersService tenantUsersService;
    @Autowired
    private TenantUserCredentialsService tenantUserCredentialsService;
    @Autowired
    private TenantDepartmentEmployeeRelationsService tenantDepartmentEmployeeRelationsService;

    @Autowired
    private SysUserInfoService sysUserInfoService;
    @Autowired
    private SysUserTenantService sysUserTenantService;
    @Autowired
    private SysTenantService sysTenantService;
    @Autowired
    private SysDeptService sysDeptService;
    @Autowired
    private SysRoleService sysRoleService;

    @Autowired
    private OrganizationOpenService organizationOpenService;
    @Autowired
    private SyncConfig syncConfig;



    /**
     * description：开通JVS
     * author：linchunpeng
     * date：2025/2/11
     */
    public String sync(String organizationId) {
        AssertUtils.isTrue(organizationOpenService.isOpenAndEffective(organizationId, OpenType.JVS.getCode()), "该组织未开通，或者开通的权限已到期");
        TenantOrganizationsEntity organizationsEntity = tenantOrganizationsService.getById(organizationId);
        AssertUtils.isNotNullObject(organizationsEntity, "企业不存在");
        TenantEmployeesEntity adminEmployeeEntity = tenantEmployeesService.getAdminByOrgId(organizationsEntity.getId());
        TenantUsersEntity adminUserEntity = tenantUsersService.getById(adminEmployeeEntity.getTenantUserId());

        String jvsOrgId = organizationId;
        if (organizationId.equals(syncConfig.getJvsMainId())) {
            jvsOrgId = "1";
        }

        StringBuffer syncResult = new StringBuffer(1024);
        //同步用户
        List<BcxEmployeeResponse> employeeResponseList = tenantEmployeesService.getEmployeeResponseListByOrganizationId(organizationId, 0);
        syncResult.append(sysUserInfoService.createBatch(employeeResponseList));
        //同步企业
        syncResult.append(sysTenantService.create(jvsOrgId, organizationsEntity.getName(), adminUserEntity.getId(), adminUserEntity.getName()));
        //初始化知识库基础角色
        syncResult.append(sysRoleService.initDocumentRoleAndPermission(jvsOrgId));
        //同步部门
        syncResult.append("\n开始同步部门");
        List<BcxDepartmentResponse> departmentTree = tenantDepartmentsService.getDepartmentTree(organizationId, null);
        Map<String, String> deptMap = sysDeptService.createAll(jvsOrgId, departmentTree, jvsOrgId);
        syncResult.append("\n同步部门完成，部门数量：").append(deptMap.size());
        //查询部门-员工关联关系列表
        List<TenantDepartmentEmployeeRelationsEntity> departmentEmployeeRelationsEntityList = tenantDepartmentEmployeeRelationsService.getListByDepartmentIdList(deptMap.keySet());
        Map<String, List<TenantDepartmentEmployeeRelationsEntity>> employeeDepartmentMap = departmentEmployeeRelationsEntityList.stream().collect(Collectors.groupingBy(TenantDepartmentEmployeeRelationsEntity::getEmployeeId));
        //用户加入企业
        syncResult.append(sysUserTenantService.createBatch(jvsOrgId, employeeResponseList, deptMap, employeeDepartmentMap));

        return syncResult.toString();
    }


    /**
     * description：同步用户
     * author：linchunpeng
     * date：2025/2/11
     */
    public void syncUser(@RequestBody @Validated UserSyncRequest request) {
        for (String userId : request.getTenantUserIdList()) {
            TenantUsersEntity tenantUsersEntity = tenantUsersService.getById(userId);
            AssertUtils.isNotNullObject(tenantUsersEntity, "用户不存在");
            List<TenantUserCredentialsEntity> credentialsEntityList = tenantUserCredentialsService.getByTenantUserId(tenantUsersEntity.getId());
            AssertUtils.isNotEmptyList(credentialsEntityList, "用户的证件为空");
            String cardNo = credentialsEntityList.get(0).getNumber();
            for (TenantUserCredentialsEntity credentials : credentialsEntityList) {
                if (credentials.getCredentialType() == 0) {
                    //优先身份证
                    cardNo = credentials.getNumber();
                    break;
                }
            }
            sysUserInfoService.create(tenantUsersEntity.getId(), cardNo, tenantUsersEntity.getName(), tenantUsersEntity.getTelephone());
        }
    }


    /**
     * description：同步部门
     * author：linchunpeng
     * date：2025/2/11
     */
    public void syncDepartment(@RequestBody @Validated DepartmentSyncRequest request) {
        AssertUtils.isTrue(organizationOpenService.isOpenAndEffective(request.getOrganizationId(), OpenType.TLJ.getCode()), "该组织未开通，或者开通的权限已到期");
        List<BcxDepartmentResponse> departmentTree = tenantDepartmentsService.getDepartmentTree(request.getOrganizationId(), request.getDepartmentIdList());
        sysDeptService.createAll(request.getOrganizationId(), departmentTree, request.getOrganizationId());
    }

    /**
     * description：同步员工
     * author：linchunpeng
     * date：2025/2/11
     */
    public void syncEmployee(@RequestBody @Validated EmployeeSyncRequest request) {
        AssertUtils.isTrue(organizationOpenService.isOpenAndEffective(request.getOrganizationId(), OpenType.TLJ.getCode()), "该组织未开通，或者开通的权限已到期");
        //先同步用户
        List<BcxEmployeeResponse> employeeResponseList = tenantEmployeesService.getEmployeeResponseListByEmployeeIdList(request.getEmployeeIdList(), null);
        for (BcxEmployeeResponse employeeResponse : employeeResponseList) {
            sysUserInfoService.create(employeeResponse.getUserId(), employeeResponse.getCredentialNumber(), employeeResponse.getEmployeeName(), employeeResponse.getTelephone());
        }
        //查询部门-员工关联关系列表
        List<TenantDepartmentEmployeeRelationsEntity> departmentEmployeeRelationsEntityList = tenantDepartmentEmployeeRelationsService.getListByEmployeeIdList(request.getEmployeeIdList());
        Map<String, List<TenantDepartmentEmployeeRelationsEntity>> employeeDepartmentMap = departmentEmployeeRelationsEntityList.stream().collect(Collectors.groupingBy(TenantDepartmentEmployeeRelationsEntity::getEmployeeId));
        List<String> departmentIdList = departmentEmployeeRelationsEntityList.stream().map(TenantDepartmentEmployeeRelationsEntity::getDepartmentId).collect(Collectors.toList());
        List<BcxDepartmentResponse> departmentResponseList = tenantDepartmentsService.getDepartmentResponseListByOrganizationId(request.getOrganizationId(), departmentIdList);
        Map<String, String> deptMap = departmentResponseList.stream().collect(Collectors.toMap(BcxDepartmentResponse::getId, BcxDepartmentResponse::getDepartmentName));
        //用户加入企业
        for (BcxEmployeeResponse employeeResponse : employeeResponseList) {
            sysUserTenantService.createAndSave(request.getOrganizationId(), employeeResponse.getUserId(), employeeResponse.getId(),
                    employeeResponse.getEmployeeName(), employeeResponse.getTelephone(), employeeResponse.getHiredDate(),
                    employeeDepartmentMap.get(employeeResponse.getId()), deptMap, null);
        }
    }

}
