'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

const unplugin = require('unplugin');
const WebpackSources = require('webpack-sources');
const pluginutils = require('@rollup/pluginutils');
const config = require('@unocss/config');
const core = require('@unocss/core');
const fs = require('node:fs/promises');
const node_path = require('node:path');
const fg = require('fast-glob');
const MagicString = require('magic-string');
const remapping = require('@ampproject/remapping');
const node_crypto = require('node:crypto');

function _interopDefaultLegacy (e) { return e && typeof e === 'object' && 'default' in e ? e["default"] : e; }

const WebpackSources__default = /*#__PURE__*/_interopDefaultLegacy(WebpackSources);
const fs__default = /*#__PURE__*/_interopDefaultLegacy(fs);
const fg__default = /*#__PURE__*/_interopDefaultLegacy(fg);
const MagicString__default = /*#__PURE__*/_interopDefaultLegacy(MagicString);
const remapping__default = /*#__PURE__*/_interopDefaultLegacy(remapping);

const INCLUDE_COMMENT = "@unocss-include";
const IGNORE_COMMENT = "@unocss-ignore";
const CSS_PLACEHOLDER = "@unocss-placeholder";

const defaultExclude = [core.cssIdRE];
const defaultInclude = [/\.(vue|svelte|[jt]sx|mdx?|astro|elm|php|phtml|html)($|\?)/];

function createContext(configOrPath, defaults = {}, extraConfigSources = [], resolveConfigResult = () => {
}) {
  let root = process.cwd();
  let rawConfig = {};
  let configFileList = [];
  const uno = core.createGenerator(rawConfig, defaults);
  let rollupFilter = pluginutils.createFilter(defaultInclude, defaultExclude);
  const invalidations = [];
  const reloadListeners = [];
  const modules = new core.BetterMap();
  const tokens = /* @__PURE__ */ new Set();
  const tasks = [];
  const affectedModules = /* @__PURE__ */ new Set();
  let ready = reloadConfig();
  async function reloadConfig() {
    const result = await config.loadConfig(root, configOrPath, extraConfigSources, defaults);
    resolveConfigResult(result);
    rawConfig = result.config;
    configFileList = result.sources;
    uno.setConfig(rawConfig);
    uno.config.envMode = "dev";
    rollupFilter = pluginutils.createFilter(
      rawConfig.include || defaultInclude,
      rawConfig.exclude || defaultExclude
    );
    tokens.clear();
    await Promise.all(modules.map((code, id) => uno.applyExtractors(code, id, tokens)));
    invalidate();
    dispatchReload();
    const presets = /* @__PURE__ */ new Set();
    uno.config.presets.forEach((i) => {
      if (!i.name)
        return;
      if (presets.has(i.name))
        console.warn(`[unocss] duplication of preset ${i.name} found, there might be something wrong with your config.`);
      else
        presets.add(i.name);
    });
    return result;
  }
  async function updateRoot(newRoot) {
    if (newRoot !== root) {
      root = newRoot;
      ready = reloadConfig();
    }
    return await ready;
  }
  function invalidate() {
    invalidations.forEach((cb) => cb());
  }
  function dispatchReload() {
    reloadListeners.forEach((cb) => cb());
  }
  async function extract(code, id) {
    if (id)
      modules.set(id, code);
    const len = tokens.size;
    await uno.applyExtractors(code, id, tokens);
    if (tokens.size > len)
      invalidate();
  }
  function filter(code, id) {
    if (code.includes(IGNORE_COMMENT))
      return false;
    return code.includes(INCLUDE_COMMENT) || code.includes(CSS_PLACEHOLDER) || rollupFilter(id.replace(/\?v=\w+$/, ""));
  }
  async function getConfig() {
    await ready;
    return rawConfig;
  }
  async function flushTasks() {
    const _tasks = [...tasks];
    await Promise.all(_tasks);
    tasks.splice(0, _tasks.length);
  }
  return {
    get ready() {
      return ready;
    },
    tokens,
    modules,
    affectedModules,
    tasks,
    flushTasks,
    invalidate,
    onInvalidate(fn) {
      invalidations.push(fn);
    },
    filter,
    reloadConfig,
    onReload(fn) {
      reloadListeners.push(fn);
    },
    uno,
    extract,
    getConfig,
    root,
    updateRoot,
    getConfigFileList: () => configFileList
  };
}

async function applyTransformers(ctx, original, id, enforce = "default") {
  if (original.includes(IGNORE_COMMENT))
    return;
  const transformers = (ctx.uno.config.transformers || []).filter((i) => (i.enforce || "default") === enforce);
  if (!transformers.length)
    return;
  let code = original;
  let s = new MagicString__default(code);
  const maps = [];
  for (const t of transformers) {
    if (t.idFilter) {
      if (!t.idFilter(id))
        continue;
    } else if (!ctx.filter(code, id)) {
      continue;
    }
    await t.transform(s, id, ctx);
    if (s.hasChanged()) {
      code = s.toString();
      maps.push(s.generateMap({ hires: true, source: id }));
      s = new MagicString__default(code);
    }
  }
  if (code !== original) {
    ctx.affectedModules.add(id);
    return {
      code,
      map: remapping__default(maps, () => null)
    };
  }
}

async function setupExtraContent(ctx, shouldWatch = false) {
  const { extraContent } = await ctx.getConfig();
  const { extract, tasks, root, filter } = ctx;
  if (extraContent?.plain) {
    await Promise.all(
      extraContent.plain.map((code, idx) => {
        return extract(code, `__extra_content_${idx}__`);
      })
    );
  }
  if (extraContent?.filesystem) {
    const files = await fg__default(extraContent.filesystem, { cwd: root });
    async function extractFile(file) {
      const code = await fs__default.readFile(file, "utf-8");
      if (!filter(code, file))
        return;
      const preTransform = await applyTransformers(ctx, code, file, "pre");
      const defaultTransform = await applyTransformers(ctx, preTransform?.code || code, file);
      await applyTransformers(ctx, defaultTransform?.code || preTransform?.code || code, file, "post");
      return await extract(preTransform?.code || code, file);
    }
    if (shouldWatch) {
      const { watch } = await import('chokidar');
      const ignored = ["**/{.git,node_modules}/**"];
      const watcher = watch(files, {
        ignorePermissionErrors: true,
        ignored,
        cwd: root,
        ignoreInitial: true
      });
      watcher.on("all", (type, file) => {
        if (type === "add" || type === "change") {
          const absolutePath = node_path.resolve(root, file);
          tasks.push(extractFile(absolutePath));
        }
      });
    }
    await Promise.all(files.map(extractFile));
  }
}

function getHash(input, length = 8) {
  return node_crypto.createHash("sha256").update(input).digest("hex").slice(0, length);
}

const VIRTUAL_ENTRY_ALIAS = [
  /^(?:virtual:)?uno(?::(.+))?\.css(\?.*)?$/
];
const LAYER_MARK_ALL = "__ALL__";
const RESOLVED_ID_WITH_QUERY_RE = /[\/\\]__uno(?:(_.*?))?\.css(\?.*)?$/;
const RESOLVED_ID_RE = /[\/\\]__uno(?:(_.*?))?\.css$/;
function resolveId(id) {
  if (id.match(RESOLVED_ID_WITH_QUERY_RE))
    return id;
  for (const alias of VIRTUAL_ENTRY_ALIAS) {
    const match = id.match(alias);
    if (match) {
      return match[1] ? `/__uno_${match[1]}.css` : "/__uno.css";
    }
  }
}
function resolveLayer(id) {
  const match = id.match(RESOLVED_ID_RE);
  if (match)
    return match[1] || LAYER_MARK_ALL;
}
const LAYER_PLACEHOLDER_RE = /(\\?")?#--unocss--\s*{\s*layer\s*:\s*(.+?);?\s*}/g;
function getLayerPlaceholder(layer) {
  return `#--unocss--{layer:${layer}}`;
}
const HASH_PLACEHOLDER_RE = /#--unocss-hash--\s*{\s*content\s*:\s*\\*"(.+?)\\*";?\s*}/g;
function getHashPlaceholder(hash) {
  return `#--unocss-hash--{content:"${hash}"}`;
}

function getPath(id) {
  return id.replace(/\?.*$/, "");
}
function isCssId(id) {
  return core.cssIdRE.test(id);
}

const PLUGIN_NAME = "unocss:webpack";
const UPDATE_DEBOUNCE = 10;
function defineConfig(config) {
  return config;
}
function WebpackPlugin(configOrPath, defaults) {
  return unplugin.createUnplugin(() => {
    const ctx = createContext(configOrPath, defaults);
    const { uno, tokens, filter, extract, onInvalidate, tasks, flushTasks } = ctx;
    let timer;
    onInvalidate(() => {
      clearTimeout(timer);
      timer = setTimeout(updateModules, UPDATE_DEBOUNCE);
    });
    const nonPreTransformers = ctx.uno.config.transformers?.filter((i) => i.enforce !== "pre");
    if (nonPreTransformers?.length) {
      console.warn(
        '[unocss] webpack integration only supports "pre" enforce transformers currently.the following transformers will be ignored\n' + nonPreTransformers.map((i) => ` - ${i.name}`).join("\n")
      );
    }
    tasks.push(setupExtraContent(ctx));
    const entries = /* @__PURE__ */ new Set();
    const hashes = /* @__PURE__ */ new Map();
    const plugin = {
      name: "unocss:webpack",
      enforce: "pre",
      transformInclude(id) {
        return filter("", id) && !id.match(/\.html$/) && !RESOLVED_ID_RE.test(id);
      },
      async transform(code, id) {
        const result = await applyTransformers(ctx, code, id, "pre");
        if (isCssId(id))
          return result;
        if (result == null)
          tasks.push(extract(code, id));
        else
          tasks.push(extract(result.code, id));
        return result;
      },
      resolveId(id) {
        const entry = resolveId(id);
        if (entry === id)
          return;
        if (entry) {
          let query = "";
          const queryIndex = id.indexOf("?");
          if (queryIndex >= 0)
            query = id.slice(queryIndex);
          entries.add(entry);
          return entry + query;
        }
      },
      loadInclude(id) {
        const layer = getLayer(id);
        return !!layer;
      },
      load(id) {
        const layer = getLayer(id);
        const hash = hashes.get(id);
        if (layer)
          return (hash ? getHashPlaceholder(hash) : "") + getLayerPlaceholder(layer);
      },
      webpack(compiler) {
        compiler.hooks.compilation.tap(PLUGIN_NAME, (compilation) => {
          compilation.hooks.optimizeAssets.tapPromise(PLUGIN_NAME, async () => {
            const files = Object.keys(compilation.assets);
            await flushTasks();
            const result = await uno.generate(tokens, { minify: true });
            for (const file of files) {
              if (file === "*")
                return;
              let code = compilation.assets[file].source().toString();
              let replaced = false;
              code = code.replace(HASH_PLACEHOLDER_RE, "");
              code = code.replace(LAYER_PLACEHOLDER_RE, (_, quote, layer) => {
                replaced = true;
                const css = layer === LAYER_MARK_ALL ? result.getLayers(void 0, Array.from(entries).map((i) => resolveLayer(i)).filter((i) => !!i)) : result.getLayer(layer) || "";
                if (!quote)
                  return css;
                let escaped = JSON.stringify(css).slice(1, -1);
                if (quote === '\\"')
                  escaped = JSON.stringify(escaped).slice(1, -1);
                return quote + escaped;
              });
              if (replaced)
                compilation.assets[file] = new WebpackSources__default.RawSource(code);
            }
          });
        });
      }
    };
    let lastTokenSize = tokens.size;
    async function updateModules() {
      if (!plugin.__vfsModules)
        return;
      await flushTasks();
      const result = await uno.generate(tokens);
      if (lastTokenSize === tokens.size)
        return;
      lastTokenSize = tokens.size;
      Array.from(plugin.__vfsModules).forEach((id) => {
        const path = decodeURIComponent(id.slice(plugin.__virtualModulePrefix.length));
        const layer = resolveLayer(path);
        if (!layer)
          return;
        const code = layer === LAYER_MARK_ALL ? result.getLayers(void 0, Array.from(entries).map((i) => resolveLayer(i)).filter((i) => !!i)) : result.getLayer(layer) || "";
        const hash = getHash(code);
        hashes.set(path, hash);
        plugin.__vfs.writeModule(id, code);
      });
    }
    return plugin;
  }).webpack();
}
function getLayer(id) {
  let layer = resolveLayer(getPath(id));
  if (!layer) {
    const entry = resolveId(id);
    if (entry)
      layer = resolveLayer(entry);
  }
  return layer;
}

exports["default"] = WebpackPlugin;
exports.defineConfig = defineConfig;
