import { dirname, resolve } from 'node:path';
import { fileURLToPath } from 'node:url';
import { defineNuxtModule, addPluginTemplate, isNuxt2, addComponentsDir, isNuxt3, extendWebpackConfig } from '@nuxt/kit';
import WebpackPlugin from '@unocss/webpack';
import VitePlugin from '@unocss/vite';
import { loadConfig } from '@unocss/config';
import presetUno from '@unocss/preset-uno';
import presetAttributify from '@unocss/preset-attributify';
import presetIcons from '@unocss/preset-icons';
import presetWebFonts from '@unocss/preset-web-fonts';
import presetTypography from '@unocss/preset-typography';
import presetTagify from '@unocss/preset-tagify';
import presetWind from '@unocss/preset-wind';
import { cssIdRE } from '@unocss/core';

const defaultExclude = [cssIdRE];

function resolveOptions(options) {
  if (options.presets == null) {
    options.presets = [];
    const presetMap = {
      uno: presetUno,
      attributify: presetAttributify,
      tagify: presetTagify,
      icons: presetIcons,
      webFonts: presetWebFonts,
      typography: presetTypography,
      wind: presetWind
    };
    for (const [key, preset] of Object.entries(presetMap)) {
      const option = options[key];
      if (option)
        options.presets.push(preset(typeof option === "boolean" ? {} : option));
    }
  }
  options.exclude = options.exclude || defaultExclude;
  if (Array.isArray(options.exclude))
    options.exclude.push(/\?macro=true/);
}

const dir = dirname(fileURLToPath(import.meta.url));
const index = defineNuxtModule({
  meta: {
    name: "unocss",
    configKey: "unocss"
  },
  defaults: {
    mode: "global",
    autoImport: true,
    preflight: false,
    components: true,
    uno: true,
    attributify: false,
    webFonts: false,
    icons: false,
    wind: false
  },
  async setup(options, nuxt) {
    resolveOptions(options);
    if (options.autoImport) {
      addPluginTemplate({
        filename: "unocss.mjs",
        getContents: () => {
          const lines = [
            "import 'uno.css'",
            isNuxt2() ? "export default () => {}" : "import { defineNuxtPlugin } from '#imports'; export default defineNuxtPlugin(() => {})"
          ];
          if (options.preflight)
            lines.unshift("import '@unocss/reset/tailwind.css'");
          return lines.join("\n");
        }
      });
    }
    if (options.components) {
      addComponentsDir({
        path: resolve(dir, "../runtime"),
        watch: false
      });
    }
    const { config: unoConfig } = await loadConfig(process.cwd(), {
      configFile: options.configFile
    }, [], options);
    if (isNuxt3() && nuxt.options.builder === "@nuxt/vite-builder" && nuxt.options.postcss.plugins.cssnano && unoConfig.transformers?.some((t) => t.name === "@unocss/transformer-directives" && t.enforce !== "pre")) {
      const preset = nuxt.options.postcss.plugins.cssnano.preset;
      nuxt.options.postcss.plugins.cssnano = {
        preset: [preset?.[0] || "default", Object.assign(
          preset?.[1] || {},
          { mergeRules: false, normalizeWhitespace: false, discardComments: false }
        )]
      };
    }
    nuxt.hook("vite:extend", ({ config }) => {
      config.plugins = config.plugins || [];
      config.plugins.unshift(...VitePlugin({ mode: options.mode }, unoConfig));
    });
    if (nuxt.options.dev) {
      nuxt.hook("devtools:customTabs", (tabs) => {
        tabs.push({
          title: "UnoCSS",
          name: "unocss",
          icon: "/__unocss/favicon.svg",
          view: {
            type: "iframe",
            src: "/__unocss/"
          }
        });
      });
    }
    if (isNuxt2()) {
      nuxt.hook("app:resolve", (config) => {
        const plugin = { src: "unocss.mjs", mode: "client" };
        if (config.plugins)
          config.plugins.push(plugin);
        else
          config.plugins = [plugin];
      });
    }
    extendWebpackConfig((config) => {
      config.plugins = config.plugins || [];
      config.plugins.unshift(WebpackPlugin({}, unoConfig));
    });
  }
});

export { index as default };
