package com.bcxin.rbac.domain.services.impls;

import com.bcxin.Infrastructures.TenantContext;
import com.bcxin.Infrastructures.UnitWork;
import com.bcxin.Infrastructures.components.EventDispatcher;
import com.bcxin.Infrastructures.exceptions.BadTenantException;
import com.bcxin.Infrastructures.exceptions.NotFoundTenantException;
import com.bcxin.rbac.domain.entities.RoleEntity;
import com.bcxin.rbac.domain.entities.SubjectEntity;
import com.bcxin.rbac.domain.entities.UserEntity;
import com.bcxin.rbac.domain.events.EnsureSubjectEvent;
import com.bcxin.rbac.domain.repositories.RoleRepository;
import com.bcxin.rbac.domain.repositories.UserRepository;
import com.bcxin.rbac.domain.services.UserService;
import com.bcxin.rbac.domain.services.commands.users.AssignUserCommand;
import org.springframework.stereotype.Service;

import java.util.Collection;
import java.util.Date;
import java.util.Optional;
import java.util.stream.Collectors;

@Service
public class UserServiceImpl implements UserService {
    private final UserRepository userRepository;
    private final RoleRepository roleRepository;
    private final EventDispatcher eventDispatcher;
    private final UnitWork unitWork;

    public UserServiceImpl(UserRepository userRepository,
                           RoleRepository roleRepository,
                           EventDispatcher eventDispatcher,
                           UnitWork unitWork) {
        this.userRepository = userRepository;
        this.roleRepository = roleRepository;
        this.eventDispatcher = eventDispatcher;
        this.unitWork = unitWork;
    }

    @Override
    public void dispatch(AssignUserCommand command) {
        command.validate();
        EnsureSubjectEvent ensureSubjectEvent =
                EnsureSubjectEvent.create(command.getSubjectReferencedId(), String.format("组织:%s", new Date()));
        this.eventDispatcher.dispatch(ensureSubjectEvent);


        SubjectEntity subject =
                TenantContext.getInstance().getValueAndRemove(ensureSubjectEvent);

        if (subject == null) {
            throw new BadTenantException("组织(Subject)初始化出错");
        }

        Collection<String> roleIds = command.getItems().stream().flatMap(ii -> ii.getRoleIds().stream())
                .collect(Collectors.toList());
        Collection<RoleEntity> wholeRoles = this.roleRepository.getByIds(roleIds);

        Collection<String> notExistRoleIds = roleIds.stream().filter(ii -> !wholeRoles.stream().anyMatch(ix -> ix.getId().equals(ii))).collect(Collectors.toList());
        if (notExistRoleIds.size() > 0) {
            throw new NotFoundTenantException(String.format("找不到角色(%s)信息", notExistRoleIds));
        }

        Collection<String> userReferencedIds = command.getItems().stream().map(ii -> ii.getReferencedId()).collect(Collectors.toList());
        Collection<UserEntity> wholeUsers = this.userRepository.getByIds(command.getSubjectReferencedId(), userReferencedIds);

        this.unitWork.executeTran(() -> {
            for (AssignUserCommand.UserCommandItem commandItem : command.getItems()) {
                Optional<UserEntity> userOptional = wholeUsers.stream().filter(ii -> ii.getReferencedId().equals(commandItem.getReferencedId())).findFirst();
                UserEntity user = null;
                if (!userOptional.isPresent()) {
                    user = UserEntity.create(subject, commandItem.getReferencedId(), commandItem.getName());
                } else {
                    user = userOptional.get();
                }

                Collection<RoleEntity> selectedRoles = wholeRoles.stream().filter(ii -> commandItem.getRoleIds().contains(ii)).collect(Collectors.toList());
                user.assignRoles(selectedRoles);

                this.userRepository.save(user);
            }
        });
    }
}
