package com.bcxin.rbac.domain.services.custom;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSONObject;
import com.bcxin.Infrastructures.UnitWork;
import com.bcxin.Infrastructures.entities.OperatorValueType;
import com.bcxin.Infrastructures.exceptions.BadTenantException;
import com.bcxin.Infrastructures.utils.AuthUtil;
import com.bcxin.Infrastructures.utils.RedisUtil;
import com.bcxin.Infrastructures.utils.SnowFlakeUtil;
import com.bcxin.rbac.domain.RbacCustomDbReader;
import com.bcxin.rbac.domain.dtos.custom.*;
import com.bcxin.rbac.domain.entities.*;
import com.bcxin.rbac.domain.repositories.custom.*;
import lombok.AllArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

@Service
@AllArgsConstructor
public class RbacPermitAppService {
    private final RbacPermitAppRepository rbacPermitAppRepository;
    private final RbacPermitOptionRepository rbacPermitOptionRepository;
    private final RbacPermitAppRoleRepository rbacPermitAppRoleRepository;
    private final RbacPermitOptionRoleRepository rbacPermitOptionRoleRepository;
    private final RbacPermitAppUserRepository rbacPermitAppUserRepository;
    private final RbacPermitOptionUserRepository rbacPermitOptionUserRepository;
    private final RbacCustomDbReader rbacCustomDbReader;
    private final RedisUtil redisUtil;
    private final UnitWork unitWork;

    private final static String RBAC_PERMIT_APP_REDIS_KEY = "bcxin:rbac:permit:app:%s";
    private final static String RBAC_PERMIT_OPTION_REDIS_KEY = "bcxin:rbac:permit:option:%s";

    /**
     * description：查询企业授权的app列表
     * author：linchunpeng
     * date：2024/2/3
     */
    public List<RbacPermitAppDto> findOrgAppList(String organizationId) {
        Object redisCache = redisUtil.get(String.format(RBAC_PERMIT_APP_REDIS_KEY, organizationId));
        if (redisCache != null) {
            return JSONObject.parseArray(redisCache.toString(), RbacPermitAppDto.class);
        }
        List<RbacPermitAppDto> list = rbacCustomDbReader.findOrgAppList(organizationId);
        //放入缓存
        redisUtil.set(String.format(RBAC_PERMIT_APP_REDIS_KEY, organizationId), JSONObject.toJSONString(list), 300);
        return list;
    }


    /**
     * description：查询应用的操作项列表
     * author：linchunpeng
     * date：2024/2/3
     */
    public List<RbacPermitOptionDto> findAppOptionList(Long appId) {
        Object redisCache = redisUtil.get(String.format(RBAC_PERMIT_OPTION_REDIS_KEY, appId));
        if (redisCache != null) {
            return JSONObject.parseArray(redisCache.toString(), RbacPermitOptionDto.class);
        }
        List<RbacPermitOptionDto> list = rbacCustomDbReader.findAppOptionList(appId);
        //放入缓存
        redisUtil.set(String.format(RBAC_PERMIT_OPTION_REDIS_KEY, appId), JSONObject.toJSONString(list), 300);
        return list;
    }

    /**
     * description：角色/职员，功能授权
     * author：linchunpeng
     * date：2024/2/4
     */
    public void authPermitOption(RbacPermitAuthRequestDto request) {
        if (request.getRoleId() == null && CollectionUtil.isEmpty(request.getEmployeeIdList())) {
            throw new BadTenantException("角色id和职员id列表，至少需要传一个");
        }
        //授权应用权限类型
        Map<Long, Integer> appPermitTypeMap = new HashMap<>();
        List<RbacPermitAppDto> appDtoList = this.findOrgAppList(request.getOrganizationId());
        List<Long> collect = appDtoList.stream().map(RbacPermitAppDto::getAppId).collect(Collectors.toList());
        Collections.sort(collect);
        Map<Long, RbacPermitAppDto> appMap = appDtoList.stream().collect(Collectors.toMap(RbacPermitAppDto::getAppId, Function.identity()));
        Map<Long, RbacPermitOptionDto> optionMap = new HashMap<>();
        //初始化授权相关数据集合
        initAuthDataMap(request, appPermitTypeMap, appMap, optionMap);
        OperatorValueType currentOperator = AuthUtil.getCurrentOperator();
        if (request.getRoleId() != null) {
            //给角色授权
            authToRole(request, appPermitTypeMap, appMap, optionMap, currentOperator);
        } else {
            //给职员授权
            for (String employeeId : request.getEmployeeIdList()) {
                authToUser(request, employeeId, appPermitTypeMap, appMap, optionMap, currentOperator);
            }
        }
    }

    /**
     * description：初始化授权相关数据集合
     * author：linchunpeng
     * date：2024/2/4
     */
    private void initAuthDataMap(RbacPermitAuthRequestDto request, Map<Long, Integer> appPermitTypeMap, Map<Long, RbacPermitAppDto> appMap, Map<Long, RbacPermitOptionDto> optionMap) {
        if (CollectionUtil.isNotEmpty(request.getAppList())) {
            for (RbacPermitAppAuthRequestDto authRequestDto : request.getAppList()) {
                if (authRequestDto.getAppId() == null) {
                    throw new BadTenantException("授权应用id不能为空");
                }
                if (CollectionUtil.isEmpty(authRequestDto.getOptionList())) {
                    throw new BadTenantException("授权操作项列表不能为空");
                }
                if (appMap.get(authRequestDto.getAppId()) == null) {
                    throw new BadTenantException(String.format("授权应用id:%s，不存在对应的应用", authRequestDto.getAppId()));
                }
                List<RbacPermitOptionDto> optionDtoList = this.findAppOptionList(authRequestDto.getAppId());
                optionMap.putAll(optionDtoList.stream().collect(Collectors.toMap(RbacPermitOptionDto::getOptionId, Function.identity())));

                //是否授权web
                boolean isAuthWeb = false;
                //是否授权app
                boolean isAuthApp = false;
                for (RbacPermitOptionAuthRequestDto optionRequestDto : authRequestDto.getOptionList()) {
                    RbacPermitOptionDto optionDto = optionMap.get(optionRequestDto.getOptionId());
                    if (optionDto == null) {
                        throw new BadTenantException(String.format("授权操作项id:%s，不存在对应的操作项", optionRequestDto.getOptionId()));
                    }
                    if (optionDto.getPermitAppId().longValue() != authRequestDto.getAppId().longValue()) {
                        throw new BadTenantException(String.format("授权应用id:%s与授权操作项id:%s，不匹配", authRequestDto.getAppId(), optionRequestDto.getOptionId()));
                    }
                    if (optionDto.getOptionType() == 1) {
                        isAuthWeb = true;
                    } else if (optionDto.getOptionType() == 2) {
                        isAuthApp = true;
                    }
                }
                if (isAuthWeb && isAuthApp) {
                    //都有
                    appPermitTypeMap.put(authRequestDto.getAppId(), 3);
                } else if (isAuthWeb) {
                    //web
                    appPermitTypeMap.put(authRequestDto.getAppId(), 1);
                } else if (isAuthApp) {
                    //app
                    appPermitTypeMap.put(authRequestDto.getAppId(), 2);
                }
            }
        }
    }

    /**
     * description：给角色授权
     * author：linchunpeng
     * date：2024/2/4
     */
    private void authToRole(RbacPermitAuthRequestDto request, Map<Long, Integer> appPermitTypeMap,
                            Map<Long, RbacPermitAppDto> appMap, Map<Long, RbacPermitOptionDto> optionMap,
                            OperatorValueType currentOperator) {
        //给角色授权
        Long roleId = request.getRoleId();
        //生成应用权限列表
        List<RbacPermitAppRoleEntity> permitAppRoleList = new ArrayList<>();
        //生成操作项权限列表
        List<RbacPermitOptionRoleEntity> permitOptionRoleList = new ArrayList<>();
        if (CollectionUtil.isNotEmpty(request.getAppList())) {
            for (RbacPermitAppAuthRequestDto appRequestDto : request.getAppList()) {
                RbacPermitAppDto appDto = appMap.get(appRequestDto.getAppId());
                //应用列表
                permitAppRoleList.add(RbacPermitAppRoleEntity.create(SnowFlakeUtil.getId(), request.getOrganizationId(),
                        appDto.getAppId(), roleId, appDto.getAppCode(), appPermitTypeMap.get(appDto.getAppId()), currentOperator.getId()));

                for (RbacPermitOptionAuthRequestDto optionRequestDto : appRequestDto.getOptionList()) {
                    RbacPermitOptionDto optionDto = optionMap.get(optionRequestDto.getOptionId());
                    //操作项列表
                    permitOptionRoleList.add(RbacPermitOptionRoleEntity.create(SnowFlakeUtil.getId(), request.getOrganizationId(),
                            optionDto.getPermitAppId(), optionDto.getOptionId(), roleId, optionDto.getAppCode(), optionDto.getOptionType(),
                            optionDto.getOptionCode(), currentOperator.getId()));
                }
            }
        }
        this.unitWork.executeTran(() -> {
            //先清除原有授权
            rbacPermitAppRoleRepository.deleteByRbacRoleId(roleId);
            rbacPermitOptionRoleRepository.deleteByRbacRoleId(roleId);
            //再保存新的授权
            if (CollectionUtil.isNotEmpty(permitAppRoleList)) {
                for (RbacPermitAppRoleEntity appRoleEntity : permitAppRoleList) {
                    rbacPermitAppRoleRepository.save(appRoleEntity);
                }
            }
            if (CollectionUtil.isNotEmpty(permitOptionRoleList)) {
                for (RbacPermitOptionRoleEntity optionRoleEntity : permitOptionRoleList) {
                    rbacPermitOptionRoleRepository.save(optionRoleEntity);
                }
            }
        });
    }

    /**
     * description：给用户授权
     * author：linchunpeng
     * date：2024/2/4
     */
    private void authToUser(RbacPermitAuthRequestDto request, String employeeId, Map<Long, Integer> appPermitTypeMap,
                            Map<Long, RbacPermitAppDto> appMap, Map<Long, RbacPermitOptionDto> optionMap,
                            OperatorValueType currentOperator) {
        //生成应用权限列表
        List<RbacPermitAppUserEntity> permitAppUserList = new ArrayList<>();
        //生成操作项权限列表
        List<RbacPermitOptionUserEntity> permitOptionUserList = new ArrayList<>();
        if (CollectionUtil.isNotEmpty(request.getAppList())) {
            for (RbacPermitAppAuthRequestDto appRequestDto : request.getAppList()) {
                RbacPermitAppDto appDto = appMap.get(appRequestDto.getAppId());
                //应用列表
                permitAppUserList.add(RbacPermitAppUserEntity.create(SnowFlakeUtil.getId(), request.getOrganizationId(),
                        appDto.getAppId(), employeeId, appDto.getAppCode(), appPermitTypeMap.get(appDto.getAppId()), currentOperator.getId()));

                for (RbacPermitOptionAuthRequestDto optionRequestDto : appRequestDto.getOptionList()) {
                    RbacPermitOptionDto optionDto = optionMap.get(optionRequestDto.getOptionId());
                    //操作项列表
                    permitOptionUserList.add(RbacPermitOptionUserEntity.create(SnowFlakeUtil.getId(), request.getOrganizationId(),
                            optionDto.getPermitAppId(), optionDto.getOptionId(), employeeId, optionDto.getAppCode(), optionDto.getOptionType(),
                            optionDto.getOptionCode(), currentOperator.getId()));
                }
            }
        }
        this.unitWork.executeTran(() -> {
            //先清除原有授权
            rbacPermitAppUserRepository.deleteByTenantEmployeeId(employeeId);
            rbacPermitOptionUserRepository.deleteByTenantEmployeeId(employeeId);
            //再保存新的授权
            if (CollectionUtil.isNotEmpty(permitAppUserList)) {
                for (RbacPermitAppUserEntity appUserEntity : permitAppUserList) {
                    rbacPermitAppUserRepository.save(appUserEntity);
                }
            }
            if (CollectionUtil.isNotEmpty(permitOptionUserList)) {
                for (RbacPermitOptionUserEntity optionUserEntity : permitOptionUserList) {
                    rbacPermitOptionUserRepository.save(optionUserEntity);
                }
            }
        });
    }


    /**
     * description：根据职员id，查询该职员所有的功能授权
     * author：linchunpeng
     * date：2024/2/5
     */
    public RbacPermitAuthResponseDto getEmployeeAuthByEmployeeId(String organizationId, String employeeId) {
        Set<Long> allAppIdSet = new HashSet<>();
        Set<Long> allOptionIdSet = new HashSet<>();
        //查询职员的授权应用
        List<RbacPermitAppUserEntity> appRoleList = rbacPermitAppUserRepository.findByTenantEmployeeId(employeeId);
        if (CollectionUtil.isNotEmpty(appRoleList)) {
            allAppIdSet.addAll(appRoleList.stream().map(RbacPermitAppUserEntity::getPermitAppId).collect(Collectors.toSet()));
        }
        //查询职员的授权操作项
        List<RbacPermitOptionUserEntity> optionRoleList = rbacPermitOptionUserRepository.findByTenantEmployeeId(employeeId);
        if (CollectionUtil.isNotEmpty(optionRoleList)) {
            allOptionIdSet.addAll(optionRoleList.stream().map(RbacPermitOptionUserEntity::getPermitOptionId).collect(Collectors.toSet()));
        }
        //合并
        List<RbacPermitAppAuthResponseDto> appResponseList = new ArrayList<>();
        if (allAppIdSet.size() > 0 && allOptionIdSet.size() > 0) {
            List<RbacPermitAppEntity> appList = rbacPermitAppRepository.findByIds(new ArrayList<>(allAppIdSet));
            List<RbacPermitOptionEntity> optionList = rbacPermitOptionRepository.findByIds(new ArrayList<>(allOptionIdSet));
            Map<Long, List<RbacPermitOptionEntity>> app_optionMap = optionList.stream().collect(Collectors.groupingBy(RbacPermitOptionEntity::getPermitAppId));
            for (RbacPermitAppEntity app : appList) {
                List<RbacPermitOptionAuthResponseDto> optionResponseList = new ArrayList<>();
                if (app_optionMap.get(app.getId()) != null) {
                    for (RbacPermitOptionEntity option : app_optionMap.get(app.getId())) {
                        optionResponseList.add(new RbacPermitOptionAuthResponseDto(option.getId(),
                                option.getCategory(), option.getOptionType(), option.getOptionCode(), option.getOptionName()));
                    }
                }

                appResponseList.add(new RbacPermitAppAuthResponseDto(app.getId(), app.getAppCode(),
                        app.getAppName(), app.getAppLogo(), app.getCategory(), optionResponseList));
            }
        }

        return new RbacPermitAuthResponseDto(appResponseList);
    }

    /**
     * description：根据角色id，查询该角色所有的功能授权
     * author：linchunpeng
     * date：2024/2/5
     */
    public RbacPermitAuthResponseDto getRoleAuthByRoleId(String organizationId, Long roleId) {
        List<Long> roleIds = Collections.singletonList(roleId);
        //查询所有角色的授权应用
        List<RbacPermitAppRoleEntity> appRoleList = rbacPermitAppRoleRepository.findByRbacRoleIdIn(roleIds);
        //查询所有角色的授权操作项
        List<RbacPermitOptionRoleEntity> optionRoleList = rbacPermitOptionRoleRepository.findByRbacRoleIdIn(roleIds);
        //添加到总列表
        Set<Long> allAppIdSet = appRoleList.stream().map(RbacPermitAppRoleEntity::getPermitAppId).collect(Collectors.toSet());
        Set<Long> allOptionIdSet = optionRoleList.stream().map(RbacPermitOptionRoleEntity::getPermitOptionId).collect(Collectors.toSet());
        //合并
        List<RbacPermitAppAuthResponseDto> appResponseList = new ArrayList<>();
        if (allAppIdSet.size() > 0 && allOptionIdSet.size() > 0) {
            List<RbacPermitAppEntity> appList = rbacPermitAppRepository.findByIds(new ArrayList<>(allAppIdSet));
            List<RbacPermitOptionEntity> optionList = rbacPermitOptionRepository.findByIds(new ArrayList<>(allOptionIdSet));
            Map<Long, List<RbacPermitOptionEntity>> app_optionMap = optionList.stream().collect(Collectors.groupingBy(RbacPermitOptionEntity::getPermitAppId));
            for (RbacPermitAppEntity app : appList) {
                List<RbacPermitOptionAuthResponseDto> optionResponseList = new ArrayList<>();
                if (app_optionMap.get(app.getId()) != null) {
                    for (RbacPermitOptionEntity option : app_optionMap.get(app.getId())) {
                        optionResponseList.add(new RbacPermitOptionAuthResponseDto(option.getId(),
                                option.getCategory(), option.getOptionType(), option.getOptionCode(), option.getOptionName()));
                    }
                }

                appResponseList.add(new RbacPermitAppAuthResponseDto(app.getId(), app.getAppCode(),
                        app.getAppName(), app.getAppLogo(), app.getCategory(), optionResponseList));
            }
        }

        return new RbacPermitAuthResponseDto(appResponseList);
    }

    /**
     * description：根据职员id，查询该职员所有的功能授权，包括个人和角色授权
     * author：linchunpeng
     * date：2024/2/5
     */
    public RbacPermitAuthResponseDto getAllAuthByEmployeeId(String organizationId, String employeeId) {
        Set<Long> allAppIdSet = new HashSet<>();
        Set<Long> allOptionIdSet = new HashSet<>();
        //查询出所有的关联角色
        List<RbacCustomRoleDto> roleList = rbacCustomDbReader.findOrgRoleList(organizationId, null, employeeId);
        if (CollectionUtil.isNotEmpty(roleList)) {
            List<Long> roleIds = roleList.stream().map(RbacCustomRoleDto::getId).collect(Collectors.toList());
            //查询所有角色的授权应用
            List<RbacPermitAppRoleEntity> appRoleList = rbacPermitAppRoleRepository.findByRbacRoleIdIn(roleIds);
            //查询所有角色的授权操作项
            List<RbacPermitOptionRoleEntity> optionRoleList = rbacPermitOptionRoleRepository.findByRbacRoleIdIn(roleIds);
            //添加到总列表
            allAppIdSet.addAll(appRoleList.stream().map(RbacPermitAppRoleEntity::getPermitAppId).collect(Collectors.toSet()));
            allOptionIdSet.addAll(optionRoleList.stream().map(RbacPermitOptionRoleEntity::getPermitOptionId).collect(Collectors.toSet()));
        }
        //查询职员的授权应用
        List<RbacPermitAppUserEntity> appRoleList = rbacPermitAppUserRepository.findByTenantEmployeeId(employeeId);
        if (CollectionUtil.isNotEmpty(appRoleList)) {
            allAppIdSet.addAll(appRoleList.stream().map(RbacPermitAppUserEntity::getPermitAppId).collect(Collectors.toSet()));
        }
        //查询职员的授权操作项
        List<RbacPermitOptionUserEntity> optionRoleList = rbacPermitOptionUserRepository.findByTenantEmployeeId(employeeId);
        if (CollectionUtil.isNotEmpty(optionRoleList)) {
            allOptionIdSet.addAll(optionRoleList.stream().map(RbacPermitOptionUserEntity::getPermitOptionId).collect(Collectors.toSet()));
        }
        //合并
        List<RbacPermitAppAuthResponseDto> appResponseList = new ArrayList<>();
        if (allAppIdSet.size() > 0 && allOptionIdSet.size() > 0) {
            List<RbacPermitAppEntity> appList = rbacPermitAppRepository.findByIds(new ArrayList<>(allAppIdSet));
            List<RbacPermitOptionEntity> optionList = rbacPermitOptionRepository.findByIds(new ArrayList<>(allOptionIdSet));
            Map<Long, List<RbacPermitOptionEntity>> app_optionMap = optionList.stream().collect(Collectors.groupingBy(RbacPermitOptionEntity::getPermitAppId));
            for (RbacPermitAppEntity app : appList) {
                List<RbacPermitOptionAuthResponseDto> optionResponseList = new ArrayList<>();
                if (app_optionMap.get(app.getId()) != null) {
                    for (RbacPermitOptionEntity option : app_optionMap.get(app.getId())) {
                        optionResponseList.add(new RbacPermitOptionAuthResponseDto(option.getId(),
                                option.getCategory(), option.getOptionType(), option.getOptionCode(), option.getOptionName()));
                    }
                }

                appResponseList.add(new RbacPermitAppAuthResponseDto(app.getId(), app.getAppCode(),
                        app.getAppName(), app.getAppLogo(), app.getCategory(), optionResponseList));
            }
        }

        return new RbacPermitAuthResponseDto(appResponseList);
    }
}
