package com.bcxin.rbac.domain.services.custom;

import cn.hutool.core.collection.CollectionUtil;
import com.bcxin.Infrastructures.Pageable;
import com.bcxin.Infrastructures.UnitWork;
import com.bcxin.Infrastructures.entities.OperatorValueType;
import com.bcxin.Infrastructures.exceptions.BadTenantException;
import com.bcxin.Infrastructures.utils.AuthUtil;
import com.bcxin.Infrastructures.utils.SnowFlakeUtil;
import com.bcxin.rbac.domain.RbacCustomDbReader;
import com.bcxin.rbac.domain.dtos.custom.RbacCustomRoleDto;
import com.bcxin.rbac.domain.dtos.custom.RbacCustomRoleUserDto;
import com.bcxin.rbac.domain.dtos.custom.RbacUserDto;
import com.bcxin.rbac.domain.dtos.custom.RbacUserSearchDto;
import com.bcxin.rbac.domain.entities.RbacCustomRoleEntity;
import com.bcxin.rbac.domain.entities.RbacCustomRoleUserEntity;
import com.bcxin.rbac.domain.entities.RbacPermitAppRoleEntity;
import com.bcxin.rbac.domain.entities.RbacPermitOptionRoleEntity;
import com.bcxin.rbac.domain.repositories.custom.RbacCustomRoleRepository;
import com.bcxin.rbac.domain.repositories.custom.RbacCustomRoleUserRepository;
import com.bcxin.rbac.domain.repositories.custom.RbacPermitAppRoleRepository;
import com.bcxin.rbac.domain.repositories.custom.RbacPermitOptionRoleRepository;
import lombok.AllArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

@Service
@AllArgsConstructor
public class RbacCustomRoleService {
    private final RbacCustomRoleRepository rbacCustomRoleRepository;
    private final RbacCustomRoleUserRepository rbacCustomRoleUserRepository;
    private final RbacPermitAppRoleRepository rbacPermitAppRoleRepository;
    private final RbacPermitOptionRoleRepository rbacPermitOptionRoleRepository;
    private final RbacCustomDbReader rbacCustomDbReader;
    private final UnitWork unitWork;


    /**
     * description：根据企业id，查询企业自定义角色列表
     * author：linchunpeng
     * date：2024/2/1
     */
    public List<RbacCustomRoleDto> findOrgRoleList(String organizationId, String roleName, String employeeId) {
        return rbacCustomDbReader.findOrgRoleList(organizationId, roleName, employeeId);
    }


    /**
     * description：分页查询用户职员列表
     * author：linchunpeng
     * date：2024/2/3
     */
    public Pageable<RbacUserDto> getRbacUserDtoList(RbacUserSearchDto searchDto) {
        return rbacCustomDbReader.getRbacUserDtoList(searchDto);
    }

    /**
     * description：企业新增角色
     * author：linchunpeng
     * date：2024/2/2
     */
    public void addRole(String organizationId, String roleName, String remark, Integer seq) {
        Optional<RbacCustomRoleEntity> exist1 = rbacCustomRoleRepository.findByOrganizationIdAndRoleName(organizationId, roleName);
        if (exist1.isPresent()) {
            throw new BadTenantException("该角色名称已存在，不能重复添加");
        }
        Optional<RbacCustomRoleEntity> exist2 = rbacCustomRoleRepository.findByOrganizationIdAndSeq(organizationId, seq);
        if (exist2.isPresent()) {
            throw new BadTenantException("该排序已存在，不能重复添加");
        }
        OperatorValueType currentOperator = AuthUtil.getCurrentOperator();
        RbacCustomRoleEntity entity = RbacCustomRoleEntity.create(SnowFlakeUtil.getId(), organizationId, roleName, remark, seq, currentOperator.getId());
        this.unitWork.executeTran(() -> {
            rbacCustomRoleRepository.save(entity);
        });
    }

    /**
     * description：企业编辑角色
     * author：linchunpeng
     * date：2024/2/2
     */
    public void editRole(String organizationId, Long id, String roleName, String remark, Integer seq) {
        Optional<RbacCustomRoleEntity> entity = rbacCustomRoleRepository.findById(id);
        if (!entity.isPresent()) {
            throw new BadTenantException("角色不存在");
        }
        Optional<RbacCustomRoleEntity> exist1 = rbacCustomRoleRepository.findByOrganizationIdAndRoleName(organizationId, roleName);
        if (exist1.isPresent() && exist1.get().getId().longValue() != id.longValue()) {
            throw new BadTenantException("该角色名称已存在，不能重复添加");
        }
        Optional<RbacCustomRoleEntity> exist2 = rbacCustomRoleRepository.findByOrganizationIdAndSeq(organizationId, seq);
        if (exist2.isPresent() && exist2.get().getId().longValue() != id.longValue()) {
            throw new BadTenantException("该排序已存在，不能重复添加");
        }
        OperatorValueType currentOperator = AuthUtil.getCurrentOperator();
        RbacCustomRoleEntity customRoleEntity = entity.get();
        customRoleEntity.update(roleName, remark, seq, currentOperator.getId());
        this.unitWork.executeTran(() -> {
            rbacCustomRoleRepository.save(customRoleEntity);
        });
    }

    /**
     * description：根据id查询角色详情
     * author：linchunpeng
     * date：2024/2/2
     */
    public RbacCustomRoleDto findById(Long id) {
        Optional<RbacCustomRoleEntity> entity = rbacCustomRoleRepository.findById(id);
        if (!entity.isPresent()) {
            throw new BadTenantException("角色不存在");
        }
        RbacCustomRoleDto dto = new RbacCustomRoleDto();
        BeanUtils.copyProperties(entity.get(), dto);
        return dto;
    }

    /**
     * description：根据id复制角色
     * author：linchunpeng
     * date：2024/2/2
     */
    public void copyRole(Long sourceId, String roleName, String remark, Integer seq) {
        Optional<RbacCustomRoleEntity> source = rbacCustomRoleRepository.findById(sourceId);
        if (!source.isPresent()) {
            throw new BadTenantException("原角色不存在");
        }
        RbacCustomRoleEntity sourceEntity = source.get();
        String organizationId = sourceEntity.getOrganizationId();

        Optional<RbacCustomRoleEntity> exist1 = rbacCustomRoleRepository.findByOrganizationIdAndRoleName(organizationId, roleName);
        if (exist1.isPresent()) {
            throw new BadTenantException("该角色名称已存在，不能重复添加");
        }
        Optional<RbacCustomRoleEntity> exist2 = rbacCustomRoleRepository.findByOrganizationIdAndSeq(organizationId, seq);
        if (exist2.isPresent()) {
            throw new BadTenantException("该排序已存在，不能重复添加");
        }
        OperatorValueType currentOperator = AuthUtil.getCurrentOperator();
        RbacCustomRoleEntity entity = RbacCustomRoleEntity.create(SnowFlakeUtil.getId(), organizationId, roleName, remark, seq, currentOperator.getId());

        //复制应用权限
        List<RbacPermitAppRoleEntity> permitAppRoleEntityList = new ArrayList<>();
        List<RbacPermitAppRoleEntity> sourcePermitAppRoleEntityList = rbacPermitAppRoleRepository.findByRbacRoleId(sourceId);
        if (CollectionUtil.isNotEmpty(sourcePermitAppRoleEntityList)) {
            for (RbacPermitAppRoleEntity appRoleEntity : sourcePermitAppRoleEntityList) {
                permitAppRoleEntityList.add(RbacPermitAppRoleEntity.create(SnowFlakeUtil.getId(), appRoleEntity.getOrganizationId(),
                        appRoleEntity.getPermitAppId(), entity.getId(), appRoleEntity.getAppCode(), appRoleEntity.getPermitType(), currentOperator.getId()));
            }
        }

        //复制菜单权限
        List<RbacPermitOptionRoleEntity> permitOptionRoleEntityList = new ArrayList<>();
        List<RbacPermitOptionRoleEntity> sourcePermitOptionRoleEntityList = rbacPermitOptionRoleRepository.findByRbacRoleId(sourceId);
        if (CollectionUtil.isNotEmpty(sourcePermitOptionRoleEntityList)) {
            for (RbacPermitOptionRoleEntity optionRoleEntity : sourcePermitOptionRoleEntityList) {
                permitOptionRoleEntityList.add(RbacPermitOptionRoleEntity.create(SnowFlakeUtil.getId(), optionRoleEntity.getOrganizationId(),
                        optionRoleEntity.getPermitAppId(), optionRoleEntity.getPermitOptionId(), entity.getId(), optionRoleEntity.getAppCode(),
                        optionRoleEntity.getOptionType(), optionRoleEntity.getOptionCode(), currentOperator.getId()));
            }
        }

        this.unitWork.executeTran(() -> {
            rbacCustomRoleRepository.save(entity);
            //复制权限
            if (CollectionUtil.isNotEmpty(permitAppRoleEntityList)) {
                for (RbacPermitAppRoleEntity rbacPermitAppRoleEntity : permitAppRoleEntityList) {
                    rbacPermitAppRoleRepository.save(rbacPermitAppRoleEntity);
                }
            }
            if (CollectionUtil.isNotEmpty(permitOptionRoleEntityList)) {
                for (RbacPermitOptionRoleEntity rbacPermitOptionRoleEntity : permitOptionRoleEntityList) {
                    rbacPermitOptionRoleRepository.save(rbacPermitOptionRoleEntity);
                }
            }
        });
    }

    /**
     * description：根据id列表删除角色
     * author：linchunpeng
     * date：2024/2/2
     */
    public void deleteRole(List<Long> idList) {
        List<RbacCustomRoleEntity> deleteList = new ArrayList<>();
        if (CollectionUtil.isEmpty(idList)) {
            throw new BadTenantException("id不能为空");
        }
        for (Long id : idList) {
            Optional<RbacCustomRoleEntity> entity = rbacCustomRoleRepository.findById(id);
            if (!entity.isPresent()) {
                throw new BadTenantException("角色不存在");
            }
            long userCount = rbacCustomRoleUserRepository.findCountByRbacRoleId(id);
            if (userCount > 0) {
                throw new BadTenantException("该角色还有成员关联，请先移除");
            }
            deleteList.add(entity.get());
        }
        if (CollectionUtil.isNotEmpty(deleteList)) {
            this.unitWork.executeTran(() -> {
                for (RbacCustomRoleEntity entity : deleteList) {
                    rbacCustomRoleRepository.delete(entity);
                }
            });
        }
    }

    /**
     * description：根据角色id、职员id列表，批量设置用户关联
     * author：linchunpeng
     * date：2024/2/2
     */
    public void addRoleAndUsers(Long roleId, List<String> employeeIdList) {
        if (roleId == null) {
            throw new BadTenantException("角色id不能为空");
        }
        if (CollectionUtil.isEmpty(employeeIdList)) {
            throw new BadTenantException("职员id不能为空");
        }
        Optional<RbacCustomRoleEntity> role = rbacCustomRoleRepository.findById(roleId);
        if (!role.isPresent()) {
            throw new BadTenantException("角色不存在");
        }
        OperatorValueType currentOperator = AuthUtil.getCurrentOperator();
        List<RbacCustomRoleUserEntity> insertList = new ArrayList<>();
        for (String employeeId : employeeIdList) {
            Optional<RbacCustomRoleUserEntity> roleUserExist = rbacCustomRoleUserRepository.findByRbacRoleIdAndTenantEmployeeId(roleId, employeeId);
            if (roleUserExist.isPresent()) {
                continue;
            }
            RbacCustomRoleUserEntity entity = RbacCustomRoleUserEntity.create(SnowFlakeUtil.getId(), role.get().getOrganizationId(), roleId, employeeId, currentOperator.getId());
            insertList.add(entity);
        }
        if (CollectionUtil.isNotEmpty(insertList)) {
            this.unitWork.executeTran(() -> {
                for (RbacCustomRoleUserEntity entity : insertList) {
                    rbacCustomRoleUserRepository.save(entity);
                }
            });
        }
    }


    /**
     * description：根据职员id、角色id列表，批量设置角色关联
     * author：linchunpeng
     * date：2024/2/2
     */
    public void addUserAndRoles(String employeeId, List<Long> roleIdList) {
        if (StringUtils.isBlank(employeeId)) {
            throw new BadTenantException("职员id不能为空");
        }
        if (CollectionUtil.isEmpty(roleIdList)) {
            throw new BadTenantException("角色id不能为空");
        }
        OperatorValueType currentOperator = AuthUtil.getCurrentOperator();
        List<RbacCustomRoleUserEntity> insertList = new ArrayList<>();
        for (Long roleId : roleIdList) {
            Optional<RbacCustomRoleEntity> role = rbacCustomRoleRepository.findById(roleId);
            if (!role.isPresent()) {
                throw new BadTenantException("角色不存在");
            }
            Optional<RbacCustomRoleUserEntity> roleUserExist = rbacCustomRoleUserRepository.findByRbacRoleIdAndTenantEmployeeId(roleId, employeeId);
            if (roleUserExist.isPresent()) {
                continue;
            }
            RbacCustomRoleUserEntity entity = RbacCustomRoleUserEntity.create(SnowFlakeUtil.getId(), role.get().getOrganizationId(), roleId, employeeId, currentOperator.getId());
            insertList.add(entity);
        }
        if (CollectionUtil.isNotEmpty(insertList)) {
            this.unitWork.executeTran(() -> {
                for (RbacCustomRoleUserEntity entity : insertList) {
                    rbacCustomRoleUserRepository.save(entity);
                }
            });
        }
    }

    /**
     * description：根据职员id列表、角色id列表，批量设置角色关联
     * author：linchunpeng
     * date：2024/3/5
     */
    public void addUsersAndRoles(List<String> employeeIdList, List<Long> roleIdList) {
        if (CollectionUtil.isEmpty(employeeIdList)) {
            throw new BadTenantException("职员id不能为空");
        }
        if (CollectionUtil.isEmpty(roleIdList)) {
            throw new BadTenantException("角色id不能为空");
        }
        OperatorValueType currentOperator = AuthUtil.getCurrentOperator();
        List<RbacCustomRoleUserEntity> insertList = new ArrayList<>();
        for (Long roleId : roleIdList) {
            Optional<RbacCustomRoleEntity> role = rbacCustomRoleRepository.findById(roleId);
            if (!role.isPresent()) {
                throw new BadTenantException("角色不存在");
            }
            for (String employeeId : employeeIdList) {
                Optional<RbacCustomRoleUserEntity> roleUserExist = rbacCustomRoleUserRepository.findByRbacRoleIdAndTenantEmployeeId(roleId, employeeId);
                if (roleUserExist.isPresent()) {
                    continue;
                }
                RbacCustomRoleUserEntity entity = RbacCustomRoleUserEntity.create(SnowFlakeUtil.getId(), role.get().getOrganizationId(), roleId, employeeId, currentOperator.getId());
                insertList.add(entity);
            }
        }
        if (CollectionUtil.isNotEmpty(insertList)) {
            this.unitWork.executeTran(() -> {
                for (RbacCustomRoleUserEntity entity : insertList) {
                    rbacCustomRoleUserRepository.save(entity);
                }
            });
        }
    }

    /**
     * description：根据角色id、职员id列表，批量删除用户关联
     * author：linchunpeng
     * date：2024/2/2
     */
    public void deleteRoleAndUsers(Long roleId, List<String> employeeIdList) {
        if (roleId == null) {
            throw new BadTenantException("角色id不能为空");
        }
        if (CollectionUtil.isEmpty(employeeIdList)) {
            throw new BadTenantException("职员id不能为空");
        }
        Optional<RbacCustomRoleEntity> role = rbacCustomRoleRepository.findById(roleId);
        if (!role.isPresent()) {
            throw new BadTenantException("角色不存在");
        }
        List<RbacCustomRoleUserEntity> deleteList = new ArrayList<>();
        for (String employeeId : employeeIdList) {
            Optional<RbacCustomRoleUserEntity> roleUserExist = rbacCustomRoleUserRepository.findByRbacRoleIdAndTenantEmployeeId(roleId, employeeId);
            roleUserExist.ifPresent(deleteList::add);
        }
        if (CollectionUtil.isNotEmpty(deleteList)) {
            this.unitWork.executeTran(() -> {
                for (RbacCustomRoleUserEntity entity : deleteList) {
                    rbacCustomRoleUserRepository.delete(entity);
                }
            });
        }
    }

    /**
     * description：根据职员id、角色id列表，批量删除用户关联
     * author：linchunpeng
     * date：2024/2/2
     */
    public void deleteUserAndRoles(String employeeId, List<Long> roleIdList) {
        if (StringUtils.isBlank(employeeId)) {
            throw new BadTenantException("职员id不能为空");
        }
        if (CollectionUtil.isEmpty(roleIdList)) {
            throw new BadTenantException("角色id不能为空");
        }
        List<RbacCustomRoleUserEntity> deleteList = new ArrayList<>();
        for (Long roleId : roleIdList) {
            Optional<RbacCustomRoleUserEntity> roleUserExist = rbacCustomRoleUserRepository.findByRbacRoleIdAndTenantEmployeeId(roleId, employeeId);
            roleUserExist.ifPresent(deleteList::add);
        }
        if (CollectionUtil.isNotEmpty(deleteList)) {
            this.unitWork.executeTran(() -> {
                for (RbacCustomRoleUserEntity entity : deleteList) {
                    rbacCustomRoleUserRepository.delete(entity);
                }
            });
        }
    }

    /**
     * description：根据角色id和职员id列表，查询角色用户关联列表
     * author：linchunpeng
     * date：2024/2/1
     */
    public List<RbacCustomRoleUserDto> findRoleUserListByRoleIdAndEmployeeIdList(Long roleId, List<String> employeeIdList) {
        List<RbacCustomRoleUserDto> resultList = new ArrayList<>();
        List<RbacCustomRoleUserEntity> roleUserEntityList = rbacCustomRoleUserRepository.findListByRbacRoleIdAndEmployeeIdList(roleId, employeeIdList);
        if (CollectionUtil.isNotEmpty(roleUserEntityList)) {
            for (RbacCustomRoleUserEntity roleUserEntity : roleUserEntityList) {
                resultList.add(new RbacCustomRoleUserDto(roleUserEntity.getId(), roleUserEntity.getRbacRoleId(), roleUserEntity.getTenantEmployeeId(), roleUserEntity.getCreateTime()));
            }
        }
        return resultList;
    }


    /**
     * description：批量导入角色成员
     * author：linchunpeng
     * date：2024/2/6
     */
    public List<Map<String, String>> importMember(String organizationId, Long roleId, List<Map<String, String>> importDataList) {
        List<Map<String, String>> result = new ArrayList<>();
        if (CollectionUtil.isEmpty(importDataList)) {
            throw new BadTenantException("导入的表格数据为空");
        }
        List<String> names = importDataList.stream().map(data -> data.get("name")).collect(Collectors.toList());
        //导入的姓名，查询职员id
        List<RbacUserDto> rbacUserDtoList = rbacCustomDbReader.getRbacUserDtoList(organizationId, names);
        Map<String, List<RbacUserDto>> userDtoListMap = rbacUserDtoList.stream().collect(Collectors.groupingBy(RbacUserDto::getUserName));
        //对应角色下的所有职员id
        List<RbacCustomRoleUserEntity> roleUserList = rbacCustomRoleUserRepository.findByRbacRoleId(roleId);
        List<String> existEmployeeIdList = roleUserList.stream().map(RbacCustomRoleUserEntity::getTenantEmployeeId).collect(Collectors.toList());

        List<String> employeeIdList = new ArrayList<>();

        for (Map<String, String> data : importDataList) {
            Map<String, String> resultMap = new HashMap<>();
            String name = data.get("name");
            String telephone = data.get("telephone");
            resultMap.put("paramName", name);
            resultMap.put("paramTelephone", telephone);

            List<RbacUserDto> dtoList = userDtoListMap.get(name);
            if (CollectionUtil.isEmpty(dtoList)) {
                resultMap.put("result", "该用户不在本企业职员列表里面");
            } else if (dtoList.size() > 1) {
                if (StringUtils.isBlank(telephone)) {
                    resultMap.put("result", "该用户姓名出现重复，且未填手机号，无法导入，请先填写手机号");
                } else {
                    Optional<RbacUserDto> userDto = dtoList.stream().filter(dto -> dto.getMobile().equals(telephone)).findFirst();
                    if (userDto.isPresent()) {
                        String employeeId = userDto.get().getEmployeeId();
                        if (existEmployeeIdList.contains(employeeId)) {
                            resultMap.put("result", "该用户已有该角色，无需重复导入");
                        } else {
                            employeeIdList.add(employeeId);
                            resultMap.put("result", "导入成功");
                        }
                    } else {
                        resultMap.put("result", "该用户姓名出现重复，且手机号匹配不上，请确认手机号是否正确");
                    }
                }
            } else {
                String employeeId = dtoList.get(0).getEmployeeId();
                if (existEmployeeIdList.contains(employeeId)) {
                    resultMap.put("result", "该用户已有该角色，无需重复导入");
                } else {
                    employeeIdList.add(employeeId);
                    resultMap.put("result", "导入成功");
                }
            }
            result.add(resultMap);
        }
        if (CollectionUtil.isNotEmpty(employeeIdList)) {
            this.addRoleAndUsers(roleId, employeeIdList);
        }
        return result;
    }

}
