package com.bcxin.identity.domain.repositories;

import com.bcxin.Infrastructures.UnitWork;
import com.bcxin.Infrastructures.entities.IAggregate;
import com.bcxin.Infrastructures.exceptions.ConflictTenantException;
import com.bcxin.Infrastructures.exceptions.TenantExceptionAbstract;
import com.bcxin.identity.domains.exceptions.IdentityExceptionConverter;
import org.springframework.stereotype.Component;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import javax.persistence.EntityManager;
import java.util.*;

@Component
public class UnitWorkImpl implements UnitWork {
    private final EntityManager entityManager;
    private final PlatformTransactionManager transactionManager;
    private static ThreadLocal<Map<String, TransactionStatus>> _container = new ThreadLocal<>();
    private final DefaultTransactionDefinition propagation_required = new DefaultTransactionDefinition(TransactionDefinition.PROPAGATION_REQUIRED);


    public UnitWorkImpl(EntityManager entityManager,
                        PlatformTransactionManager transactionManager) {

        this.entityManager = entityManager;
        this.transactionManager = transactionManager;
        /*
        设置过期时间
         */
        this.propagation_required.setTimeout(3000);
    }


    @Override
    public String beginTransaction() {
        return this.generateTranId();
    }

    @Override
    public void commit(String tid) {
        TransactionStatus status = getTranStatus(tid);
        if (status != null) {
            this.transactionManager.commit(status);
        }
    }

    @Override
    public void detachAll() {
        this.entityManager.clear();
    }

    public void rollback(String tid) {
        TransactionStatus status = getTranStatus(tid);
        if (status != null) {
            this.transactionManager.rollback(status);
        }
    }

    @Override
    public void executeNewTran(Runnable runnable) {
        TransactionStatus status = null;
        boolean hasCommitted=false;
        try {
            status = this.transactionManager.getTransaction(new DefaultTransactionDefinition(TransactionDefinition.PROPAGATION_REQUIRES_NEW));
            runnable.run();
            this.entityManager.flush();

            hasCommitted=true;
            this.transactionManager.commit(status);
        }
        catch (Exception ex) {
            if (!hasCommitted && status != null) {
                this.transactionManager.rollback(status);
            }

            throw ex;
        }
    }

    @Override
    public void detach(IAggregate aggregate) {
        this.entityManager.detach(aggregate);
    }

    @Override
    public <T extends IAggregate> void detachAll(Collection<T> aggregates) {
        if (aggregates != null) {
            aggregates.forEach(ii -> {
                this.entityManager.detach(ii);
            });
        }
    }

    //@Transactional(propagation = Propagation.REQUIRES_NEW)
    @Override
    public void executeTran(Runnable runnable) {
        String tranId = null;
        boolean hasCommitted = false;
        try {
            tranId = this.beginTransaction();
            //TransactionStatus status = this.transactionManager.getTransaction(propagation_required);
            runnable.run();
            //this.transactionManager.commit(status);
           // this.entityManager.flush();
            hasCommitted = true;
            this.commit(tranId);
        } catch (Exception ex) {
            if (!hasCommitted) {
                this.rollback(tranId);
            }

            TenantExceptionAbstract tenantException = IdentityExceptionConverter.cast(ex);
            if (tenantException instanceof ConflictTenantException) {
               throw new ConflictTenantException("用户名已经存在!");
            }

            throw ex;
        }
    }


    private String generateTranId() {
        String uuId = UUID.randomUUID().toString();
        Map<String, TransactionStatus> transactionStatusMap = _container.get();
        if (transactionStatusMap == null) {
            transactionStatusMap = new HashMap<>();
        }

        if (transactionStatusMap.size() == 0) {
            transactionStatusMap.put(uuId, this.transactionManager.getTransaction(propagation_required));
        }

        _container.set(transactionStatusMap);

        return uuId;
    }

    /**
     * 获取完事务之后，直接进行删除
     * @param tranId
     * @return
     */
    private TransactionStatus getTranStatus(String tranId) {
        Map<String, TransactionStatus> transactionStatusMap = _container.get();
        if (transactionStatusMap == null || transactionStatusMap.size() == 0) {
            return null;
        }

        TransactionStatus status = transactionStatusMap.get(tranId);
        transactionStatusMap.remove(tranId);
        _container.set(transactionStatusMap);

        return status;
    }
}
