package com.bcxin.tenant.domain.repository.readers;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSON;
import com.bcxin.Infrastructures.Pageable;
import com.bcxin.Infrastructures.TenantContext;
import com.bcxin.Infrastructures.TenantUserContext;
import com.bcxin.Infrastructures.components.JsonProvider;
import com.bcxin.Infrastructures.enums.*;
import com.bcxin.Infrastructures.exceptions.ArgumentTenantException;
import com.bcxin.Infrastructures.exceptions.NotSupportTenantException;
import com.bcxin.Infrastructures.utils.AuthUtil;
import com.bcxin.api.interfaces.identities.requests.GetCredentialRequest;
import com.bcxin.api.interfaces.tenants.criterias.ContractCriteria;
import com.bcxin.api.interfaces.tenants.criterias.EmployeeLeaveCriteria;
import com.bcxin.api.interfaces.tenants.criterias.TenantUserRegionCriteria;
import com.bcxin.api.interfaces.tenants.requests.operatelog.SearchOperateLogRequest;
import com.bcxin.api.interfaces.tenants.requests.tenantUsers.IMContactCriteria;
import com.bcxin.api.interfaces.tenants.requests.tenantUsers.QueryCredentialRequest;
import com.bcxin.api.interfaces.tenants.requests.tenantUsers.UpdateTenantUserRealNameRequest;
import com.bcxin.api.interfaces.tenants.responses.*;
import com.bcxin.tenant.domain.dto.EmployeeOccupationTypeValidationBasicDto;
import com.bcxin.tenant.domain.dto.EmployeeTenantUserIdNumDto;
import com.bcxin.tenant.domain.dto.TenantUserTelephoneCredentialDto;
import com.bcxin.tenant.domain.entities.*;
import com.bcxin.tenant.domain.entities.valueTypes.ApprovedInformationValueType;
import com.bcxin.tenant.domain.enums.EventAction;
import com.bcxin.tenant.domain.enums.EventProcessedStatus;
import com.bcxin.tenant.domain.readers.TenantDbReader;
import com.bcxin.tenant.domain.readers.criterias.*;
import com.bcxin.tenant.domain.readers.dtos.OrganizationDto;
import com.bcxin.tenant.domain.readers.dtos.*;
import com.bcxin.tenant.domain.readers.tmps.EmployeeDepartData;
import com.bcxin.tenant.domain.repositories.OrganizationRepository;
import com.bcxin.tenant.domain.repositories.TenantUserRepository;
import com.bcxin.tenant.domain.repositories.criterias.ExternalMemberSearchCriteria;
import com.bcxin.tenant.domain.repositories.dtos.*;
import com.bcxin.tenant.domain.repository.readers.tmps.DepartAdminReadDto;
import com.bcxin.tenant.domain.repository.readers.tmps.DepartmentReadDto;
import com.bcxin.tenant.domain.repository.readers.tmps.EmployeeContactReadDto;
import com.bcxin.tenant.domain.repository.translates.DataTranslate;
import com.bcxin.tenant.domain.snapshots.DepartImAllowedDepartSnapshot;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.time.DateUtils;
import org.hibernate.SQLQuery;
import org.hibernate.transform.Transformers;
import org.springframework.beans.BeanUtils;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.persistence.EntityManager;
import javax.persistence.NoResultException;
import javax.persistence.Query;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.*;
import javax.transaction.Transactional;
import java.sql.Timestamp;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Slf4j
@Component
public class TenantDbReaderImpl implements TenantDbReader {
    private final EntityManager entityManager;
    private final JsonProvider jsonProvider;
    private final DataTranslate dataTranslate;
    private final TenantUserRepository tenantUserRepository;
    private final OrganizationRepository organizationRepository;

    public TenantDbReaderImpl(EntityManager entityManager, JsonProvider jsonProvider, DataTranslate dataTranslate, TenantUserRepository tenantUserRepository, OrganizationRepository organizationRepository) {
        this.entityManager = entityManager;
        this.jsonProvider = jsonProvider;
        this.dataTranslate = dataTranslate;
        this.tenantUserRepository = tenantUserRepository;
        this.organizationRepository = organizationRepository;
    }

    /**
     * 获取员工信息
     *
     * @param organizationId
     * @param criteria
     * @return
     */
    @Override
    public Pageable<EmployeeDto> find(String organizationId, EmployeeCriteria criteria) {
        StringBuilder sb = new StringBuilder();
        Map<String, Object> parameters = new HashMap<>();
        if (StringUtils.hasLength(criteria.getKeyword())) {
            String keyword = criteria.getKeyword();
            String substrings = keyword.substring(0, 1);
            Pattern p = Pattern.compile("[\u4e00-\u9fa5]");
            Matcher m = p.matcher(substrings);
            if (m.find()) {
                sb.append(" and (t.name like :keyword)");
                parameters.put("keyword", criteria.getKeyword() + "%");
            } else if (keyword.length() >= 12) {
                sb.append(" and (s.number like :keyword)");
                parameters.put("keyword", criteria.getKeyword() + "%");
            } else {
                sb.append(" and (t.telephone like :keyword or s.number like :keyword)");
                parameters.put("keyword", criteria.getKeyword() + "%");
            }
        }

        /**
         * 证书类型
         */
        if (!CollectionUtils.isEmpty(criteria.getCredentialTypes())) {
            sb.append(" and s.credentialType in :credentialType");
            parameters.put("credentialType", criteria.getCredentialTypes());
        }

        /**
         * 实名认证
         */
        if (!CollectionUtils.isEmpty(criteria.getAuthenticatedStatuses())) {
            sb.append(" and t.authenticateStatus in :authenticateStatus");
            parameters.put("authenticateStatus", criteria.getAuthenticatedStatuses());
        }
        /**
         * 证件核验状态
         */
        if (!CollectionUtils.isEmpty(criteria.getCheckedStatuses())) {
            sb.append(" and t.checkedStatus in :checkedStatus");
            parameters.put("checkedStatus", criteria.getCheckedStatuses());
        }

        /**
         * 职业类型
         */
        if (!CollectionUtils.isEmpty(criteria.getOccupationTypes())) {
            sb.append(" and e.occupationType in :occupationType");
            parameters.put("occupationType", criteria.getOccupationTypes());
        }

        if (criteria.getHiredDate() != null) {
            sb.append(" and e.hiredDate=:hiredDate ");
            parameters.put("hiredDate", criteria.getHiredDate());
        }

        if (StringUtils.hasLength(criteria.getPosition())) {
            sb.append(" and e.position = :position ");
            parameters.put("position", criteria.getPosition());
        }

        if (criteria.getStartDate() != null) {
            sb.append(" and e.hiredDate >=:startDate ");
            parameters.put("startDate", criteria.getStartDate());
        }

        if (criteria.getEndDate() != null) {
            sb.append(" and e.hiredDate <=:endDate ");
            parameters.put("endDate", criteria.getEndDate());
        }

        if ("1".equals(criteria.getContractStatus())) {
            sb.append(" and exists(select ct from ContractEntity ct where ct.employee = e and ct.status=1 and CURDATE() between ct.beginDate and ct.endDate) ");
        }
        if ("0".equals(criteria.getContractStatus())) {
            sb.append(" and not exists(select ct from ContractEntity ct where ct.employee = e and ct.status=1 and CURDATE() between ct.beginDate and ct.endDate) ");
        }
        Optional<OrganizationEntity> organizationOptional = this.organizationRepository.findById(organizationId);
        String areaCode = "EMPTY";
        if (organizationOptional.isPresent()) {
            OrganizationEntity organizationEntity = organizationOptional.get();
            areaCode = organizationEntity.getSuperviseRegionCode();
        }
        if (StrUtil.isNotEmpty(criteria.getCerStatus())) {
            String exists = " ";
            if ("0".equals(criteria.getCerStatus())) {
                exists = " not ";
            }
            sb.append(" and " + exists + " exists(select tucd from  TenantUserCredentialDetailsEntity tucd " +
                    "where t.id=tucd.tenantUserId  and tucd.state = '1' and tucd.active = true and tucd.certificateType='1'");
            if (!AuthUtil.isUnDistinguishArea()) {
                sb.append(" and tucd.areaCode like :areaCode ");
                parameters.put("areaCode", AuthUtil.getShortAreaCode(areaCode) + "%");
            }

            sb.append(") ");
        }

        if (StrUtil.isNotEmpty(criteria.getGradeCerStatus()) || CollUtil.isNotEmpty(criteria.getLevel())) {
            String exists = " ";
            if ("0".equals(criteria.getCerStatus())) {
                exists = " not ";
            }
            sb.append(" and " + exists + " exists(select tucd from  TenantUserCredentialDetailsEntity tucd  " +
                    "where t.id=tucd.tenantUserId and  tucd.state = '1' and tucd.active = true and tucd.certificateType='2'");
            if (StringUtils.hasLength(areaCode)) {
                sb.append(" and tucd.areaCode like :areaCode ");
                parameters.put("areaCode", AuthUtil.getShortAreaCode(areaCode) + "%");
            }
            if (CollUtil.isNotEmpty(criteria.getLevel())) {
                sb.append(" and tucd.appraisalGrade in :gradeLevels");
                Collection<Integer> level = criteria.getLevel();
                List<String> gradeLevels = new ArrayList<>();
                for (Integer i : level) {
                    gradeLevels.add(i.toString());
                }
                parameters.put("gradeLevels", gradeLevels);
            }
            sb.append(") ");
        }
        //年龄转成日期
        if (criteria.getBeginAge() != null) {
            LocalDate now = LocalDate.now();
            LocalDate bTime = now.minus(criteria.getBeginAge(), ChronoUnit.YEARS);
            sb.append(" and  t.birthdate <= :bTime");
            parameters.put("bTime", Date.from(bTime.atStartOfDay(ZoneId.systemDefault()).toInstant()));
        }

        if (criteria.getEndAge() != null) {
            LocalDate now = LocalDate.now();
            LocalDate eTime = now.minus(criteria.getEndAge() + 1, ChronoUnit.YEARS);
            sb.append(" and  t.birthdate >= :eTime");
            parameters.put("eTime", Date.from(eTime.atStartOfDay(ZoneId.systemDefault()).toInstant()));
        }


        //性别
        if (criteria.getSex() != null) {
            sb.append("  and  t.sex = :sex");
            parameters.put("sex", criteria.getSex());
        }
        //是否投保
        if (criteria.getInsure() != null) {
            if (criteria.getInsure().equals(TrueFalseStatus.False)) {
                sb.append("  and  (e.insure = :insure or e.insure is null)");
            } else {
                sb.append("  and  e.insure = :insure");
            }
            parameters.put("insure", criteria.getInsure());
        }
        if (!CollectionUtils.isEmpty(criteria.getBackgroundScreeningStatus())) {
            sb.append(" and t.backgroundScreeningStatus in :backgroundScreeningStatus");
            parameters.put("backgroundScreeningStatus", criteria.getBackgroundScreeningStatus());
        }
        String qString = "select new com.bcxin.tenant.domain.readers.dtos.EmployeeDto(t.id,e.organization.name,e.id,t.name,t.telephone,s.credentialType," +
                "e.occupationType, s.number,t.checkedStatus,t.userType,t.authenticateStatus,e.hiredDate," +
                "e.position,t.authenticatedResult,t.sex,t.birthdate,t.nation,t.education,t.politicsStatus," +
                "t.householdType,t.militaryStatus,t.nativePlace,t.maritalStatus,t.stature,t.emergencyContact," +
                "t.emergencyPhone,t.licenseLevel,t.backgroundScreeningStatus,e.insure," +
                "est.name,e.positiveDate,s.validDateFrom,s.validDateTo,s.address,e.createdTime,e.domainAdmin,e.organization.id," +
                "e.personStatus,e.probation,e.planPositiveDate,t.placeOfNow" +
                ",e.hiredOperator,t.lastCheckedStatusTime" +
                ") " +
                " from EmployeeEntity e join e.tenantUser t left join t.selectedCredential s " +
                " left join e.superior es left join es.tenantUser est " +
//                " left join ContractEntity c on c.employee = e and c.status = com.bcxin.Infrastructures.enums.ValidStatus.Valid" +
//                " left join t.credentials tc on tc.credentialType=com.bcxin.Infrastructures.enums.CredentialType.QualificationCer " +
                " where e.status!=com.bcxin.Infrastructures.enums.EmploymentStatus.OffJob ";

        String cString = "select count(1) from EmployeeEntity e join e.tenantUser t left join t.selectedCredential s  " +
                " where e.status!=com.bcxin.Infrastructures.enums.EmploymentStatus.OffJob ";

        if (StrUtil.isNotEmpty(organizationId)) {
            qString += " and e.organization.id=:organId ";
            cString += " and e.organization.id=:organId ";
            parameters.put("organId", organizationId);
        }

        if (StrUtil.isNotEmpty(criteria.getOrganName())) {
            qString += " and e.organization.name like '%:organName%' ";
            cString += " and e.organization.name like '%:organName%' ";
            parameters.put("organName", criteria.getOrganName());
        }


        qString += sb;
        cString += sb;

        TenantUserContext userContext = TenantContext.getInstance().getUserContext();
        if (!criteria.isIgnorePermission() && userContext != null && userContext.get() != null && !userContext.get().isMaster() &&
                StringUtils.hasLength(userContext.get().getEmployeeId())) {
            if (userContext.get().isDepartAdmin()) {
                String deptAdminSql = getDeptAdminSql(criteria.getTreeCodes());
                if (!CollectionUtils.isEmpty(criteria.getDepartIds())) {
                    qString += " and ((e.id=:selfEmployeeId and exists(select k from e.departmentEmployeeRelations k where k.department.id in (:selfDepartmentIds))) or exists (SELECT k FROM e.departmentEmployeeRelations k WHERE k.department.id IN (:departmentIds) " + deptAdminSql + " ))";
                    cString += " and ((e.id=:selfEmployeeId and exists(select k from e.departmentEmployeeRelations k where k.department.id in (:selfDepartmentIds))) or exists (SELECT k FROM e.departmentEmployeeRelations k WHERE k.department.id IN (:departmentIds) " + deptAdminSql + " ))";
                    parameters.put("departmentIds", criteria.getDepartIds());
                    parameters.put("selfDepartmentIds", criteria.getDepartIds());
                } else {
                    qString += " and (e.id=:selfEmployeeId or exists (SELECT k FROM e.departmentEmployeeRelations k WHERE 1=1 " + deptAdminSql + " ))";
                    cString += " and (e.id=:selfEmployeeId or exists (SELECT k FROM e.departmentEmployeeRelations k WHERE 1=1 " + deptAdminSql + "))";
                }
                parameters.put("selfEmployeeId", userContext.get().getEmployeeId());
            } else {
                if (!CollectionUtils.isEmpty(criteria.getDepartIds())) {
                    qString += " and ((e.id=:selfEmployeeId and exists(select k from e.departmentEmployeeRelations k where k.department.id in (:selfDepartmentIds))))";
                    cString += " and ((e.id=:selfEmployeeId and exists(select k from e.departmentEmployeeRelations k where k.department.id in (:selfDepartmentIds))))";
                    parameters.put("selfDepartmentIds", criteria.getDepartIds());
                } else {
                    qString += " and (e.id=:selfEmployeeId)";
                    cString += " and (e.id=:selfEmployeeId)";
                }
                parameters.put("selfEmployeeId", userContext.get().getEmployeeId());
            }
        } else {
            /**
             * 所在部门
             */
            if (!CollectionUtils.isEmpty(criteria.getDepartIds())) {
                qString += " and exists(select de from DepartmentEmployeeRelationEntity de where de.employee.id=e.id and de.department.id in (:departmentIds))";
                cString += " and exists(select de from DepartmentEmployeeRelationEntity de where de.employee.id=e.id and de.department.id in (:departmentIds))";

                parameters.put("departmentIds", criteria.getDepartIds());
            }
        }

        //加上排序
        qString += " order by e.hiredDate desc ";
        TypedQuery<EmployeeDto> employeeTypedQuery = this.entityManager.createQuery(qString, EmployeeDto.class);
        TypedQuery<Long> totalCountQuery = this.entityManager.createQuery(cString, Long.class);

        for (String key : parameters.keySet()) {
            employeeTypedQuery = employeeTypedQuery.setParameter(key, parameters.get(key));
            totalCountQuery = totalCountQuery.setParameter(key, parameters.get(key));
        }


        long totalCount = totalCountQuery.getSingleResult();
        Collection<EmployeeDto> result = null;
        if (totalCount > 0) {
            result = employeeTypedQuery
                    .setFirstResult(criteria.getSkip())
                    .setMaxResults(criteria.getPageSize())
                    .getResultList();
            Collection<String> employeeIds = result.stream().map(ii -> ii.getId()).collect(Collectors.toList());

            if (employeeIds.size() > 0) {
                TypedQuery<DepartDto> mapTypedQuery =
                        this.entityManager.createQuery("select new com.bcxin.tenant.domain.readers.dtos.DepartDto(de.employee.id as employeeId, dp.id as departId, dp.name as departName, de.leaderType as departLeader) from DepartmentEmployeeRelationEntity de join de.department dp where de.employee.id in (:employeeIds)",
                                DepartDto.class);
                mapTypedQuery.setParameter("employeeIds", employeeIds);
                Collection<DepartDto> departDtos = mapTypedQuery.getResultList();
                Map<String, List<DepartDto>> departMap = departDtos.stream().collect(Collectors.groupingBy(DepartDto::getEmployeeId));
                for (EmployeeDto rt : result) {
                    List<DepartDto> departDtoList = departMap.get(rt.getId());
                    if (departDtoList != null) {
                        Optional<DepartDto> leaderDept = departDtoList.stream().filter(dto -> dto.getDepartLeader() != null && dto.getDepartLeader()).findFirst();
                        rt.assignDepart(departDtoList.stream().map(DepartDto::getDepartId).collect(Collectors.joining(", ")),
                                departDtoList.stream().map(DepartDto::getDepartName).collect(Collectors.joining(", ")),
                                leaderDept.map(DepartDto::getDepartLeader).orElse(null));
                    }
                }

                TypedQuery<ContractDto> contractDtoTypedQuery =
                        this.entityManager.createQuery("select new com.bcxin.tenant.domain.readers.dtos.ContractDto(de.employee.id as employeeId, de.id as contractId, de.endDate as endDate) from ContractEntity de where de.employee.id in (:employeeIds) and de.status = com.bcxin.Infrastructures.enums.ValidStatus.Valid and CURDATE() between de.beginDate and de.endDate ",
                                ContractDto.class);
                contractDtoTypedQuery.setParameter("employeeIds", employeeIds);
                Collection<ContractDto> contractDtos = contractDtoTypedQuery.getResultList();
                Map<String, List<ContractDto>> contractMap = contractDtos.stream().collect(Collectors.groupingBy(ContractDto::getEmployeeId));
                result.forEach(rt -> {
                    if (contractMap.get(rt.getId()) != null) {
                        List<ContractDto> contractDtoList = contractMap.get(rt.getId());
                        contractDtoList = contractDtoList.stream().sorted(Comparator.comparing(ContractDto::getValidDateTo)).collect(Collectors.toList());
                        rt.assignContractStatus(true, contractDtoList.get(contractDtoList.size() - 1).getValidDateTo());
                    } else {
                        rt.assignContractStatus(false, null);
                    }
                });

                mapTypedQuery = this.entityManager.createQuery("select new com.bcxin.tenant.domain.readers.dtos.DepartDto(de.employee.id as employeeId, dp.id as departId, dp.name as departName) from DepartmentAdminEntity de join de.department dp where de.employee.id in (:employeeIds)",
                        DepartDto.class);
                mapTypedQuery.setParameter("employeeIds", employeeIds);
                departDtos = mapTypedQuery.getResultList();
                departMap = departDtos.stream().collect(Collectors.groupingBy(DepartDto::getEmployeeId));
                for (EmployeeDto rt : result) {
                    if (departMap.get(rt.getId()) != null) {
                        rt.assignAdminDepart(departMap.get(rt.getId()).stream().map(DepartDto::getDepartName).collect(Collectors.joining(", ")));
                    }
                }
                List<TenantUserCredentialDetailsEntity> cerDetails = this.getCerDetailsByUserIds(CerType.Qualification.getTypeValue(), result.stream().map(EmployeeDto::getUserId).collect(Collectors.toList()));
                List<TenantUserCredentialDetailsEntity> gradeDetails = this.getCerDetailsByUserIds(CerType.Grade.getTypeValue(), result.stream().map(EmployeeDto::getUserId).collect(Collectors.toList()));
                if (cerDetails != null && cerDetails.size() > 0) {
                    Map<String, List<TenantUserCredentialDetailsEntity>> cerDetailMap = cerDetails.stream().collect(Collectors.groupingBy(i -> i.getTenantUserId()));
                    for (EmployeeDto rt : result) {
                        List<TenantUserCredentialDetailsEntity> cerDetailList = cerDetailMap.get(rt.getUserId());
                        if (cerDetailList != null) {
                            String finalAreaCode = areaCode;
                            for (TenantUserCredentialDetailsEntity tenantUserCredentialDetailsEntity : cerDetailList) {
                                if (StrUtil.isNotEmpty(tenantUserCredentialDetailsEntity.getAreaCode()) && tenantUserCredentialDetailsEntity.getAreaCode().length() >= 2) {
                                    if (AuthUtil.getShortAreaCode(tenantUserCredentialDetailsEntity.getAreaCode()).equals(AuthUtil.getShortAreaCode(finalAreaCode))) {
                                        rt.setCerStatus(true);
                                        List<TenantUserCredentialDetailsEntity> tenantUserCredentialDetailsEntities = cerDetailMap.get(rt.getUserId());
                                        for (TenantUserCredentialDetailsEntity userCredentialDetailsEntity : tenantUserCredentialDetailsEntities) {
                                            if (AuthUtil.getShortAreaCode(userCredentialDetailsEntity.getAreaCode()).equals(AuthUtil.getShortAreaCode(finalAreaCode))) {
                                                rt.setCerNo(userCredentialDetailsEntity.getCerNo());
                                            }
                                        }
                                    } else if (tenantUserCredentialDetailsEntity.getAreaCode() == "#") {
                                        rt.setCerStatus(false);
                                    }
                                    ;
                                }
                            }
                        }
                    }
                }

                if (gradeDetails != null && gradeDetails.size() > 0) {
                    Map<String, List<TenantUserCredentialDetailsEntity>> cerDetailMap = gradeDetails.stream().collect(Collectors.groupingBy(i -> i.getTenantUserId()));
                    for (EmployeeDto rt : result) {
                        List<TenantUserCredentialDetailsEntity> cerDetailList = cerDetailMap.get(rt.getUserId());
                        if (cerDetailList != null) {
                            String finalAreaCode = areaCode;
                            for (TenantUserCredentialDetailsEntity tenantUserCredentialDetailsEntity : cerDetailList) {
                                if (StrUtil.isNotEmpty(tenantUserCredentialDetailsEntity.getAreaCode()) && tenantUserCredentialDetailsEntity.getAreaCode().length() >= 2) {
                                    if (AuthUtil.getShortAreaCode(tenantUserCredentialDetailsEntity.getAreaCode()).equals(AuthUtil.getShortAreaCode(finalAreaCode))) {
                                        rt.setGradeCerStatus(true);
                                        List<TenantUserCredentialDetailsEntity> tenantUserCredentialDetailsEntities = cerDetailMap.get(rt.getUserId());
                                        for (TenantUserCredentialDetailsEntity userCredentialDetailsEntity : tenantUserCredentialDetailsEntities) {
                                            rt.assignGradeCer(userCredentialDetailsEntity.getCerNo(), userCredentialDetailsEntity.getAppraisalGrade());
                                            rt.setGradeLevel(userCredentialDetailsEntity.getGradeLevel().getTypeValue());
                                        }
                                    } else if (tenantUserCredentialDetailsEntity.getAreaCode() == "#") {
                                        rt.setGradeCerStatus(false);
                                    }
                                    ;
                                }
                            }
                        }
                    }
                }
            }
        } else {
            result = Collections.emptyList();
        }

        return Pageable.create(criteria.getPageIndex(), criteria.getPageSize(), (int) totalCount, result);
    }

    /**
     * description：获取我管理的部门的所有职员id
     * author：linchunpeng
     * date：2024/4/24
     */
    @Override
    public List<String> getMyDeptEmployeeIdList(Set<String> treeCodes) {
        String deptAdminSql = getDeptAdminSql(treeCodes);
        String sql = "SELECT k.employee.id FROM DepartmentEmployeeRelationEntity k WHERE 1=1 " + deptAdminSql + "";
        TypedQuery<String> typedQuery = this.entityManager.createQuery(sql, String.class);
        return typedQuery.getResultList();
    }

    @Override
    public Collection<MyOrganizationProfileDto> getMyOrganizationProfiles(String userId) {
        CriteriaBuilder builder = this.entityManager.getCriteriaBuilder();
        CriteriaQuery<MyOrganizationProfileDto> criteria = builder.createQuery(MyOrganizationProfileDto.class);
        Root<EmployeeEntity> root = criteria.from(EmployeeEntity.class);
        Join<EmployeeEntity, OrganizationEntity> employeeOrganizationEntityJoin = root.join("organization", JoinType.INNER);


        Selection<MyOrganizationProfileDto> myOrganizationProfileDtoSelection =
                builder.construct(MyOrganizationProfileDto.class,
                        employeeOrganizationEntityJoin.get("id"),
                        employeeOrganizationEntityJoin.get("name"),
                        employeeOrganizationEntityJoin.get("industryCode"),
                        employeeOrganizationEntityJoin.get("institutionalCode"),
                        employeeOrganizationEntityJoin.get("superviseRegionCode"),
                        employeeOrganizationEntityJoin.<TenantUserEntity>get("tenantUser").get("id"),
                        root.get("id"),
                        root.<TenantUserEntity>get("tenantUser").get("id"),
                        root.get("occupationType"),
                        root.get("status"),
                        root.get("hiredDate"),
                        root.get("masterSlaveType"),
                        root.get("domainAdmin"),
                        employeeOrganizationEntityJoin.get("organizationLevel")

                );

        CriteriaBuilder.In<EmploymentStatus> employmentStatusIn = builder.in(root.get("status"));
        employmentStatusIn.value(EmploymentStatus.OnJob);
        employmentStatusIn.value(EmploymentStatus.PendingForOffJob);
        criteria = criteria.where(builder.equal(root.<TenantUserEntity>get("tenantUser").get("id"), userId), employmentStatusIn,
                builder.equal(employeeOrganizationEntityJoin.<ApprovedInformationValueType>get("approvedInformationValueType").get("status"), ApprovedStatus.Passed)
        ).select(myOrganizationProfileDtoSelection);

        Collection<MyOrganizationProfileDto> result = entityManager.createQuery(criteria).getResultList();
        if (result.size() > 0) {
            List<String> employeeIds = result.stream().filter(i -> !i.isDomainAdmin()).map(MyOrganizationProfileDto::getEmployeeId).collect(Collectors.toList());
            /**
             * 查出是否部门管理员
             */
            if (employeeIds != null && employeeIds.size() > 0) {
                TypedQuery<String> selectedDepartAdminQuery =
                        entityManager.createQuery("select new java.lang.String(e.employee.id) from DepartmentAdminEntity e where e.employee.id in (?1) group by e.employee.id", String.class);
                selectedDepartAdminQuery.setParameter(1, employeeIds);
                employeeIds = selectedDepartAdminQuery.getResultList();
                if (employeeIds.size() > 0) {
                    Map<String, String> employeeMap = employeeIds.stream().collect(Collectors.toMap(String::new, Function.identity()));
                    for (MyOrganizationProfileDto myOrganizationProfileDto : result) {
                        myOrganizationProfileDto.setDepartAdmin(employeeMap.get(myOrganizationProfileDto.getEmployeeId()) != null);
                    }
                }
            }
        }

        return result;
    }

    /**
     * 判断是不是指定部门，是返回0，否则返回其它
     *
     * @param organId
     * @param tenantUserId
     * @return
     */
    @Override
    public int isAssign(String organId, String tenantUserId) {
        /**
         * 查出该租户再该公司的为职员信息
         */
        TypedQuery<EmployeeDepartData> selectedDepartmentQuery =
                entityManager.createQuery("select new com.bcxin.tenant.domain.readers.tmps.EmployeeDepartData(d.id,d.permissionType,d.permissionConfig,e.id) " +
                                " from EmployeeEntity e join e.departmentEmployeeRelations c join c.department d  where e.status <> com.bcxin.Infrastructures.enums.EmploymentStatus.OffJob and e.organization.id=?1 and e.tenantUser.id=?2",
                        EmployeeDepartData.class);

        selectedDepartmentQuery.setParameter(1, organId);
        selectedDepartmentQuery.setParameter(2, tenantUserId);
        List<EmployeeDepartData> employeeDepartDataList = selectedDepartmentQuery.getResultList();

        if (CollectionUtils.isEmpty(employeeDepartDataList)) {
            return 1;
        }
        return DepartImPermissionType.Special.compareTo(employeeDepartDataList.get(0).getPermissionType());
    }

    /**
     * 获取权限部门信息
     *
     * @param organId
     * @param tenantUserId
     * @return
     */
    @Override
    public Collection<MyImDepartDto> getMyDeparts(String organId, String tenantUserId) {
        /**
         * 查出该租户再该公司的为职员信息
         */
        TypedQuery<EmployeeDepartData> selectedDepartmentQuery =
                entityManager.createQuery("select new com.bcxin.tenant.domain.readers.tmps.EmployeeDepartData(d.id,d.permissionType,d.permissionConfig,e.id) " +
                                " from EmployeeEntity e join e.departmentEmployeeRelations c join c.department d  where e.status <> com.bcxin.Infrastructures.enums.EmploymentStatus.OffJob and e.organization.id=?1 and e.tenantUser.id=?2 and d.deleted = false",
                        EmployeeDepartData.class);

        selectedDepartmentQuery.setParameter(1, organId);
        selectedDepartmentQuery.setParameter(2, tenantUserId);
        List<EmployeeDepartData> employeeDepartDataList = selectedDepartmentQuery.getResultList();
        if (CollectionUtils.isEmpty(employeeDepartDataList)) {
            return Collections.EMPTY_LIST;
        }

        /**
         * 执行通讯录的查询操作
         */
        List<String> likeDepartIds = new ArrayList<>();
        List<String> equalMatchedDepartIds = new ArrayList<>();
        AtomicBoolean showWhole = new AtomicBoolean(false);

        StringBuilder selfForDepartCondition = new StringBuilder();
        AtomicInteger paramIndex = new AtomicInteger(2);
        Map<Integer, Object> parameters = new HashMap<>();
        parameters.put(1, organId);
        int self = 0;
        for (EmployeeDepartData ed : employeeDepartDataList) {
            switch (ed.getPermissionType()) {
                case Whole:
                    showWhole.set(true);
                    break;
                case OnDepartAndSub:
                    likeDepartIds.add(ed.getId());
                    break;
                case JustOnDepart:
                    equalMatchedDepartIds.add(ed.getId());
                    break;
                case Special:
                    DepartImAllowedDepartSnapshot snapshot = jsonProvider.toObject(DepartImAllowedDepartSnapshot.class, ed.getPermissionConfig());
                    if (snapshot != null && !CollectionUtils.isEmpty(snapshot.getDepartIds())) {
                        equalMatchedDepartIds.addAll(snapshot.getDepartIds());
                    }
                    break;
                case JustSelf:
                    self = 1;
                    if (selfForDepartCondition.length() > 0) {
                        selfForDepartCondition.append(" OR ");
                    }
                    selfForDepartCondition.append(String.format(" (d.id=?%s and c.employee.id=?%s)", paramIndex.get(), paramIndex.get() + 1));
                    parameters.put(paramIndex.get(), ed.getId());
                    parameters.put(paramIndex.get() + 1, ed.getEmployeeId());

                    paramIndex.incrementAndGet();
                    paramIndex.incrementAndGet();
                    break;
            }
        }


        StringBuilder noSelfForDepartCondition = new StringBuilder();
        if (!showWhole.get()) {

            /**
             * 查询: 可见所在部门以及下级部门成员
             */
            if (!CollectionUtils.isEmpty(likeDepartIds)) {
                StringBuilder sbLikeCondition = new StringBuilder();
                sbLikeCondition.append(" (");
                for (int lindex = 0; lindex < likeDepartIds.size(); lindex++) {
                    if (lindex > 0) {
                        sbLikeCondition.append(" OR ");
                    }
                    sbLikeCondition.append(String.format(" d.indexTree like ?%s", paramIndex.get()));
                    parameters.put(paramIndex.get(), "%" + likeDepartIds.get(lindex) + "%");

                    paramIndex.incrementAndGet();
                }
                sbLikeCondition.append(") ");

                noSelfForDepartCondition.append(sbLikeCondition);
            }

            /**
             * 可见所在部门 + 可见指定部门
             */
            if (!CollectionUtils.isEmpty(equalMatchedDepartIds)) {
                if (noSelfForDepartCondition.length() > 0) {
                    noSelfForDepartCondition.append(" OR ");
                }
                noSelfForDepartCondition.append(String.format(" d.id in (?%s)", paramIndex.get()));
                parameters.put(paramIndex.get(), equalMatchedDepartIds);
            }
        }
        //正常情况只查部门表
        String qString = "select new com.bcxin.tenant.domain.repository.readers.tmps.DepartmentReadDto(d.id,d.name,d.displayOrder,d.parent.id)" +
                " from DepartmentEntity d where d.organization.id=?1 and d.deleted=false ";
        //如果权限是只能看到自己，查询语句就加入departmentEmployeeRelations关联表
        if (self == 1) {
            qString = "select new com.bcxin.tenant.domain.repository.readers.tmps.DepartmentReadDto(d.id,d.name,d.displayOrder,d.parent.id)" +
                    " from DepartmentEntity d left join d.departmentEmployeeRelations c where d.organization.id=?1 and d.deleted=false ";
        }

        Collection<String> whereItems = new ArrayList<>();
        whereItems.add(selfForDepartCondition.toString());
        whereItems.add(noSelfForDepartCondition.toString());
        whereItems = whereItems.stream().filter(ii -> StringUtils.hasLength(ii)).collect(Collectors.toList());

        if (whereItems.size() > 0) {
            qString = String.format("%s and (%s)", qString, whereItems.stream().collect(Collectors.joining(" or ")));
        }
        TypedQuery<DepartmentReadDto> departEmployeeQuery =
                entityManager.createQuery(qString, DepartmentReadDto.class);

        for (Integer key : parameters.keySet()) {
            departEmployeeQuery = departEmployeeQuery.setParameter(key, parameters.get(key));
        }
        List<DepartmentReadDto> result = departEmployeeQuery.getResultList();

        return this.dataTranslate.translate2MyImDeparts(result);
    }

    /**
     * 获取部门底下人员信息
     *
     * @param request
     * @return
     */
    @Override
    public Collection<MyImContactDto> getMyImContacts(IMContactCriteria request) {
        Map<String, String> parameters = new HashMap<>();
        parameters.put("organId", request.getOrganizationId());
        parameters.put("departId", request.getDepartId());

        String qString = "select new com.bcxin.tenant.domain.repository.readers.tmps.EmployeeContactReadDto(t.id,c.employee.id,t.name,t.telephone,t.headPhoto,t.sex,t.imIdentity)" +
                " from DepartmentEntity d join d.departmentEmployeeRelations c join c.employee e join e.tenantUser t where d.organization.id=:organId and d.id=:departId and e.status <>1 ";

        if (!this.isSpecialDepart(request)) {
            qString += " and (d.permissionType <> com.bcxin.Infrastructures.enums.DepartImPermissionType.JustSelf or (" +
                    " d.permissionType = com.bcxin.Infrastructures.enums.DepartImPermissionType.JustSelf and t.id =:tenantUserId" +
                    " )" +
                    " or exists (select 1 from DepartmentEntity tt1 join tt1.departmentEmployeeRelations tt2 join tt2.employee.tenantUser tt3 " +
                    " where tt1.organization.id=:organId  " +
                    " and tt3.id =:tenantUserId  " +
                    " and ( tt1.permissionType =3 " +
                    " or (tt1.permissionType =1 and d.indexTree like concat(tt1.indexTree,'%')) " +
                    ")) " +
                    ")";
            parameters.put("tenantUserId", request.getUserId());
        }

        TypedQuery<EmployeeContactReadDto> employeeQuery =
                entityManager.createQuery(qString, EmployeeContactReadDto.class);

        for (String key : parameters.keySet()) {
            employeeQuery.setParameter(key, parameters.get(key));
        }

        List<EmployeeContactReadDto> result = employeeQuery.getResultList();
        return this.dataTranslate.translate2MyImContacts(result);
    }

    public Collection<MyImContactDto> getMyImContactSearch(IMContactCriteria request) {
        String qString = "select new com.bcxin.tenant.domain.repository.readers.tmps.EmployeeContactReadDto(t.id,c.employee.id,t.name,t.telephone,t.headPhoto,t.sex,t.imIdentity)" +
                " from DepartmentEntity d join d.departmentEmployeeRelations c join c.employee e join e.tenantUser t join t.selectedCredential s where d.organization.id=:organId  and e.status <>1 and d.deleted = false ";

        Map<String, Object> parameters = new HashMap<>();
        parameters.put("organId", request.getOrganizationId());

        if (StrUtil.isNotEmpty(request.getDepartId())) {
            qString += " and d.id=:departId ";
            parameters.put("departId", request.getDepartId());
        }

        if (StrUtil.isNotEmpty(request.getKeyword())) {
            qString += " and (t.name like :keyword or t.telephone like :keyword or s.number like :keyword)";
            parameters.put("keyword", "%" + request.getKeyword() + "%");
        }

        List<String> specialDeparts = this.getSpecialDeparts(request);
        if (specialDeparts.size() == 0) {
            specialDeparts.add("specialDeparts");
        }
        qString += " and ((d.id in (:specialDeparts)) ";
        parameters.put("specialDeparts", specialDeparts);

        qString += " or ((" +
                " d.permissionType = com.bcxin.Infrastructures.enums.DepartImPermissionType.JustSelf and t.id =:tenantUserId" +
                " )" +
                " or exists (select 1 from DepartmentEntity tt1 join tt1.departmentEmployeeRelations tt2 join tt2.employee.tenantUser tt3 " +
                " where tt1.organization.id=:organId  " +
                " and tt3.id =:tenantUserId  " +
                " and ( tt1.permissionType =3 " +
                " or (tt1.permissionType =0 and d.id = tt1.id) " +
                " or (tt1.permissionType =1 and d.indexTree like concat(tt1.indexTree,'%')) " +
                ")) " +
                "))";
        parameters.put("tenantUserId", request.getUserId());

        TypedQuery<EmployeeContactReadDto> employeeQuery =
                entityManager.createQuery(qString, EmployeeContactReadDto.class);

        for (String key : parameters.keySet()) {
            employeeQuery.setParameter(key, parameters.get(key));
        }

        List<EmployeeContactReadDto> result = employeeQuery.getResultList();
        return this.dataTranslate.translate2MyImContacts(result);
    }

    @Override
    public Pageable<CompanyCredentialResponse> getCompanys(OrganizationCriteria organizationCriteria) {
        if (StrUtil.isEmpty(organizationCriteria.getName())) {
            return Pageable.create(organizationCriteria.getPageIndex(), organizationCriteria.getPageSize(), 0, null);
        }
        String qString = "select t from CompanyCredentialViewEntity t where t.comName = '" + organizationCriteria.getName() + "'";
        String cString = "select count(1) from CompanyCredentialViewEntity t where t.comName = '" + organizationCriteria.getName() + "'";
        TypedQuery<CompanyCredentialViewEntity> companyQuery = entityManager.createQuery(qString, CompanyCredentialViewEntity.class);
        TypedQuery<Long> cquery = this.entityManager.createQuery(cString, Long.class);
        long totalCount = cquery.getSingleResult();
        if (totalCount > 0) {
            List<CompanyCredentialViewEntity> result = companyQuery.setFirstResult(organizationCriteria.getSkip()).setMaxResults(organizationCriteria.getPageSize()).getResultList();
            List<CompanyCredentialResponse> comList = new ArrayList<>();
            for (CompanyCredentialViewEntity companyCredential : result) {
                comList.add(CompanyCredentialResponse.create(companyCredential.getComName(),
                        companyCredential.getLegalPerson(),
                        companyCredential.getTel(),
                        companyCredential.getRegisteredMoney(),
                        companyCredential.getCerNo(),
                        companyCredential.getAddress(),
                        companyCredential.getLocation(),
                        companyCredential.getCerDate(),
                        companyCredential.getSecurityScopeType()));
            }

            return Pageable.create(organizationCriteria.getPageIndex(), organizationCriteria.getPageSize(), (int) totalCount, comList);
        }
        return Pageable.create(organizationCriteria.getPageIndex(), organizationCriteria.getPageSize(), (int) totalCount, null);
    }


    public Pageable<ImContactDto> contactSearch(IMContactCriteria criteria) {
        String qString = "select new com.bcxin.tenant.domain.readers.dtos.ImContactDto(t.id,t.name,t.headPhoto,t.telephone,s.number,t.imIdentity,t.sex,t.oneInchColorWhitePhoto)" +
                " from TenantUserEntity t join t.selectedCredential s where 1=1  ";
        String cString = "select count(1) " +
                " from TenantUserEntity t join t.selectedCredential s where 1=1  ";


        Map<String, Object> parameters = new HashMap<>();
        if (StrUtil.isNotEmpty(criteria.getKeyword())) {
            qString += " and (t.telephone =:keyword or s.number =:keyword)";
            cString += " and (t.telephone =:keyword or s.number =:keyword)";
            parameters.put("keyword", criteria.getKeyword());
        }
        TypedQuery<ImContactDto> employeeQuery = entityManager.createQuery(qString, ImContactDto.class);
        TypedQuery<Long> cquery = this.entityManager.createQuery(cString, Long.class);
        for (String key : parameters.keySet()) {
            employeeQuery.setParameter(key, parameters.get(key));
            cquery.setParameter(key, parameters.get(key));

        }
        List<ImContactDto> result = employeeQuery.setFirstResult(criteria.getSkip()).setMaxResults(criteria.getPageSize()).getResultList();
        long totalCount = cquery.getSingleResult();
        return Pageable.create(criteria.getPageIndex(), criteria.getPageSize(), (int) totalCount, result);
    }

    private List<String> getSpecialDeparts(IMContactCriteria request) {
        List<String> result = new ArrayList<>();
        String sql = "select t3.permissionConfig from EmployeeEntity t1 join t1.departmentEmployeeRelations t2 join t2.department t3  where t1.tenantUser.id = :tenantUserId and t3.permissionType = com.bcxin.Infrastructures.enums.DepartImPermissionType.Special";
        Map<String, String> parameters = new HashMap<>();
        parameters.put("tenantUserId", request.getUserId());
        TypedQuery<String> employeeLeaveDtoTypedQuery = this.entityManager.createQuery(sql, String.class);
        for (String key : parameters.keySet()) {
            employeeLeaveDtoTypedQuery.setParameter(key, parameters.get(key));
        }
        List<String> list = employeeLeaveDtoTypedQuery.getResultList();
        DepartImAllowedDepartSnapshot snapshot = null;
        if (list.size() > 0) {
            for (String permissionConfig : list) {
                snapshot = JSON.parseObject(permissionConfig, DepartImAllowedDepartSnapshot.class);
                result.addAll(snapshot.getDepartIds());
            }
        }
        return result;
    }

    private boolean isSpecialDepart(IMContactCriteria request) {
        boolean result = false;
        List<String> list = this.getSpecialDeparts(request);
        if (list.size() > 0 && list.contains(request.getDepartId())) {
            result = true;
        }

        return result;
    }

    /**
     * 查询部门管理员功能
     *
     * @param organizationId
     * @param criteria
     * @return
     */
    @Override
    public Pageable<DepartAdminDto> findDepartAdmins(String organizationId, DepartAdminCriteria criteria) {
        /**
         * 获取部门管理员信息
         */
        String qString =
                "select new com.bcxin.tenant.domain.repository.readers.tmps.DepartAdminReadDto(e.id,t.name,t.telephone,p.department.name,d.department.name) " +
                        " from DepartmentAdminEntity d join d.employee e left join e.tenantUser t join e.departmentEmployeeRelations p  " +
                        " where e.status<>com.bcxin.Infrastructures.enums.EmploymentStatus.OffJob and d.organization.id=:organizationId";

        Map<String, Object> parameters = new HashMap<>();

        parameters.put("organizationId", organizationId);

        if (CollUtil.isNotEmpty(criteria.getDepartId())) {
            qString += " AND p.department.id in( :departId )  ";
            parameters.put("departId", criteria.getDepartId());
        }

        if (CollUtil.isNotEmpty(criteria.getAdminDepartIds())) {
            qString += " AND d.department.id in( :adminDepartIds )  ";
            parameters.put("adminDepartIds", criteria.getAdminDepartIds());
        }

        if (StrUtil.isNotEmpty(criteria.getKeyword())) {
            qString += " and (t.name like :keyword or t.telephone like :keyword )";
            parameters.put("keyword", "%" + criteria.getKeyword() + "%");
        }

        TenantUserContext userContext = TenantContext.getInstance().getUserContext();
        if (!criteria.isIgnorePermission() && userContext != null && userContext.get() != null && !userContext.get().isMaster() &&
                StringUtils.hasLength(userContext.get().getEmployeeId())) {
            qString += " and exists(select da from DepartmentAdminEntity da where da.employee.id=:employeeId and exists(select 1 from d.department dp where dp.indexTree like concat(da.department.indexTree ,'%')))";
            qString += " and exists(select 1 from p.department dp join DepartmentEntity dade on dp.indexTree LIKE concat(dade.indexTree,'%') join DepartmentAdminEntity da on da.employee.id=:employeeId and da.department.id = dade.id )";
            parameters.put("employeeId", userContext.get().getEmployeeId());
        }

        TypedQuery<DepartAdminReadDto> adminReadDtoTypedQuery =
                this.entityManager.createQuery(qString, DepartAdminReadDto.class);
        for (String key : parameters.keySet()) {
            adminReadDtoTypedQuery.setParameter(key, parameters.get(key));
        }

        Collection<DepartAdminReadDto> departAdminReads = adminReadDtoTypedQuery.getResultList();

        /**
         * 获取部门管理员信息
         */
        Collection<DepartAdminDto> wholeDepartAdmins = this.dataTranslate.translate2DepartAdmins(departAdminReads);

        Pageable<DepartAdminDto> pageable = Pageable.create(criteria.getPageIndex(), criteria.getPageSize(),
                wholeDepartAdmins.size(),
                wholeDepartAdmins.stream().skip(criteria.getSkip()).limit(criteria.getPageSize())
                        .collect(Collectors.toList())
        );
        return pageable;
    }

    /**
     * 查询导出部门管理员功能
     *
     * @param organizationId
     * @param criteria
     * @return
     */
    @Override
    public List<DepartmentAdminExportDto> findExportDepartAdmins(String organizationId, DepartAdminCriteria criteria) {
        /**
         * 获取部门管理员信息
         */
        String qString =
                "select new com.bcxin.tenant.domain.repository.readers.tmps.DepartAdminReadDto(e.id,t.name,t.telephone,p.department.name,d.department.name) " +
                        " from DepartmentAdminEntity d join d.employee e left join e.tenantUser t join e.departmentEmployeeRelations p  " +
                        " where e.status<>com.bcxin.Infrastructures.enums.EmploymentStatus.OffJob and d.organization.id=:organizationId";

        Map<String, Object> parameters = new HashMap<>();

        parameters.put("organizationId", organizationId);

        if (CollUtil.isNotEmpty(criteria.getDepartId())) {
            qString += " AND p.department.id in( :departId )  ";
            parameters.put("departId", criteria.getDepartId());
        }

        if (CollUtil.isNotEmpty(criteria.getAdminDepartIds())) {
            qString += " AND d.department.id in( :adminDepartIds )  ";
            parameters.put("adminDepartIds", criteria.getAdminDepartIds());
        }

        if (StrUtil.isNotEmpty(criteria.getKeyword())) {
            qString += " and (t.name like :keyword or t.telephone like :keyword )";
            parameters.put("keyword", "%" + criteria.getKeyword() + "%");
        }

        TenantUserContext userContext = TenantContext.getInstance().getUserContext();
        if (!criteria.isIgnorePermission() && userContext != null && userContext.get() != null && !userContext.get().isMaster() &&
                StringUtils.hasLength(userContext.get().getEmployeeId())) {
            qString += " and exists(select da from DepartmentAdminEntity da where da.employee.id=:employeeId and exists(select 1 from d.department dp where dp.indexTree like concat(da.department.indexTree ,'%')))";
            qString += " and exists(select 1 from p.department dp join DepartmentEntity dade on dp.indexTree LIKE concat(dade.indexTree,'%') join DepartmentAdminEntity da on da.employee.id=:employeeId and da.department.id = dade.id )";
            parameters.put("employeeId", userContext.get().getEmployeeId());
        }

        TypedQuery<DepartAdminReadDto> adminReadDtoTypedQuery =
                this.entityManager.createQuery(qString, DepartAdminReadDto.class);
        for (String key : parameters.keySet()) {
            adminReadDtoTypedQuery.setParameter(key, parameters.get(key));
        }

        Collection<DepartAdminReadDto> departAdminReads = adminReadDtoTypedQuery.getResultList();

        /**
         * 获取部门管理员信息
         */
        List<DepartmentAdminExportDto> wholeDepartAdmins = this.dataTranslate.translate2DepartExportAdmins(departAdminReads);

        return wholeDepartAdmins;
    }

    @Override
    public UserOrganBasicDto getUserOrganBasicDto(String organizationId, String tenantUserId, PersonResourceType resourceType) {
        try {
            TypedQuery<UserOrganBasicDto> userOrganBasicDtoTypedQuery = null;
            UserOrganBasicDto userOrganBasicDto = null;
            if (resourceType == PersonResourceType.Member) {
                userOrganBasicDtoTypedQuery = this.entityManager.createQuery(
                        "select new com.bcxin.tenant.domain.readers.dtos.UserOrganBasicDto(t.id,t.name, o.id,o.name, o.superviseRegionCode,e.id,e.memberType) " +
                                " from ExternalMemberEntity e join e.tenantUser t join OrganizationEntity o on e.referenceNumber=o.id" +
                                " where t.id=?1 and o.id=?2 and " +
                                " e.approvedInformationValueType.status = com.bcxin.Infrastructures.enums.ApprovedStatus.Passed", UserOrganBasicDto.class);
                userOrganBasicDtoTypedQuery.setParameter(1, tenantUserId);
                userOrganBasicDtoTypedQuery.setParameter(2, organizationId);
                userOrganBasicDto = userOrganBasicDtoTypedQuery.getSingleResult();

            } else {
                userOrganBasicDtoTypedQuery = this.entityManager.createQuery(
                        "select new com.bcxin.tenant.domain.readers.dtos.UserOrganBasicDto(t.id,t.name, o.id,o.name, o.superviseRegionCode,e.id,e.masterSlaveType,e.domainAdmin) " +
                                " from EmployeeEntity e join e.tenantUser t join e.organization o" +
                                " where t.id=?1 and o.id=?2 and " +
                                " e.status<>com.bcxin.Infrastructures.enums.EmploymentStatus.OffJob", UserOrganBasicDto.class);
                userOrganBasicDtoTypedQuery.setParameter(1, tenantUserId);
                userOrganBasicDtoTypedQuery.setParameter(2, organizationId);

                userOrganBasicDto = userOrganBasicDtoTypedQuery.getSingleResult();
                if (userOrganBasicDto != null) {
                    /**
                     * 查出是否部门管理员
                     */
                    TypedQuery<String> selectedDepartAdminQuery =
                            entityManager.createQuery("select new java.lang.String(e.employee.id) from DepartmentAdminEntity e where e.employee.id = ?1 group by e.employee.id", String.class);
                    selectedDepartAdminQuery.setParameter(1, userOrganBasicDto.getEmployeeId());
                    List<String> employeeIds = selectedDepartAdminQuery.getResultList();
                    if (employeeIds.size() > 0) {
                        userOrganBasicDto.setDepartAdmin(true);
                    }
                }
            }

            return userOrganBasicDto;
        } catch (NoResultException ex) {
            return null;
        }
    }

    @Override
    public Pageable<EmployeeLeaveDto> findLeaveEmployees(String organizationId, LeaveEmployeeCriteria criteria) {
        /*
        String qString = "select new com.bcxin.tenant.domain.readers.dtos.EmployeeLeaveDto(c.id,t.name,t.telephone,c.leaveNote,s.credentialType,s.number,c.leaveTime,c.leaveNote,c.id,c.leaveRequestTime) " +
                " from EmployeeEntity c join c.tenantUser t left join t.selectedCredential s where c.status=com.bcxin.Infrastructures.enums.EmploymentStatus.OffJob " +
                " and c.organization.id=:organizationId";

         */
        String qString = "select new map(c.id as id,t.name as name,t.telephone as telephone," +
                "s.credentialType as credentialType,x.name as departName,c.insure as insure," +
                "s.number as number,c.hiredDate as hiredDate,c.leaveTime as leaveTime,c.leaveNote as leaveNote," +
                "c.leaveOperator.name as operatorName,c.leaveOperator.createdTime as leaveRequestTime) " +
                " from EmployeeEntity c join c.tenantUser t left join t.selectedCredential s left join c.defaultDepartment x" +
                " where c.status=com.bcxin.Infrastructures.enums.EmploymentStatus.OffJob " +
                " and c.organization.id=:organizationId";

        String cString = "select count(1) from EmployeeEntity c join c.tenantUser t left join t.selectedCredential s where c.status=com.bcxin.Infrastructures.enums.EmploymentStatus.OffJob " +
                " and c.organization.id=:organizationId";
        Map<String, Object> parameters = new HashMap<>();
        parameters.put("organizationId", organizationId);

        if (StringUtils.hasLength(criteria.getKeyword())) {
            qString += " and (t.name like :keyword or t.telephone like :keyword or s.number like :keyword)";
            cString += " and (t.name like :keyword or t.telephone like :keyword or s.number like :keyword)";
            parameters.put("keyword", "%" + criteria.getKeyword() + "%");
        }


        if (criteria.getStartDate() != null) {
            qString += " and c.leaveTime >=:startDate ";
            cString += " and c.leaveTime >=:startDate ";
            parameters.put("startDate", criteria.getStartDate());
        }

        if (criteria.getEndDate() != null) {
            qString += " and c.leaveTime <=:endDate ";
            cString += " and c.leaveTime <=:endDate ";
            parameters.put("endDate", criteria.getEndDate());
        }

        if (criteria.getInsure() != null) {
            if (criteria.getInsure().equals(TrueFalseStatus.False)) {
                qString += " and (c.insure = :insure or c.insure is null) ";
                cString += " and (c.insure = :insure or c.insure is null) ";
            } else {
                qString += " and c.insure =:insure ";
                cString += " and c.insure =:insure ";
            }

            parameters.put("insure", criteria.getInsure());
        }


        TenantUserContext userContext = TenantContext.getInstance().getUserContext();
        if (!criteria.isIgnorePermission() && userContext != null && userContext.get() != null && !userContext.get().isMaster() &&
                StringUtils.hasLength(userContext.get().getEmployeeId())) {

            if (userContext.get().isDepartAdmin()) {
                String deptAdminSql = getDeptAdminSql(criteria.getTreeCodes());
                if (!CollectionUtils.isEmpty(criteria.getDepartIds())) {
                    qString += " and ((c.id=:selfEmployeeId and exists(select k from c.departmentEmployeeRelations k where k.department.id in (:selfDepartmentIds))) or exists (SELECT k FROM c.departmentEmployeeRelations k WHERE k.department.id IN (:departmentIds) " + deptAdminSql + " ))";
                    cString += " and ((c.id=:selfEmployeeId and exists(select k from c.departmentEmployeeRelations k where k.department.id in (:selfDepartmentIds))) or exists (SELECT k FROM c.departmentEmployeeRelations k WHERE k.department.id IN (:departmentIds) " + deptAdminSql + " ))";
                    parameters.put("departmentIds", criteria.getDepartIds());
                    parameters.put("selfDepartmentIds", criteria.getDepartIds());
                } else {
                    qString += " and (c.id=:selfEmployeeId or exists (SELECT k FROM c.departmentEmployeeRelations k WHERE 1=1 " + deptAdminSql + " ))";
                    cString += " and (c.id=:selfEmployeeId or exists (SELECT k FROM c.departmentEmployeeRelations k WHERE 1=1 " + deptAdminSql + " ))";
                }
                parameters.put("selfEmployeeId", userContext.get().getEmployeeId());
            } else {
                if (!CollectionUtils.isEmpty(criteria.getDepartIds())) {
                    qString += " and ((c.id=:selfEmployeeId and exists(select k from c.departmentEmployeeRelations k where k.department.id in (:selfDepartmentIds))))";
                    cString += " and ((c.id=:selfEmployeeId and exists(select k from c.departmentEmployeeRelations k where k.department.id in (:selfDepartmentIds))))";
                    parameters.put("selfDepartmentIds", criteria.getDepartIds());
                } else {
                    qString += " and (c.id=:selfEmployeeId)";
                    cString += " and (c.id=:selfEmployeeId)";
                }
                parameters.put("selfEmployeeId", userContext.get().getEmployeeId());
            }
        } else {
            /**
             * 所在部门
             */
            if (!CollectionUtils.isEmpty(criteria.getDepartIds())) {
                qString += " and exists(select de from DepartmentEmployeeRelationEntity de where de.employee.id=c.id and de.department.id in (:departmentIds))";
                cString += " and exists(select de from DepartmentEmployeeRelationEntity de where de.employee.id=c.id and de.department.id in (:departmentIds))";

                parameters.put("departmentIds", criteria.getDepartIds());
            }
        }

        qString += " order by c.leaveTime desc ";

        TypedQuery<Map> employeeLeaveDtoTypedQuery =
                this.entityManager.createQuery(qString, Map.class);
        TypedQuery<Long> totalCountQuery =
                this.entityManager.createQuery(cString, Long.class);

        for (String key : parameters.keySet()) {
            employeeLeaveDtoTypedQuery.setParameter(key, parameters.get(key));
            totalCountQuery.setParameter(key, parameters.get(key));
        }

        long totalCount = totalCountQuery.getSingleResult();
        List<Map> result =
                employeeLeaveDtoTypedQuery.setFirstResult(criteria.getSkip())
                        .setMaxResults(criteria.getPageSize()).getResultList();

        List<EmployeeLeaveDto> list = Collections.emptyList();
        if (result.size() > 0) {
            list = result.stream().map(ii -> {
                Timestamp leveTime = (Timestamp) ii.get("leaveTime");
                Date hiredDate = (Date) ii.get("hiredDate");
                Timestamp leaveRequestTime = (Timestamp) ii.get("leaveRequestTime");
                return new EmployeeLeaveDto(
                        (String) ii.get("id"),
                        (String) ii.get("name"),
                        (String) ii.get("telephone"),
                        "",
                        (CredentialType) ii.get("credentialType"),
                        (String) ii.get("number"),
                        hiredDate,
                        leveTime,
                        (String) ii.get("leaveNote"),
                        (String) ii.get("operatorName"),
                        leaveRequestTime,
                        (TrueFalseStatus) ii.get("insure")
                );
            }).collect(Collectors.toList());
            Collection<String> employeeIds = list.stream().map(ii -> ii.getId()).collect(Collectors.toList());

            TypedQuery<DepartDto> mapTypedQuery =
                    this.entityManager.createQuery("select new com.bcxin.tenant.domain.readers.dtos.DepartDto(de.employee.id as employeeId, dp.id as departId, dp.name as departName) from DepartmentEmployeeRelationEntity de join de.department dp where de.employee.id in (:employeeIds)",
                            DepartDto.class);
            mapTypedQuery.setParameter("employeeIds", employeeIds);
            Collection<DepartDto> departDtos = mapTypedQuery.getResultList();
            Map<String, List<DepartDto>> departMap = departDtos.stream().collect(Collectors.groupingBy(DepartDto::getEmployeeId));
            for (EmployeeLeaveDto rt : list) {
                if (departMap.get(rt.getId()) != null) {
                    rt.assignDepart(departMap.get(rt.getId()).stream().map(DepartDto::getDepartName).collect(Collectors.joining(", ")));
                }
            }
        }

        return Pageable.create(criteria.getPageIndex(), criteria.getPageSize(), (int) totalCount, list);
    }

    @Override
    public Pageable<InvitedToJoinQueuesDto> findInvitedToJoinQueuesMember(String organizationId, InvitedToJoinQueuesCriteria criteria) {
        String qString = "select new com.bcxin.tenant.domain.readers.dtos.InvitedToJoinQueuesDto(i.id,i.name,i.telephone,i.credentialType," +
                "i.credentialNumber,i.inviter,i.invitedType,i.departmentId,i.departmentName," +
                "i.occupationType,i.attendanceSiteName,i.invitedStatus) from InvitedToJoinQueuesEntity i " +
                " where i.organizationId = :organizationId and i.invitedStatus = 0";
        String cString = "select count(1) from InvitedToJoinQueuesEntity i where i.organizationId=:organizationId and i.invitedStatus = 0";
        Map<String, Object> parameters = new HashMap<>();
        parameters.put("organizationId", organizationId);
        if (StringUtils.hasLength(criteria.getKeyword())) {
            qString += " and (i.name like :keyword or i.telephone like :keyword or i.credentialNumber like :keyword)";
            cString += " and (i.name like :keyword or i.telephone like :keyword or i.credentialNumber like :keyword)";
            parameters.put("keyword", "%" + criteria.getKeyword() + "%");
        }
        if (!CollectionUtils.isEmpty(criteria.getOccupationTypes())) {
            qString += " and i.occupationType in :occupationType";
            cString += " and i.occupationType in :occupationType";
            parameters.put("occupationType", criteria.getOccupationTypes());
        }

        TenantUserContext userContext = TenantContext.getInstance().getUserContext();
        if (!criteria.isIgnorePermission() && userContext != null && userContext.get() != null && !userContext.get().isMaster() &&
                StringUtils.hasLength(userContext.get().getEmployeeId())) {
            if (userContext.get().isDepartAdmin()) {
                if (!CollectionUtils.isEmpty(criteria.getDepartIds())) {
                    qString += " and exists(select 1 from DepartmentAdminEntity x where x.department.id=i.departmentId and x.employee.id=:employeeId and i.departmentId in :departmentId)";
                    cString += " and exists(select 1 from DepartmentAdminEntity x where x.department.id=i.departmentId and x.employee.id=:employeeId and i.departmentId in :departmentId)";
                    parameters.put("departmentId", criteria.getDepartIds());
                    parameters.put("employeeId", userContext.get().getEmployeeId());
                } else {
                    qString += " and exists(select 1 from DepartmentAdminEntity x where x.department.id=i.departmentId and x.employee.id=:employeeId)";
                    cString += " and exists(select 1 from DepartmentAdminEntity x where x.department.id=i.departmentId and x.employee.id=:employeeId)";
                    parameters.put("employeeId", userContext.get().getEmployeeId());
                }
            } else {
                qString += " and i.id=:invalidId";
                cString += " and i.id=:invalidId";
                parameters.put("invalidId", 0l);
            }
        } else if (!CollectionUtils.isEmpty(criteria.getDepartIds())) {
            qString += " and i.departmentId in :departmentId";
            cString += " and i.departmentId in :departmentId";
            parameters.put("departmentId", criteria.getDepartIds());
        }

        TypedQuery<InvitedToJoinQueuesDto> invitedToJoinQueuesDtoTypedQuery =
                this.entityManager.createQuery(qString, InvitedToJoinQueuesDto.class);
        TypedQuery<Long> totalCountQuery =
                this.entityManager.createQuery(cString, Long.class);
        for (String key : parameters.keySet()) {
            invitedToJoinQueuesDtoTypedQuery.setParameter(key, parameters.get(key));
            totalCountQuery.setParameter(key, parameters.get(key));
        }
        long totalCount = totalCountQuery.getSingleResult();
        List<InvitedToJoinQueuesDto> result =
                invitedToJoinQueuesDtoTypedQuery.setFirstResult(criteria.getSkip())
                        .setMaxResults(criteria.getPageSize()).getResultList();

        List<InvitedToJoinQueuesDto> list = Collections.emptyList();
        if (result.size() > 0) {
            list = result.stream().map(ii -> {
                return new InvitedToJoinQueuesDto(
                        ii.getId(),
                        ii.getName(),
                        ii.getTelephone(),
                        ii.getCredentialType(),
                        ii.getCredentialNumber(),
                        ii.getInviter(),
                        ii.getInvitedType(),
                        ii.getDepartmentId(),
                        ii.getDepartmentName(),
                        ii.getOccupationType(),
                        ii.getAttendanceSiteName(),
                        ii.getInvitedStatus()
                );
            }).collect(Collectors.toList());
        }
        return Pageable.create(criteria.getPageIndex(), criteria.getPageSize(), (int) totalCount, list);
    }

    /**
     * description: 获取treecode拼接的hql
     * author: linchunpeng
     * date:  2023-05-16 15:17
     */
    private String getDeptAdminSql(Collection<String> treeCodes) {
        String deptAdminSql = "";
        StringBuffer deptAdminSqlSb = new StringBuffer();
        if (!CollectionUtils.isEmpty(treeCodes)) {
            deptAdminSqlSb.append(" and ( ");
            for (String treeCode : treeCodes) {
                deptAdminSqlSb.append(" k.departmentIndexTree LIKE concat('").append(treeCode).append("', '%') ").append(" or");
            }
            deptAdminSql = deptAdminSqlSb.substring(0, deptAdminSqlSb.length() - 2).concat(") ");
        }
        return deptAdminSql;
    }

    @Override
    public List<LeaveEmployeeExportDto> findLeaveEmployeeList(String organizationId, EmployeeLeaveCriteria criteria) {
        String qString = "select new com.bcxin.tenant.domain.repositories.dtos.LeaveEmployeeExportDto(c.id,t.name,t.telephone,c.insure," +
                "s.credentialType,s.number,c.hiredDate,c.leaveTime," +
                "c.leaveNote,c.leaveOperator.name,c.leaveOperator.createdTime) " +
                " from EmployeeEntity c join c.tenantUser t left join t.selectedCredential s left join c.defaultDepartment x" +
                " where c.status=com.bcxin.Infrastructures.enums.EmploymentStatus.OffJob " +
                " and c.organization.id=:organizationId";

        Map<String, Object> parameters = new HashMap<>();
        parameters.put("organizationId", organizationId);

        if (StringUtils.hasLength(criteria.getKeyword())) {
            qString += " and (t.name like :keyword or t.telephone like :keyword or s.number like :keyword)";
            parameters.put("keyword", "%" + criteria.getKeyword() + "%");
        }

        if (criteria.getStartDate() != null) {
            qString += " and c.leaveTime >=:startDate ";
            parameters.put("startDate", criteria.getStartDate());
        }

        if (criteria.getEndDate() != null) {
            qString += " and c.leaveTime <=:endDate ";
            parameters.put("endDate", criteria.getEndDate());
        }

        if (criteria.getInsure() != null) {
            if (criteria.getInsure().equals(TrueFalseStatus.False)) {
                qString += " and (c.insure = :insure or c.insure is null) ";
            } else {
                qString += " and c.insure =:insure ";
            }

            parameters.put("insure", criteria.getInsure());
        }


        TenantUserContext userContext = TenantContext.getInstance().getUserContext();
        if (!criteria.isIgnorePermission() && userContext != null && userContext.get() != null && !userContext.get().isMaster() &&
                StringUtils.hasLength(userContext.get().getEmployeeId())) {

            if (userContext.get().isDepartAdmin()) {
                String deptAdminSql = getDeptAdminSql(criteria.getTreeCodes());
                if (!CollectionUtils.isEmpty(criteria.getDepartIds())) {
                    qString += " and ((c.id=:selfEmployeeId and exists(select k from c.departmentEmployeeRelations k where k.department.id in (:selfDepartmentIds))) or exists (SELECT k FROM c.departmentEmployeeRelations k WHERE k.department.id IN (:departmentIds) " + deptAdminSql + " ))";
                    parameters.put("departmentIds", criteria.getDepartIds());
                    parameters.put("selfDepartmentIds", criteria.getDepartIds());
                } else {
                    qString += " and (c.id=:selfEmployeeId or exists (SELECT k FROM c.departmentEmployeeRelations k WHERE 1=1 " + deptAdminSql + " ))";
                }
                parameters.put("selfEmployeeId", userContext.get().getEmployeeId());
            } else {
                if (!CollectionUtils.isEmpty(criteria.getDepartIds())) {
                    qString += " and ((c.id=:selfEmployeeId and exists(select k from c.departmentEmployeeRelations k where k.department.id in (:selfDepartmentIds))))";
                    parameters.put("selfDepartmentIds", criteria.getDepartIds());
                } else {
                    qString += " and (c.id=:selfEmployeeId)";
                }

                parameters.put("selfEmployeeId", userContext.get().getEmployeeId());
            }


        } else {
            /**
             * 所在部门
             */
            if (!CollectionUtils.isEmpty(criteria.getDepartIds())) {
                qString += " and exists(select de from DepartmentEmployeeRelationEntity de where de.employee.id=c.id and de.department.id in (:departmentIds))";
                parameters.put("departmentIds", criteria.getDepartIds());
            }
        }

        qString += " order by c.leaveTime desc ";
        TypedQuery<LeaveEmployeeExportDto> employeeLeaveDtoTypedQuery =
                this.entityManager.createQuery(qString, LeaveEmployeeExportDto.class);

        for (String key : parameters.keySet()) {
            employeeLeaveDtoTypedQuery.setParameter(key, parameters.get(key));
        }

        List<LeaveEmployeeExportDto> list = employeeLeaveDtoTypedQuery.getResultList();

        if (list.size() > 0) {
            Collection<String> employeeIds = list.stream().map(ii -> ii.getId()).collect(Collectors.toList());

            TypedQuery<DepartDto> mapTypedQuery =
                    this.entityManager.createQuery("select new com.bcxin.tenant.domain.readers.dtos.DepartDto(de.employee.id as employeeId, dp.id as departId, dp.name as departName) from DepartmentEmployeeRelationEntity de join de.department dp where de.employee.id in (:employeeIds)",
                            DepartDto.class);
            mapTypedQuery.setParameter("employeeIds", employeeIds);
            Collection<DepartDto> departDtos = mapTypedQuery.getResultList();
            Map<String, List<DepartDto>> departMap = departDtos.stream().collect(Collectors.groupingBy(DepartDto::getEmployeeId));
            for (LeaveEmployeeExportDto rt : list) {
                if (departMap.get(rt.getId()) != null) {
                    rt.assignDepart(departMap.get(rt.getId()).stream().map(DepartDto::getDepartName).collect(Collectors.joining(", ")));
                }
            }
        }


        return list;
    }

    @Override
    public List<ContractExportDto> findContractList(String organizationId, ContractCriteria criteria) {
        String qString = "select new com.bcxin.tenant.domain.repositories.dtos.ContractExportDto(e.id,c.name,t.name,t.telephone,sc.credentialType,sc.number,c.aName,c.bName, " +
                " c.beginDate,c.endDate,c.dateLimitless,c.status,c.note,c.attachment,e.status,c.lastModifier,c.creator) " +
                " from ContractEntity c join c.employee e join e.tenantUser t left join t.selectedCredential sc where c.organization.id=:organizationId ";

        Map<String, Object> parameters = new HashMap<>();
        parameters.put("organizationId", organizationId);

        if (StringUtils.hasLength(criteria.getKeyword())) {
            qString += " and (t.name like :keyword or t.telephone like :keyword or t.selectedCredential.number like :keyword)";
            parameters.put("keyword", "%" + criteria.getKeyword() + "%");
        }

        if (StringUtils.hasLength(criteria.getEmployeeId())) {
            qString += " and e.id=:employeeId1 ";
            parameters.put("employeeId1", criteria.getEmployeeId());
        }

        if (criteria.getOccupationTypes() != null && criteria.getOccupationTypes().size() > 0) {
            qString += " and e.occupationType in :occupationTypes";
            parameters.put("occupationTypes", criteria.getOccupationTypes());
        }

        if (criteria.getEmploymentStatus() != null) {
            qString += " and e.status=:employmentStatus";
            parameters.put("employmentStatus", criteria.getEmploymentStatus());
        }

        if (criteria.getStatus() != null) {
            switch (criteria.getStatus()) {
                case INCOMPLETE:
                    qString += " and c.status = 1 and e.status <> 1 and (c.attachment is null or c.attachment ='') ";
                    break;
                case BEEFFECT:
                    qString += " and c.status = 1 and e.status <> 1 and c.beginDate > now() ";
                    break;
                case EFFECT:
                    qString += " and c.status = 1 and e.status <> 1 and now() between c.beginDate and c.endDate and c.attachment is not null and c.attachment !='' ";
                    break;
                case STOP:
                    qString += " and (c.status = 0 or e.status = 1 or (c.dateLimitless=0 and c.endDate < now())) ";
                    break;
                case DUE:
                    Date dueDate = DateUtils.addMonths(new Date(), 1);
                    qString += " and c.status = 1 and e.status <> 1 and c.dateLimitless=0 and c.endDate > now() and c.endDate < :dueDate ";
                    parameters.put("dueDate", dueDate);
                    break;
                default:
                    break;
            }
        }

        if (criteria.getDepartIds() != null && criteria.getDepartIds().size() > 0) {
            qString += " and exists(select d from DepartmentEmployeeRelationEntity d where d.employee.id=e.id and d.department.id in :departIds)";
            parameters.put("departIds", criteria.getDepartIds());
        }

        if (criteria.getBeginDate() != null) {
            qString += " and c.beginDate >=:beginDate";
            parameters.put("beginDate", criteria.getBeginDate());
        }

        if (criteria.getEndDate() != null) {
            qString += " and c.endDate <=:endDate";
            parameters.put("endDate", criteria.getEndDate());
        }

        if (criteria.getBeginCreatedTime() != null) {
            qString += " and DATE(c.creator.createdTime) >= DATE(:beginCreatedTime)";
            parameters.put("beginCreatedTime", criteria.getBeginCreatedTime());
        }

        if (criteria.getEndCreatedTime() != null) {
            qString += " and DATE(c.creator.createdTime) <= DATE(:endCreatedTime)";
            parameters.put("endCreatedTime", criteria.getEndCreatedTime());
        }

        TenantUserContext userContext = TenantContext.getInstance().getUserContext();

        if (userContext != null && userContext.get() != null && !userContext.get().isMaster() &&
                StringUtils.hasLength(userContext.get().getEmployeeId())) {
            qString += " and exists(select 1 from e.departmentEmployeeRelations k join k.department dp join DepartmentEntity dade on dp.indexTree LIKE concat(dade.indexTree,'%') join DepartmentAdminEntity da on da.employee.id=:employeeId and da.department.id = dade.id )";
            parameters.put("employeeId", userContext.get().getEmployeeId());
        }

        qString += " order by c.endDate ";

        TypedQuery<ContractExportDto> employeeContractDtoTypedQuery = this.entityManager.createQuery(qString, ContractExportDto.class);

        for (String key : parameters.keySet()) {
            employeeContractDtoTypedQuery.setParameter(key, parameters.get(key));
        }

        List<ContractExportDto> list = employeeContractDtoTypedQuery.getResultList();

        if (list.size() > 0) {
            Map<String, DepartmentDto> departMaps = this.getDepartMaps(organizationId);
            TypedQuery<DepartDto> mapTypedQuery =
                    this.entityManager.createQuery("select new com.bcxin.tenant.domain.readers.dtos.DepartDto(de.employee.id as employeeId, dp.id as departId, dp.name as departName) from DepartmentEmployeeRelationEntity de join de.department dp where de.employee.id in :employeeIds",
                            DepartDto.class);
            List<String> employeeIds = list.stream().map(ContractExportDto::getEmployeeId).collect(Collectors.toList());
            mapTypedQuery.setParameter("employeeIds", employeeIds);
            List<DepartDto> departDtos = mapTypedQuery.getResultList();
            for (DepartDto departDto : departDtos) {
                if (departMaps.get(departDto.getDepartId()) != null) {
                    departDto.resetDepartName(departMaps.get(departDto.getDepartId()).getIndexTree());
                }
            }
            Map<String, List<DepartDto>> departMap = departDtos.stream().collect(Collectors.groupingBy(DepartDto::getEmployeeId));
            for (ContractExportDto employeeContractDto : list) {
                if (departMap.get(employeeContractDto.getEmployeeId()) != null) {
                    employeeContractDto.setEmployeeDepartName(departMap.get(employeeContractDto.getEmployeeId()).stream().map(DepartDto::getDepartName).collect(Collectors.joining(",")));
                }
            }
        }
        return list;
    }

    private Map<String, DepartmentDto> getDepartMaps(String organizationId) {
        Collection<DepartmentDto> departmentDtos = this.getAllManagedDepartmentDtos(organizationId);
        for (DepartmentDto department : departmentDtos) {
            department.resetIndexTree("/" + department.getIndexTree().replaceAll("--", "#").replaceAll("-", "/") + "/");
        }
        for (DepartmentDto departmentDto : departmentDtos) {
            for (DepartmentDto department : departmentDtos) {
                department.resetIndexTree(department.getIndexTree().replace("/" + departmentDto.getId().replaceAll("--", "#") + "/", "/" + departmentDto.getName() + "/"));
            }
        }
        for (DepartmentDto department : departmentDtos) {
            department.resetIndexTree(department.getIndexTree().substring(1, department.getIndexTree().length() - 1));
        }
        return departmentDtos.stream().collect(Collectors.toMap(DepartmentDto::getId, Function.identity()));
    }


    @Override
    public EmployeeReportDto findReport(String organizationId, EmployeeCriteria criteria) {
        StringBuilder sb = new StringBuilder();
        Map<String, Object> parameters = new HashMap<>();
        if (StringUtils.hasLength(criteria.getKeyword())) {
            sb.append(" and (t.name like :keyword or t.telephone like :keyword or s.number like :keyword)");
            parameters.put("keyword", "%" + criteria.getKeyword() + "%");
        }

        /**
         * 证书类型
         */
        if (!CollectionUtils.isEmpty(criteria.getCredentialTypes())) {
            sb.append(" and s.credentialType in :credentialType");
            parameters.put("credentialType", criteria.getCredentialTypes());
        }

        /**
         * 实名认证
         */
        if (!CollectionUtils.isEmpty(criteria.getAuthenticatedStatuses())) {
            sb.append(" and t.authenticateStatus in :authenticateStatus");
            parameters.put("authenticateStatus", criteria.getAuthenticatedStatuses());
        }
        /**
         * 证件核验状态
         */
        if (!CollectionUtils.isEmpty(criteria.getCheckedStatuses())) {
            sb.append(" and t.checkedStatus in :checkedStatus");
            parameters.put("checkedStatus", criteria.getCheckedStatuses());
        }

        /**
         * 职业类型
         */
        if (!CollectionUtils.isEmpty(criteria.getOccupationTypes())) {
            sb.append(" and e.occupationType in :occupationType");
            parameters.put("occupationType", criteria.getOccupationTypes());
        }

        if (criteria.getHiredDate() != null) {
            sb.append(" and e.hiredDate=:hiredDate ");
            parameters.put("hiredDate", criteria.getHiredDate());
        }

        if (StringUtils.hasLength(criteria.getPosition())) {
            sb.append(" and e.position = :position ");
            parameters.put("position", criteria.getPosition());
        }

        if ("1".equals(criteria.getContractStatus())) {
            sb.append(" and exists(select ct from ContractEntity ct where ct.employee = e and ct.status=1 and CURDATE() between ct.beginDate and ct.endDate) ");
        }
        if ("0".equals(criteria.getContractStatus())) {
            sb.append(" and not exists(select ct from ContractEntity ct where ct.employee = e and ct.status=1 and CURDATE() between ct.beginDate and ct.endDate) ");
        }

        if ("1".equals(criteria.getCerStatus())) {
            sb.append(" and exists(select tuc from TenantUserCredentialsEntity tuc where tuc.tenantUser = t and tuc.credentialType=1) ");
        }
        if ("0".equals(criteria.getCerStatus())) {
            sb.append(" and not exists(select tuc from TenantUserCredentialsEntity tuc where tuc.tenantUser = t and tuc.credentialType=1) ");
        }

        if ("1".equals(criteria.getGradeCerStatus())) {
            sb.append(" and exists(select tuc from TenantUserCredentialsEntity tuc where tuc.tenantUser = t and tuc.credentialType=8) ");
        }
        if ("0".equals(criteria.getGradeCerStatus())) {
            sb.append(" and not exists(select tuc from TenantUserCredentialsEntity tuc where tuc.tenantUser = t and tuc.credentialType=8) ");
        }

        String qString = "select new com.bcxin.tenant.domain.readers.dtos.EmployeeConDto(e.id,t.checkedStatus, e.occupationType, t.authenticateStatus," +
                "(select count(1) from ContractEntity c where c.employee = e and c.status = com.bcxin.Infrastructures.enums.ValidStatus.Valid and CURDATE() between c.beginDate and c.endDate)" +
                ") " +
                " from EmployeeEntity e join e.tenantUser t left join t.selectedCredential s " +
                " where e.status!=com.bcxin.Infrastructures.enums.EmploymentStatus.OffJob";

        if (StrUtil.isNotEmpty(organizationId)) {
            qString += " and e.organization.id=:organId ";
            parameters.put("organId", organizationId);
        }

        if (StrUtil.isNotEmpty(criteria.getOrganName())) {
            qString += " and e.organization.name like '%:organName%' ";
            parameters.put("organName", criteria.getOrganName());
        }
        qString += sb;

        TenantUserContext userContext = TenantContext.getInstance().getUserContext();
        if (!criteria.isIgnorePermission() && userContext != null && userContext.get() != null && !userContext.get().isMaster() &&
                StringUtils.hasLength(userContext.get().getEmployeeId())) {
            String deptAdminSql = getDeptAdminSql(criteria.getTreeCodes());
            if (!CollectionUtils.isEmpty(criteria.getDepartIds())) {
                qString += " and ((e.id=:selfEmployeeId and exists(select k from e.departmentEmployeeRelations k where k.department.id in (:selfDepartmentIds))) or exists (SELECT k FROM e.departmentEmployeeRelations k WHERE k.department.id IN (:departmentIds) " + deptAdminSql + " ))";
                parameters.put("departmentIds", criteria.getDepartIds());
                parameters.put("selfDepartmentIds", criteria.getDepartIds());
            } else {
                qString += " and (e.id=:selfEmployeeId or exists (SELECT k FROM e.departmentEmployeeRelations k WHERE 1=1 " + deptAdminSql + " ))";
            }
            parameters.put("selfEmployeeId", userContext.get().getEmployeeId());
        } else {
            /**
             * 所在部门
             */
            if (!CollectionUtils.isEmpty(criteria.getDepartIds())) {
                qString += " and exists(select de from DepartmentEmployeeRelationEntity de where de.employee.id=e.id and de.department.id in (:departmentIds))";
                parameters.put("departmentIds", criteria.getDepartIds());
            }
        }

        TypedQuery<EmployeeConDto> employeeTypedQuery = this.entityManager.createQuery(qString, EmployeeConDto.class);

        for (String key : parameters.keySet()) {
            employeeTypedQuery = employeeTypedQuery.setParameter(key, parameters.get(key));
        }
        List<EmployeeConDto> employeeConDtos = employeeTypedQuery.getResultList();

        return new EmployeeReportDto((long) employeeConDtos.size(),
                employeeConDtos.stream().filter(employeeConDto -> UserCheckedStatus.None.equals(employeeConDto.getCheckedStatus())).count(),
                employeeConDtos.stream().filter(employeeConDto -> UserCheckedStatus.Matched.equals(employeeConDto.getCheckedStatus())).count(),
                employeeConDtos.stream().filter(employeeConDto -> UserCheckedStatus.Commit.equals(employeeConDto.getCheckedStatus())).count(),
                employeeConDtos.stream().filter(employeeConDto -> employeeConDto.getContractCount() > 0).count(),
                employeeConDtos.stream().filter(employeeConDto -> OccupationType.SecurityGuard.equals(employeeConDto.getOccupationType())).count(),
                employeeConDtos.stream().filter(employeeConDto -> OccupationType.SecurityGuard.equals(employeeConDto.getOccupationType()) && UserCheckedStatus.None.equals(employeeConDto.getCheckedStatus())).count(),
                employeeConDtos.stream().filter(employeeConDto -> OccupationType.SecurityGuard.equals(employeeConDto.getOccupationType()) && UserCheckedStatus.Matched.equals(employeeConDto.getCheckedStatus())).count(),
                employeeConDtos.stream().filter(employeeConDto -> OccupationType.SecurityGuard.equals(employeeConDto.getOccupationType()) && RealNameAuthenticatedStatus.Authenticating.equals(employeeConDto.getAuthenticateStatus())).count(),
                employeeConDtos.stream().filter(employeeConDto -> OccupationType.SecurityGuard.equals(employeeConDto.getOccupationType()) && RealNameAuthenticatedStatus.Passed.equals(employeeConDto.getAuthenticateStatus())).count(),
                employeeConDtos.stream().filter(employeeConDto -> OccupationType.SecurityGuard.equals(employeeConDto.getOccupationType()) && RealNameAuthenticatedStatus.Failed.equals(employeeConDto.getAuthenticateStatus())).count(),
                employeeConDtos.stream().filter(employeeConDto -> OccupationType.SecurityGuard.equals(employeeConDto.getOccupationType()) && employeeConDto.getContractCount() > 0).count()
        );
    }

    @Override
    public ContractReportDto findContractReport(String organizationId, ContractCriteria criteria) {
        Date dueDate = DateUtils.addMonths(new Date(), 1);
        String qString = "select new com.bcxin.tenant.domain.repositories.dtos.ContractReportDto(" +
                "count(CASE WHEN c.id is not null THEN 1 END)," +
                "count(CASE WHEN c.status = 1 and e.status <> 1 and (c.attachment is null or c.attachment ='') THEN 1 END)," +
                "count(CASE WHEN c.status = 1 and e.status <> 1 and now() between c.beginDate and c.endDate and c.attachment is not null and c.attachment !='' THEN 1 END)," +
                "count(CASE WHEN c.status = 1 and e.status <> 1 and c.beginDate > now() THEN 1 END)," +
                "count(CASE WHEN c.status = 1 and e.status <> 1 and c.dateLimitless=0 and c.endDate > now() and c.endDate < :dueDate THEN 1 END)," +
                "count(CASE WHEN c.id is not null and (c.status = 0 or e.status = 1 or (c.dateLimitless=0 and c.endDate < now())) THEN 1 END)," +
                "count(CASE WHEN c.id is null THEN 1 END)," +
                "count(CASE WHEN c.id is null and e.occupationType = 1 THEN 1 END)," +
                "count(CASE WHEN c.id is not null and e.status = 1 THEN 1 END)" +
                ") " +
                " from EmployeeEntity e inner join e.tenantUser t left join ContractEntity c on c.employee=e where e.organization.id=:organizationId ";


        Map<String, Object> parameters = new HashMap<>();
        parameters.put("organizationId", organizationId);
        parameters.put("dueDate", dueDate);

        if (StringUtils.hasLength(criteria.getKeyword())) {
            qString += " and (t.name like :keyword or t.telephone like :keyword or t.selectedCredential.number like :keyword)";
            parameters.put("keyword", "%" + criteria.getKeyword() + "%");
        }

        if (StringUtils.hasLength(criteria.getEmployeeId())) {
            qString += " and e.id=:employeeId1 ";
            parameters.put("employeeId1", criteria.getEmployeeId());
        }

        if (criteria.getOccupationTypes() != null && criteria.getOccupationTypes().size() > 0) {
            qString += " and e.occupationType in :occupationTypes";
            parameters.put("occupationTypes", criteria.getOccupationTypes());
        }

        if (criteria.getEmploymentStatus() != null) {
            qString += " and e.status=:employmentStatus";
            parameters.put("employmentStatus", criteria.getEmploymentStatus());
        }

        if (criteria.getStatus() != null) {
            switch (criteria.getStatus()) {
                case INCOMPLETE:
                    qString += " and c.status = 1 and e.status <> 1 and (c.attachment is null or c.attachment ='') ";
                    break;
                case BEEFFECT:
                    qString += " and c.status = 1 and e.status <> 1 and c.beginDate > now() ";
                    break;
                case EFFECT:
                    qString += " and c.status = 1 and e.status <> 1 and now() between c.beginDate and c.endDate and c.attachment is not null and c.attachment !='' ";
                    break;
                case STOP:
                    qString += " and (c.status = 0 or e.status = 1 or (c.dateLimitless=0 and c.endDate < now())) ";
                    break;
                case DUE:
                    qString += " and c.status = 1 and e.status <> 1 and c.dateLimitless=0 and now() between :dueDate and c.endDate ";
                    parameters.put("dueDate", dueDate);
                    break;
                default:
                    break;
            }
        }

        if (criteria.getDepartIds() != null && criteria.getDepartIds().size() > 0) {
            qString += " and d.department.id in :departIds";
            parameters.put("departIds", criteria.getDepartIds());
        }

        if (criteria.getBeginDate() != null) {
            qString += " and c.beginDate >=:beginDate";
            parameters.put("beginDate", criteria.getBeginDate());
        }

        if (criteria.getEndDate() != null) {
            qString += " and c.endDate <=:endDate";
            parameters.put("endDate", criteria.getEndDate());
        }

        TenantUserContext userContext = TenantContext.getInstance().getUserContext();

        if (userContext != null && userContext.get() != null && !userContext.get().isMaster() &&
                StringUtils.hasLength(userContext.get().getEmployeeId())) {
            qString += " and exists(select 1 from e.departmentEmployeeRelations k join k.department dp join DepartmentEntity dade on dp.indexTree LIKE concat(dade.indexTree,'%') join DepartmentAdminEntity da on da.employee.id=:employeeId and da.department.id = dade.id )";
            parameters.put("employeeId", userContext.get().getEmployeeId());
        }

        TypedQuery<ContractReportDto> employeeContractDtoTypedQuery =
                this.entityManager.createQuery(qString, ContractReportDto.class);

        for (String key : parameters.keySet()) {
            employeeContractDtoTypedQuery.setParameter(key, parameters.get(key));
        }

        return employeeContractDtoTypedQuery.getSingleResult();
    }

    @Override
    public Collection<EmployeeBasicDto> getSuperiorEmployeeBasics(String organizationId, Collection<String> ids) {
        TypedQuery<EmployeeBasicDto> employeeBasicDtoTypedQuery =
                this.entityManager.createQuery("select new com.bcxin.tenant.domain.readers.dtos.EmployeeBasicDto(e.id,'','',true, e.status,e.masterSlaveType) " +
                        " from EmployeeEntity e where e.organization.id=?1 and e.id in (?2) and exists(select 1 from EmployeeEntity e2 where e2.superior.id=e.id and e2.status = 0) ", EmployeeBasicDto.class);
        employeeBasicDtoTypedQuery.setParameter(1, organizationId);
        employeeBasicDtoTypedQuery.setParameter(2, ids);

        return employeeBasicDtoTypedQuery.getResultList();
    }

    @Override
    public Collection<EmployeeCompositeDto> getEmployeeComposites(String organizationId, Collection<String> ids) {

        TypedQuery<EmployeeCompositeDto> employeeCompositeDtoTypedQuery =
                this.entityManager.createQuery(
                        "select new com.bcxin.tenant.domain.readers.dtos.EmployeeCompositeDto(e,(select true from EmployeeEntity e2 where e2.superior.id=e.id)) " +
                                " from EmployeeEntity e where e.organization.id=?1 and e.id in (?2) ", EmployeeCompositeDto.class);
        employeeCompositeDtoTypedQuery.setParameter(1, organizationId);
        employeeCompositeDtoTypedQuery.setParameter(2, ids);

        return employeeCompositeDtoTypedQuery.getResultList();
    }

    @Override
    public Pageable<EmployeeContractDto> findEmployeeContracts(String organizationId, EmployeeContractCriteria criteria) {
        String qString = "select new com.bcxin.tenant.domain.readers.dtos.EmployeeContractDto(" +
                "c.id,c.name,sc.number,sc.credentialType,c.aName,c.bName,e.id," +
                "t.name,t.telephone,e.occupationType,e.status,c.beginDate,c.endDate, " +
                " c.dateLimitless,c.attachment,c.note,c.status,c.lastModifier, c.creator) " +
                " from ContractEntity c join c.employee e join e.tenantUser t left join t.selectedCredential sc where c.organization.id=:organizationId ";

        String cString = "select count(1) from ContractEntity c join c.employee e join e.tenantUser t left join t.selectedCredential sc where c.organization.id=:organizationId ";

        Map<String, Object> parameters = new HashMap<>();
        parameters.put("organizationId", organizationId);

        if (StringUtils.hasLength(criteria.getKeyword())) {
            qString += " and (t.name like :keyword or t.telephone like :keyword or t.selectedCredential.number like :keyword)";
            cString += " and (t.name like :keyword or t.telephone like :keyword or t.selectedCredential.number like :keyword)";
            parameters.put("keyword", "%" + criteria.getKeyword() + "%");
        }

        if (StringUtils.hasLength(criteria.getEmployeeId())) {
            qString += " and e.id=:employeeId1 ";
            cString += " and e.id=:employeeId1 ";
            parameters.put("employeeId1", criteria.getEmployeeId());
        }

        if (criteria.getOccupationTypes() != null && criteria.getOccupationTypes().size() > 0) {
            qString += " and e.occupationType in :occupationTypes";
            cString += " and e.occupationType in :occupationTypes";
            parameters.put("occupationTypes", criteria.getOccupationTypes());
        }

        if (criteria.getEmploymentStatus() != null) {
            qString += " and e.status=:employmentStatus";
            cString += " and e.status=:employmentStatus";
            parameters.put("employmentStatus", criteria.getEmploymentStatus());
        }

        if (criteria.getStatus() != null) {
            switch (criteria.getStatus()) {
                case INCOMPLETE:
                    qString += " and c.status = 1 and e.status <> 1  and c.endDate >= now() and (c.attachment is null or c.attachment ='') ";
                    cString += " and c.status = 1 and e.status <> 1  and c.endDate >= now()  and (c.attachment is null or c.attachment ='') ";
                    break;
                case BEEFFECT:
                    qString += " and c.status = 1 and e.status <> 1 and c.beginDate > now() ";
                    cString += " and c.status = 1 and e.status <> 1 and c.beginDate > now() ";
                    break;
                case EFFECT:
                    qString += " and c.status = 1 and e.status <> 1 and now() between c.beginDate and c.endDate and c.attachment is not null and c.attachment !='' ";
                    cString += " and c.status = 1 and e.status <> 1 and now() between c.beginDate and c.endDate and c.attachment is not null and c.attachment !='' ";
                    break;
                case STOP:
                    qString += " and (c.status = 0 or e.status = 1 or (c.dateLimitless=0 and c.endDate < now())) ";
                    cString += " and (c.status = 0 or e.status = 1 or (c.dateLimitless=0 and c.endDate < now())) ";
                    break;
                case DUE:
                    Date dueDate = DateUtils.addMonths(new Date(), 1);
                    qString += " and c.status = 1 and e.status <> 1 and c.dateLimitless=0 and c.endDate between now() and  :dueDate";
                    cString += " and c.status = 1 and e.status <> 1 and c.dateLimitless=0 and  c.endDate between now() and  :dueDate";
                    parameters.put("dueDate", dueDate);
                    break;
                default:
                    break;
            }
        }

        if (criteria.getDepartIds() != null && criteria.getDepartIds().size() > 0) {
            qString += " and exists(select d from DepartmentEmployeeRelationEntity d where d.employee.id=e.id and d.department.id in :departIds)";
            cString += " and exists(select d from DepartmentEmployeeRelationEntity d where d.employee.id=e.id and d.department.id in :departIds)";
            parameters.put("departIds", criteria.getDepartIds());
        }

        if (criteria.getBeginDate() != null) {
            qString += " and c.beginDate >=:beginDate";
            cString += " and c.beginDate >=:beginDate";
            parameters.put("beginDate", criteria.getBeginDate());
        }

        if (criteria.getEndDate() != null) {
            qString += " and c.endDate <=:endDate";
            cString += " and c.endDate <=:endDate";
            parameters.put("endDate", criteria.getEndDate());
        }

        if (criteria.getBeginCreatedTime() != null) {
            qString += " and DATE(c.creator.createdTime) >= DATE(:beginCreatedTime)";
            cString += " and DATE(c.creator.createdTime) >= DATE(:beginCreatedTime)";
            parameters.put("beginCreatedTime", criteria.getBeginCreatedTime());
        }

        if (criteria.getEndCreatedTime() != null) {
            qString += " and DATE(c.creator.createdTime) <= DATE(:endCreatedTime)";
            cString += " and DATE(c.creator.createdTime) <= DATE(:endCreatedTime)";
            parameters.put("endCreatedTime", criteria.getEndCreatedTime());
        }

        TenantUserContext userContext = TenantContext.getInstance().getUserContext();

        if (userContext != null && userContext.get() != null && !userContext.get().isMaster() &&
                StringUtils.hasLength(userContext.get().getEmployeeId())) {
            qString += " and exists(select 1 from e.departmentEmployeeRelations k join k.department dp join DepartmentEntity dade on dp.indexTree LIKE concat(dade.indexTree,'%') join DepartmentAdminEntity da on da.employee.id=:employeeId and da.department.id = dade.id )";
            cString += " and exists(select 1 from e.departmentEmployeeRelations k join k.department dp join DepartmentEntity dade on dp.indexTree LIKE concat(dade.indexTree,'%') join DepartmentAdminEntity da on da.employee.id=:employeeId and da.department.id = dade.id )";
            parameters.put("employeeId", userContext.get().getEmployeeId());
        }
        qString += " order by c.endDate ";

        TypedQuery<EmployeeContractDto> employeeContractDtoTypedQuery =
                this.entityManager.createQuery(qString, EmployeeContractDto.class);

        TypedQuery<Long> totalCountQuery =
                this.entityManager.createQuery(cString, Long.class);

        for (String key : parameters.keySet()) {
            employeeContractDtoTypedQuery.setParameter(key, parameters.get(key));
            totalCountQuery.setParameter(key, parameters.get(key));
        }

        long totalCount = totalCountQuery.getSingleResult();
        List<EmployeeContractDto> list =
                employeeContractDtoTypedQuery.setFirstResult(criteria.getSkip())
                        .setMaxResults(criteria.getPageSize()).getResultList();

        if (list.size() > 0) {
            TypedQuery<DepartDto> mapTypedQuery =
                    this.entityManager.createQuery("select new com.bcxin.tenant.domain.readers.dtos.DepartDto(de.employee.id as employeeId, dp.id as departId, dp.name as departName) from DepartmentEmployeeRelationEntity de join de.department dp where de.employee.id in :employeeIds",
                            DepartDto.class);
            List<String> employeeIds = list.stream().map(EmployeeContractDto::getEmployeeId).collect(Collectors.toList());
            mapTypedQuery.setParameter("employeeIds", employeeIds);
            List<DepartDto> departDtos = mapTypedQuery.getResultList();
            Map<String, List<DepartDto>> departMap = departDtos.stream().collect(Collectors.groupingBy(DepartDto::getEmployeeId));
            for (EmployeeContractDto employeeContractDto : list) {
                employeeContractDto.setContractStatus(criteria.getStatus());
                if (departMap.get(employeeContractDto.getEmployeeId()) != null) {
                    employeeContractDto.setEmployeeDepartName(departMap.get(employeeContractDto.getEmployeeId()).stream().map(DepartDto::getDepartName).collect(Collectors.joining(",")));
                }
            }
        }

        return Pageable.create(criteria.getPageIndex(), criteria.getPageSize(), (int) totalCount, list);
    }

    @Override
    public Collection<Map<String, EmployeeEntity>> getEmployeesByTelephones(String organizationId, Collection<String> telephones) {
        String qString = "select new map(t.telephone,u) from EmployeeEntity u join u.tenantUser t where u.organization.id=?1 and t.telephone in (?2)";
        TypedQuery<Map> mapTypedQuery = this.entityManager.createQuery(qString, Map.class);
        mapTypedQuery.setParameter(1, organizationId);
        mapTypedQuery.setParameter(2, telephones);

        Collection<Map> mapList = mapTypedQuery.getResultList();

        Collection<Map<String, EmployeeEntity>> response = new ArrayList<>();
        for (Map item : mapList) {
            Map<String, EmployeeEntity> mpItem = new HashMap<>();
            mpItem.put((String) item.get("0"), (EmployeeEntity) item.get("1"));

            response.add(mpItem);
        }

        return response;
    }

    @Override
    public Collection<Map<String, EmployeeEntity>> getEmployeesByIdNums(String organizationId, Collection<String> idNums) {
        String qString = "select new map(sc.number,u) from EmployeeEntity u join u.tenantUser t join t.selectedCredential sc where u.organization.id=?1 and sc.number in (?2)";
        TypedQuery<Map> mapTypedQuery = this.entityManager.createQuery(qString, Map.class);
        mapTypedQuery.setParameter(1, organizationId);
        mapTypedQuery.setParameter(2, idNums);

        Collection<Map> mapList = mapTypedQuery.getResultList();

        Collection<Map<String, EmployeeEntity>> response = new ArrayList<>();
        for (Map item : mapList) {
            Map<String, EmployeeEntity> mpItem = new HashMap<>();
            mpItem.put((String) item.get("0"), (EmployeeEntity) item.get("1"));

            response.add(mpItem);
        }

        return response;
    }

    @Override
    public Collection<String> getMyOrganApps(String organId) {
        try {
            TypedQuery<String> typedQuery =
                    this.entityManager.createQuery("select t.bindApplication from TDomainEntity t where t.id=?1", String.class);
            typedQuery.setParameter(1, organId);

            String bindAppJson = typedQuery.getSingleResult();

            return this.jsonProvider.toObjects(String.class, bindAppJson);
        } catch (NoResultException ex) {
            return Collections.EMPTY_LIST;
        }
    }

    @Override
    public EmployeeDetailDto getByEmployeeId(String organId, String employeeId) {
        TypedQuery<EmployeeDetailDto> employeeDetailDtoTypedQuery =
                this.entityManager.createQuery(
                        "select new com.bcxin.tenant.domain.readers.dtos.EmployeeDetailDto(" +
                                "e.id,e.insure, xd.id,xd.name,s.id, t2.name," +
                                "e.hiredDate,e.positiveDate,e.position,e.occupationType," +
                                "t.id,t.name,t.email,t.wechatNicky, t.telephone, t.lonLatJson, t.checkedStatus,t.authenticateStatus,t.authenticatedResult,t.headPhoto,t.sex,t.nation,t.workYear,t.diseasesHistory,t.politicsStatus," +
                                "t.stature,t.militaryStatus,t.birthdate,t.education,t.householdType,t.nativePlace,t.maritalStatus," +
                                "t.oneInchColorWhitePhoto,t.twoInchColorBluePhoto,e.masterSlaveType,t.cid," +
                                "t.emergencyContact,t.emergencyPhone,t.licenseLevel,t.placeOfNow," +
                                "sc.credentialType,sc.name, sc.number,sc.validDateFrom,sc.validDateTo,sc.frontPhoto,sc.reversePhoto,sc.address,sc.headPhoto," +
                                "e.leaveTime,e.leaveNote,t.thirdPartyLoginNo,t.imIdentity,e.interview,e.personStatus,e.probation,e.planPositiveDate,t.CertificateImage,e.salary) " +
                                " from EmployeeEntity e join e.tenantUser t left join e.departmentEmployeeRelations x left join x.department xd " +
                                " left join t.selectedCredential sc " +
                                " left join e.superior s left join s.tenantUser t2" +
                                " where e.organization.id=?1 and e.id=?2", EmployeeDetailDto.class);
        employeeDetailDtoTypedQuery.setParameter(1, organId);
        employeeDetailDtoTypedQuery.setParameter(2, employeeId);

        Collection<EmployeeDetailDto> employeeDetailDtos = employeeDetailDtoTypedQuery.getResultList();
        Optional<EmployeeDetailDto> employeeDetailOptional = employeeDetailDtos.stream().findFirst();
        if (employeeDetailOptional.isPresent()) {
            EmployeeDetailDto employeeDetailDto = employeeDetailOptional.get();

            TypedQuery<DepartDto> mapTypedQuery =
                    this.entityManager.createQuery("select new com.bcxin.tenant.domain.readers.dtos.DepartDto(de.employee.id as employeeId, dp.id as departId, dp.name as departName) from DepartmentEmployeeRelationEntity de join de.department dp where de.employee.id =:employeeId",
                            DepartDto.class);
            mapTypedQuery.setParameter("employeeId", employeeDetailDto.getEmployeeId());
            List<DepartDto> departDtos = mapTypedQuery.getResultList();
            employeeDetailDto.assignDepart(departDtos);
            return employeeDetailDto;
        } else {
            return null;
        }
    }

    @Override
    public InviteUserDto getByOranIdAndPhone(String organId, String phone) {
        TypedQuery<InviteUserDto> inviteUserDtoTypedQuery =
                this.entityManager.createQuery(
                        "select new com.bcxin.tenant.domain.readers.dtos.InviteUserDto(" +
                                "e.id,t2.name,e.hiredDate,e.occupationType,sc.credentialType,sc.number) " +
                                " from EmployeeEntity e join e.tenantUser t" +
                                " left join t.selectedCredential sc " +
                                " left join e.superior s left join s.tenantUser t2" +
                                " where e.organization.id=?1 and t.telephone=?2", InviteUserDto.class);
        inviteUserDtoTypedQuery.setParameter(1, organId);
        inviteUserDtoTypedQuery.setParameter(2, phone);

        Collection<InviteUserDto> inviteUsers = inviteUserDtoTypedQuery.getResultList();
        Optional<InviteUserDto> inviteUserOptional = inviteUsers.stream().findFirst();
        if (inviteUserOptional.isPresent()) {
            return inviteUserOptional.get();
        } else {
            return null;
        }
    }

    @Override
    public InviteUserDto getByPhone(String phone) {
        TypedQuery<InviteUserDto> inviteUserDtoTypedQuery =
                this.entityManager.createQuery(
                        "select new com.bcxin.tenant.domain.readers.dtos.InviteUserDto(" +
                                "t.name,sc.credentialType,sc.number,t.cid) " +
                                " from TenantUserEntity t left join t.selectedCredential sc" +
                                " where t.telephone=?1", InviteUserDto.class);
        inviteUserDtoTypedQuery.setParameter(1, phone);

        Collection<InviteUserDto> inviteUserDtos = inviteUserDtoTypedQuery.getResultList();
        Optional<InviteUserDto> inviteUserOptional = inviteUserDtos.stream().findFirst();
        if (inviteUserOptional.isPresent()) {
            return inviteUserOptional.get();
        } else {
            return null;
        }
    }

    @Override
    public InviteUserDto getByOranIdAndIdNum(String organId, String idNum) {
        TypedQuery<InviteUserDto> inviteUserDtoTypedQuery =
                this.entityManager.createQuery(
                        "select new com.bcxin.tenant.domain.readers.dtos.InviteUserDto(" +
                                "e.id,t2.name,e.hiredDate,e.occupationType,sc.credentialType,sc.number) " +
                                " from EmployeeEntity e join e.tenantUser t" +
                                " left join t.selectedCredential sc " +
                                " left join e.superior s left join s.tenantUser t2" +
                                " where e.organization.id=?1 and sc.number=?2", InviteUserDto.class);
        inviteUserDtoTypedQuery.setParameter(1, organId);
        inviteUserDtoTypedQuery.setParameter(2, idNum);

        Collection<InviteUserDto> inviteUsers = inviteUserDtoTypedQuery.getResultList();
        Optional<InviteUserDto> inviteUserOptional = inviteUsers.stream().findFirst();
        if (inviteUserOptional.isPresent()) {
            return inviteUserOptional.get();
        } else {
            return null;
        }
    }

    @Override
    public InviteUserDto getByIdNum(String idNum) {
        TypedQuery<InviteUserDto> inviteUserDtoTypedQuery =
                this.entityManager.createQuery(
                        "select new com.bcxin.tenant.domain.readers.dtos.InviteUserDto(" +
                                "t.name,sc.credentialType,sc.number) " +
                                " from TenantUserEntity t left join t.selectedCredential sc" +
                                " where sc.number=?1", InviteUserDto.class);
        inviteUserDtoTypedQuery.setParameter(1, idNum);

        Collection<InviteUserDto> inviteUserDtos = inviteUserDtoTypedQuery.getResultList();
        Optional<InviteUserDto> inviteUserOptional = inviteUserDtos.stream().findFirst();
        if (inviteUserOptional.isPresent()) {
            return inviteUserOptional.get();
        } else {
            return null;
        }
    }

    @Override
    public Pageable<OrganizationAdminDto> searchOrganAdmins(OrganizationAdminCriteria criteria) {
        String qString = "select new com.bcxin.tenant.domain.readers.dtos.OrganizationAdminDto(e.id,t.name,t.telephone," +
                " (select max(k.department.name) from e.departmentEmployeeRelations k ) " +
                ") from EmployeeEntity e join e.tenantUser t where e.domainAdmin=com.bcxin.Infrastructures.enums.TrueFalseStatus.True and e.organization.id=:id ";
        String cString = "select count(1) from EmployeeEntity e join e.tenantUser t where e.domainAdmin=com.bcxin.Infrastructures.enums.TrueFalseStatus.True and e.organization.id=:id  ";

        Map<String, String> params = new HashMap<>();
        params.put("id", criteria.getOrganizationId());

        if (StringUtils.hasLength(criteria.getKeyword())) {
            qString += " and (t.name like :keyword or t.telephone like :keyword) ";
            cString += " and (t.name like :keyword or t.telephone like :keyword) ";

            params.put("keyword", criteria.getKeyword());
        }

        TypedQuery<OrganizationAdminDto> organizationAdminDtoTypedQuery =
                this.entityManager.createQuery(qString, OrganizationAdminDto.class);

        TypedQuery<Long> longTypedQuery =
                this.entityManager.createQuery(cString, Long.class);

        for (String key : params.keySet()) {
            organizationAdminDtoTypedQuery.setParameter(key, params.get(key));
            longTypedQuery.setParameter(key, params.get(key));
        }

        Long totalCount = longTypedQuery.getSingleResult();
        Collection<OrganizationAdminDto> organizations = organizationAdminDtoTypedQuery.setFirstResult(criteria.getSkip())
                .setMaxResults(criteria.getPageSize())
                .getResultList();

        return Pageable.create(
                criteria.getPageIndex(), criteria.getPageSize(), totalCount.intValue(),
                organizations
        );
    }

    @Override
    public Collection<String> getDuplicatedEmployeeRelationIds(Collection<String> sourceDepartIds, String destDepartId) {
        String qString = "select x.id from DepartmentEmployeeRelationEntity x where x.department.id in (?1) and x.employee.id in (select j.employee.id from DepartmentEmployeeRelationEntity j where j.department.id =?2)";
        TypedQuery<String> typedQuery = this.entityManager.createQuery(qString, String.class);
        typedQuery.setParameter(1, sourceDepartIds);
        typedQuery.setParameter(2, destDepartId);


        return typedQuery.getResultList();
    }

    @Override
    public boolean checkIfNotMatchLeaveCondition(Collection<String> employeeIds) {
        Query notMatchedEmployeeLeaveQuery = this.entityManager.createNativeQuery(
                "select count(1) from vm_not_matched_employee_leave_condition k where k.id in (?1)");

        notMatchedEmployeeLeaveQuery.setParameter(1, employeeIds);
        Object result = notMatchedEmployeeLeaveQuery.getSingleResult();

        Long count = Long.parseLong(String.valueOf(result));
        return count <= 0;
    }

    @Override
    public List<EmployeeConditionDto> matchLeaveCondition(Collection<String> employeeIds) {
        Query notMatchedEmployeeLeaveQuery = this.entityManager.createNativeQuery("select k.id as employeeId,k.info from vm_not_matched_employee_leave_condition k where k.id in (?1) ");
        notMatchedEmployeeLeaveQuery.unwrap(SQLQuery.class).setResultTransformer(Transformers.aliasToBean(EmployeeConditionDto.class));
        notMatchedEmployeeLeaveQuery.setParameter(1, employeeIds);
        return notMatchedEmployeeLeaveQuery.getResultList();
    }

    @Override
    public List<EmployeeConditionDto> matchForAttendSite(Collection<String> employeeIds) {
        //判断监管归属地是否是北京
        Query notMatchedEmployeeLeaveQuery = this.entityManager.createNativeQuery("select k.employeeId,k.info from vm_not_matched_employee_leave_condition_attend_site k where k.employeeId in (?1) ");
        notMatchedEmployeeLeaveQuery.unwrap(SQLQuery.class).setResultTransformer(Transformers.aliasToBean(EmployeeConditionDto.class));
        notMatchedEmployeeLeaveQuery.setParameter(1, employeeIds);
        return notMatchedEmployeeLeaveQuery.getResultList();
    }

    /*@Override
    public List<EmployeeConditionDto> matchForAttendSitePerson(Collection<String> employeeIds) {
        Query notMatchedEmployeeLeaveQuery = this.entityManager.createNativeQuery("select k.employeeId,k.info from vm_not_matched_employee_leave_condition_attend_site_person k where k.employeeId in (?1) ");
        notMatchedEmployeeLeaveQuery.unwrap(SQLQuery.class).setResultTransformer(Transformers.aliasToBean(EmployeeConditionDto.class));
        notMatchedEmployeeLeaveQuery.setParameter(1, employeeIds);
        return notMatchedEmployeeLeaveQuery.getResultList();
    }*/

    @Override
    public List<OrganizationExportDto> searchOrgList(com.bcxin.api.interfaces.tenants.criterias.OrganizationCriteria criteria) {

        StringBuilder sqlStr = new StringBuilder();
        sqlStr.append("select new com.bcxin.tenant.domain.repositories.dtos.OrganizationExportDto" +
                "(o.name,o.approvedInformationValueType.status,o.industryCode," +
                "o.institutionalCode,o.placeOfRegister,o.placeOfBusiness,o.createdTime," +
                "o.approvedInformationValueType.lastUpdatedTime," +
                "(select regionFullName from RegionEntity r where r.id = o.superviseRegionCode), " +
                "o.superviseDepartName)");
        sqlStr.append(" from OrganizationEntity o where 1=1 ");

        Map<String, Object> parameters = new HashMap<>();
        if (StringUtils.hasLength(criteria.getName())) {
            parameters.put("name", "%".concat(criteria.getName()).concat("%"));
            sqlStr.append(" and o.name like :name");
        }
        if (StringUtils.hasLength(criteria.getIndustryCode())) {
            parameters.put("industryCode", criteria.getIndustryCode());
            sqlStr.append(" and o.industryCode = :industryCode");
        }

        if (criteria.getPlaceOfSupervise() != null
                && criteria.getPlaceOfSupervise().getDistrict() != null
                && StringUtils.hasLength(criteria.getPlaceOfSupervise().getDistrict().getCode())) {
            parameters.put("superviseRegionCode", criteria.getPlaceOfSupervise().getDistrict().getCode());
            sqlStr.append(" and o.superviseRegionCode = :superviseRegionCode ");
        }
        if (criteria.getStatuses() != null && criteria.getStatuses().size() > 0) {
            Collection<ApprovedStatus> statuses = criteria.getStatuses();
            if (criteria.getStatuses().contains(ApprovedStatus.Init)) {
                sqlStr.append(" and (o.approvedInformationValueType.status=null or o.approvedInformationValueType.status in (:status)) ");
            } else {
                sqlStr.append(" and o.approvedInformationValueType.status in (:status) ");
            }

            parameters.put("status", statuses);

        }

        if (StrUtil.isNotEmpty(criteria.getStartDate())) {
            sqlStr.append(" and o.createdTime >='" + criteria.getStartDate() + "' ");
        }

        if (StrUtil.isNotEmpty(criteria.getEndDate())) {
            sqlStr.append(" and o.createdTime <='" + criteria.getEndDate() + " 23:59:59' ");
        }

        TypedQuery<OrganizationExportDto> organizationQuery =
                this.entityManager.createQuery(sqlStr.toString() + " order by o.createdTime desc,o.approvedInformationValueType.status asc",
                        OrganizationExportDto.class);

        for (String key : parameters.keySet()) {
            organizationQuery.setParameter(key, parameters.get(key));
        }

        List<OrganizationExportDto> organizations = organizationQuery.getResultList();


        return organizations;
    }

    @Override
    public List<OrganizationExportDto> searchOrgList(com.bcxin.api.interfaces.tenants.criterias.OrganizationCriteria criteria, Collection<String> industryCodes) {
        StringBuilder sqlStr = new StringBuilder();
        sqlStr.append("select new com.bcxin.tenant.domain.repositories.dtos.OrganizationExportDto" +
                "(o.name,o.approvedInformationValueType.status,o.industryCode," +
                "o.institutionalCode,o.placeOfRegister,o.placeOfBusiness,o.createdTime," +
                "o.approvedInformationValueType.lastUpdatedTime," +
                "(select regionFullName from RegionEntity r where r.id = o.superviseRegionCode), " +
                "o.superviseDepartName)");
        sqlStr.append(" from OrganizationEntity o where 1=1 ");

        Map<String, Object> parameters = new HashMap<>();
        if (StringUtils.hasLength(criteria.getName())) {
            parameters.put("name", "%".concat(criteria.getName()).concat("%"));
            sqlStr.append(" and o.name like :name");
        }
        if (industryCodes != null && !industryCodes.isEmpty()) {
            parameters.put("industryCodes", industryCodes);
            sqlStr.append(" and o.industryCode in (:industryCodes)");
        }

        if (criteria.getPlaceOfSupervise() != null
                && criteria.getPlaceOfSupervise().getDistrict() != null
                && StringUtils.hasLength(criteria.getPlaceOfSupervise().getDistrict().getCode())) {
            parameters.put("superviseRegionCode", criteria.getPlaceOfSupervise().getDistrict().getCode());
            sqlStr.append(" and o.superviseRegionCode = :superviseRegionCode ");
        }
        if (criteria.getStatuses() != null && criteria.getStatuses().size() > 0) {
            Collection<ApprovedStatus> statuses = criteria.getStatuses();
            if (criteria.getStatuses().contains(ApprovedStatus.Init)) {
                sqlStr.append(" and (o.approvedInformationValueType.status=null or o.approvedInformationValueType.status in (:status)) ");
            } else {
                sqlStr.append(" and o.approvedInformationValueType.status in (:status) ");
            }

            parameters.put("status", statuses);

        }

        if (StrUtil.isNotEmpty(criteria.getStartDate())) {
            sqlStr.append(" and o.createdTime >='" + criteria.getStartDate() + "' ");
        }

        if (StrUtil.isNotEmpty(criteria.getEndDate())) {
            sqlStr.append(" and o.createdTime <='" + criteria.getEndDate() + " 23:59:59' ");
        }

        TypedQuery<OrganizationExportDto> organizationQuery =
                this.entityManager.createQuery(sqlStr.toString() + " order by o.createdTime desc,o.approvedInformationValueType.status asc",
                        OrganizationExportDto.class);

        for (String key : parameters.keySet()) {
            organizationQuery.setParameter(key, parameters.get(key));
        }

        List<OrganizationExportDto> organizations = organizationQuery.getResultList();

        return organizations;
    }

    @Override
    public List<EmployeeExportViewEntity> searchEmployeeList(String organizationId, EmployeeCriteria criteria) {
        StringBuilder sb = new StringBuilder();
        Map<String, Object> parameters = new HashMap<>();
        if (StringUtils.hasLength(criteria.getKeyword())) {
            sb.append(" and (t.name like :keyword or t.telephone like :keyword or s.number like :keyword)");
            parameters.put("keyword", "%" + criteria.getKeyword() + "%");
        }

        /**
         * 证书类型
         */
        if (!CollectionUtils.isEmpty(criteria.getCredentialTypes())) {
            sb.append(" and s.credentialType in :credentialType");
            parameters.put("credentialType", criteria.getCredentialTypes());
        }

        /**
         * 实名认证
         */
        if (!CollectionUtils.isEmpty(criteria.getAuthenticatedStatuses())) {
            sb.append(" and t.authenticateStatus in :authenticateStatus");
            parameters.put("authenticateStatus", criteria.getAuthenticatedStatuses());
        }
        /**
         * 证件核验状态
         */
        if (!CollectionUtils.isEmpty(criteria.getCheckedStatuses())) {
            sb.append(" and t.checkedStatus in :checkedStatus");
            parameters.put("checkedStatus", criteria.getCheckedStatuses());
        }

        /**
         * 职业类型
         */
        if (!CollectionUtils.isEmpty(criteria.getOccupationTypes())) {
            sb.append(" and e.occupationType in :occupationType");
            parameters.put("occupationType", criteria.getOccupationTypes());
        }

        if (criteria.getHiredDate() != null) {
            sb.append(" and e.hiredDate=:hiredDate ");
            parameters.put("hiredDate", criteria.getHiredDate());
        }

        if (StringUtils.hasLength(criteria.getPosition())) {
            sb.append(" and e.position = :position ");
            parameters.put("position", criteria.getPosition());
        }

        if (criteria.getStartDate() != null) {
            sb.append(" and e.hiredDate >=:startDate ");
            parameters.put("startDate", criteria.getStartDate());
        }

        if (criteria.getEndDate() != null) {
            sb.append(" and e.hiredDate <=:endDate ");
            parameters.put("endDate", criteria.getEndDate());
        }

        if ("1".equals(criteria.getContractStatus())) {
            sb.append(" and exists(select ct from ContractEntity ct where ct.employee = e and ct.status=1 and CURDATE() between ct.beginDate and ct.endDate) ");
        }
        if ("0".equals(criteria.getContractStatus())) {
            sb.append(" and not exists(select ct from ContractEntity ct where ct.employee = e and ct.status=1 and CURDATE() between ct.beginDate and ct.endDate) ");
        }
        Optional<OrganizationEntity> organizationOptional = this.organizationRepository.findById(organizationId);
        OrganizationEntity organizationEntity = organizationOptional.get();
        String areaCode = organizationEntity.getSuperviseRegionCode();
        if (StrUtil.isNotEmpty(criteria.getCerStatus())) {
            String exists = " ";
            if ("0".equals(criteria.getCerStatus())) {
                exists = " not ";
            }
            sb.append(" and " + exists + " exists(select tucd from  TenantUserCredentialDetailsEntity tucd " +
                    "where t.id=tucd.tenantUserId  and tucd.state = '1' and tucd.active = true and tucd.certificateType='1'");
            if (!AuthUtil.isUnDistinguishArea()) {
                sb.append(" and tucd.areaCode like :areaCode ");
                parameters.put("areaCode", AuthUtil.getShortAreaCode(areaCode) + "%");
            }

            sb.append(") ");
        }

        if (StrUtil.isNotEmpty(criteria.getGradeCerStatus()) || CollUtil.isNotEmpty(criteria.getLevel())) {
            String exists = " ";
            if ("0".equals(criteria.getCerStatus())) {
                exists = " not ";
            }
            sb.append(" and " + exists + " exists(select tucd from  TenantUserCredentialDetailsEntity tucd  " +
                    "where t.id=tucd.tenantUserId and  tucd.state = '1' and tucd.active = true and tucd.certificateType='2'");
            if (StringUtils.hasLength(areaCode)) {
                sb.append(" and tucd.areaCode like :areaCode ");
                parameters.put("areaCode", AuthUtil.getShortAreaCode(areaCode) + "%");
            }
            if (CollUtil.isNotEmpty(criteria.getLevel())) {
                sb.append(" and tucd.appraisalGrade in :gradeLevels");
                Collection<Integer> level = criteria.getLevel();
                List<String> gradeLevels = new ArrayList<>();
                for (Integer i : level) {
                    gradeLevels.add(i.toString());
                }
                parameters.put("gradeLevels", gradeLevels);
            }
            sb.append(") ");
        }
        //年龄转成日期
        if (criteria.getBeginAge() != null) {
            LocalDate now = LocalDate.now();
            LocalDate bTime = now.minus(criteria.getBeginAge(), ChronoUnit.YEARS);
            sb.append(" and  t.birthdate <= :bTime");
            parameters.put("bTime", Date.from(bTime.atStartOfDay(ZoneId.systemDefault()).toInstant()));
        }

        if (criteria.getEndAge() != null) {
            LocalDate now = LocalDate.now();
            LocalDate eTime = now.minus(criteria.getEndAge() + 1, ChronoUnit.YEARS);
            sb.append(" and  t.birthdate >= :eTime");
            parameters.put("eTime", Date.from(eTime.atStartOfDay(ZoneId.systemDefault()).toInstant()));
        }

        //是否投保
        if (criteria.getInsure() != null) {
            if (criteria.getInsure().equals(TrueFalseStatus.False)) {
                sb.append(" and  (e.insure = :insure or e.insure is null) ");
            } else {
                sb.append(" and  e.insure = :insure ");
            }
            parameters.put("insure", criteria.getInsure());
        }
        if (!CollectionUtils.isEmpty(criteria.getBackgroundScreeningStatus())) {
            sb.append(" and t.backgroundScreeningStatus in :backgroundScreeningStatus");
            parameters.put("backgroundScreeningStatus", criteria.getBackgroundScreeningStatus());
        }
        String qString = "select new com.bcxin.tenant.domain.entities.EmployeeExportViewEntity(" +
                "e.id,t.id,t.name,t.telephone,s.credentialType,s.number,t.checkedStatus,t.lastCheckedStatusTime,t.authenticateStatus,t.authenticatedResult,e.occupationType,p.name," +
                "e.position,e.hiredDate,e.positiveDate,t.sex,t.birthdate,t.nation,t.education,t.politicsStatus,t.householdType," +
                "t.stature,t.nativePlace,t.militaryStatus,t.maritalStatus,s.validDateFrom,s.validDateTo,s.address," +
                "(select count(1) from ContractEntity ct where ct.employee = e and ct.status=1 and CURDATE() between ct.beginDate and ct.endDate)>0," +
//                "(select count(1) from TenantUserCredentialsEntity tuc join TenantUserCredentialDetailsEntity tucd where tuc.tenantUser = t and tuc.credentialType=7 and tuc.number = tucd.zsbh and tucd.areaCode like :areaCode )>0," +
//                "(select count(1) from TenantUserCredentialsEntity tuc where tuc.tenantUser = t and tuc.credentialType=8 and CURDATE() <  tuc.validDateTo)>0," +
                "e.insure,e.createdTime,e.domainAdmin,e.organization.id," +
                "e.personStatus,e.probation,e.planPositiveDate,t.emergencyContact,t.emergencyPhone,t.licenseLevel,t.placeOfNow" +
                ",e.hiredOperator" +
                ") " +
                " from EmployeeEntity e join e.tenantUser t left join e.superior.tenantUser p left join t.selectedCredential s " +
                " where e.status!=com.bcxin.Infrastructures.enums.EmploymentStatus.OffJob ";


        if (StrUtil.isNotEmpty(organizationId)) {
            qString += " and e.organization.id=:organId ";
            parameters.put("organId", organizationId);
        }

        if (StrUtil.isNotEmpty(criteria.getOrganName())) {
            qString += " and e.organization.name like '%:organName%' ";
            parameters.put("organName", criteria.getOrganName());
        }


        qString += sb;

        TenantUserContext userContext = TenantContext.getInstance().getUserContext();
        if (!criteria.isIgnorePermission() && userContext != null && userContext.get() != null && !userContext.get().isMaster() &&
                StringUtils.hasLength(userContext.get().getEmployeeId())) {


            if (userContext.get().isDepartAdmin()) {
                String deptAdminSql = getDeptAdminSql(criteria.getTreeCodes());
                if (!CollectionUtils.isEmpty(criteria.getDepartIds())) {
                    qString += " and ((e.id=:selfEmployeeId and exists(select k from e.departmentEmployeeRelations k where k.department.id in (:selfDepartmentIds))) or exists (SELECT k FROM e.departmentEmployeeRelations k WHERE k.department.id IN (:departmentIds) " + deptAdminSql + " ))";
                    parameters.put("departmentIds", criteria.getDepartIds());
                    parameters.put("selfDepartmentIds", criteria.getDepartIds());
                } else {
                    qString += " and (e.id=:selfEmployeeId or exists (SELECT k FROM e.departmentEmployeeRelations k WHERE 1=1 " + deptAdminSql + " ))";
                }
                parameters.put("selfEmployeeId", userContext.get().getEmployeeId());
            } else {
                if (!CollectionUtils.isEmpty(criteria.getDepartIds())) {
                    qString += " and ((e.id=:selfEmployeeId and exists(select k from e.departmentEmployeeRelations k where k.department.id in (:selfDepartmentIds))))";
                    parameters.put("selfDepartmentIds", criteria.getDepartIds());
                } else {
                    qString += " and (e.id=:selfEmployeeId)";
                }
                parameters.put("selfEmployeeId", userContext.get().getEmployeeId());
            }


        } else {
            /**
             * 所在部门
             */
            if (!CollectionUtils.isEmpty(criteria.getDepartIds())) {
                qString += " and exists(select de from DepartmentEmployeeRelationEntity de where de.employee.id=e.id and de.department.id in (:departmentIds))";

                parameters.put("departmentIds", criteria.getDepartIds());
            }
        }

        TypedQuery<EmployeeExportViewEntity> employeeTypedQuery = this.entityManager.createQuery(qString, EmployeeExportViewEntity.class);

        for (String key : parameters.keySet()) {
            employeeTypedQuery = employeeTypedQuery.setParameter(key, parameters.get(key));
        }
        Map<String, DepartmentDto> departMaps = this.getDepartMaps(criteria.getOrganizationId());

        List<EmployeeExportViewEntity> result = employeeTypedQuery.getResultList();
        Collection<String> employeeIds = result.stream().map(ii -> ii.getId()).collect(Collectors.toList());
        if (employeeIds.size() > 0) {
            TypedQuery<DepartDto> mapTypedQuery =
                    this.entityManager.createQuery("select new com.bcxin.tenant.domain.readers.dtos.DepartDto(de.employee.id as employeeId, dp.id as departId, dp.name as departName,dp.indexTree) from DepartmentEmployeeRelationEntity de join de.department dp where de.employee.id in (:employeeIds)",
                            DepartDto.class);
            mapTypedQuery.setParameter("employeeIds", employeeIds);
            Collection<DepartDto> departDtos = mapTypedQuery.getResultList();
            for (DepartDto departDto : departDtos) {
                if (departMaps.get(departDto.getDepartId()) != null) {
                    departDto.resetDepartName(departMaps.get(departDto.getDepartId()).getIndexTree());
                }
            }
            Map<String, List<DepartDto>> departMap = departDtos.stream().collect(Collectors.groupingBy(DepartDto::getEmployeeId));
            for (EmployeeExportViewEntity rt : result) {
                if (departMap.get(rt.getId()) != null) {
                    rt.assignDepart(departMap.get(rt.getId()).stream().map(DepartDto::getDepartName).collect(Collectors.joining(", ")));
                }
            }

            mapTypedQuery = this.entityManager.createQuery("select new com.bcxin.tenant.domain.readers.dtos.DepartDto(de.employee.id as employeeId, dp.id as departId, dp.name as departName) from DepartmentAdminEntity de join de.department dp where de.employee.id in (:employeeIds)",
                    DepartDto.class);
            mapTypedQuery.setParameter("employeeIds", employeeIds);
            departDtos = mapTypedQuery.getResultList();
            departMap = departDtos.stream().collect(Collectors.groupingBy(DepartDto::getEmployeeId));
            for (EmployeeExportViewEntity rt : result) {
                if (departMap.get(rt.getId()) != null) {
                    rt.assignAdminDepart(departMap.get(rt.getId()).stream().sorted(Comparator.comparing(DepartDto::getDepartName, Comparator.nullsLast(String::compareTo))).map(DepartDto::getDepartName).collect(Collectors.joining(", ")));
                }
            }


            List<TenantUserCredentialDetailsEntity> cerDetails = this.getCerDetailsByUserIds(CerType.Qualification.getTypeValue(), result.stream().map(EmployeeExportViewEntity::getUserId).collect(Collectors.toList()));
            List<TenantUserCredentialDetailsEntity> gradeDetails = this.getCerDetailsByUserIds(CerType.Grade.getTypeValue(), result.stream().map(EmployeeExportViewEntity::getUserId).collect(Collectors.toList()));
            if (cerDetails != null && cerDetails.size() > 0) {
                Map<String, List<TenantUserCredentialDetailsEntity>> cerDetailMap = cerDetails.stream().collect(Collectors.groupingBy(i -> i.getTenantUserId()));
                for (EmployeeExportViewEntity rt : result) {
                    List<TenantUserCredentialDetailsEntity> cerDetailList = cerDetailMap.get(rt.getUserId());
                    if (cerDetailList != null) {
                        String finalAreaCode = areaCode;
                        for (TenantUserCredentialDetailsEntity tenantUserCredentialDetailsEntity : cerDetailList) {
                            if (StrUtil.isNotEmpty(tenantUserCredentialDetailsEntity.getAreaCode()) && tenantUserCredentialDetailsEntity.getAreaCode().length() >= 2) {
                                if (AuthUtil.getShortAreaCode(tenantUserCredentialDetailsEntity.getAreaCode()).equals(AuthUtil.getShortAreaCode(finalAreaCode))) {
                                    rt.setCerStatus(TrueFalseStatus.True);
                                    List<TenantUserCredentialDetailsEntity> tenantUserCredentialDetailsEntities = cerDetailMap.get(rt.getUserId());
                                    for (TenantUserCredentialDetailsEntity userCredentialDetailsEntity : tenantUserCredentialDetailsEntities) {
                                        rt.setCerNo(userCredentialDetailsEntity.getCerNo());
                                    }
                                } else if (tenantUserCredentialDetailsEntity.getAreaCode() == "#") {
                                    rt.setCerStatus(TrueFalseStatus.False);
                                }
                                ;
                            }
                        }
                    }
                }
            }

            if (gradeDetails != null && gradeDetails.size() > 0) {
                Map<String, List<TenantUserCredentialDetailsEntity>> cerDetailMap = gradeDetails.stream().collect(Collectors.groupingBy(i -> i.getTenantUserId()));
                for (EmployeeExportViewEntity rt : result) {
                    List<TenantUserCredentialDetailsEntity> cerDetailList = cerDetailMap.get(rt.getUserId());
                    if (cerDetailList != null) {
                        String finalAreaCode = areaCode;
                        for (TenantUserCredentialDetailsEntity tenantUserCredentialDetailsEntity : cerDetailList) {
                            if (StrUtil.isNotEmpty(tenantUserCredentialDetailsEntity.getAreaCode()) && tenantUserCredentialDetailsEntity.getAreaCode().length() >= 2) {
                                if (AuthUtil.getShortAreaCode(tenantUserCredentialDetailsEntity.getAreaCode()).equals(AuthUtil.getShortAreaCode(finalAreaCode))) {
                                    rt.setGradeCerStatus(TrueFalseStatus.True);
                                    List<TenantUserCredentialDetailsEntity> tenantUserCredentialDetailsEntities = cerDetailMap.get(rt.getUserId());
                                    for (TenantUserCredentialDetailsEntity userCredentialDetailsEntity : tenantUserCredentialDetailsEntities) {
                                        rt.setGradeCerNo(userCredentialDetailsEntity.getCerNo());
                                        rt.setGradeLevel(userCredentialDetailsEntity.getGradeLevel().getTypeValue());
                                    }
                                } else if (tenantUserCredentialDetailsEntity.getAreaCode() == "#") {
                                    rt.setGradeCerStatus(TrueFalseStatus.False);
                                }
                                ;
                            }
                        }
                    }
                }
            }
        }
        return result.stream().sorted(Comparator.comparing(EmployeeExportViewEntity::getDepartName, Comparator.nullsLast(String::compareTo))).collect(Collectors.toList());
    }

    private List<TenantUserCredentialDetailsEntity> getCerDetailsByUserIds(String cerType, List<String> userIds) {
        if (userIds == null || userIds.size() == 0) {
            return null;
        }

        StringBuilder sql = new StringBuilder("select de from  TenantUserCredentialDetailsEntity de ");
        sql.append(" where de.tenantUserId in :userIds and de.certificateType= :cerType and de.state = '1' and de.active = true");
        TypedQuery<TenantUserCredentialDetailsEntity> mapTypedQuery = this.entityManager.createQuery(sql.toString(), TenantUserCredentialDetailsEntity.class);
        mapTypedQuery.setParameter("userIds", userIds);
        mapTypedQuery.setParameter("cerType", cerType);
        return mapTypedQuery.getResultList();
    }

    @Override
    public Collection<DepartmentDto> getManagedDepartmentDtos(String organId) {
        String qString = "select new com.bcxin.tenant.domain.repositories.dtos.DepartmentDto(d.id,d.code,d.name,d.parent.id,d.level," +
                " -1l," +
                " d.displayOrder,d.permissionType,d.permissionConfig,d.indexTree) " +
                " from DepartmentEntity d where d.organization.id=?1 and d.deleted=false ";
        TenantUserContext userContext = TenantContext.getInstance().getUserContext();
        Map<Integer, Object> parameters = new HashMap<>();
        parameters.put(1, organId);
        if (userContext != null && userContext.get() != null && !userContext.get().isMaster() &&
                StringUtils.hasLength(userContext.get().getEmployeeId())) {
            qString += " and exists (select da from DepartmentAdminEntity da where da.employee.id=?2 and d.indexTree like concat(da.department.indexTree ,'%'))";
            parameters.put(2, userContext.get().getEmployeeId());
        }

        TypedQuery<DepartmentDto> departmentDtoTypedQuery =
                this.entityManager.createQuery(qString, DepartmentDto.class);
        for (Integer pIndex : parameters.keySet()) {
            departmentDtoTypedQuery.setParameter(pIndex, parameters.get(pIndex));
        }

        return departmentDtoTypedQuery.getResultList();
    }

    @Override
    public Collection<DepartmentDto> getAllManagedDepartmentDtos(String organId) {
        String qString = "select new com.bcxin.tenant.domain.repositories.dtos.DepartmentDto(d.id,d.code,d.name,d.parent.id,d.level," +
                " -1l," +
                " d.displayOrder,d.permissionType,d.permissionConfig,d.indexTree) " +
                " from DepartmentEntity d where d.organization.id=?1 and d.deleted=false ";
        Map<Integer, Object> parameters = new HashMap<>();
        parameters.put(1, organId);
        TypedQuery<DepartmentDto> departmentDtoTypedQuery =
                this.entityManager.createQuery(qString, DepartmentDto.class);
        for (Integer pIndex : parameters.keySet()) {
            departmentDtoTypedQuery.setParameter(pIndex, parameters.get(pIndex));
        }

        return departmentDtoTypedQuery.getResultList();
    }

    @Override
    public Collection<EmployeeDepartIdDto> getEmployeeDepartIds(String organizationId, Collection<String> employeeIds) {
        TypedQuery<EmployeeDepartIdDto> employeeDepartIdDtoTypedQuery =
                this.entityManager.createQuery(
                        "select new com.bcxin.tenant.domain.readers.dtos.EmployeeDepartIdDto(d.department.id,d.employee.id) " +
                                " from DepartmentEmployeeRelationEntity d where d.employee.id in (?1)",
                        EmployeeDepartIdDto.class);
        employeeDepartIdDtoTypedQuery.setParameter(1, employeeIds);

        return employeeDepartIdDtoTypedQuery.getResultList();
    }

    @Override
    public int checkIfTelephoneExists(String telephone) {
        TypedQuery<Long> tenantUserCountQuery =
                this.entityManager.createQuery("select count(t) from TenantUserEntity t where t.telephone=?1", Long.class);
        tenantUserCountQuery.setParameter(1, telephone);
        Long count = tenantUserCountQuery.getSingleResult();
        return count.intValue();
    }

    @Override
    public Pageable<OrganizationDto> searchOrganizations(OrganizationCriteria criteria) {
        String qString =
                "select new com.bcxin.tenant.domain.readers.dtos.OrganizationDto(o.id,o.code,o.name,o.approvedInformationValueType.status," +
                        "o.industryCode,o.institutionalCode,o.placeOfRegister,o.placeOfBusiness,o.createdTime,o.approvedInformationValueType.lastUpdatedTime,o.lonLatJson," +
                        "(select regionFullName from RegionEntity r where r.id = o.superviseRegionCode), o.superviseDepartName) " +
                        " from OrganizationEntity o where 1=1 ";

        String cString =
                "select count(1) from OrganizationEntity o where 1=1 ";

        Map<String, Object> parameters = new HashMap<>();
        if (StringUtils.hasLength(criteria.getName())) {
            parameters.put("name", "%".concat(criteria.getName()).concat("%"));
            qString += " and o.name like :name";
            cString += " and o.name like :name";
        }
        if (criteria.getIndustryCodes() != null && !criteria.getIndustryCodes().isEmpty()) {
            parameters.put("industryCodes", criteria.getIndustryCodes());
            qString += " and o.industryCode in (:industryCodes)";
            cString += " and o.industryCode in (:industryCodes)";
        } else if (StringUtils.hasLength(criteria.getIndustryCode())) {
            parameters.put("industryCode", criteria.getIndustryCode());
            qString += " and o.industryCode = :industryCode";
            cString += " and o.industryCode = :industryCode";
        }
        if (StringUtils.hasLength(criteria.getSuperviseRegionCode())) {
            parameters.put("superviseRegionCode", criteria.getSuperviseRegionCode());
            qString += " and o.superviseRegionCode = :superviseRegionCode";
            cString += " and o.superviseRegionCode = :superviseRegionCode";
        }

        if (criteria.getStatuses() != null && criteria.getStatuses().size() > 0) {
            Collection<ApprovedStatus> statuses = criteria.getStatuses();
            if (criteria.getStatuses().contains(ApprovedStatus.Init)) {
                qString += " and (o.approvedInformationValueType.status=null or o.approvedInformationValueType.status in (:status)) ";
                cString += " and (o.approvedInformationValueType.status=null or o.approvedInformationValueType.status in (:status)) ";
            } else {
                qString += " and o.approvedInformationValueType.status in (:status) ";
                cString += " and o.approvedInformationValueType.status in (:status) ";
            }

            parameters.put("status", statuses);

        }

        if (StrUtil.isNotEmpty(criteria.getStartDate())) {
            qString += " and o.createdTime >='" + criteria.getStartDate() + "' ";
            cString += " and o.createdTime >='" + criteria.getStartDate() + "' ";
        }

        if (StrUtil.isNotEmpty(criteria.getEndDate())) {
            qString += " and o.createdTime <='" + criteria.getEndDate() + " 23:59:59' ";
            cString += " and o.createdTime <='" + criteria.getEndDate() + " 23:59:59' ";
        }

        TypedQuery<OrganizationDto> organizationQuery =
                this.entityManager.createQuery(qString + " order by o.createdTime desc,o.approvedInformationValueType.status asc",
                        OrganizationDto.class);
        TypedQuery<Long> organizationCountQuery = this.entityManager.createQuery(cString, Long.class);

        for (String key : parameters.keySet()) {
            organizationQuery.setParameter(key, parameters.get(key));
            organizationCountQuery.setParameter(key, parameters.get(key));
        }

        Long totalCount = organizationCountQuery.getSingleResult();
        Collection<OrganizationDto> organizations = organizationQuery.setFirstResult(criteria.getSkip())
                .setMaxResults(criteria.getPageSize())
                .getResultList();

        return Pageable.create(criteria.getPageIndex(), criteria.getPageSize(), totalCount.intValue(),
                organizations
        );
    }

    @Override
    public TenantUserCredentialsDto getCertificateByEmployeeId(String employeeId) {
        TypedQuery<TenantUserCredentialsDto> tenantUserCredentialsQuery =
                this.entityManager.createQuery(
                        "select new com.bcxin.tenant.domain.readers.dtos.TenantUserCredentialsDto(" +
                                "t.name,t.number) " +
                                " from EmployeeEntity sc join TenantUserCredentialsEntity t on t.tenantUser.id = sc.tenantUser.id and t.credentialType = com.bcxin.Infrastructures.enums.CredentialType.IdCard" +
                                " where sc.id=?1", TenantUserCredentialsDto.class);
        tenantUserCredentialsQuery.setParameter(1, employeeId);

        try {
            return tenantUserCredentialsQuery.getSingleResult();
        } catch (NoResultException ex) {
            return null;
        }
    }

    @Override
    public TenantUserDto getTenantUserById(String id) {
        TypedQuery<TenantUserDto> tenantDtoTypedQuery =
                this.entityManager.createQuery(
                        "select new com.bcxin.tenant.domain.readers.dtos.TenantUserDto(" +
                                "t.id,t.name,t.email,t.wechatNicky, t.telephone, t.lonLatJson, t.checkedStatus,t.authenticateStatus,t.authenticatedResult,t.headPhoto," +
                                "t.sex,t.nation,t.workYear,t.diseasesHistory,t.politicsStatus,t.stature,t.militaryStatus,t.birthdate,t.education,t.householdType,t.nativePlace,t.maritalStatus," +
                                "t.oneInchColorWhitePhoto,t.twoInchColorBluePhoto,t.cid," +
                                "t.emergencyContact,t.emergencyPhone,t.licenseLevel,t.placeOfNow," +
                                "sc.credentialType,sc.name, sc.number,sc.validDateFrom,sc.validDateTo,sc.frontPhoto,sc.reversePhoto,sc.address,sc.headPhoto,t.thirdPartyLoginNo,t.imIdentity,t.CertificateImage) " +
                                " from TenantUserEntity t left join t.selectedCredential sc " +
                                " where t.id=?1", TenantUserDto.class);

        tenantDtoTypedQuery.setParameter(1, id);

        try {
            return tenantDtoTypedQuery.getSingleResult();
        } catch (NoResultException ex) {
            return null;
        }
    }

    @Override
    public TenantUserDto getTenantUserByThirdPartyLoginNo(String id) {
        TypedQuery<TenantUserDto> tenantDtoTypedQuery =
                this.entityManager.createQuery(
                        "select new com.bcxin.tenant.domain.readers.dtos.TenantUserDto(" +
                                "t.id,t.name,t.email,t.wechatNicky, t.telephone, t.lonLatJson, t.checkedStatus,t.authenticateStatus,t.authenticatedResult,t.headPhoto," +
                                "t.sex,t.nation,t.workYear,t.diseasesHistory,t.politicsStatus,t.stature,t.militaryStatus,t.birthdate,t.education,t.householdType,t.nativePlace,t.maritalStatus," +
                                "t.oneInchColorWhitePhoto,t.twoInchColorBluePhoto,t.cid," +
                                "t.emergencyContact,t.emergencyPhone,t.licenseLevel,t.placeOfNow," +
                                "sc.credentialType,sc.name, sc.number,sc.validDateFrom,sc.validDateTo,sc.frontPhoto,sc.reversePhoto,sc.address,sc.headPhoto,t.thirdPartyLoginNo,t.imIdentity,t.CertificateImage) " +
                                " from TenantUserEntity t left join t.selectedCredential sc " +
                                " where t.thirdPartyLoginNo=?1", TenantUserDto.class);

        tenantDtoTypedQuery.setParameter(1, id);

        try {
            return tenantDtoTypedQuery.getSingleResult();
        } catch (NoResultException ex) {
            return null;
        }
    }

    @Override
    public List<CompanyDto> exactSearch(String name, String unifySocialCreditCode) {
        String sql = "select new com.bcxin.tenant.domain.readers.dtos.CompanyDto(u.id,u.name,c.unifySocialCreditCode)";
        sql += " from OrganizationEntity u left join CompanyEntity c on c.id = u.id where 1=1 ";

        int index = 0;
        if (!StringUtils.isEmpty(name)) {
            sql += " and u.name=?" + (++index);
        }

        if (!StringUtils.isEmpty(unifySocialCreditCode)) {
            sql += " and c.unifySocialCreditCode=?" + (++index);
        }
        TypedQuery<com.bcxin.tenant.domain.readers.dtos.CompanyDto> companyQuery =
                this.entityManager.createQuery(sql, com.bcxin.tenant.domain.readers.dtos.CompanyDto.class);

        if (!StringUtils.isEmpty(name)) {
            companyQuery.setParameter(1, name);
        }

        if (!StringUtils.isEmpty(unifySocialCreditCode)) {
            companyQuery.setParameter(index, unifySocialCreditCode);
        }

        return companyQuery.getResultList();
    }

    @Override
    public Collection<TenantEventEntity> getTopNTenantEvents(Long startVersion, Collection<String> mapKeys, Collection<String> eventIds, int topN) {
        String qString = "select t from TenantEventEntity t where t.version>?2 and t.mapKey in (?1)  ";

        Map<Integer, Object> parameters = new HashMap<>();
        parameters.put(1, mapKeys);


        if (!CollectionUtils.isEmpty(eventIds)) {
            parameters.put(2, 0L);
            qString += " and t.id in (?3) ";
            parameters.put(3, eventIds);
        } else {
            parameters.put(2, startVersion);
            qString += " and t.createdTime<?3 ";
            /**
             * 只处理延迟的部分, 避免重复执行
             */
            parameters.put(3, Timestamp.from(Instant.now().minus(3, ChronoUnit.MINUTES)));
        }

        qString += " order by t.version asc";

        TypedQuery<TenantEventEntity> eventEntityTypedQuery =
                this.entityManager.createQuery(qString,
                        TenantEventEntity.class);

        for (Integer key : parameters.keySet()) {
            eventEntityTypedQuery.setParameter(key, parameters.get(key));
        }

        eventEntityTypedQuery = eventEntityTypedQuery.setMaxResults(topN);

        return eventEntityTypedQuery.getResultList();
    }

    @Override
    public Collection<String> getTopNPendingTenantEvents(Collection<String> mapKeys, boolean isForFailed, int topN) {
        String qString = "select t.id from TenantEventEntity t where t.mapKey in (?1) and " +
                " (t.lastProcessedTime<?2 or (t.lastProcessedTime is null and t.createdTime<?3)) and t.status=?4 ";

        Map<Integer, Object> parameters = new HashMap<>();
        parameters.put(1, mapKeys);
        // 优化：将延迟从5分钟改为5秒，减少事件处理延迟
        // 5秒足够确保数据库事务已提交，同时大幅降低处理延迟
        Timestamp comparedTimestamp = Timestamp.from(Instant.now().minus(5, ChronoUnit.SECONDS));
        parameters.put(2, comparedTimestamp);
        parameters.put(3, comparedTimestamp);

        if (isForFailed) {
            parameters.put(4, EventProcessedStatus.Failed);
        } else {
            parameters.put(4, EventProcessedStatus.Init);
        }

        qString += " order by t.lastProcessedTime asc";

        TypedQuery<String> eventEntityTypedQuery =
                this.entityManager.createQuery(qString, String.class);

        for (Integer key : parameters.keySet()) {
            eventEntityTypedQuery.setParameter(key, parameters.get(key));
        }

        eventEntityTypedQuery = eventEntityTypedQuery.setMaxResults(topN);

        return eventEntityTypedQuery.getResultList();
    }

    @Override
    public Collection<EmployeeEntity> getEmployeesByIds(Collection<String> ids) {
        TypedQuery<EmployeeEntity> eventEntityTypedQuery =
                this.entityManager.createQuery("select t from EmployeeEntity t where t.id in (?1)",
                        EmployeeEntity.class);

        eventEntityTypedQuery.setParameter(1, ids);

        return eventEntityTypedQuery.getResultList();
    }

    @Override
    public List<EmployeeEntity> getEmployeesByUserId(String userId) {
        TypedQuery<EmployeeEntity> eventEntityTypedQuery =
                this.entityManager.createQuery("select t from EmployeeEntity t where t.tenantUser.id =?1 and t.status <> 1",
                        EmployeeEntity.class);

        eventEntityTypedQuery.setParameter(1, userId);

        return eventEntityTypedQuery.getResultList();
    }

    public EmployeeBasicDto getEmployeeBasic(String organizationId, String id) {
        try {
            TypedQuery<EmployeeBasicDto> query =
                    this.entityManager.createQuery("select new com.bcxin.tenant.domain.readers.dtos.EmployeeBasicDto(e.id,t.telephone,sc.number,false,e.status,e.masterSlaveType)" +
                            " from EmployeeEntity e join e.tenantUser t join t.selectedCredential sc " +
                            " where e.organization.id = ?1 and e.id = ?2 ", EmployeeBasicDto.class);
            query.setParameter(1, organizationId);
            query.setParameter(2, id);
            List<EmployeeBasicDto> resultList = query.getResultList();
            if (resultList.size() > 0) {
                return resultList.get(0);
            }
            return null;
        } catch (Exception ex) {
            throw ex;
        }

    }

    @Override
    public Pageable<CredentialResponse> getCredentials(String id, String organizationId, GetCredentialRequest request) {
        Map<String, Object> parameters = new HashMap<>();
        parameters.put("id", id);
        String sqlStr = "select a.id,a.create_time as created_time,a.fzjgmc as address," +
                "case a.certificateType when '1' then 7 when '2' then 8 when '3' then 9 when '11' then 11 when '12' then 12 when '13' then 13 else 0 end credential_type," +
                "a.electronCerUrl as front_photo,a.zsbh as number,'' as reverse_photo,fzrq as valid_date_from,'9999-12-31' as valid_date_to,a.tenant_user_id,'' as head_photo,xm as name,0 as selected " +
                "from tenant_user_credential_details a join tenant_employees as b on b.tenant_user_id = a.tenant_user_id   where b.id=:id and a.state = '1'  ";

        String sqlCount = "select count(1) as sum from tenant_user_credential_details a join tenant_employees as b on b.tenant_user_id = a.tenant_user_id  where b.id=:id and a.state = '1' ";
        Collection<String> credentialDetailType = new ArrayList<>();
        if (CollUtil.isNotEmpty(request.getCredentialTypes())) {
            credentialDetailType =
                    request.getCredentialTypes().stream().map(credentialType -> {
                        switch (credentialType) {
                            case "7":
                                return ("1");
                            case "8":
                                return ("2");
                            case "9":
                                return ("3");
                        }

                        return credentialType;
                    }).collect(Collectors.toList());
            sqlStr += " and a.certificatetype in (:credentialDetailType)";
            sqlCount += " and a.certificatetype in (:credentialDetailType)";
            parameters.put("credentialDetailType", credentialDetailType);
        }
        if (CollUtil.isEmpty(credentialDetailType)) {
            throw new NotSupportTenantException("不支持该证件类型");
        }
        if (StringUtils.hasLength(request.getName())) {
            sqlStr += " and a.xm = :name";
            sqlCount += " and a.xm = :name";
            parameters.put("name", request.getName());
        }
        if (StringUtils.hasLength(request.getNumber())) {
            sqlStr += " and a.zsbh = :number";
            sqlCount += " and a.zsbh =:number";
            parameters.put("number", request.getNumber());
        }
        Optional<OrganizationEntity> organizationOptional = this.organizationRepository.findById(organizationId);
        if (request.getSkipAreaCodeFilter() != null) {
            if (request.getSkipAreaCodeFilter()) {
                sqlStr += "and a.areacode like :areaCode";
                sqlCount += "and a.areacode like :areaCode";
                parameters.put("areaCode", AuthUtil.getShortAreaCode(organizationOptional.get().getSuperviseRegionCode()) + "%");
            }
        }

        Query query = this.entityManager.createNativeQuery(sqlStr, TenantUserCredentialsEntity.class);
        Query cquery = this.entityManager.createNativeQuery(sqlCount);

        for (String key : parameters.keySet()) {
            query.setParameter(key, parameters.get(key));
            cquery.setParameter(key, parameters.get(key));
        }
        query.setFirstResult(request.getSkip()).setMaxResults(request.getPageSize());
        ArrayList<CredentialResponse> credentialResponses = new ArrayList<>();
        List<TenantUserCredentialsEntity> resultList = query.getResultList();
        int sum = Integer.valueOf(cquery.getResultList().get(0).toString());
        for (TenantUserCredentialsEntity tenantUserCredentialsEntity : resultList) {
            CredentialResponse credentialResponse = new CredentialResponse();
            BeanUtil.copyProperties(tenantUserCredentialsEntity, credentialResponse);
            credentialResponse.setCredentialTypeName(tenantUserCredentialsEntity.getCredentialType().getTypeName());
            credentialResponse.setCreatedTime(tenantUserCredentialsEntity.getCreatedTime());

            //查询详细表信息
            String sqlDetail = "select id,native_code,active,create_time,update_time,update_by,isOldData,xm,companyName,address,csrq,idnum,zsbh,fzrq,zzrq,zzzt,securitypersonid,fzjgmc,fzjgbh,isprint,havephoto,printcount,printTime,updateflag,receiveState,rkkflag,state,examDate,score,cancelDate,cancelReason,cancelOrgName,cancelOrg,revokeReason,revokeOrgName,revokeOrg,revokeDate,snapshotId,isDraw,userid,areaCode,nation,personId,orgId,sex,phone,trainId,trainName,companyId,populationAddress,push,seq,headImg,electronCerUrl,certificateType,trainTime,appraisalTime,appraisalGrade,companyCode,securityCertificateNo,securityCertificateId,uploadCompanyName,trainType,pushDate,personGradeId,profession,trainStartTime,trainEndTime,tenant_user_id,organization_id from tenant_user_credential_details where id  = ?1";
            Query detailsQuery = this.entityManager.createNativeQuery(sqlDetail);
            //返回证书详细表数据
            detailsQuery.unwrap(SQLQuery.class).setResultTransformer(Transformers.ALIAS_TO_ENTITY_MAP);
            detailsQuery.setParameter(1, tenantUserCredentialsEntity.getId());
            credentialResponse.setDetail(detailsQuery.getResultList());
            credentialResponses.add(credentialResponse);
        }
        return Pageable.create(request.getPageIndex(), request.getPageSize(), sum, credentialResponses);
    }

    @Override
    public List<CurrentCredentialResponse> getCredentials(QueryCredentialRequest queryRequest) {
        StringBuilder sql = new StringBuilder("select de from TenantUserCredentialDetailsEntity de  ");
        sql.append(" where de.tenantUserId =:userId  and de.state = '1' and de.active = true ");
        Map<String, Object> parameters = new HashMap<>();
        parameters.put("userId", queryRequest.getUserId());
        TypedQuery<TenantUserCredentialDetailsEntity> query = this.entityManager.createQuery(sql.toString(), TenantUserCredentialDetailsEntity.class);
        for (String key : parameters.keySet()) {
            query.setParameter(key, parameters.get(key));
        }

        List<TenantUserCredentialDetailsEntity> cerList = query.getResultList();
        if (cerList.size() > 0) {
            List<CurrentCredentialResponse> resultList = new ArrayList<>();
            List<EmployeeEntity> employees = this.getEmployeesByUserId(queryRequest.getUserId());
            List<OrganizationEntity> organizationEntityList = new ArrayList<>();
            if (employees.size() > 0) {
                organizationEntityList = employees.stream().map(ii -> ii.getOrganization()).collect(Collectors.toList());
            }

            for (TenantUserCredentialDetailsEntity credentialDetailsEntity : cerList) {
                CurrentCredentialResponse cerResponse = CurrentCredentialResponse.create(credentialDetailsEntity.getName(),
                        credentialDetailsEntity.getCerTypeName(),
                        credentialDetailsEntity.getCerNo(),
                        credentialDetailsEntity.getOrgName(),
                        credentialDetailsEntity.getCerDate());
                resultList.add(cerResponse);
                if (organizationEntityList.size() == 0) {
                    continue;
                }
                List<String> comIds = new ArrayList<>();
                for (OrganizationEntity organizationEntity : organizationEntityList) {
                    if (AuthUtil.isUnDistinguishArea(organizationEntity.getSuperviseRegionCode()) || organizationEntity.getSuperviseRegionCode().equals(credentialDetailsEntity.getAreaCode())) {
                        comIds.add(organizationEntity.getId());
                    }
                }
                cerResponse.setComIds(comIds);
            }

            return resultList;
        }

        return null;
    }

    @Override
    public List<QualificationCredentialResponse> getQualificationCredentials(QueryCredentialRequest request) {
        Map<String, Object> parameters = new HashMap<>();
        String sqlStr = "select a.* from tenant_user_credential_details a where ";
        String idNumValue = request.getIdnum();
        if (StringUtils.hasLength(idNumValue)) {
            if (idNumValue.matches("^\\d{17}[\\dXx]$|^\\d{15}$")) {
                // 身份证号码格式 (15位或18位数字，18位最后一位可以是数字或X)
                sqlStr += " a.idnum = :idnum";
                parameters.put("idnum", idNumValue);
            } else if (idNumValue.matches("^[\\u4e00-\\u9fa5]+$")) {
                // 纯中文姓名 TODO 不通过姓名查
                return new ArrayList<>();
            } else {
                // 其他情况使用证书编号
                sqlStr += " a.zsbh = :zsbh";
                parameters.put("zsbh", idNumValue);
            }
        }else {
            return new ArrayList<>();
        }

        sqlStr+=" and a.areacode like :areacode and a.state = '1' and a.active = 1 ";
        if(CredentialType.GradeCer.compareTo(request.getCredentialType()) == 0){
            sqlStr+=" and a.certificateType = 2 ";
        } else {
            sqlStr+=" and a.certificateType = 1 ";
        }
        parameters.put("areacode", request.getAreaCode()+"%");
        Query query = this.entityManager.createNativeQuery(sqlStr, TenantUserCredentialDetailsEntity.class);

        for (String key : parameters.keySet()) {
            query.setParameter(key, parameters.get(key));
        }

        ArrayList<QualificationCredentialResponse> credentialResponses = new ArrayList<>();
        List<TenantUserCredentialDetailsEntity> resultList = query.getResultList();
        for (TenantUserCredentialDetailsEntity tenantUserCredentialsEntity : resultList) {
            credentialResponses.add(QualificationCredentialResponse.create(tenantUserCredentialsEntity.getName()
                    ,tenantUserCredentialsEntity.getIdnum()
                    ,tenantUserCredentialsEntity.getOrgName()
                    ,tenantUserCredentialsEntity.getCerNo()
                    ,""
                    ,tenantUserCredentialsEntity.getCerDate()));
        }

        return credentialResponses;
    }

    @Override
    public List<TenantUserCredentialDetailsEntity> getQualificationCredential(String idnum) {
        TypedQuery<TenantUserCredentialDetailsEntity> query =
                this.entityManager.createQuery("select t from TenantUserCredentialDetailsEntity t where t.idnum = :idnum " +
                                " and (t.state='1' or t.state is null) and t.active = true ",
                        TenantUserCredentialDetailsEntity.class);
        query.setParameter("idnum", idnum);
        List<TenantUserCredentialDetailsEntity> credentialDetailsEntities = query.getResultList();
        if (credentialDetailsEntities.size() > 0) {
            return credentialDetailsEntities;
        }
        return null;
    }

    @Override
    public List<DepartDto> findDepartAdminsByEmployeeId(String employeeId) {
        TypedQuery<DepartDto> mapTypedQuery =
                this.entityManager.createQuery("select new com.bcxin.tenant.domain.readers.dtos.DepartDto(de.employee.id as employeeId, dp.id as departId, dp.name as departName) from DepartmentAdminEntity de join de.department dp where de.employee.id =:employeeId",
                        DepartDto.class);
        mapTypedQuery.setParameter("employeeId", employeeId);
        return mapTypedQuery.getResultList();
    }

    @Override
    public LoginUserDto getLoginUserByIdNum(String idNum) {
        try {
            TypedQuery<LoginUserDto> query =
                    this.entityManager.createQuery("select  new com.bcxin.tenant.domain.readers.dtos.LoginUserDto(t.id,t.name,t.telephone) from TenantUserEntity t " +
                            " where t.selectedCredential.number = ?1", LoginUserDto.class);
            query.setParameter(1, idNum);
            List<LoginUserDto> resultList = query.getResultList();
            if (resultList.size() > 0) {
                return resultList.get(0);
            }
            return null;
        } catch (Exception ex) {
            throw ex;
        }
    }

    @Override
    public boolean isUserByUserName(String userName) {
        IdentityUserpasswordEntity user = tenantUserRepository.getByUserName(userName);
        if(user == null){
            return false;
        }
        return true;
    }

    @Override
    public List<EmployeeEntity> getAllByIdNums(Collection<String> idNums, String areaCode) {
        String sql = "select e from EmployeeEntity e join e.tenantUser t join e.organization o " +
                " where t.selectedCredential.number in (:idNums) ";

        Map<String, Object> parameters = new HashMap<>();
        parameters.put("idNums", idNums);
        if (StrUtil.isNotEmpty(areaCode) && areaCode.length() >= 2) {
            sql += " and o.superviseRegionCode like concat(:areaCode,'%') ";
            parameters.put("areaCode", areaCode.substring(0, 2));
        }

        TypedQuery<EmployeeEntity> query = this.entityManager.createQuery(sql, EmployeeEntity.class);
        for (String key : parameters.keySet()) {
            query.setParameter(key, parameters.get(key));
        }

        List<EmployeeEntity> resultList = query.getResultList();
        return resultList;
    }

    @Override
    public List<EmployeeEntity> getAllByTelephones(Collection<String> telephones, String areaCode) {
        String sql = "select e from EmployeeEntity e join e.tenantUser t join e.organization o " +
                " where t.telephone in (:telephones) ";

        Map<String, Object> parameters = new HashMap<>();
        parameters.put("telephones", telephones);
        if (StrUtil.isNotEmpty(areaCode) && areaCode.length() >= 2) {
            sql += " and o.superviseRegionCode like concat(:areaCode,'%') ";
            parameters.put("areaCode", areaCode.substring(0, 2));
        }

        TypedQuery<EmployeeEntity> query = this.entityManager.createQuery(sql, EmployeeEntity.class);
        for (String key : parameters.keySet()) {
            query.setParameter(key, parameters.get(key));
        }

        List<EmployeeEntity> resultList = query.getResultList();
        return resultList;
    }

    @Override
    public List<EmployeeEntity> getSecurityGuardByIdNums(Collection<String> idNums, String areaCode) {
        String sql = "select e from EmployeeEntity e join e.tenantUser t join e.organization o " +
                " where t.selectedCredential.number in (:idNums) and e.occupationType=com.bcxin.Infrastructures.enums.OccupationType.SecurityGuard and e.status <> 1 ";

        Map<String, Object> parameters = new HashMap<>();
        parameters.put("idNums", idNums);
        if (StrUtil.isNotEmpty(areaCode) && areaCode.length() >= 2) {
            sql += " and o.superviseRegionCode like concat(:areaCode,'%') ";
            parameters.put("areaCode", areaCode.substring(0, 2));
        }

        TypedQuery<EmployeeEntity> query = this.entityManager.createQuery(sql, EmployeeEntity.class);
        for (String key : parameters.keySet()) {
            query.setParameter(key, parameters.get(key));
        }

        List<EmployeeEntity> resultList = query.getResultList();
        return resultList;
    }


    @Override
    public List<UserRegionDto> getUsersByRegion(TenantUserRegionCriteria criteria) {
        Query regionQuery = this.entityManager.createNativeQuery("select name,number,oneInchColorWhitePhoto  from  auth_users_view  where region like :region");

        regionQuery.setParameter("region", "%" + criteria.getRegion() + "%");

        List res = (List) regionQuery.unwrap(SQLQuery.class).setMaxResults(100).setResultTransformer(Transformers.ALIAS_TO_ENTITY_MAP).getResultList().stream().map(item -> {
            Map<String, String> map = BeanUtil.toBean(item, Map.class);
            return new UserRegionDto(map.get("NAME"), map.get("number"), map.get("oneInchColorWhitePhoto"));
        }).collect(Collectors.toList());

        return res;
    }

    @Override
    @Modifying
    @Transactional
    public void updateTenantUserRealNameRequest(List<UpdateTenantUserRealNameRequest> requests) {

        for (UpdateTenantUserRealNameRequest request : requests) {
            Query query = this.entityManager.createNativeQuery(String.format("UPDATE tenant_users u, tenant_user_credentials tuc set u.authenticated_status = %s, authenticated_result = '%s', authenticated_time = NOW()  where u.selected_credential_id = tuc.id  and tuc.number = '%s';", request.getStatus().ordinal(), request.getResult(), request.getNumber()));
            int i = query.executeUpdate();
        }
    }


    @Override
    public UserCredentialDto queryUserCredential(String id, Integer credentialType, String credentialNumber) {
        boolean b = true;
        String sqlStr = "SELECT u.`name`,u.authenticated_status AS authenticatedStatus,u.checked_status AS checkedStatus," +
                "u.telephone,(CAST(uc.credential_type as UNSIGNED)) AS credentialType ,uc.number,uc.reverse_photo AS reversePhoto,uc.front_photo AS frontPhoto,u.head_photo AS headPhoto," +
                "uc.number as idnum,(CAST(null as CHAR)) as fzjgmc,(cast(null as date)) as fzrq,uc.address AS address " +
                "FROM tenant_users u  " +
                "LEFT JOIN tenant_user_credentials uc ON u.id = uc.tenant_user_id   " +
                "WHERE uc.tenant_user_id = ?1 AND uc.credential_type = ?2 AND uc.number = ?3 and u.id = ?4";

        final Collection<CredentialType> excludeCredentialTypes =
                Stream.of(CredentialType.IdCard, CredentialType.IdCardOfXiangGang,
                                CredentialType.IdCardOfAoMen, CredentialType.IdCardOfTaiwan,
                                CredentialType.Passport,
                                CredentialType.Arms,
                                CredentialType.PoliceNo)
                        .collect(Collectors.toList());


        /**
         * 非身份证信息不额外返回其他数据
         */
        if (!excludeCredentialTypes.stream().anyMatch(ii -> ii.ordinal() == credentialType)) {
            sqlStr = "SELECT u.`name`,u.authenticated_status AS authenticatedStatus,u.checked_status AS checkedStatus," +
                    "u.telephone," +
                    "(CASE WHEN ucd.certificateType = '1' THEN 7 WHEN ucd.certificateType = '2' THEN 8 WHEN ucd.certificateType = '3' THEN 9 WHEN ucd.certificateType = '11' THEN 11 ELSE 0 END) as credentialType," +
                    "ucd.zsbh as number,'' AS reversePhoto,ucd.electronCerUrl AS frontPhoto,u.head_photo AS headPhoto," +
                    "ucd.idnum,ucd.fzjgmc,ucd.fzrq,ucp.address " +
                    "FROM tenant_users u  " +
                    "LEFT JOIN tenant_user_credentials ucp ON ucp.tenant_user_id = u.id AND ucp.credential_type = 0 " +
                    "LEFT JOIN tenant_user_credential_details ucd ON ucd.tenant_user_id = u.id " +
                    "WHERE ucd.tenant_user_id = ?1 AND ucd.certificatetype = ?2 AND ucd.zsbh = ?3 AND u.id = ?4";
            b = false;
        }
        String credentialDetailType = "";
        if (!b) {
            switch (credentialType) {
                case 7:
                    credentialDetailType = "1";
                    break;
                case 8:
                    credentialDetailType = "2";
                    break;
                case 9:
                    credentialDetailType = "3";
                    break;
                case 11:
                    credentialDetailType = "11";
                    break;
                default:
                    credentialDetailType = "";
            }
            if (StringUtils.isEmpty(credentialDetailType)) {
                throw new NotSupportTenantException("不支持该证件类型");
            }
        }
        Query nativeQuery = this.entityManager.createNativeQuery(sqlStr);
        nativeQuery.unwrap(SQLQuery.class).setResultTransformer(Transformers.aliasToBean(UserCredentialDto.class));
        nativeQuery.setParameter(1, id);
        if (b) {
            nativeQuery.setParameter(2, credentialType);
        } else {
            nativeQuery.setParameter(2, credentialDetailType);
        }
        nativeQuery.setParameter(3, credentialNumber);
        nativeQuery.setParameter(4, id);

        List<UserCredentialDto> list = nativeQuery.getResultList();
        return list != null && !list.isEmpty() ? list.get(0) : null;
    }

    @Override
    public Collection<String> queryCanBeDeletedEventSubscriberLogs(int pageSize) {
        Collection<String> handlers = Stream.of(
                EventAction.TenantUserAfterCreatedEventForSms.name()
        ).collect(Collectors.toList());

        TypedQuery<String> eventSubscriberActionLogQuery =
                this.entityManager.createQuery("select ev.id from EventSubscribedActionLogEntity ev where ev.subscriber.handler not in (:handlers) and ev.createdTime<:createdTime", String.class);
        Calendar nowCalendar = Calendar.getInstance();
        nowCalendar.add(Calendar.MONTH, -2);
        DateTime expiredDateTime = DateTime.of(nowCalendar);
        eventSubscriberActionLogQuery.setParameter("handlers", handlers);
        eventSubscriberActionLogQuery.setParameter("createdTime", expiredDateTime);

        return eventSubscriberActionLogQuery.setMaxResults(pageSize).getResultList();
    }

    @Override
    public EmployeeStatusDto getEmployeeIdByOrganIdAndIdNumAndType(String organId, CredentialType credentialType, String credentialNumber) {
        Query employeeIdQuery =
                this.entityManager.createNativeQuery(
                        "select ee.id,ee.status from tenant_employees ee join tenant_user_credentials cc on ee.tenant_user_id=cc.tenant_user_id\n" +
                                "where cc.number=?1 and cc.credential_type=?2 and ee.organization_id=?3");
        employeeIdQuery.unwrap(SQLQuery.class).setResultTransformer(Transformers.aliasToBean(EmployeeStatusDto.class));

        employeeIdQuery.setParameter(1, credentialNumber);
        employeeIdQuery.setParameter(2, credentialType.ordinal());
        employeeIdQuery.setParameter(3, organId);
        List<EmployeeStatusDto> result = employeeIdQuery.getResultList();
        if (CollectionUtils.isEmpty(result)) {
            return null;
        }

        return result.get(0);
    }

    @Override
    public EmployeeStatusDto getEmployeeIdByOrganIdAndTelephone(String organId, String telephone) {
        TypedQuery<EmployeeStatusDto> employeeIdQuery =
                this.entityManager.createQuery(
                        "select new com.bcxin.tenant.domain.readers.dtos.EmployeeStatusDto(ee.id,ee.status) from EmployeeEntity ee join ee.tenantUser u " +
                                " where ee.organization.id=:organId and u.telephone=:telephone",
                        EmployeeStatusDto.class);
        employeeIdQuery.setParameter("telephone", telephone);
        employeeIdQuery.setParameter("organId", organId);
        List<EmployeeStatusDto> result = employeeIdQuery.getResultList();
        if (CollectionUtils.isEmpty(result)) {
            return null;
        }

        return result.get(0);
    }

    /**
     * 根据用户获取在职的职员列表
     *
     * @param tenantUserId
     * @param areaCode
     * @return
     */
    @Override
    public List<BatchEmployeeValidationDto> getOnDutyEmployeeValidationDtoByIdAndAreaCode(String tenantUserId, String areaCode) {
        if (!StringUtils.hasLength(areaCode)) {
            throw new ArgumentTenantException("区域编码不能为空!");
        }
        if (areaCode.length() < 2) {
            throw new ArgumentTenantException(String.format("该组织区域(%s)编码无效!", areaCode));
        }

        String sql =
                "select new com.bcxin.tenant.domain.readers.dtos.BatchEmployeeValidationDto(" +
                        "e.id,o.id,o.name, e.occupationType,e.status,e.hiredDate,e.leaveTime) " +
                        "from EmployeeEntity e join e.organization o  " +
                        " where e.status=com.bcxin.Infrastructures.enums.EmploymentStatus.OnJob and e.tenantUser.id=:tenantUserId " +
                        " and o.superviseRegionCode like concat(:areaCode,'%')";
        Map<String, Object> parameters = new HashMap<>();
        parameters.put("tenantUserId", tenantUserId);
        parameters.put("areaCode", areaCode.substring(0, 2));

        TypedQuery<BatchEmployeeValidationDto> query =
                this.entityManager.createQuery(sql, BatchEmployeeValidationDto.class);
        for (String key : parameters.keySet()) {
            query.setParameter(key, parameters.get(key));
        }

        List<BatchEmployeeValidationDto> resultList = query.getResultList();

        return resultList;
    }

    @Override
    public Collection<EmployeeTenantUserIdNumDto> getTenantUserIdMapByIdNums(Collection<String> idNums) {
        String sql =
                "select new com.bcxin.tenant.domain.dto.EmployeeTenantUserIdNumDto(tc.tenantUser.id,tc.number) from TenantUserCredentialsEntity tc  " +
                        " where tc.credentialType=com.bcxin.Infrastructures.enums.CredentialType.IdCard and tc.number in (:idNums) " +
                        "";

        Map<String, Object> parameters = new HashMap<>();
        parameters.put("idNums", idNums);

        TypedQuery<EmployeeTenantUserIdNumDto> query =
                this.entityManager.createQuery(sql, EmployeeTenantUserIdNumDto.class);
        for (String key : parameters.keySet()) {
            query.setParameter(key, parameters.get(key));
        }

        return query.getResultList();
    }

    @Override
    public Collection<EmployeeOccupationTypeValidationBasicDto> getOnDutyEmployeeValidationDtoByIdsAndAreaCode(
            String areaCode,
            String organizationId,
            Collection<String> tenantUserIds) {
        if (CollectionUtils.isEmpty(tenantUserIds)) {
            return Collections.EMPTY_LIST;
        }

        String sql =
                "select new com.bcxin.tenant.domain.dto.EmployeeOccupationTypeValidationBasicDto(e.tenantUser.id,tc.credentialType,tc.number, e.occupationType,o.id,o.name, o.superviseRegionCode,e.status) " +
                        "from EmployeeEntity e join e.organization o " +
                        "join TenantUserCredentialsEntity tc on tc.tenantUser.id = e.tenantUser.id and tc.credentialType in (:credentialTypes)" +
                        " where e.status=com.bcxin.Infrastructures.enums.EmploymentStatus.OnJob and " +
                        " e.tenantUser.id in (:tenantUserIds)";
        Map<String, Object> parameters = new HashMap<>();
        parameters.put("tenantUserIds", tenantUserIds);
        Collection<CredentialType> credentialTypes =
                Stream.of(
                        CredentialType.IdCard, CredentialType.IdCardOfAoMen, CredentialType.IdCardOfXiangGang,
                        CredentialType.IdCardOfAoMen, CredentialType.IdCardOfTaiwan, CredentialType.Passport,
                        CredentialType.Arms, CredentialType.PoliceNo
                ).collect(Collectors.toList());
        parameters.put("credentialTypes", credentialTypes);

        if (StringUtils.hasLength(areaCode) && areaCode.length() > 2) {
            sql += " and o.superviseRegionCode like concat(:areaCode,'%')";
            parameters.put("areaCode", areaCode.substring(0, 2));
        }

        TypedQuery<EmployeeOccupationTypeValidationBasicDto> query =
                this.entityManager.createQuery(sql, EmployeeOccupationTypeValidationBasicDto.class);
        for (String key : parameters.keySet()) {
            query.setParameter(key, parameters.get(key));
        }

        List<EmployeeOccupationTypeValidationBasicDto> resultList = query.getResultList();

        return resultList;
    }

    @Override
    public String getTelephoneByCredentialNumber(CredentialType credentialType, String number) {
        String sql = "select tc.tenantUser.telephone " +
                "from TenantUserCredentialsEntity tc " +
                " where tc.credentialType=:credentialType " +
                " and tc.number=:number";
        Map<String, Object> parameters = new HashMap<>();
        parameters.put("credentialType", credentialType);
        parameters.put("number", number);

        TypedQuery<String> query =
                this.entityManager.createQuery(sql, String.class);
        for (String key : parameters.keySet()) {
            query.setParameter(key, parameters.get(key));
        }

        return query.getSingleResult();
    }

    @Override
    public Collection<TenantUserTelephoneCredentialDto> getTenantUserTelephoneCredentialsByNumberOrTel(Collection<String> numbers, Collection<String> telephones) {
        String sql = "select new com.bcxin.tenant.domain.dto.TenantUserTelephoneCredentialDto(u.id,u.telephone,tc.credentialType,tc.number) " +
                "from TenantUserEntity u join u.credentials tc" +
                " where tc.number in (:numbers) or u.telephone in (:telephones) ";
        Map<String, Object> parameters = new HashMap<>();
        parameters.put("numbers", numbers);
        parameters.put("telephones", telephones);

        TypedQuery<TenantUserTelephoneCredentialDto> query =
                this.entityManager.createQuery(sql, TenantUserTelephoneCredentialDto.class);
        for (String key : parameters.keySet()) {
            query.setParameter(key, parameters.get(key));
        }

        Collection<TenantUserTelephoneCredentialDto> result = query.getResultList();

        return result;
    }

    @Override
    public Pageable<EmployeeRecordDto> searchEmployeeRecords(EmployeeRecordCriteria criteria) {
        String sql = "select new com.bcxin.tenant.domain.readers.dtos.EmployeeRecordDto(" +
                " ee.tenantUserId,u.name,u.credentialType,u.number,ee.occupationType," +
                " ee.actionTime,ee.status,ee.actionNote,ee.operator.name,ee.operator.createdTime) " +
                "from TenantUserCredentialsEntity u join EmployeeRecordEntity ee on u.tenantUser.id=ee.tenantUserId and u.selected=com.bcxin.Infrastructures.enums.TrueFalseStatus.True " +
                " where ee.employeeId =:employeeId and ee.organizationId=:organizationId order by ee.id desc ";

        String countSql = "select count(ee) " +
                "from TenantUserCredentialsEntity u join EmployeeRecordEntity ee on u.tenantUser.id=ee.tenantUserId and u.selected=com.bcxin.Infrastructures.enums.TrueFalseStatus.True " +
                " where ee.employeeId =:employeeId and ee.organizationId=:organizationId  ";

        Map<String, Object> parameters = new HashMap<>();
        parameters.put("employeeId", criteria.getEmployeeId());
        parameters.put("organizationId", criteria.getOrganizationId());

        TypedQuery<EmployeeRecordDto> query =
                this.entityManager.createQuery(sql, EmployeeRecordDto.class);

        long totalCount = 0;
        TypedQuery<Long> countQuery =
                this.entityManager.createQuery(countSql, Long.class);
        for (String key : parameters.keySet()) {
            query.setParameter(key, parameters.get(key));
            countQuery.setParameter(key, parameters.get(key));
        }

        totalCount = countQuery.getSingleResult();

        Collection<EmployeeRecordDto> result = new ArrayList<>();

        if (totalCount > 0) {
            result = query
                    .setFirstResult(criteria.getSkip())
                    .setMaxResults(criteria.getPageSize())
                    .getResultList();
        }

        return Pageable.create(criteria.getPageIndex(), criteria.getPageSize(), (int) totalCount, result);
    }

    /**
     * description：判断是否激活分销商
     * author：linchunpeng
     * date：2023/10/31
     */
    @Override
    public boolean isOrganizationPromoter(String organizationId) {
        Query isOrganizationPromoterQuery = this.entityManager.createNativeQuery(
                "select count(1) from shoppingrules.tlk_activate t where t.ITEM_COMPANY_ID = ?1 and ((t.ITEM_START_TIME <= CURDATE() and t.ITEM_END_TIME >= CURDATE()) or t.ITEM_END_TIME is NULL) ");
        isOrganizationPromoterQuery.setParameter(1, organizationId);

        Object result = isOrganizationPromoterQuery.getSingleResult();

        Long count = Long.parseLong(String.valueOf(result));
        return count > 0;
    }


    /**
     * description：查询操作日志
     * author：linchunpeng
     * date：2023/12/15
     */
    @Override
    public Pageable<OperateLogResponse> getOperateLogList(SearchOperateLogRequest request, String orgId) {
        Map<String, Object> parameters = new HashMap<>();
        //查询sql
        StringBuffer sql = new StringBuffer(1024);
        sql.append("select id, tenant_user_id, user_name, real_name, operate_type, operate_time, operate_content, operate_result, ip_address from t_operate_log tol where 1=1 ");
        //计数sql
        StringBuffer sqlCount = new StringBuffer(1024);
        sqlCount.append("select count(1) as total from t_operate_log tol where 1=1 ");

        if (StringUtils.hasLength(orgId)) {
            sql.append(" and tol.tenant_user_id in (select te.tenant_user_id from tenant_employees te where te.status = 0 and te.organization_id = :orgId) ");
            sqlCount.append(" and tol.tenant_user_id in (select te.tenant_user_id from tenant_employees te where te.status = 0 and te.organization_id = :orgId) ");
            parameters.put("orgId", orgId);
        }

        if (StringUtils.hasLength(request.getTenantUserId())) {
            sql.append(" and tol.tenant_user_id = :tenantUserId");
            sqlCount.append(" and tol.tenant_user_id = :tenantUserId");
            parameters.put("tenantUserId", request.getTenantUserId());
        }

        if (StringUtils.hasLength(request.getRealName())) {
            sql.append(" and tol.real_name like :realName");
            sqlCount.append(" and tol.real_name like :realName");
            parameters.put("realName", "%" + request.getRealName() + "%");
        }

        if (StringUtils.hasLength(request.getUserName())) {
            sql.append(" and tol.user_name like :userName");
            sqlCount.append(" and tol.user_name like :userName");
            parameters.put("userName", "%" + request.getUserName() + "%");
        }

        if (request.getOperateType() != null) {
            sql.append(" and tol.operate_type = :operateType");
            sqlCount.append(" and tol.operate_type = :operateType");
            parameters.put("operateType", request.getOperateType());
        }

        if (StringUtils.hasLength(request.getOperateTimeStart())) {
            sql.append(" and tol.operate_time >= :operateTimeStart");
            sqlCount.append(" and tol.operate_time >= :operateTimeStart");
            parameters.put("operateTimeStart", request.getOperateTimeStart().concat(" 00:00:00"));
        }

        if (StringUtils.hasLength(request.getOperateTimeEnd())) {
            sql.append(" and tol.operate_time <= :operateTimeEnd");
            sqlCount.append(" and tol.operate_time <= :operateTimeEnd");
            parameters.put("operateTimeEnd", request.getOperateTimeEnd().concat(" 23:59:59"));
        }
        sql.append(" order by tol.operate_time desc ");
        Query query = this.entityManager.createNativeQuery(sql.toString(), OperateLogEntity.class);
        Query countQuery = this.entityManager.createNativeQuery(sqlCount.toString());
        //设置参数
        for (String key : parameters.keySet()) {
            query.setParameter(key, parameters.get(key));
            countQuery.setParameter(key, parameters.get(key));
        }
        //设置分页
        query.setFirstResult(request.getSkip()).setMaxResults(request.getPageSize());

        List<OperateLogResponse> responseList = new ArrayList<>();
        List<OperateLogEntity> resultList = query.getResultList();

        //总数
        int total = Integer.valueOf(countQuery.getResultList().get(0).toString());
        //转为DTO
        for (OperateLogEntity operateLogEntity : resultList) {
            OperateLogResponse response = new OperateLogResponse();
            BeanUtils.copyProperties(operateLogEntity, response);
            response.setOperateTimeStr(DateUtil.formatDateTime(operateLogEntity.getOperateTime()));
            responseList.add(response);
        }
        return Pageable.create(request.getPageIndex(), request.getPageSize(), total, responseList);
    }

    @Override
    public Collection<OrganizationAdminiInfoDto> getOrganizationAdminInfo(String organizationId) {
        String sql = "select new com.bcxin.tenant.domain.repositories.dtos.OrganizationAdminiInfoDto(e.id,u.id,u.name,u.telephone)" +
                " from EmployeeEntity e join e.tenantUser as u" +
                " where e.organization.id = :organizationId and e.status = 0 and e.domainAdmin = 1";
        TypedQuery<OrganizationAdminiInfoDto> query = this.entityManager.createQuery(sql, OrganizationAdminiInfoDto.class);
        query.setParameter("organizationId", organizationId);
        return query.getResultList();
    }

    @Override
    public Collection<UserAppealDto> getUserAppeals(UserAppealsCriteria criteria) {
        String sql = "select new com.bcxin.tenant.domain.readers.dtos.UserAppealDto(" +
                " a.createdTime,a.status,a.result,a.lastApproverDepartName,a.lastApproverName,a.lastApproverTime)" +
                " from TenantUserAppealsEntity a where a.tenantUser.id =:userId order by a.createdTime ";
        TypedQuery<UserAppealDto> query = this.entityManager.createQuery(sql, UserAppealDto.class);
        query.setParameter("userId", criteria.getUserId());
        Collection<UserAppealDto> result = query.setFirstResult(criteria.getSkip())
                .setMaxResults(criteria.getPageSize()).getResultList();
        return result;
    }

    @Override
    public Collection<ExternalMemberDTO> search(ExternalMemberSearchCriteria criteria) {
        String sql = "select new com.bcxin.tenant.domain.repositories.dtos.ExternalMemberDTO(" +
                "e.id,e.inviteType,e.inviteCode,u.id,u.name,u.telephone,c.credentialType,c.number," +
                "e.approvedInformationValueType.status,e.approvedInformationValueType.note,e.inviteGroupId," +
                "e.inviteGroupName,e.createdTime,e.memberType ) " +
                " from ExternalMemberEntity e join e.tenantUser u left join u.selectedCredential c where e.referenceType=:referenceType and e.referenceNumber=:referenceNumber ";

        Map<String, Object> parameters = new HashMap<>();
        parameters.put("referenceType", ResourceReferenceType.Organization);
        parameters.put("referenceNumber", criteria.getOrganizationId());

        Collection<ApprovedStatus> statuses = criteria.getStatuses();
        if (CollectionUtils.isEmpty(criteria.getStatuses())) {
            statuses = Stream.of(ApprovedStatus.Passed, ApprovedStatus.Init, ApprovedStatus.NoPassed).collect(Collectors.toList());
        }

        /**
         * 默认情况下; 不查询已被删除的数据
         */
        sql += " and e.approvedInformationValueType.status in :statuses";
        parameters.put("statuses", statuses.stream().collect(Collectors.toList()));


        if (!CollectionUtils.isEmpty(criteria.getGroupIds())) {
            if (criteria.getGroupIds().stream().anyMatch(ii -> ii == null)) {
                sql += " and JSON_LENGTH(e.groupIdsJson) = 0";
            } else {
                List<String> groupIds = criteria.getGroupIds().stream().collect(Collectors.toList());
                /*
                StringBuilder inValue = new StringBuilder();


                for (int index = 0; index < groupIds.size(); index++) {
                    String ii = groupIds.get(index);
                    inValue.append(String.format("\"%s\"", ii.replace("'", "\"").replace("`", "")));
                    if (index < groupIds.size() - 1) {
                        inValue.append(",");
                    }
                }

                sql += String.format(" and JSON_CONTAINS(e.groupIdsJson, '[%s]')", inValue);

                 */
                sql += " and e.groupIdsJson like :groupIds";
                parameters.put("groupIds", "%".concat(groupIds.get(0).concat("%")));
                // parameters.put("groupIds", criteria.getGroupIds());
            }
        }

        if (StringUtils.hasLength(criteria.getKeyword())) {
            sql += " and u.name like :name";
            parameters.put("name", criteria.getKeyword() + "%");
        }

        if (statuses.contains(ApprovedStatus.Init) && statuses.size() == 1) {
            sql += " order by e.createdTime desc ";
        } else {
            sql += " order by e.approvedInformationValueType.status asc, e.createdTime desc";
        }

        TypedQuery<ExternalMemberDTO> query = this.entityManager.createQuery(sql, ExternalMemberDTO.class);
        for (String key : parameters.keySet()) {
            query.setParameter(key, parameters.get(key));
        }

        Collection<ExternalMemberDTO> result = query.setFirstResult(criteria.getSkip())
                .setMaxResults(criteria.getPageSize()).getResultList();

        Collection<String> memberIds =
                result.stream().map(ii -> ii.getId())
                        .distinct().collect(Collectors.toList());

        if (!CollectionUtils.isEmpty(memberIds)) {
            String groupSql = "select new com.bcxin.tenant.domain.readers.dtos.ExternalGroupDto(g.id,g.principal.id) from ExternalGroupEntity g where g.principal.id in (:principalIds)";
            TypedQuery<ExternalGroupDto> groupGroupQuery =
                    this.entityManager.createQuery(groupSql, ExternalGroupDto.class);
            groupGroupQuery.setParameter("principalIds", memberIds);

            Collection<ExternalGroupDto> groups = groupGroupQuery.getResultList();

            for(ExternalMemberDTO em:result) {
                Collection<String> matchedGroupIds =
                        groups.stream().filter(ii -> em.getId().equalsIgnoreCase(ii.getPrincipalId()))
                                .map(ii -> ii.getId()).distinct().collect(Collectors.toList());
                em.addPrincipalGroupIds(matchedGroupIds);
            }
        }

        return result;
    }



    @Override
    public Collection<MyExternalMemberRecordDTO> getMyExternalMemberRecords(MyExternalMemberRecordCriteria criteria) {
        String sql = "select new com.bcxin.tenant.domain.readers.dtos.MyExternalMemberRecordDTO(e.id,o.id,o.name,e.approvedInformationValueType.status,e.approvedInformationValueType.note,e.approvedInformationValueType.lastUpdatedTime,e.createdTime) " +
                " from ExternalMemberEntity e join OrganizationEntity o on e.referenceType=com.bcxin.Infrastructures.enums.ResourceReferenceType.Organization" +
                " and e.referenceNumber=o.id where e.tenantUser.id=:userId";

        Map<String, Object> parameters = new HashMap<>();
        parameters.put("userId", criteria.getTenantUserId());

        if(StringUtils.hasLength(criteria.getKeyword())) {
            sql += " and o.name like :keyword";
            parameters.put("keyword", "%".concat(criteria.getKeyword().concat("%")));
        }

        TypedQuery<MyExternalMemberRecordDTO> query = this.entityManager.createQuery(sql, MyExternalMemberRecordDTO.class);
        for (String key : parameters.keySet()) {
            query.setParameter(key, parameters.get(key));
        }

        Collection<MyExternalMemberRecordDTO> result = query.setFirstResult(criteria.getSkip())
                .setMaxResults(criteria.getPageSize()).getResultList();

        return result;
    }

    @Override
    public Collection<MyTeamDto> getMyTeamsByUserId(String userId) {
        String sql =
                "select new com.bcxin.tenant.domain.readers.dtos.MyTeamDto(" +
                        "oo.id,oo.name,oo.industryCode,oo.institutionalCode,oo.superviseRegionCode,ee.createdTime, oo.organizationLevel," +
                        "ee.tenantUser.id, ee.id,ee.memberType" +
                        ") from ExternalMemberEntity ee join OrganizationEntity oo " +
                        " on ee.referenceType = com.bcxin.Infrastructures.enums.ResourceReferenceType.Organization and " +
                        " ee.referenceNumber=oo.id where oo.approvedInformationValueType.status in (1) and ee.approvedInformationValueType.status in (1) and ee.tenantUser.id=:userId " ;

        TypedQuery<MyTeamDto> query = this.entityManager.createQuery(sql, MyTeamDto.class);

        query.setParameter("userId", userId);
        Collection<MyTeamDto> result = query.getResultList();

        return result;
    }

    @Override
    public Collection<MyRegistrationOrganizationDto> getMyRegistrationsByUserId(String userId,int pageIndex,int pageSize) {
        String sql =
                "select new com.bcxin.tenant.domain.readers.dtos.MyRegistrationOrganizationDto(" +
                        "oo.id,oo.name,oo.approvedInformationValueType.status," +
                        "oo.approvedInformationValueType.lastUpdatedTime,oo.approvedInformationValueType.note," +
                        "oo.institutionalCode,oo.industryCode,oo.createdTime" +
                        ") from OrganizationEntity oo where oo.tenantUser.id=:userId";
        sql += " order by oo.createdTime desc ";

        TypedQuery<MyRegistrationOrganizationDto> query = this.entityManager.createQuery(sql, MyRegistrationOrganizationDto.class);
        query.setParameter("userId", userId);

        if (pageIndex < 1) {
            pageIndex = 1;
        }

        if (pageSize <= 0) {
            pageSize = 10;
        }

        int skip = (pageIndex - 1) * pageSize;

        query = query.setFirstResult(skip)
                .setMaxResults(pageSize);

        return query.getResultList();
    }
}
